// ellipfilt.c

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <math.h>
#include "application.h"
#include "localdefs.h"
#include "ellipfilt.h"
#include "query.h"
#include "request.h"

extern "C" {
int setell_(float*, float*, float*, float*, float*, float*, float*, long*);
}

EllipticalFilter::EllipticalFilter(Data* output, double pband, double stopband,
	double bpstopband, double ripple, double atten, double amp)
		: SimpleFunction(output), array(new float[461]), gain(amp) {
	float sr=sampRate(), p=pband, s=stopband, b=bpstopband, r=ripple, a=atten;
	setell_(&sr, &p, &s, &b, &r, &a, array, &numberOfSections);
	initialize();
}

EllipticalFilter::EllipticalFilter(Data* output)
	: SimpleFunction(output), array(new float[461]) {}

EllipticalFilter::~EllipticalFilter() {
	delete [] array;
}

const QueryInfo *
EllipticalFilter::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Apply Elliptical Filter to Selected Region:" },
		{ nil }
	};	
	static QueryValueInfo valueInfo[] = {
		{ "Passband Cutoff (Hz.):", "1000.0", CharCheck::posNumsOnly },
		{ "StopBand Cutoff (Hz.):", "2000.0", CharCheck::posNumsOnly },
		{ "BandPass StopBand (Hz.):", "0", CharCheck::posNumsOnly },
		{ "PassBand Ripple (db.):", "0.5", nil, "0.1|12.0" },
		{ "StopBand Attenuation (db.):", "60.0", nil, "0.0|90.0" },
		{ "Gain Factor:", "1.00", CharCheck::numsOnly },
		{ nil }
	};	
	static QueryInfo info[] = {
		{ labelInfo, "", valueInfo },
		{ nil }
	};
	return info;
}

boolean
EllipticalFilter::setValues(Request& request) {
	boolean status = true;
	const int nvals = 6;
	QueryValue vr[nvals];
	request.retrieveValues(vr, nvals);
	double passband = vr[0];
	double stopband = vr[1];
	double bpstopband = vr[2];
	double ripple = vr[3];
	double atten = vr[4];
	// cutoff freq check
	if(passband > sampRate()/2.0 || stopband > sampRate()/2.0) {
		Application::alert("Pass and stop bands must be between 0 and the Nyquist.");
		return false;
	}
	if(bpstopband != 0.0 && passband >= stopband) {
		Application::alert("For bandpass mode, stop band must be > pass band.");
		return false;
	}
	gain = vr[5];
	float sr=sampRate(), p=passband, s=stopband, b=bpstopband, r=ripple, a=atten;
	setell_(&sr, &p, &s, &b, &r, &a, array, &numberOfSections);
	return status;
}

void
EllipticalFilter::initialize() {
	SimpleFunction::initialize();
}

void
EllipticalFilter::restoreState() {
	int i=0;
	for(int m=0; m < numberOfSections; m++) {
		for(int j=0;j<4;j++)  {    
			coeff[m][j] = array[i++];
			past[m][j] = 0;
		}
	}
	normalizer = array[i] * gain;
/*	char msg[64];
	sprintf(msg, "Number of filter sections: %d\n", numberOfSections);
	Application::inform(msg, true);
	sprintf(msg, "Nrmlztn. factor: %f; No. of terms: %d\n", array[i], i);
	Application::inform(msg, true);
*/
}

double
EllipticalFilter::operator () (double input) {
	register double val = input;
	for(int m=0; m<numberOfSections; m++) {
		double* p = &past[m][0];
		float* c = &coeff[m][0];
		double op = val + *c * *p + *(c+2) * *(p+1)
		       - *(c+1) * *(p+2) - *(c+3) * *(p+3);
		*(p+1) = *p;
		*p = val;
		*(p+3) = *(p+2);
		*(p+2) = op;
		val = op;
	}
	return(val*normalizer);
}
