// editor.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/button.h>
#include <InterViews/world.h>
#include <X11/keysym.h>
#include "localdefs.h"
#include <sys/time.h>
#include <assert.h>
#include "datafile.h"
#include "application.h"
#include "header.h"
#include "editor.h"
#include "comment.h"
#include "controller.h"
#include "data.h"
#include "delay.h"
#include "filename.h"
#include "filecommand.h"
#include "sound.h"
#include "lpcdata.h"
#include "formantfilter.h"
#include "pchdata.h"
#include "fftdata.h"
#include "envelope.h"
#include "datamodifier.h"
#include "analysisfun.h"
#include "optionsetter.h"
#include "phraser.h"
#include "ellipfilt.h"
#include "fftfun.h"
#include "rescaler.h"
#include "interpolater.h"
#include "lowpassfilter.h"
#include "pulsegen.h"
#include "statusaction.h"
#include "textwindow.h"
#include "pvoceditor.h"
#include "soundeditor.h"
#include "curvegen.h"
#include "crossfader.h"
#include "commenteditor.h"

const char *
scat(const char *s1, const char *s2) {
	static char str[1024];
	sprintf(str, "%s%s", s1, s2);
	return str;
}

const char *
scat(const char *s1, const char *s2, const char *s3) {
	static char str[1024];
	sprintf(str, "%s%s%s", s1, s2, s3);
	return str;
}

Data *
DataEditor::internalBuffer = nil;

DataEditor::C_Constructor 
DataEditor::ctor_table1[] = {
	nil,
	SoundEditor::new_DataEditor1,
	LPCEditor::new_DataEditor1,
	PCHEditor::new_DataEditor1,
	FFTEditor::new_DataEditor1,
	EnvelopeEditor::new_DataEditor1,
	PvocEditor::new_DataEditor1
};

DataEditor::C_D_Constructor 
DataEditor::ctor_table2[] = {
	nil,
	SoundEditor::new_DataEditor2,
	LPCEditor::new_DataEditor2,
	PCHEditor::new_DataEditor2,
	FFTEditor::new_DataEditor2,
	EnvelopeEditor::new_DataEditor2,
	PvocEditor::new_DataEditor2
};

DataEditor::DataEditor(Controller *c, const Data *d)
		: controller(c), data(nil), selection(nil) {
	init((Data *) d);
}

DataEditor *
DataEditor::create(Controller *c, const char* name) {
	int type_index = int(FileName::fileType(name));
	static const int tableEnd = sizeof(ctor_table1) / sizeof(*ctor_table1);
	assert(0 < type_index && type_index < tableEnd);
	return (*DataEditor::ctor_table1[type_index])(c);
}

DataEditor *
DataEditor::create(Controller *c, const Data *d) {
	int type_index = int(d->fileType());
	static const int tableEnd = sizeof(ctor_table2) / sizeof(*ctor_table2);
	assert(0 < type_index && type_index < tableEnd);
	return (*DataEditor::ctor_table2[type_index])(c, d);
}

void
DataEditor::init(Data* d) {
	reset();
	if(d)
		setData(d);
}

DataEditor::~DataEditor() {
	setCopyBuffer(nil);
	setSelection(nil);
	setData(nil);
}

void
DataEditor::setData(Data* d) {
	if(d != data) {
		Resource::unref(data);
		data = d;
		if(data != nil) {
			data->ref();
			location = data->frameRange();
			channels = data->channelRange();
		}
	}
}

Data *
DataEditor::createModel(DataFile *f) {
	boolean status = false;
	Data *d = newModel();
	if(f->readable()) {
		Application::inform(scat("Reading ", f->name(), "..."));
		if(!d->read(f)) {
			if(d->length() > 0 &&		// until Exceptions enabled
					Application::globalResourceIsTrue("ReadRawFiles")) {
				Header* header = nil;
				if((header = d->queryForHeader(controller)) != nil) {
					f->clear();
					status = d->read(f, header);
					Resource::unref(header);
				}
			}
		}
		else
			status = true;
	}
	else
		Application::error(scat("Unable to read file ", f->name()));
	f->failif(status != true);		// calling routine checks file state
	return d;
}

const char *
DataEditor::defaultDir() {
	const char* sfd = getenv("SFDIR");
	const char *a = 
		Application::getGlobalResource(data->defaultDirAttribute());
	return a ? a : sfd ? sfd : ".";
}

boolean
DataEditor::readFile(DataFile* file) {
	boolean status = true;
	file->ref();
	setData(createModel(file));
	status = file->good();
	Resource::unref(file);
	return status;
}

void
DataEditor::setInsertPoint(int point, int chan) {
	if(selectionMade() || chan < 0) // for first call after setting
		reset();
	setLocation(Range(point, max(point, data->length() - 1)));
	addChannel(chan);
	setSelection(nil);
}

void
DataEditor::setEditRegion(const Range &region, int chan) {
	if(!selectionMade()) {	// for first call after unsetting
		reset();
		isSelected = true;
	}
	setLocation(region);
	addChannel(chan);
	setSelection(nil);
	// when new region set in any window, it becomes default target rather
	// than any previously copied internal buffer.  Not so for insert point.
	setCopyBuffer(nil);
}

void
DataEditor::setLocation(const Range& newloc) {
	setTimeStamp();
	if(newloc != location)
		location = newloc;
}

void
DataEditor::addChannel(int chan) {
	if(chan >= 0) {
		int first = channels.intMin();
		int last = channels.intMax();
		int nfirst = (first < 0 || chan < first) ? chan : first;
		setChannels(nfirst, chan > last ? chan : last);
	}
}

Range
DataEditor::currentChannels() {
	// for now, either just one, or all channels
	// set to all if none chosen (-1 is flag for this)
	if(nchans() > 1 || channels.includes(-1))
		setChannels(model()->channelRange());
	return channels;
}

void
DataEditor::setTimeStamp() {
	struct timeval tp;
	struct timezone tzp;
	gettimeofday(&tp,&tzp);
	timestamp = tp.tv_sec;
}

void
DataEditor::setSelection(Data *s) {
	if(selection != s) {
		Resource::unref(selection);
		if((selection = s) != nil)
			selection->ref();
	}
}

void
DataEditor::setCopyBuffer(Data *b) {
	if(internalBuffer != b) {
		Resource::unref(internalBuffer);
		if((internalBuffer = b) != nil)
			internalBuffer->ref();
	}
}

Data *
DataEditor::currentSelection() {
	if(selection == nil)
		setSelection(data->clone(currentRegion(), currentChannels()));
	return selection;
}

Data *
DataEditor::currentSource() {
	Data *source = (copyBuffer() != nil) ?
		copyBuffer() : controller->findSelection();
	if(source == nil)
		Application::alert(
			"You must select a source to use this operation", 
			"or copy a selection into the internal buffer."
		);
	else if(*source == *data) {
		// must make deep copy if source is same as target
		Data *copyOfSource = source->copyOf();
		Resource::unref(source);
		source = copyOfSource;
		source->ref();
	}
	else
		source->ref();
	return source;
}

void
DataEditor::freeSource(Data *sel) {
	if(sel != internalBuffer)	// only unref source if it was not internal
		Resource::unref(sel);
}

Data *
DataEditor::getSelection(long &timestamp) {
	timestamp = timeStamp();
	Data *sel = nil;
	if (selectionMade())
		sel = data->clone(currentRegion(), currentChannels());
	return sel;
}

// re-select edit region (after reverting to saved, for instance)
// unless that region is no longer within the data bounds

void
DataEditor::reselect() {
	if(selectionMade() && data->frameRange().includes(currentRegion()))
			setSelection(data->clone(currentRegion(), currentChannels()));
	else {
		controller->unselectView();	// do to reset even if no selection made
		setSelection(nil);
	}
}

// static method -- used by "static constructor" methods

boolean
DataEditor::applyModifierUsing(Modifier &mod, Controller* controller) {
	boolean status = false;
	if(mod.configure(controller))
		status = mod.apply();
	return status;
}

boolean
DataEditor::applyModifier(Modifier &mod) {
	return applyModifierUsing(mod, controller);
}

boolean
DataEditor::applyModifierToNew(Modifier &mod) {
	boolean status;
	if((status = applyModifier(mod)) == true)
		displayInternalBuffer();
	setCopyBuffer(nil);
	return status;
}

void
DataEditor::markEditRegion(int size) {
	controller->showEditRegion(
		Range(currentInsert(), currentInsert() + size - 1),
		currentChannels()
	);
}

// keysym is checked first by controller, then by view, then by editor
// subclass, and lastly by editor base class.  Checking stops when matched.

boolean
DataEditor::keyCommand(unsigned long sym) {
	boolean interested = true;
	switch (sym) {
	case XK_period:
		setDefaultDir();
		break;
	case XK_n:
		newFile();
		break;
	case XK_nobreakspace:
		newLPCFile();
		break;
	case XK_exclamdown:
		newSoundFile();
		break;
	case XK_cent:
		newEnvelopeFile();
		break;
	case XK_currency:
		newPvocFile();
		break;
	case XK_o:
		openNewFile();
		break;
	case XK_s:
		saveFile();
		break;
	case XK_S:
		saveToFile();
		break;
	case XK_U:
		revertToSaved();
		break;
	case XK_quotedbl:
		changeComment();
		break;
	case XK_question:
		information();
		break;
	case XK_braceright:
		dataDump();
		break;
	case XK_I:
		displayInternalBuffer();
		break;
	case XK_c:
		copy();
		break;
	case XK_C:
		copyToNew();
		break;
	case XK_r:
		remove();
		break;
	case XK_y:
		removeToNew();
		break;
	case XK_e:
		erase();
		break;
	case XK_x:
		spliceOut();
		break;
	case XK_X:
		spliceOutToNew();
		break;
	case XK_d:
		zap();
		break;
	case XK_m:
		mix();
		break;
	case XK_R:
		replace();
		break;
	case XK_g:
		crossfade();
		break;
	case XK_v:
		spliceIn();
		break;
	case XK_P:
		scaleValues();
		break;
	case XK_b:
		reverse();
		break;
	case XK_i:
		insertSpace();
		break;
	case XK_E:
		applyEnvelope();
		break;
	case XK_O:
		vertOffset();
		break;
	case XK_bracketright:
		rescale();
		break;
	case XK_D:
		delay();
		break;
	case XK_t:
		interpolate();
		break;
	case XK_L:
		lowPassFilter();
		break;
	case XK_bracketleft:
		normalize();
		break;
	case XK_l:
		changeLength();
		break;
	case XK_0:
		findZeroCrossing();
		break;
	case XK_1:
		showPeakFrame();
		break;
	case XK_2:
		extractEnvelope();
		break;
	case XK_degree:
		setDataOptions();
		break;
	default:
		interested = false;
		break;
	}
	return interested;
}

Controller*
DataEditor::openNewFile(const char* windowtitle) {
	FileOpener fo(defaultDir(), windowtitle);
	applyModifier(fo);
	return fo.getNewController();
}

// the following are the menu-called functions

void
DataEditor::openNewFile() {
	openNewFile("Select File to Open:");
}

void
DataEditor::newLPCFile() {
	LPCEditor::new_File(controller);
}

void
DataEditor::newSoundFile() {
	SoundEditor::new_File(controller);
}

void
DataEditor::newEnvelopeFile() {
	EnvelopeEditor::new_File(controller);
}

void
DataEditor::newPvocFile() {
	PvocEditor::new_File(controller);
}

Response
DataEditor::closeFile() {
	Response r = No;
	// warn if file modified and if this is the last open view of it
	if(data->modified() && data->currentViews() == 1) {
		r = Application::choice("Do you wish to save changes to",
			controller->fileName(), "before closing?", Cancel);
		switch(r) {
		case Yes:
			saveFile();
			if(data->modified())
				r = Cancel;		// assumes cancel or error during save
			break;
		case No:
		case Cancel:
			break;
  		default:
			break;
		}
	}
	return r;
}
	
void
DataEditor::saveFile() {
	FileSaver fs(controller->fileName(), defaultDir(), model());
	applyModifier(fs);
}

void
DataEditor::saveToFile() {
	FileSaver fs(defaultDir(), model());
	applyModifier(fs);
}

void
DataEditor::revertToSaved() {
	const char* filename = controller->fileName();
	if(!data->modified())
		Application::alert("File has not been edited since last save.");
	else if(FileName::isTempName(filename))
		Application::alert("This is a temporary file and does not",
			"exist yet on disk.  You must save it first.");
	else if(Application::confirm(
			"Are you sure you wish to revert to the saved version?")) {
		DataFile f(filename, "r");
		if(f.readable()) {
			Application::inform("Re-reading file from disk...");
			data->read(&f);
		}
		controller->resetScaleTimes();
		reselect();
	}
}

void
DataEditor::displayInternalBuffer() {
	if(copyBuffer() != nil) {
		if(copyBuffer()->length() > 3) {
			Controller *newCtlr = new Controller(copyBuffer());
			newCtlr->display(controller->world());
			copyBuffer()->Notify();			// to assure rescan of amps
		}
		else
			Application::alert("Cannot display data with length < 4 frames.");
		// once displayed, not available internally to avoid confusion
		setCopyBuffer(nil);
	}
	else Application::alert("Internal buffer is currently empty.");
}

void
DataEditor::changeComment() {
	TextWindow* commentWindow = new TextWindow(
		scat(controller->fileName(), ": Comment"),
		new CommentEditor(new ButtonState, model())
	); 
	commentWindow->display(controller->world());
}

void
DataEditor::dataDump() {
	DataDumper dumper(defaultDir(), currentSelection());
	applyModifier(dumper);
}

void
DataEditor::information() {
	data->information(controller);
}

// previous internal buffer is kept until actual copy of new buffer takes place

void
DataEditor::copy() {
	Application::inform("Copying...");
	boolean was = Data::deferRescan(true);
	setCopyBuffer(currentSelection()->copyOf());
	Data::deferRescan(was);
}

void
DataEditor::copyToNew() {
	copy();
	displayInternalBuffer();
}

void
DataEditor::remove() {
	copy();
	erase();
}

void
DataEditor::removeToNew() {
	remove();
	displayInternalBuffer();
}

void
DataEditor::erase() {
	currentSelection()->erase();
}

void
DataEditor::spliceOut() {
	copy();
	zap();
}

void
DataEditor::spliceOutToNew() {
	spliceOut();
	displayInternalBuffer();
}

void
DataEditor::zap() {
	if(selectionMade()
		|| Application::confirm("Splice out everything from this point",
			"to the end of the file?")) {
		OutSplicer splicer(model(), currentRegion());
		if(applyModifier(splicer))
			controller->showInsertPoint(currentInsert(), currentChannels());
	}
}

void
DataEditor::mix() {
	// for now this mixes at unity gain
	Data* src = nil;
	if((src = currentSource()) != nil) {
		Mixer mixer(currentSelection(), src);
		if(applyModifier(mixer))
			markEditRegion(min(currentSelection()->length(), src->length()));
	}
	freeSource(src);
}

void
DataEditor::replace() {
	Data* src = nil;
	if((src = currentSource()) != nil) {
		Replacer replacer(currentSelection(), src);
		if(applyModifier(replacer))
			markEditRegion(min(currentSelection()->length(), src->length()));
	}
	freeSource(src);
}

void
DataEditor::crossfade() {
	Data* src = nil;
	if((src = currentSource()) != nil) {
		Crossfader crossfader(currentSelection(), src);
		if(applyModifier(crossfader))
			markEditRegion(min(currentSelection()->length(), src->length()));
	}
	freeSource(src);
}

void
DataEditor::spliceIn() {
	Data* src = nil;
	if((src = currentSource()) != nil) {
		Splicer splicer(currentSelection(), src);
		if(applyModifier(splicer)) {
			Range edit = src->frameRange();
			edit += currentInsert();		// offset for highlight display
			controller->showEditRegion(edit, data->channelRange());
		}
	}
	freeSource(src);
}

void
DataEditor::scaleValues() {
	Scaler s(currentSelection());
	applyModifier(s);
}

void
DataEditor::insertSpace() {
	SpaceInsert s(currentSelection());
	applyModifier(s);
}

void
DataEditor::reverse() {
	Reverser r(currentSelection());
	applyModifier(r);
}

void
DataEditor::vertOffset() {
	Offset o(currentSelection());
	applyModifier(o);
}

void
DataEditor::rescale() {
	Rescaler r(currentSelection());
	applyModifier(r);
}

void
DataEditor::interpolate() {
	// create new buffer which will be resized by Interpolater
	setCopyBuffer(currentSelection()->newData(1));
	Interpolater i(currentSelection(), copyBuffer());
	applyModifierToNew(i);
}

void
DataEditor::changeLength() {
	LengthChanger l(model());
	applyModifier(l);
	reselect();
}

void
DataEditor::delay() {
	Delay d(currentSelection());
	applyModifier(d);
}

void
DataEditor::lowPassFilter() {
	LowPassFilter l(currentSelection(), 0.5);
	applyModifier(l);
}

void
DataEditor::applyEnvelope() {
	Envelope *envelope = nil;
	Data *source = nil;
	Controller* ctlr = nil;
	if((source = currentSource()) != nil) {
		;
	}
	else if((ctlr = openNewFile("Select Envelope File to Apply:")) != nil) {
		source = ctlr->model();
		source->ref();
	}
	if(source) {
		// make copy of source so that it may be normalized, etc.
		envelope = new Envelope(source->length());
		envelope->copyFrom(source);
		envelope->ref();
		Application::inform("Scaling envelope between -1 and 1...");
		envelope->normalize();
		Phraser p(currentSelection(), envelope);
		if(p.configure(controller))
			p.apply();
		Resource::unref(envelope);
	}
	freeSource(source);
}

void
DataEditor::normalize() {
	Normalizer n(currentSelection());
	applyModifier(n);
}

void
DataEditor::changeSampleRate() {
	SampRateChanger s(model());
	if(applyModifier(s)) {
		controller->resetScaleTimes();
		reselect();
	}
}

void
DataEditor::showPeakFrame() {
	int peakchan, peakloc;
	data->maxValue(&peakchan, &peakloc);
	Range chan(peakchan, peakchan);
	controller->showInsertPoint(peakloc, chan);
}

void
DataEditor::findZeroCrossing() {
	int offset = currentSelection()->zeroCrossing();
	if(offset > 0) {
		Range chan(currentChannels().intMin(), currentChannels().intMin());
		controller->showInsertPoint(currentInsert() + offset, chan);
	}
	else
		Application::alert("No zero crossings found.");
}

void
DataEditor::extractEnvelope() {
	EnvelopeExtracter e(currentSelection());
	if(applyModifier(e)) {
		Controller *newctlr = new Controller(e.getAnalysis());
		newctlr->display(controller->world());
	}
}

//**************

Data *
LPCEditor::newModel() { return new LPCData(); }

Range
LPCEditor::currentChannels() {
	if(nchans() == 1 && !channels.includes(-1))
		;
	else if(channels.includesZero() || channels.includes(-1))  // default
		setChannels(model()->channelRange());
	else if(channels.intMin() >= 4)		// coefficient display
		setChannels(4, model()->channels() - 1);
	return channels;
}

boolean
LPCEditor::keyCommand(unsigned long sym) {
	boolean interested = true;
	switch (sym) {
	case XK_asterisk:
		stabilizeFrames();
		break;
	case XK_A:
		displayFrameAmplitudes();
		break;
	case XK_F:
		displayFrameFormants();
		break;
	case XK_k:
		changeSampleRate();
		break;
	case XK_M:
		mergePitchData();
		break;
	case XK_p:
		adjustPitchDeviation();
		break;
	default:
		interested = DataEditor::keyCommand(sym);
		break;
	}
	return interested;
}

void
LPCEditor::newFile() {
	LPCEditor::new_File(controller);
}

void
LPCEditor::saveFile() {
	LPCFileSaver lfs(controller->fileName(), defaultDir(), model());
	applyModifier(lfs);
}

void
LPCEditor::saveToFile() {
	LPCFileSaver lfs(defaultDir(), model());
	applyModifier(lfs);
}

// static constructor

Controller *
LPCEditor::new_File(Controller* controller) {
	LPCFileCreator lfc;
	applyModifierUsing(lfc, controller);
	return lfc.getNewController();
}

void
LPCEditor::stabilizeFrames() {
	FrameStabilizer fs(model());
	applyModifier(fs);
}

void
LPCEditor::displayFrameAmplitudes() {
	const int pulseFrameSize = 1000;
	const int pulsePerFrame = 10;
	LPCData* selected = (LPCData *) currentSelection();
	Application::inform("Creating test pattern...");
	int lpcLen = selected->length();
	int srate = selected->sRate();
	Sound* pulses = new Sound(lpcLen * pulseFrameSize, srate, 1, FloatData);
	pulses->ref();
	// add pulsePerFrame pulses per original LPC frame
	PulseGenerator pgen(pulses, pulseFrameSize/pulsePerFrame);
	pgen.apply();
	FormantFilter filter(pulses, pulses, selected, 1.0);
	filter.apply();
	Envelope* amplitudes = new Envelope(lpcLen * pulsePerFrame);
	amplitudes->setFrameRangeLabel("LPC Analysis Frames");
	amplitudes->setRangeFactor(1.0/pulsePerFrame);
	Application::inform("Extracting amplitudes...");
	pulses->getEnvelope(amplitudes, 0, AbsoluteMagnitude);
	Resource::unref(pulses);
	Controller* ampDisplay = new Controller(amplitudes, 
		scat(controller->windowName(), ":  Frame Amplitudes"));
	ampDisplay->display(controller->world());
}

void
LPCEditor::displayFrameFormants() {
	const int pulseFrameSize = 1024;
	LPCData* selected = (LPCData *) currentSelection();
	int lpcLen = selected->length();
	int srate = selected->sRate();
	Sound* pulses = new Sound(lpcLen * pulseFrameSize, srate, 1, FloatData);
	pulses->ref();
	PulseGenerator pgen(pulses, pulseFrameSize);
	Application::inform("Creating test pattern...");
	pgen.apply();
	FormantFilter filter(pulses, pulses, selected, 1.0);
	filter.apply();
	Application::inform("Analyzing formants...");
	// fft size depends on npoles
	FFT_Function analyzer(
		pulses, selected->nPoles() > 32 ? 128 : 64, pulseFrameSize
	);
	if(analyzer.apply()) {
		Controller* fftDisplay = new Controller(
			analyzer.getAnalysis(), 
			scat(controller->windowName(), ":  Formant Frequencies"));
		fftDisplay->display(controller->world());
	}
	Resource::unref(pulses);
}

void
LPCEditor::mergePitchData() {
	// regardless of selection, select only channel 3 for merge
	const int pitchChannel = 3;
	setChannels(pitchChannel, pitchChannel);
	LPCData *lpc = (LPCData *) currentSelection();
	Data *pitches = nil;
	Controller* newController = nil;
	if((pitches = currentSource()) != nil) {
		;
	}
	else if((newController = openNewFile("Select Pitch File for Merge:")) != nil)
		pitches = newController->model();
	if(pitches && Application::confirm("Please confirm merge operation.")) {
		Application::inform("Merging pitch data...");
		lpc->mergePitchData(pitches);	
		markEditRegion(lpc->length());
	}
	freeSource(pitches);
}

#if 0

void
LPCEditor::warpFrames() {
	Data *source = nil;
	Controller* ctlr = nil;
	if((source = currentSource()) != nil) {
		copy();
		Envelope* evp = new Envelope(1);
		evp->ref();
		Interpolater i(source, evp, copyBuffer()->length());
		i.apply();		// stretch source to equal current sel's length
		FrameWarper f(evp, copyBuffer());
		if(applyModifierToNew(f)) {
			Application::alert("A new file of filter modifying values",
				"is being created, and will be needed for any",
				"subsequent warped LPC formant filtering.");
			ctlr = new Controller(evp, "new_warp.evp");
			ctlr->display(controller->world());
		}
		Resource::unref(evp);
	}
	freeSource(source);
}

#endif /* 0 */

void
LPCEditor::setDataOptions() {
	LPCOptionSetter setter;
	applyModifier(setter);
}

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
LPCEditor::new_DataEditor1(Controller *c) {
	return new LPCEditor(c);
}

DataEditor *
LPCEditor::new_DataEditor2(Controller *c, const Data *d) {
	return new LPCEditor(c, d);
}

//**************

Data *
PCHEditor::newModel() { return new PCHData(); }

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
PCHEditor::new_DataEditor1(Controller *c) {
	return new PCHEditor(c);
}

DataEditor *
PCHEditor::new_DataEditor2(Controller *c, const Data *d) {
	return new PCHEditor(c, d);
}

//**************

Data *
FFTEditor::newModel() { return new FFTData(); }

void
FFTEditor::setDataOptions() {}

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
FFTEditor::new_DataEditor1(Controller *c) {
	return new FFTEditor(c);
}

DataEditor *
FFTEditor::new_DataEditor2(Controller *c, const Data *d) {
	return new FFTEditor(c, d);
}

//**************

Data *
EnvelopeEditor::newModel() { return new Envelope(); }

boolean
EnvelopeEditor::keyCommand(unsigned long sym) {
	boolean interested = true;
	switch (sym) {
	case XK_underscore:
		createLinearCurve();
		break;
	case XK_asterisk:
		createExponentialCurve();
		break;
	default:
		interested = DataEditor::keyCommand(sym);
		break;
	}
	return interested;
}

void
EnvelopeEditor::newFile() {
	EnvelopeEditor::new_File(controller);
}

void
EnvelopeEditor::setDataOptions() {}

// static constructor

Controller *
EnvelopeEditor::new_File(Controller* controller) {
	EnvelopeFileCreator efc;
	applyModifierUsing(efc, controller);
	return efc.getNewController();
}

void
EnvelopeEditor::createLinearCurve() {
	LinearCurveGenerator l(currentSelection());
	applyModifier(l);
}

void
EnvelopeEditor::createExponentialCurve() {
	ExponentialCurveGenerator e(currentSelection());
	applyModifier(e);
}

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
EnvelopeEditor::new_DataEditor1(Controller *c) {
	return new EnvelopeEditor(c);
}

DataEditor *
EnvelopeEditor::new_DataEditor2(Controller *c, const Data *d) {
	return new EnvelopeEditor(c, d);
}
