#include <stdio.h>
#include <varargs.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "lxt.h"

extern Menu *xm_menus;
extern Void *xmret_void;

/*VARARGS*/
Menu_item *
menuitem_create(va_alist)
/*
   User-callable.
   Creates a new Menu_item.
*/
va_dcl
{
	va_list varg_ptr;
	int nargs, attr;
	Menu_item *mi;

	va_start(varg_ptr);
	for (nargs= 0;; nargs++) {

		if (nargs == 0) {
			if ((mi= (Menu_item *) calloc(1, sizeof(Menu_item))) == (Menu_item *) NULL) {
				(void) fprintf(stderr, "menuitem_create: memory allocation error\n");
				return((Menu_item *) NULL);
			}
			mi->xmi_magic= LX_MENUITEM;
			if (menuitem_filldefaults(mi) != LX_SUCCESS) {
				cfree((char *) mi);
				return((Menu_item *) NULL);
			}
		}

		/* attribute list */
		else {
			char *c;
			int n;

			attr= va_arg(varg_ptr, int);
			if (attr == LXMI_NULL)
				break;

			switch (attr) {

			case LXMI_STRING:
				c= va_arg(varg_ptr, char *);
				if (c != (char *) NULL) {
					if ((mi->xmi_string= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
						(void) fprintf(stderr, "menuitem_create: memory allocation error\n");
						return((Menu_item *) NULL);
					}
					(void) strcpy(mi->xmi_string, c);
				}
				break;
			case LXMI_PROC:
				mi->xmi_proc= (void (*)()) va_arg(varg_ptr, char *);
				break;
			case LXMI_PULLRIGHT:
				mi->xmi_pullright= va_arg(varg_ptr, char *);
				break;
				break;
			case LXMI_IMAGE:
				mi->xmi_image= va_arg(varg_ptr, XImage *);
				break;
			case LXMI_STATE:
				n= va_arg(varg_ptr, int);
				switch (n) {
				case LXMI_ACTIVE:
				case LXMI_INACTIVE:
					mi->xmi_state= n;
					break;
				default:
					(void) fprintf(stderr, "menuitem_create: invalid item state\n");
					break;
				}
				break;
			case LXMI_CLIENTDATA:
				mi->xmi_clientdata= va_arg(varg_ptr, char *);
				break;
			default:
				(void) fprintf(stderr, "menuitem_create: ignoring unrecognized attribute\n");
				break;
			}
		}
	}
	va_end(varg_ptr);
	return(mi);
}

int
menuitem_filldefaults(mi)
/*
   Internal function.
   Initialize defaults for a newly-created item.
*/
Menu_item *mi;
{
	mi->xmi_image= (XImage *) NULL;
	mi->xmi_proc= (void (*)()) NULL;
	mi->xmi_pullright= (char *) NULL;
	mi->xmi_state= LXMI_ACTIVE;
	return(LX_SUCCESS);
}

/*VARARGS*/
int
menuitem_set(va_alist)
/*
   User-callable.
   Changes an attribute of a previously created Menu_item.
*/
va_dcl
{
	va_list varg_ptr;
	Menu_item *mi;
	int nargs, attr;
	void menuitem_adjhist();

	va_start(varg_ptr);
	for (nargs= 0;; nargs++) {

		/* get item */
		if (nargs == 0) {
			mi= va_arg(varg_ptr, Menu_item *);
			if (mi == (Menu_item *) NULL) {
				(void) fprintf(stderr, "menuitem_set: null menu item\n");
				return(LX_ERROR);
			}
			if (mi->xmi_magic != LX_MENUITEM) {
				(void) fprintf(stderr, "menuitem_set: object is not a menu item\n");
				return(LX_ERROR);
			}
		}

		/* get attribute */
		else {
			char *c;
			int n;

			attr= va_arg(varg_ptr, int);
			if (attr == LXMI_NULL)
				break;

			switch (attr) {

			case LXMI_STRING:
				c= va_arg(varg_ptr, char *);
				if (mi->xmi_string != (char *) NULL)
					cfree(mi->xmi_string);
				if (c == (char *) NULL)
					mi->xmi_string= (char *) NULL;
				else {
					if ((mi->xmi_string= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
						(void) fprintf(stderr, "menuitem_set: memory allocation error\n");
						return(LX_ERROR);
					}
					(void) strcpy(mi->xmi_string, c);
				}
				break;
			case LXMI_PROC:
				mi->xmi_proc= (void (*)()) va_arg(varg_ptr, char *);
				break;
			case LXMI_PULLRIGHT:
				mi->xmi_pullright= va_arg(varg_ptr, char *);
				break;
			case LXMI_IMAGE:
				mi->xmi_image= va_arg(varg_ptr, XImage *);
				break;
			case LXMI_STATE:
				n= va_arg(varg_ptr, int);
				switch (n) {
				case LXMI_ACTIVE:
					mi->xmi_state= n;
					break;
				case LXMI_INACTIVE:
					mi->xmi_state= n;
					menuitem_adjhist((Menu *) NULL, mi);
					break;
				default:
					(void) fprintf(stderr, "menuitem_set: unrecognized state\n");
					return(LX_ERROR);
					break;
				}
				break;
			case LXMI_CLIENTDATA:
				mi->xmi_clientdata= va_arg(varg_ptr, char *);
				break;
			default:
				(void) fprintf(stderr, "menuitem_set: unrecognized menu attribute %d\n", attr);
				return(LX_ERROR);
				break;
			}
		}
	}
	va_end(varg_ptr);
	return(LX_SUCCESS);
}

Void *
menuitem_get(mi, attr)
/*
   User-callable.
   Returns the value of an attribute or a pointer thereto.
*/
Menu_item *mi;
int attr;
{
	xmret_void= (Void *) NULL;
	if (mi == (Menu_item *) NULL) {
		(void) fprintf(stderr, "menuitem_get: null menu item\n");
		return(xmret_void);
	}
	if (mi->xmi_magic != LX_MENUITEM) {
		(void) fprintf(stderr, "menuitem_get: object is not a menu item\n");
		return(xmret_void);
	}
	if (attr == LXMI_NULL)
		return(xmret_void);

	switch (attr) {

	case LXMI_STRING:
		xmret_void= (Void *) mi->xmi_string;
		break;
	case LXMI_PROC:
		xmret_void= (Void *) mi->xmi_proc;
		break;
	case LXMI_PULLRIGHT:
		xmret_void= (Void *) mi->xmi_pullright;
		break;
	case LXMI_IMAGE:
		xmret_void= (Void *) mi->xmi_image;
		break;
	case LXMI_STATE:
		xmret_void= (Void *) &(mi->xmi_state);
		break;
	case LXMI_CLIENTDATA:
		xmret_void= (Void *) mi->xmi_clientdata;
		break;
	case LXMI_TIMESTAMP:
		xmret_void= (Void *) &(mi->xmi_timestamp);
		break;
	default:
		(void) fprintf(stderr, "menuitem_get: unrecognized attribute\n");
		break;
	}
	return(xmret_void);
}

/*VARARGS*/
Menu_item *
menuitem_find(va_alist)
/*
   User-callable.
   Returns the first item within a menu whose
   named attribute matches the supplied argument.
*/
va_dcl
{
	va_list varg_ptr;
	Menu *m;
	int nargs, attribute;
	Menu_itemptr *mip;

	va_start(varg_ptr);
	for (nargs= 0; nargs < 3; nargs++) {

		/* get item */
		if (nargs == 0) {
			m= va_arg(varg_ptr, Menu *);
			if (m == (Menu *) NULL) {
				(void) fprintf(stderr, "menuitem_find: null menu\n");
				return((Menu_item *) NULL);
			}
			if (m->xm_magic != LX_MENU) {
				(void) fprintf(stderr, "menuitem_find: object is not a menu\n");
				return((Menu_item *) NULL);
			}
		}

		/* get attribute */
		else {
			char *c;
			int n;
			Menu *mp;
			XImage *im;
			void (*proc)();

			attribute= va_arg(varg_ptr, int);
			if (attribute == LXMI_NULL)
				break;

			switch (attribute) {

			case LXMI_STRING:
				c= va_arg(varg_ptr, char *);
				for (mip= m->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next) {
					if (mip->xmip_item->xmi_string != (char *) NULL)
						if (!strcmp(c, mip->xmip_item->xmi_string))
							return(mip->xmip_item);
				}
				return((Menu_item *) NULL);
				break;
			case LXMI_PROC:
				proc= (void (*)()) va_arg(varg_ptr, char *);
				for (mip= m->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next)
					if (proc == mip->xmip_item->xmi_proc)
						return(mip->xmip_item);
				return((Menu_item *) NULL);
				break;
			case LXMI_PULLRIGHT:
				mp= va_arg(varg_ptr, Menu *);
				for (mip= m->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next)
					if (mp == (Menu *) (mip->xmip_item->xmi_pullright))
						return(mip->xmip_item);
				return((Menu_item *) NULL);
				break;
			case LXMI_IMAGE:
				im= va_arg(varg_ptr, XImage *);
				for (mip= m->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next)
					if (im == mip->xmip_item->xmi_image)
						return(mip->xmip_item);
				return((Menu_item *) NULL);
				break;
			case LXMI_STATE:
				n= va_arg(varg_ptr, int);
				for (mip= m->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next)
					if (n == mip->xmip_item->xmi_state)
						return(mip->xmip_item);
				return((Menu_item *) NULL);
				break;
			case LXMI_CLIENTDATA:
				c= va_arg(varg_ptr, char *);
				for (mip= m->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next) {
					if (mip->xmip_item->xmi_clientdata == c)
						return(mip->xmip_item);
				}
				return((Menu_item *) NULL);
				break;
			default:
				(void) fprintf(stderr, "menuitem_find: invalid menu item search attribute %d\n", attribute);
				return((Menu_item *) NULL);
				break;
			}
		}
	}
	va_end(varg_ptr);
	return((Menu_item *) NULL);
}

int
menuitem_insert(m, mi)
/*
   User-callable.
   Inserts a Menu_item into a previously created menu.
*/
Menu *m;
Menu_item *mi;
{
	Menu_itemptr *mip, *nip;

	if (m == (Menu *) NULL) {
		(void) fprintf(stderr, "menuitem_insert: null menu\n");
		return(LX_ERROR);
	}
	if (m->xm_magic != LX_MENU) {
		(void) fprintf(stderr, "menuitem_insert: object is not a menu\n");
		return(LX_ERROR);
	}
	if (mi == (Menu_item *) NULL) {
		(void) fprintf(stderr, "menuitem_insert: null menu item\n");
		return(LX_ERROR);
	}
	if (mi->xmi_magic != LX_MENUITEM) {
		(void) fprintf(stderr, "menuitem_insert: object is not a menu item\n");
		return(LX_ERROR);
	}

	if ((mip= (Menu_itemptr *) calloc(1, sizeof(Menu_itemptr))) == (Menu_itemptr *) NULL) {
		(void) fprintf(stderr, "menuitem_insert: memory allocation error\n");
		return(LX_ERROR);
	}
	if (m->xm_items != (Menu_itemptr *) NULL) {
		for (nip= m->xm_items; nip->xmip_next != (Menu_itemptr *) NULL; nip= nip->xmip_next);
		nip->xmip_next= mip;
		mip->xmip_prev= nip;
	}
	else {
		m->xm_items= mip;
		mip->xmip_prev= (Menu_itemptr *) NULL;
	}
	mip->xmip_next= (Menu_itemptr *) NULL;
	mip->xmip_item= mi;
	return(LX_SUCCESS);
}

void
menuitem_resize(m, mip, font_ht)
/*
   Internal function.
   Determines the size of the label or image
   attached to mip to be displayed in m.
*/
Menu *m;
Menu_itemptr *mip;
int font_ht;
{
	if (mip->xmip_item->xmi_image != (XImage *) NULL) {
		mip->xmip_w= mip->xmip_item->xmi_image->width;
		mip->xmip_h= mip->xmip_item->xmi_image->height+m->xm_vmargin;
	}
	else {
		if (mip->xmip_item->xmi_string != (char *) NULL)
			mip->xmip_w= XTextWidth(m->xm_font, mip->xmip_item->xmi_string, strlen(mip->xmip_item->xmi_string));
		else
			mip->xmip_w= XTextWidth(m->xm_font, "   ", 3);
		if (mip->xmip_item->xmi_pullright != (char *) NULL)
			mip->xmip_w+= LXMI_PRARROWLEN+LXMI_PRARROWGAP;
		mip->xmip_h= font_ht+m->xm_vmargin;
	}
}

Menu_item *
menu_findsel(m, x, y)
/*
   Internal function.
   Returns a pointer to the currently selected item.
*/
Menu *m;
int x, y;
{
	int font_ht;
	Menu_itemptr *mip;

	if (m == (Menu *) NULL)
		return((Menu_item *) NULL);
	font_ht= m->xm_font->max_bounds.ascent+m->xm_font->max_bounds.descent;
	if (m->xm_title != (char *) NULL) {
		if (y <= font_ht+(2*m->xm_vmargin))
			return((Menu_item *) NULL);
	}
	else if (y <= m->xm_vmargin)
		return((Menu_item *) NULL);

	for (mip= m->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next) {
		if (y <= mip->xmip_y+mip->xmip_h) {
			if (mip->xmip_item->xmi_state == LXMI_ACTIVE)
				return(mip->xmip_item);
			else
				return((Menu_item *) NULL);
		}
	}
	return((Menu_item *) NULL);
}

int
menuitem_getord(m, mi)
/*
   User-callable.
   Returns the ordinal position of
   a menu item within a menu (-1 if nonexistent).
*/
Menu *m;
Menu_item *mi;
{
	int i;
	Menu_itemptr *mip;

	if (m == (Menu *) NULL)
		return(-1);
	if (m->xm_magic != LX_MENU)
		return(-1);
	if (mi == (Menu_item *) NULL)
		return(-1);
	if (mi->xmi_magic != LX_MENUITEM)
		return(-1);

	for (mip= m->xm_items, i= 0; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next, i++) {
		if (mi == mip->xmip_item)
			return(i);
	}
	return(-1);
}

int
menuitem_delete(m, mi)
/*
   User-callable.
   Removes a Menu_item from a menu.
*/
Menu *m;
Menu_item *mi;
{
	Menu_itemptr *xmip, *ymip;
	void menuitem_adjhist();

	if (m == (Menu *) NULL) {
		(void) fprintf(stderr, "menuitem_delete: null menu\n");
		return(LX_ERROR);
	}
	if (m->xm_magic != LX_MENU) {
		(void) fprintf(stderr, "menuitem_delete: object is not a menu\n");
		return(LX_ERROR);
	}
	if (mi == (Menu_item *) NULL) {
		(void) fprintf(stderr, "menuitem_delete: null menu item\n");
		return(LX_ERROR);
	}
	if (mi->xmi_magic != LX_MENUITEM) {
		(void) fprintf(stderr, "menuitem_delete: object is not a menu item\n");
		return(LX_ERROR);
	}

	menuitem_adjhist(m, mi);
	for (xmip= ymip= m->xm_items; xmip != (Menu_itemptr *) NULL; xmip= xmip->xmip_next) {
		if (xmip->xmip_item == mi)
			break;
		else
			ymip= xmip;
	}
	if (xmip == (Menu_itemptr *) NULL) {
		(void) fprintf(stderr, "menuitem_delete: item not found within menu\n");
		return(LX_ERROR);
	}
	else {
		if (xmip == m->xm_items)
			m->xm_items= xmip->xmip_next;
		else
			ymip->xmip_next= xmip->xmip_next;
		if (xmip->xmip_next != (Menu_itemptr *) NULL)
			xmip->xmip_next->xmip_prev= ymip;
		cfree((char *) xmip);
	}
	return(LX_SUCCESS);
}

int
menuitem_destroy(mi)
/*
   User-callable.
   Destroys a Menu_item. Does not check to see
   if the item is still linked into a menu!
*/
Menu_item *mi;
{
	void menuitem_adjhist();

	if (mi == (Menu_item *) NULL) {
		(void) fprintf(stderr, "menuitem_destroy: null menu item\n");
		return(LX_ERROR);
	}
	if (mi->xmi_magic != LX_MENUITEM) {
		(void) fprintf(stderr, "menuitem_destroy: object is not a menu item\n");
		return(LX_ERROR);
	}

	menuitem_adjhist((Menu *) NULL, mi);
	if (mi->xmi_string != (char *) NULL)
		cfree(mi->xmi_string);
	cfree((char *) mi);
	return(LX_SUCCESS);
}

void
menuitem_adjhist(m, mi)
/*
   Internal function.
   Check all history chains to delete any
   references to the specified menu/item pair.
   If no menu is specified, delete all references
   to the specified item.
*/
Menu *m;
Menu_item *mi;
{
	Menu *n;
	Menu_hist *mh, *nh;
	void menu_dsthist();

	for (n= xm_menus; n != (Menu *) NULL; n= n->xm_next) {
		for (mh= (Menu_hist *) n->xm_hist; mh != (Menu_hist *) NULL; mh= mh->xmh_next) {
			if (m == (Menu *) NULL) {
				if (mh->xmh_menuitem == mi)
					break;
			}
			else {
				if ((mh->xmh_menu == m) && (mh->xmh_menuitem == mi))
					break;
			}
		}
		if (mh != (Menu_hist *) NULL) {
			if (mh == (Menu_hist *) n->xm_hist)
				menu_dsthist(n);
			else {
				nh= mh->xmh_next;
				mh->xmh_menuitem= (Menu_item *) NULL;
				mh->xmh_next= (Menu_hist *) NULL;
				for (mh= nh; mh != (Menu_hist *) NULL; mh= nh) {
					nh= mh->xmh_next;
					cfree((char *) mh);
				}
			}
		}
	}
	return;
}
