#include <stdio.h>
#include <varargs.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "lxt.h"

extern Panel *xp_blocked;
extern char xpkey_cret;

static Display *alert_dpy;
static Window alert_win;
static Panel *alert_panel;
static int *alert_valptr;
static Alert_item *alert_labels;
static Alert_item *alert_texts;
static Alert_item *alert_buttons;
static Alert_item *alert_defbutton;
static boolean alert_fontset= FALSE;
static char alert_fontnm[LX_MAXFONTNMLEN+1];
static XFontStruct *alert_font= (XFontStruct *) NULL;
static Visual *alert_vis= (Visual *) NULL;
static Colormap alert_cmap= (Colormap) None;
static boolean alert_fgset= FALSE;
static boolean alert_bgset= FALSE;
static unsigned long alert_fg, alert_bg;
static int alert_w, alert_h;
static boolean alert_bwset= FALSE;
static unsigned int alert_bw;
static void (*alert_proc)();
static int (*alert_kpproc)();

boolean alert_buttonactive= FALSE;
int alert_xwarp, alert_ywarp;

static XWMHints alert_wmh= {
	(InputHint | StateHint),
	False, NormalState,
	0, 0, 0, 0, 0, 0,
};

/*VARARGS*/
int
alert_setdefaults(va_alist)
/*
   User-callable.
   Sets up default attributes for all alert panels.
*/
va_dcl
{
	va_list varg_ptr;
	int attr;
	char *c;

	va_start(varg_ptr);
	for (;;) {
		attr= va_arg(varg_ptr, int);
		if (attr == LXA_NULL)
			break;

		switch (attr) {
		case LXA_FOREGROUND:
			alert_fg= va_arg(varg_ptr, unsigned long);
			alert_fgset= TRUE;
			break;
		case LXA_BACKGROUND:
			alert_bg= va_arg(varg_ptr, unsigned long);
			alert_bgset= TRUE;
			break;
		case LXA_BWIDTH:
			alert_bw= va_arg(varg_ptr, unsigned int);
			alert_bwset= TRUE;
			break;
		case LXA_FONT:
			c= va_arg(varg_ptr, char *);
			if (c == (char *) NULL) {
				(void) fprintf(stderr, "alert_setdefaults: null font name\n");
				return(LX_ERROR);
			}
			(void) strncpy(alert_fontnm, c, LX_MAXFONTNMLEN);
			alert_fontset= TRUE;
			break;
		case LXA_VISUAL:
			alert_vis= va_arg(varg_ptr, Visual *);
			break;
		case LXA_COLORMAP:
			alert_cmap= va_arg(varg_ptr, Colormap);
			break;
		case LXA_PROC:
		case LXA_LABEL:
		case LXA_TEXT:
		case LXA_BUTTON:
		case LXA_DEFAULTBUTTON:
			(void) fprintf(stderr, "alert_setdefaults: non-default attribute specified\n");
			return(LX_ERROR);
			break;
		default:
			(void) fprintf(stderr, "alert_setdefaults: unrecognized attribute specified\n");
			return(LX_ERROR);
			break;
		}
	}
	va_end(varg_ptr);

	return(LX_SUCCESS);
}

/*VARARGS*/
int
alert_prompt(va_alist)
/*
   User-callable.
   Creates, displays and destroys an alert panel.
*/
va_dcl
{
	va_list varg_ptr;
	int nargs;
	XSizeHints *xsh;
#ifdef R3
	XSizeHints xsh_prealloc;
#endif R3
	int screen, depth, attr, ntexts;
	char *name, *f;
	Alert_item *ai;
	XSetWindowAttributes xswa;
	Alert_item *alert_labelalloc();
	Alert_item *alert_textalloc();
	Alert_item *alert_buttonalloc();
	void alert_geometry(), alert_destroylists();
	int alert_textproc();

	if (xp_blocked != (Panel *) NULL) {
		(void) fprintf(stderr, "alert_prompt: blocked panel already displayed\n");
		return(LX_ERROR);
	}

	va_start(varg_ptr);
	for (nargs= 0;; nargs++) {

		/* get program name */
		if (nargs == 0) {
			name= va_arg(varg_ptr, char *);
			if (name == (char *) NULL) {
				(void) fprintf(stderr, "alert_prompt: null program name\n");
				return(LX_ERROR);
			}
		}

		/* get display */
		else if (nargs == 1) {
			alert_dpy= va_arg(varg_ptr, Display *);
			if (alert_dpy == (Display *) NULL) {
				(void) fprintf(stderr, "alert_prompt: null display\n");
				return(LX_ERROR);
			}

			screen= DefaultScreen(alert_dpy);
			depth= DefaultDepth(alert_dpy, screen);
			if (alert_vis == (Visual *) NULL)
				alert_vis= DefaultVisual(alert_dpy, screen);
			if (alert_cmap == (Colormap) None)
				alert_cmap= DefaultColormap(alert_dpy, screen);
			if (alert_bgset == FALSE) {
				alert_bg= WhitePixel(alert_dpy, screen);
				alert_bgset= TRUE;
			}
			if (alert_fgset == FALSE) {
				alert_fg= BlackPixel(alert_dpy, screen);
				alert_fgset= TRUE;
			}
			if (alert_bwset == FALSE) {
				alert_bw= LXADEF_BWIDTH;
				alert_bwset= TRUE;
			}
			alert_font= (XFontStruct *) NULL;
			alert_proc= (void (*)()) NULL;
			alert_kpproc= (int (*)()) NULL;
			alert_labels= alert_texts= alert_buttons= (Alert_item *) NULL;
			alert_defbutton= (Alert_item *) NULL;
			ntexts= 0;
		}

		/* get pointer to return value */
		else if (nargs == 2) {
			alert_valptr= va_arg(varg_ptr, int *);
			if (alert_valptr == (int *) NULL) {
				(void) fprintf(stderr, "alert_prompt: null return value pointer\n");
				return(LX_ERROR);
			}
		}

		else {
			char *c, *str, *val;
			int n;

			attr= va_arg(varg_ptr, int);
			if (attr == LXA_NULL)
				break;

			switch (attr) {
			case LXA_FOREGROUND:
				alert_fg= va_arg(varg_ptr, unsigned long);
				alert_fgset= TRUE;
				break;
			case LXA_BACKGROUND:
				alert_bg= va_arg(varg_ptr, unsigned long);
				alert_bgset= TRUE;
				break;
			case LXA_BWIDTH:
				alert_bw= va_arg(varg_ptr, unsigned int);
				alert_bwset= TRUE;
				break;
			case LXA_PROC:
				alert_proc= (void (*)()) va_arg(varg_ptr, char *);
				break;
			case LXA_FONT:
				c= va_arg(varg_ptr, char *);
				if (c == (char *) NULL) {
					(void) fprintf(stderr, "alert_prompt: null font name\n");
					return(LX_ERROR);
				}
				(void) strncpy(alert_fontnm, c, LX_MAXFONTNMLEN);
				if ((alert_font= XLoadQueryFont(alert_dpy, alert_fontnm)) == NULL) {
					(void) fprintf(stderr, "alert_prompt: cannot load font %s\n", alert_fontnm);
					return(LX_ERROR);
				}
				alert_fontset= TRUE;
				break;
			case LXA_LABEL:
				str= va_arg(varg_ptr, char *);
				if (str == (char *) NULL) {
					(void) fprintf(stderr, "alert_prompt: null string\n");
					return(LX_ERROR);
				}
				if (alert_labelalloc(str) == (Alert_item *) NULL) {
					(void) fprintf(stderr, "alert_prompt: memory allocation error\n");
					return(LX_ERROR);
				}
				break;
			case LXA_TEXT:
				str= va_arg(varg_ptr, char *);
				val= va_arg(varg_ptr, char *);
				c= va_arg(varg_ptr, char *);
				if (c == (char *) NULL) {
					(void) fprintf(stderr, "alert_prompt: null text return pointer\n");
					return(LX_ERROR);
				}
				if (alert_textalloc(str, val, c) == (Alert_item *) NULL) {
					(void) fprintf(stderr, "alert_prompt: memory allocation error\n");
					return(LX_ERROR);
				}
				ntexts++;
				break;
			case LXA_BUTTON:
			case LXA_DEFAULTBUTTON:
				str= va_arg(varg_ptr, char *);
				if (str == (char *) NULL) {
					(void) fprintf(stderr, "alert_prompt: null string\n");
					return(LX_ERROR);
				}
				n= va_arg(varg_ptr, int);
				if ((ai= alert_buttonalloc(str, n)) == (Alert_item *) NULL) {
					(void) fprintf(stderr, "alert_prompt: memory allocation error\n");
					return(LX_ERROR);
				}
				if ((alert_defbutton == (Alert_item *) NULL) || (attr == LXA_DEFAULTBUTTON))
					alert_defbutton= ai;
				break;
			case LXA_VISUAL:
				alert_vis= va_arg(varg_ptr, Visual *);
				break;
			case LXA_COLORMAP:
				alert_cmap= va_arg(varg_ptr, Colormap);
				break;
			default:
				break;
			}
		}
	}
	va_end(varg_ptr);

	/* must be at least one button or exactly one text */
	if (alert_buttons == (Alert_item *) NULL) {
		if (ntexts == 1)
			alert_kpproc= alert_textproc;
		else {
			(void) fprintf(stderr, "alert_prompt: must be at least one button item or exactly one text item\n");
			alert_destroylists();
			return(LX_ERROR);
		}
	}
	if (alert_font == (XFontStruct *) NULL) {
		if (alert_fontset == FALSE) {
			if ((f= XGetDefault(alert_dpy, name, "Font")) == (char *) NULL)
				(void) strcpy(alert_fontnm, LXADEF_FONT);
			else
				(void) strncpy(alert_fontnm, f, LX_MAXFONTNMLEN);
			alert_fontset= TRUE;
		}
		if ((alert_font= XLoadQueryFont(alert_dpy, alert_fontnm)) == NULL) {
			(void) fprintf(stderr, "alert_prompt: cannot load font %s\n", alert_fontnm);
			alert_destroylists();
			return(LX_ERROR);
		}
	}

	/* compute size and location of panel and items */
	alert_geometry();

	/* create enclosing window of panel */
#ifndef R3
	if ((xsh= XAllocSizeHints()) == (XSizeHints *) NULL) {
		(void) fprintf(stderr, "alert_prompt: memory allocation error\n");
		alert_destroylists();
		return(LX_ERROR);
        }
#else
	xsh= &xsh_prealloc;
#endif R3
	xsh->flags= (PPosition | PSize);
	xsh->width= alert_w;
	xsh->height= alert_h;
	xsh->x= (DisplayWidth(alert_dpy, DefaultScreen(alert_dpy)) - xsh->width) / 2;
	xsh->y= (DisplayHeight(alert_dpy, DefaultScreen(alert_dpy)) - xsh->height) / 2;
	xswa.background_pixel= alert_bg;
	xswa.border_pixel= alert_fg;
	xswa.colormap= alert_cmap;
	alert_win= XCreateWindow(alert_dpy, DefaultRootWindow(alert_dpy), xsh->x, xsh->y, xsh->width, xsh->height, alert_bw, depth, InputOutput, alert_vis, CWBackPixel | CWBorderPixel | CWColormap, &xswa);
	XSetStandardProperties(alert_dpy, alert_win, name, "alert", None, (char *) NULL, 0, xsh);
	XSetWMHints(alert_dpy, alert_win, &alert_wmh);
#ifndef R3
	XFree((char *) xsh);
#endif R3

	/* create panel */
	if ((alert_panel= panel_create(name, alert_dpy, alert_win,
			LXP_FOREGROUND, alert_fg,
			LXP_BACKGROUND, alert_bg,
			LXP_WIDTH, alert_w,
			LXP_HEIGHT, alert_h,
			LXP_PROC, alert_proc,
			LXP_FONT, alert_fontnm,
			LXP_NULL)) == (Panel *) NULL) {
		XFreeFont(alert_dpy, alert_font);
		alert_destroylists();
		XDestroyWindow(alert_dpy, alert_win);
		return(LX_ERROR);
	}

	/* create panel items */
	if (alert_createitems() != LX_SUCCESS) {
		alert_destroylists();
		(void) panel_destroy(alert_panel);
		XFreeFont(alert_dpy, alert_font);
		XDestroyWindow(alert_dpy, alert_win);
		return(LX_ERROR);
	}

	/* display blocking panel */
	if (alert_buttons != (Alert_item *) NULL)
		alert_buttonactive= TRUE;
	else
		alert_buttonactive= FALSE;
	XBell(alert_dpy, 50);
	if (panel_block(alert_panel, alert_valptr) != LX_SUCCESS) {
		alert_destroylists();
		(void) panel_destroy(alert_panel);
		XFreeFont(alert_dpy, alert_font);
		XDestroyWindow(alert_dpy, alert_win);
		alert_buttonactive= FALSE;
		return(LX_ERROR);
	}
	alert_buttonactive= FALSE;

	/* destroy data objects */
	alert_destroylists();
	(void) panel_destroy(alert_panel);
	XFreeFont(alert_dpy, alert_font);
	XDestroyWindow(alert_dpy, alert_win);

	return(LX_SUCCESS);
}

void
alert_geometry()
/*
   Internal function.
   Setup alert panel geometry.
*/
{
	int x, y, n, diff;
	int font_ht, line_ht, button_ht, max_width;
	Alert_item *ai, *bi;
	void alert_buttonproc();

	font_ht= alert_font->max_bounds.ascent+alert_font->max_bounds.descent;
	line_ht= font_ht+(font_ht/6);
	button_ht= font_ht+(2*LXPDEF_BUTTONVMARGIN)+4+6;
	max_width= 0;
	y= LXADEF_VMARGIN;

	/* configure label items */
	for (ai= alert_labels; ai != (Alert_item *) NULL; ai= ai->xa_next, y+= line_ht) {
		ai->xa_w= XTextWidth(alert_font, ai->xa_string, strlen(ai->xa_string));
		if (max_width < ai->xa_w)
			max_width= ai->xa_w;
		ai->xa_x= LXADEF_HMARGIN;
		ai->xa_y= y;
	}
	if (alert_labels != (Alert_item *) NULL)
		y+= line_ht;

	/* configure text items */
	for (ai= alert_texts; ai != (Alert_item *) NULL; ai= ai->xa_next, y+= (6*line_ht)/5) {
		if (ai->xa_string != (char *) NULL)
			ai->xa_w= XTextWidth(alert_font, ai->xa_string, strlen(ai->xa_string));
		else
			ai->xa_w= 0;
		ai->xa_w+= LXPDEF_TEXTGAP+((LXADEF_MAXDISPLAY+1)*XTextWidth(alert_font, "X", 1));
		if (max_width < ai->xa_w)
			max_width= ai->xa_w;
		ai->xa_x= LXADEF_HMARGIN;
		ai->xa_y= y;
	}
	if (alert_texts != (Alert_item *) NULL)
		y+= line_ht;
	if (max_width < LXADEF_WIDTH-(2*LXADEF_HMARGIN))
		max_width= LXADEF_WIDTH-(2*LXADEF_HMARGIN);

	/* configure button items */
	/* first pass -- fit as many buttons
	   as possible onto each line */
	x= 0;
	for (ai= alert_buttons; ai != (Alert_item *) NULL; ai= ai->xa_next) {
		ai->xa_w= XTextWidth(alert_font, ai->xa_string, strlen(ai->xa_string))+(2*LXPDEF_BUTTONHMARGIN)+4;
		if (max_width < ai->xa_w) {
			max_width= ai->xa_w;
			if (x > 0)
				y+= button_ht;
			x= 0;
		}
		else if (max_width < x+ai->xa_w) {
			y+= button_ht;
			x= 0;
		}
		ai->xa_x= x+LXADEF_HMARGIN;
		ai->xa_y= y;
		x+= ai->xa_w+LXADEF_BUTTONGAP;
	}

	/* second pass -- space buttons evenly on each line */
	for (ai= alert_buttons; ai != (Alert_item *) NULL; ai= bi->xa_next) {
		for (bi= ai, n= 1; bi->xa_next != (Alert_item *) NULL; bi= bi->xa_next, n++) {
			if (bi->xa_next->xa_y > ai->xa_y)
				break;
		}
		if (n > 1) {
			diff= max_width-(bi->xa_x-LXADEF_HMARGIN+bi->xa_w);
			if (diff <= 0)
				continue;
			diff/= n-1;
			x= diff;
			for (ai= ai->xa_next; ai != bi->xa_next; ai= ai->xa_next) {
				ai->xa_x+= x;
				x+= diff;
			}
		}
	}
	if (alert_buttons != (Alert_item *) NULL) {
		y+= button_ht;
		alert_xwarp= alert_defbutton->xa_x+(alert_defbutton->xa_w/2);
		alert_ywarp= alert_defbutton->xa_y+(button_ht/2);
	}
	y+= LXADEF_VMARGIN;

	alert_w= max_width+(2*LXADEF_HMARGIN);
	alert_h= y;
	return;
}


int
alert_createitems()
/*
   Internal function.
   Create items.
*/
{
	Alert_item *ai;
	void alert_buttonproc();

	/* create label items */
	for (ai= alert_labels; ai != (Alert_item *) NULL; ai= ai->xa_next) {
		if ((ai->xa_pi= panelitem_create(alert_panel, LXPI_LABEL,
				LXPI_X, ai->xa_x, LXPI_Y, ai->xa_y,
				LXPI_STRING, ai->xa_string,
				LXPI_NULL)) == (Panel_item *) NULL)
			return(LX_ERROR);
	}

	/* create text items */
	for (ai= alert_texts; ai != (Alert_item *) NULL; ai= ai->xa_next) {
		if ((ai->xa_pi= panelitem_create(alert_panel, LXPI_TEXT,
				LXPI_X, ai->xa_x, LXPI_Y, ai->xa_y,
				LXPI_STRING, ai->xa_string,
				LXPI_CLIENTDATA, ai->xa_textvalptr,
				LXPTEXT_VALUE, ai->xa_textval,
				LXPTEXT_PROC, alert_kpproc,
				LXPTEXT_MAXSTORE, LXADEF_MAXSTORE,
				LXPTEXT_MAXDISPLAY, LXADEF_MAXDISPLAY,
				LXPI_NULL)) == (Panel_item *) NULL)
			return(LX_ERROR);
	}

	/* create buttons */
	for (ai= alert_buttons; ai != (Alert_item *) NULL; ai= ai->xa_next) {
		if ((ai->xa_pi= panelitem_create(alert_panel, LXPI_BUTTON,
				LXPI_X, ai->xa_x, LXPI_Y, ai->xa_y,
				LXPI_STRING, ai->xa_string,
				LXPI_PROC, alert_buttonproc,
				LXPI_CLIENTDATA, (char *) &(ai->xa_buttonval),
				LXPI_NULL)) == (Panel_item *) NULL)
			return(LX_ERROR);
	}
	return(LX_SUCCESS);
}

void
alert_destroylists()
/*
   Internal function.
   Destroy alert lists.
*/
{
	Alert_item *ai, *bi;

	for (ai= alert_labels; ai != (Alert_item *) NULL; ai= bi) {
		bi= ai->xa_next;
		if (ai->xa_string != (char *) NULL)
			cfree((char *) ai->xa_string);
		cfree((char *) ai);
	}
	for (ai= alert_texts; ai != (Alert_item *) NULL; ai= bi) {
		bi= ai->xa_next;
		if (ai->xa_string != (char *) NULL)
			cfree((char *) ai->xa_string);
		if (ai->xa_textval != (char *) NULL)
			cfree((char *) ai->xa_textval);
		cfree((char *) ai);
	}
	for (ai= alert_buttons; ai != (Alert_item *) NULL; ai= bi) {
		bi= ai->xa_next;
		if (ai->xa_string != (char *) NULL)
			cfree((char *) ai->xa_string);
		cfree((char *) ai);
	}
}

void
alert_buttonproc(p, pi)
/*
   Internal function.
   Procedure called upon all alert button invocations.
*/
Panel *p;
Panel_item *pi;
{
	Alert_item *ai;

	for (ai= alert_texts; ai != (Alert_item *) NULL; ai= ai->xa_next)
		(void) strcpy(ai->xa_textvalptr, (char *) panelitem_get(alert_panel, ai->xa_pi, LXPTEXT_VALUE));
	(void) panel_unblock(alert_panel, (int) *((int *) pi->xpi_clientdata));
	return;
}

int
alert_textproc(p, pi, str)
/*
   Internal function.
   Procedure called upon all alert text item key presses
   when there is only one text item -- causes the alert
   to return when '\n' is encountered.
*/
Panel *p;
Panel_item *pi;
char *str;
{
	Alert_item *ai;

	if (*str != xpkey_cret)
		return;

	for (ai= alert_texts; ai != (Alert_item *) NULL; ai= ai->xa_next)
		(void) strcpy(ai->xa_textvalptr, (char *) panelitem_get(alert_panel, ai->xa_pi, LXPTEXT_VALUE));
	(void) panel_unblock(alert_panel, 0);
	return;
}

Alert_item *
alert_labelalloc(string)
/*
   Internal function.
   Allocate and initialize a label Alert_item.
*/
char *string;
{
	Alert_item *ai, *new;

	if ((new= (Alert_item *) calloc(1, sizeof(Alert_item))) == (Alert_item *) NULL)
		return((Alert_item *) NULL);
	if ((new->xa_string= calloc((unsigned) (strlen(string)+1), sizeof(char))) == (char *) NULL) {
		cfree((char *) new);
		return((Alert_item *) NULL);
	}
	(void) strcpy(new->xa_string, string);
	new->xa_next= (Alert_item *) NULL;

	if (alert_labels == (Alert_item *) NULL)
		alert_labels= new;
	else {
		for (ai= alert_labels; ai->xa_next != (Alert_item *) NULL; ai= ai->xa_next);
		ai->xa_next= new;
	}
	return(new);
}

Alert_item *
alert_textalloc(prompt, value, valptr)
/*
   Internal function.
   Allocate and initialize a text Alert_item.
*/
char *prompt, *value, *valptr;
{
	Alert_item *ai, *new;

	if ((new= (Alert_item *) calloc(1, sizeof(Alert_item))) == (Alert_item *) NULL)
		return((Alert_item *) NULL);

	if (prompt == (char *) NULL)
		new->xa_string= (char *) NULL;
	else {
		if ((new->xa_string= calloc((unsigned) (strlen(prompt)+1), sizeof(char))) == (char *) NULL) {
			cfree((char *) new);
			return((Alert_item *) NULL);
		}
		(void) strcpy(new->xa_string, prompt);
	}

	if (value == (char *) NULL)
		new->xa_textval= (char *) NULL;
	else {
		if ((new->xa_textval= calloc((unsigned) (strlen(value)+1), sizeof(char))) == (char *) NULL) {
			cfree(new->xa_string);
			cfree((char *) new);
			return((Alert_item *) NULL);
		}
		(void) strcpy(new->xa_textval, value);
	}
	new->xa_textvalptr= valptr;
	new->xa_next= (Alert_item *) NULL;

	if (alert_texts == (Alert_item *) NULL)
		alert_texts= new;
	else {
		for (ai= alert_texts; ai->xa_next != (Alert_item *) NULL; ai= ai->xa_next);
		ai->xa_next= new;
	}
	return(new);
}

Alert_item *
alert_buttonalloc(string, val)
/*
   Internal function.
   Allocate and initialize a button Alert_item.
*/
char *string;
int val;
{
	Alert_item *ai, *new;

	if ((new= (Alert_item *) calloc(1, sizeof(Alert_item))) == (Alert_item *) NULL)
		return((Alert_item *) NULL);
	if ((new->xa_string= calloc((unsigned) (strlen(string)+1), sizeof(char))) == (char *) NULL) {
		cfree((char *) new);
		return((Alert_item *) NULL);
	}
	(void) strcpy(new->xa_string, string);
	new->xa_buttonval= val;
	new->xa_next= (Alert_item *) NULL;

	if (alert_buttons == (Alert_item *) NULL)
		alert_buttons= new;
	else {
		for (ai= alert_buttons; ai->xa_next != (Alert_item *) NULL; ai= ai->xa_next);
		ai->xa_next= new;
	}
	return(new);
}
