/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, I1.HR                           *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jul 92                                                   *
*  Last Update : Sep 92                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : timing.c                                                 *
*                                                                         *
*  Function    : Time Measurement for Benchmarking and so on              *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*                                                                         *
*     void dalib_clear_timer_ (n)                                         *
*     void dalib_start_timer_ (n)                                         *
*     void dalib_stop_timer_ (n)                                          *
*     void dalib_print_timer_ (n)                                         *
*                                                                         *
*     void dalib_walltime (t)                                             *
*     float *t;                                                           *
*                                                                         *
**************************************************************************/

#if defined(OS2)
#include <time.h>
#endif
#include <sys/types.h>
#include <sys/times.h>
#include <sys/time.h>
#if defined(OS2)
#include <sys/param.h>
#endif

/* undef   MULTI for timing1.c,  single node,
   define  MULTI for timing.c,   many nodes    */

#define MULTI

#ifdef MULTI
#include "system.h"
#endif

#ifndef OS2
#define HZ 60
#endif
#define TIMERS 10

#if defined(ALLIANT)
extern struct hrcval hrcounter;
#endif

#if defined(IPB)
#include <cm/cmmd.h>
#endif

/**************************************************************************
*                                                                         *
*  struct tms {                                                           *
*          clock_t tms_utime;              / user time /                  *
*          clock_t tms_stime;              / system time /                *
*          clock_t tms_cutime;             / user time, children /        *
*          clock_t tms_cstime;             / system time, children /      *
*  }                                                                      *
**************************************************************************/

          /*************************************************
          *                                                *
          *  values for each TIMER                         *
          *                                                *
          *   starttime [timer]                            *
          *   stoptime  [timer]                            *
          *   ms_start  [timer]                            *
          *   ms_stop   [timer]                            *
          *                                                *
          *   usertime  [timer]   user time of timer       *
          *   systemtime[timer]   system time of timer     *
          *   mean_ms   [timer]   mean time of timer       *
          *                                                *
          *************************************************/

struct tms starttime[TIMERS], stoptime[TIMERS];
long   ms_start[TIMERS], ms_stop[TIMERS];

long start_walltime;                         /* walltime function */

#if defined(IPSC)
double first_time;
#endif

float usertime[TIMERS], systemtime[TIMERS];
long mean_ms[TIMERS];                        /* measured in milliseconds */

#ifdef MULTI
void time_synchronization ()

{ int x, sender, op;

  /* synchronize all processors */
  x = 0;
  if (target_model == 0) /* WITH HOST */
     sender = 0;
   else
     sender = 1;
#if defined(GC)
  process_broadcast (&x, 1, sender);  /* cannot send 0 bytes */
#else
  process_broadcast (&x, 0, sender);
#endif
  op = 7;
  dalib_reduction__ (&x, &op);           /* int_sum */
}
#endif

          /*************************************************
          *                                                *
          *  long dalib_timestamp ()                       *
          *                                                *
          *    - get a time stamp of current walltime      *
          *                                                *
          *  counts tick, tick is usually every 10 us      *
          *                                                *
          *  machine dependent implementation              *
          *                                                *
          *************************************************/

/* general function to get a time stamp */

#if defined(SUN4) || defined(IBM) || defined(SRM) || defined(KSR1) || defined(SGI) || defined(OS2)
long dalib_timestamp ()
{ long t;
  struct timeval tp;
  struct timezone tzp;
  (void) gettimeofday(&tp,&tzp);
  t = tp.tv_sec * 100000 + tp.tv_usec / 10;
  return (t);
}
#endif

#if defined(ALLIANT)
long dalib_timestamp ()
{ long t;
  t = hrcounter.uuu.ddd.d_hv_low;  /* tick every 10 us */
  return (t);
}
#endif

#if defined(IPSC)
long dalib_timestamp ()
{ long t;
  double d, dclock_ ();

  d = dclock_() - first_time;
  t = d * ((double) 100000.0);  /* tick every 10 us */
  return (t);
}
#endif

#if defined(HMEIKO)
long dalib_timestamp ()
{ long t;
  t = MEIKO_time((void *)NULL)/100000;
  return (t);
}
#endif

#if defined(MEIKO)
long dalib_timestamp ()
{ long t;
  double dzero, dsecnds_();
  dzero = 0.0;
  d = dsecnds_(&dzero);
  t = d * ((double) 100000.0);  /* tick every 10 us */
  return (t);
}
#endif

#if defined(GC)
long dalib_timestamp ()
{ long t;
  t = TimeNowLow ();               /* tick every 64 us */
  return (t);
}
#endif

#if defined(IPB)
long dalib_timestamp ()
{ long t;
  double d;
  CMMD_node_timer_stop (0);
  d = CMMD_node_timer_elapsed (0);
  CMMD_node_timer_start (0);
  t = d * 100000.0;                 /* tick for 10 us */
  return (t);
}
#endif

          /*************************************************
          *                                                *
          *  void dalib_init_walltime ()                   *
          *                                                *
          *************************************************/

void dalib_init_walltime ()

{ /* walltime is measured in multiple of 10 microseconds */

#if defined(IPSC)
  double dclock_ ();
  first_time = dclock_ ();
#endif
#if defined(IPB)
  CMMD_node_timer_clear (0);
#endif
  start_walltime = dalib_timestamp ();
}

          /*************************************************
          *                                                *
          *  void dalib_clear_timer   ()                   *
          *                                                *
          *************************************************/

void dalib_clear_timer__ (timer)
int *timer;
{
  int tim;
  tim = *timer; if (tim < 0) tim = 0; if (tim>=TIMERS) tim = 0;

  mean_ms   [tim] = 0;
  usertime  [tim] = 0.0;
  systemtime[tim] = 0.0;
}

          /*************************************************
          *                                                *
          *  void dalib_start_timer   ()                   *
          *                                                *
          *************************************************/

void dalib_start_timer__ (timer)
int *timer;
{
  int tim;
	
  tim = *timer; if (tim < 0) tim = 0; if (tim>=TIMERS) tim = 0;

#ifdef MULTI
  time_synchronization ();
#endif

#if defined(IPB)
  starttime[tim].tms_utime = 0;
  starttime[tim].tms_stime = 0;
#else
  times (starttime+tim);
#endif

  ms_start[tim] = dalib_timestamp ();
}

          /*************************************************
          *                                                *
          *  void dalib_stop_timer   ()                    *
          *                                                *
          *  - adds the new values                         *
          *                                                *
          *************************************************/

void dalib_stop_timer__ (timer)
int *timer;

{ float h;
  int tim;

  tim = *timer; if (tim < 0) tim = 0; if (tim>=TIMERS) tim = 0;

#ifdef MULTI
  time_synchronization ();
#endif

  /* stop timer */
#if defined(IPB)
  stoptime[tim].tms_utime = 0;
  stoptime[tim].tms_stime = 0;
#else
  times (stoptime+tim);
#endif

  ms_stop[tim] = dalib_timestamp ();

  /* add user time */
  h = stoptime[tim].tms_utime - starttime[tim].tms_utime;
  h = h / HZ;
  usertime[tim] += h;

  /* add system time */
  h = stoptime[tim].tms_stime - starttime[tim].tms_stime;
  h = h / HZ;
  systemtime[tim] += h;

  /* add whole time */
  mean_ms[tim]    += (ms_stop[tim] - ms_start[tim]);
}

          /*************************************************
          *                                                *
          *  void dalib_print_timer   ()                   *
          *                                                *
          *************************************************/

void dalib_print_timer__ (timer)
int *timer;
{
 int tim, printer;
 long milli_seconds;
 tim = *timer; if (tim < 0) tim = 0; if (tim>=TIMERS) tim = 0;

#if defined(ALLIANT) || defined(SUN4) || defined(IBM) || defined(SRM) || defined(IPSC) || defined (MEIKO) || defined (KSR1) || defined (SGI) || defined (IPB) || defined(OS2)
  milli_seconds = mean_ms[tim] / 100;
#endif
#if defined(GC)
  milli_seconds = mean_ms[tim] * 1000 / CLK_TCK_LOW;
#endif


#ifdef MULTI
 if (target_model == 0) /* WITH HOST */
    printer = 0;
  else
    printer = 1;

 if (pcb.i == printer)
  printf ("Timer %d: User time: %f, System time: %f, Mean time: %d ms\n",
           tim, usertime[tim], systemtime[tim], milli_seconds);
#else
  printf ("Timer %d: User time: %f, System time: %f, Mean time: %d ms\n",
           tim, usertime[tim], systemtime[tim], milli_seconds);
#endif
}

          /*************************************************
          *                                                *
          *  void dalib_walltime   ()                      *
          *                                                *
          *************************************************/

void dalib_walltime_ (t)
float *t;

{ long us;

  int from, length;

#ifdef MULTI
  time_synchronization ();
#endif

  us = dalib_timestamp ();
  us -= start_walltime;

  /* justify to seconds */

#if defined(IPSC) || defined (SRM) || defined (MEIKO) || defined (IPB)
  *t = ((float) us ) / 100000.0;
#endif
#if defined(ALLIANT) || defined(SUN4) || defined(IBM) || defined(SRM) || defined (KSR1) || defined (SGI) || defined(OS2)
  *t = ((float) us ) / 100000.0;
#endif
#if defined(GC)
  *t = ((float) us ) / ((float) CLK_TCK_LOW);
#endif

  /* it might be very important that all processors have the same time */

#ifdef MULTI
  from = target_model;
#if defined(SRM) || defined(IPSC) || defined(MEIKO) || defined(HMEIKO) || defined(IPB)
  /* node has only accurate timer */
  from = 1;
#endif
  length = 4;
#if defined(KSR1)
  length = 8;   /* float has length 8 on KSR machines */
#endif
  dalib_broadcast__ (t, &length, &from);
#endif
}
