
/* Functions for accessing data from the GCC2 environment variable */

const char *getenv (const char *);
char *strdup (const char *);
char *strtok (char *, const char *);
char *strchr (const char *, char);

typedef struct _gcc2_info gcc2_info;
struct _gcc2_info
{
  gcc2_info *next;
  const char *option_name;
  const char *option;
  const char *unixish;
};

static gcc2_info *info_list = 0;

static void init_gcc2()
{
  static int init = 0;
  const char *env;
  char *gcc2, *p, *tok;
  gcc2_info *info;

  /* Init only once */
  if (init)
    return;
  init = 1;

  /* Copy the gcc2 env variable into a new data area */
  env = getenv("GCC2");
  if (env == 0)
    return;
  gcc2 = strdup(env);
  if (gcc2 == 0)
    return;

  /* Deblank the environment variable */
  while (isspace(*gcc2))
    gcc2++;
  if (*gcc2 == '\0')
    return;
  p = gcc2 + strlen(gcc2) - 1;
  while (isspace(*p))
    {
      *p = '\0';
      p--;
    }

  /* Split the variable into it's subparts */
  for (tok = strtok (gcc2, ","); tok != 0; tok = strtok ((void *)0, ","))
    {
      if (tok[0] == '\0' || tok[1] == '\0' || tok[2] == '\0' || tok[0] == ':')
	continue; /* Illegal, so ignore it */

      if (tok[1] == ':')
	{
	  /* We have a path variable */
	  info = (gcc2_info *) malloc (sizeof (gcc2_info));
	  if (info == 0)
	    return;
	  info->unixish = strdup (tok);
	  if (info->unixish == 0)
	    return;
	  info->next = info_list;
	  info_list = info;
	  info->option_name = 0;
	  info->option = tok;
	  for (p = strchr (info->option, '/'); p != 0; p = strchr (p, '/'))
	    *p = '\\';
	  for (p = strchr (info->unixish, '\\'); p != 0; p = strchr (p, '\\'))
	    *p = '/';
	}
      else
	{
	  /* We have a variable declaration */
	  p = strchr(tok, ':');
	  if (p == 0)
	    continue; /* Illegal */
	  *p++ = '\0';
	  info = (gcc2_info *) malloc (sizeof (gcc2_info));
	  if (info == 0)
	    return;
	  info->next = info_list;
	  info_list = info;
	  info->option_name = tok;
	  info->option = p;
	}
    }
}

const char *_gcc2_getpath()
{
  gcc2_info *info;
  init_gcc2();
  for (info = info_list; info != 0; info = info->next)
    if (info->option_name == 0)
      return info->option;
  return 0;
}

const char *_gcc2_get_unixish_path()
{
  gcc2_info *info;
  init_gcc2();
  for (info = info_list; info != 0; info = info->next)
    if (info->option_name == 0)
      return info->unixish;
  return 0;
}

const char *_gcc2_getoption(const char *option_name)
{
  gcc2_info *info;
  init_gcc2();
  for (info = info_list; info != 0; info = info->next)
    if (info->option_name != 0 && !strcmp (option_name, info->option_name))
      return info->option;
  return 0;
}

void _gcc2_getargv(const char *option_name, int *argc, const char ***argv)
{
  const char *option = _gcc2_getoption (option_name);
  char *optdup = 0;
  int alloc;
  char *tok;

  if (option == 0)
    goto err;
  optdup = strdup (option);
  if (optdup == 0)
    goto err;
    
  alloc = 16;
  *argv = (const char **) malloc (alloc * sizeof (char *));
  if (*argv == 0)
    goto err;
  *argc = 0;
  for (tok = strtok (optdup, " \t"); tok != 0; tok = strtok ((void *)0, " \t"))
    {
      if (*argc >= alloc-2)
	{
	  alloc += 16;
	  *argv = (const char **) realloc (*argv, alloc * sizeof (char *));
	  if (*argv == 0)
	    goto err;
	}
      tok = strdup (tok);
      if (tok == 0)
	goto err;
      (*argv)[(*argc)++] = tok;
    }
  (*argv)[*argc] = 0;

  /* Cleanup */
  free (optdup);
  return;
 err:
  if (optdup)
    free (optdup);
  *argc = 0;
  *argv = 0;
}

static void merge_args (int *argc, const char ***argv, 
			int argc1, const char **argv1,
			int argc2, const char **argv2)
{
  int i;
  *argv = (const char **) malloc ((argc1 + argc2 + 1) * sizeof (char *));
  *argc = 0;
  if (*argv == 0)
    return;
  for (i=0; i < argc1; i++)
    (*argv)[(*argc)++] = argv1[i];
  for (i=0; i < argc2; i++)
    (*argv)[(*argc)++] = argv2[i];
  (*argv)[*argc] = 0;
}

void _gcc2_append_argv (const char *option_name, 
			int *argc, const char ***argv)
{
  const char **env_argv, **nargv;
  int env_argc, nargc;
  _gcc2_getargv (option_name, &env_argc, &env_argv);
  if (env_argv == 0)
    return;
  merge_args (&nargc, &nargv, *argc, *argv, env_argc, env_argv);
  if (nargv == 0)
    return;
  *argc = nargc;
  *argv = nargv;
}

void _insert_env (const char *env, const char *prepend, char separator)
{
  const char *value = getenv (env);
  if (value == 0)
    {
      char newenv[strlen (env) + 1 + strlen (prepend) + 1];
      sprintf (newenv, "%s=%s", env, prepend);
      putenv (newenv);
    }
  else
    {
      char newenv[strlen (env) + 1 + strlen (prepend) + 1 + strlen (value)
		  + 1];
      sprintf (newenv, "%s=%s%c%s", env, prepend, separator, value);
      putenv (newenv);
    }
}
