// <><><><><><><><><><><><><>  Plot3D.cpp  <><><><><><><><><><><><><><>
//
// ----------------------------------------------------
// (C) Copyright Clark Dailey, Inc. 1999
//
// C++ Class Name:
//    Plot3D
//
// Purpose:
//    Used as a base class for several Plot3D classes
//
// ----------------------------------------------------

#include "stdafx.h"	// precompiled header
#include "plot3d.h"

// ------------------------------------------------------------------------------------------
Plot3D::Plot3D()	// constructor
{
	// default axis info
	m_show_axis = 0;	// no axis
	memset((char*)&m_axis,0,sizeof(m_axis));
	m_axis.nxticks = 3;
	m_axis.nyticks = 3;
	m_axis.nzticks = 3;
	m_axis.axis_color = RGB(0,0,0);	// black
	strcpy(m_axis.font_name,"Arial"); // needs to be TrueType
	m_axis.font_height = 12;
	m_axis.font_width = 7;
	m_axis.tick_length = 20;
	m_axis.tick_spacing = 20;

	// set to defaults
	SetRanges();
	SetViewingPosition();
	SetVideoRes();
	PlotOutputOn();
	CalcTM();
	CalcCorners();

	m_abort_plotting = 0;
	m_pending_moveto = 0;
	m_xv_cur = 0;
	m_yv_cur = 0;
	m_color_cur = 1;
}

// ------------------------------------------------------------------------------------------
Plot3D::~Plot3D()	// destructor
{
}

// ------------------------------------------------------------------------------------------
// settings
int Plot3D::SetRanges(double xmin,double xmax,double ymin,double ymax,double zmin,double zmax)
{
	int rc=0;
	if (xmin>xmax || ymin>ymax || zmin>zmax) return(1);
	m_xmin = xmin;
	m_xmax = xmax;
	m_ymin = ymin;
	m_ymax = ymax;
	m_zmin = zmin;
	m_zmax = zmax;
	m_xmid = (m_xmin + m_xmax)/2.;
	m_ymid = (m_ymin + m_ymax)/2.;
	m_zmid = (m_zmin + m_zmax)/2.;
	return(rc);
}

// ------------------------------------------------------------------------------------------
// angles in degrees
int Plot3D::SetViewingPosition(double xytilt,double zrotate)
{
	m_xytilt = xytilt;
	m_zrotate = zrotate;
	return(0);
}

// ------------------------------------------------------------------------------------------
int Plot3D::SetVideoRes(int xvmin,int yvmin,int xvmax,int yvmax,double scale)
{
	if (xvmin>=xvmax || yvmin>=yvmax) return(1);
	if (scale>1.0) scale=1.0;
	if (scale<.01) scale=.01;
	m_xvmin = xvmin;
	m_yvmin = yvmin;
	m_xvmax = xvmax;
	m_yvmax = yvmax;
	m_xvmid = (m_xvmin + m_xvmax)/2;
	m_yvmid = (m_yvmin + m_yvmax)/2;
	m_scale = scale;
	return(0);
}

// ------------------------------------------------------------------------------------------
void Plot3D::SetAxisParameters(unsigned ShowAxis,AXIS_PLOT_INFO* pAxisInfo)
{
	m_show_axis  = ShowAxis;
	if (pAxisInfo==0) return;

	memcpy((char*)&m_axis,(char*)pAxisInfo,sizeof(m_axis));
}

// ------------------------------------------------------------------------------------------
void Plot3D::Dump()
{
	unsigned i;
	DBGPLOT("------------------ Plot3D Dump Begin -----------------------\n");
	DBGPLOT("Inputs:\n");
	DBGPLOT("  Xmin=%10.4f Xmax=%10.4f\n", m_xmin, m_xmax);
	DBGPLOT("  Ymin=%10.4f Ymax=%10.4f\n", m_ymin, m_ymax);
	DBGPLOT("  Zmin=%10.4f Zmax=%10.4f\n", m_zmin, m_zmax);
	DBGPLOT("  Xytilt=%10.4f Zrotate=%10.4f\n",m_xytilt, m_zrotate);
	DBGPLOT("  Xvmin=%d  Xvmax=%d\n", m_xvmin, m_xvmax);
	DBGPLOT("  Yvmin=%d  Yvmax=%d\n", m_yvmin, m_yvmax);
	DBGPLOT("  Scale=%10.4f\n",m_scale);

	DBGPLOT("Transformation Matrix:\n");
	DBGPLOT("  [0,0]=%10.4f  [0,1]=%10.4f  [0,2]=%10.4f\n", tm[0][0],tm[0][1],tm[0][2]);
	DBGPLOT("  [1,0]=%10.4f  [1,1]=%10.4f  [1,2]=%10.4f\n", tm[1][0],tm[1][1],tm[1][2]);
	DBGPLOT("  [2,0]=%10.4f  [2,1]=%10.4f  [2,2]=%10.4f\n", tm[2][0],tm[2][1],tm[2][2]);

	DBGPLOT("Mins/Mids/Maxs:\n");
	DBGPLOT("  xpmin=%10.4f xpmid=%10.4f xpmax=%10.4f\n", m_xpmin,m_xpmid,m_xpmax);
	DBGPLOT("  ypmin=%10.4f ypmid=%10.4f ypmax=%10.4f\n", m_ypmin,m_ypmid,m_ypmax);
	DBGPLOT("  zpmin=%10.4f zpmid=%10.4f zpmax=%10.4f\n", m_zpmin,m_zpmid,m_zpmax);
	DBGPLOT("  xvmid=%10d   yvmid=%10d   p2v  =%10.4f\n", m_xvmid,m_yvmid,m_p2v);

	DBGPLOT("Pen Location:\n");
	DBGPLOT("  Yv=%d  Yvx=%d  Color=%ld\n", m_xv_cur, m_yv_cur, m_color_cur);
	DBGPLOT("  PendMoveTo=%d  ToPlot=%d\n", m_pending_moveto, m_to_plotter);

	DBGPLOT("Corners:\n");
	for (i=0;i<8;i++)
	{
		DBGPLOT("  [%d] xp=%10.4f  yp=%10.4f  zp=%10.4f\n", i,m_xpcorner[i],m_ypcorner[i],m_zpcorner[i]);
	}
	DBGPLOT("------------------- Plot3D Dump End ------------------------\n");
}

// ------------------------------------------------------------------------------------------
// calculate transformation matrix
void Plot3D::CalcTM()
{
	double	zrotate,sin_zrotate,cos_zrotate;
	double	xytilt,sin_xytilt,cos_xytilt;

	zrotate = Deg2Rad(m_zrotate);
	xytilt  = Deg2Rad(m_xytilt);
	sin_zrotate = sin(zrotate);
	cos_zrotate = cos(zrotate);
	sin_xytilt  = sin(xytilt);
	cos_xytilt  = cos(xytilt);

	tm[0][0] =  cos_zrotate;
	tm[0][1] =  sin_zrotate;
	tm[0][2] =  0;
	tm[1][0] = -sin_zrotate * cos_xytilt;
	tm[1][1] =  cos_zrotate * cos_xytilt;
	tm[1][2] =                sin_xytilt;
	tm[2][0] =  sin_zrotate * sin_xytilt;
	tm[2][1] = -cos_zrotate * sin_xytilt;
	tm[2][2] =                cos_xytilt;
} // CalcTM

// ------------------------------------------------------------------------------------------
// calculate projections of corners
void Plot3D::CalcCorners()
{
	double xc[2], yc[2], zc[2];
	double x,y,z;
	unsigned ix,iy,iz,ic;

	xc[0] = m_xmin;
	xc[1] = m_xmax;
	yc[0] = m_ymin;
	yc[1] = m_ymax;
	zc[0] = m_zmin;
	zc[1] = m_zmax;

	for (ic=0,iz=0;iz<2;iz++)
	{
		z = zc[iz];
		for (iy=0;iy<2;iy++)
		{
			y = yc[iy];
			for (ix=0;ix<2;ix++)
			{
				// corner values
				m_xcorner[ic] = x = xc[ix];
				m_ycorner[ic] = y;
				m_zcorner[ic] = z;
				// projection of corners
				m_xpcorner[ic] = xpf(x,y,z);
				m_ypcorner[ic] = ypf(x,y,z);
				m_zpcorner[ic] = zpf(x,y,z);
				ic++;
			} // for ix
		} // for iy
	} // for iz

	// find min,max plot coordinates 
	m_xpmin = m_xpmax = m_xpcorner[0];
	m_ypmin = m_ypmax = m_ypcorner[0];
	m_zpmin = m_zpmax = m_zpcorner[0];
	for (unsigned i=1;i<8;i++)
	{
		if (m_xpcorner[i]<m_xpmin) m_xpmin = m_xpcorner[i];
		if (m_xpcorner[i]>m_xpmax) m_xpmax = m_xpcorner[i];
		if (m_ypcorner[i]<m_ypmin) m_ypmin = m_ypcorner[i];
		if (m_ypcorner[i]>m_ypmax) m_ypmax = m_ypcorner[i];
		if (m_zpcorner[i]<m_zpmin) m_zpmin = m_zpcorner[i];
		if (m_zpcorner[i]>m_zpmax) m_zpmax = m_zpcorner[i];
	}
	m_xpmid = (m_xpmin + m_xpmax)/2.;
	m_ypmid = (m_ypmin + m_ypmax)/2.;
	m_zpmid = (m_zpmin + m_zpmax)/2.;

	// calc plot to video scaling
	if (m_xpmax == m_xpmin)
		m_p2v = 1.;	// avoid division by zero
	else
		m_p2v = (m_scale * (double)(m_xvmax - m_xvmin)) / (double)(m_xpmax - m_xpmin);
} // CalcCorners

// ------------------------------------------------------------------------------------------
void Plot3D::FilterMoveTo(int xv,int yv)
{
	if (m_xv_cur==xv && m_yv_cur==yv) return; // already there
	if (m_to_plotter)
		MoveTo(xv,yv);
	else
		m_pending_moveto = 1;
	m_xv_cur = xv;
	m_yv_cur = yv;
} // FilterMoveTo

// ------------------------------------------------------------------------------------------
void Plot3D::FilterDrawTo(int xv,int yv)
{
	if (m_xv_cur==xv && m_yv_cur==yv) return; // already there
	if (m_to_plotter)
	{
		if (m_pending_moveto) MoveTo(m_xv_cur,m_yv_cur);
		m_pending_moveto = 0;
		DrawTo(xv,yv);
	}
	m_xv_cur = xv;
	m_yv_cur = yv;
} // FilterDrawTo

// ------------------------------------------------------------------------------------------
void Plot3D::FilterColorTo(long color)
{
	if (color==m_color_cur) return; // already got this color
	if (m_to_plotter) ColorTo(color);
	m_color_cur = color;
} // FilterColorTo

// ------------------------------------------------------------
void Plot3D::MapPoint(double x,double y,double z,int* xv,int* yv)
{
	double	xp,yp;
	// Note: invert z because of plot inversion
	xp = xpf( x, y, z);
	yp = ypf( x, y, z);
	*xv = xvf(xp);
	*yv = yvf(yp);
}

// ------------------------------------------------------------
void Plot3D::MapAndDrawLine(double x1,double y1,double z1,double x2,double y2,double z2)
{
	int	xv1,yv1,xv2,yv2;

	// Map points
	MapPoint(x1,y1,z1,&xv1,&yv1);
	MapPoint(x2,y2,z2,&xv2,&yv2);

	// draw the line
	FilterMoveTo(xv1,yv1);
	FilterDrawTo(xv2,yv2);
}

/*	<><><><><><><><><><><><><>  Plot3D.cpp  <><><><><><><><><><><><><><> */



