#region (C) Wout Ware 2005
//
// File: DxfWriteTest.cs
// Author: Wout de Zeeuw
// Creation: 1-10-2004
//
// (c) 2005 Wout Ware All Rights Reserved.
//
#endregion

using System;
using System.Drawing;
using System.IO;

using CadLib.Base;
using CadLib.IO;
using CadLib.Model;
using CadLib.Model.Entities;
using CadLib.Model.Tables;

using Sharp3D.Math.Core;

namespace DxfWriteTest
{
	class DxfWriteTest
	{
		/// <summary>
		/// Example of how to create a line, circle, text (DXF release 12).
		/// </summary>
		private static void WriteTestR12() {
			CadData data = new CadData(DxfVersion.Dxf12);

			// Line.
			Line line1 = new Line(new Vector3D(1d, 2d, 3d), new Vector3D(4d, 2.1d, 3d));
			data.Entities.Add(line1);

			// Circle.
			Circle circle = new Circle(new Vector3D(1d, 2d, 3d), 3d);
			circle.Color = Color.Green;
			data.Entities.Add(circle);
			
			// Invisible blue circle.
			Circle circle2 = new Circle(new Vector3D(3d, 2d, 3d), 4d);
			circle2.Color = Color.Blue;
			circle2.Visible = false;
			data.Entities.Add(circle2);
			
			// Extruded circle with line type.
			LineType lineType = 
				new LineType(
					"SpecialLineType", 
					"description", 
					new double[] {0.5d, -0.3d, 0, -0.3d}
				);
			data.LineTypes.Add(lineType);
			Circle circle3 = new Circle(new Vector3D(3d, 2d, 3d), 3d);
			circle3.Color = Color.Red;
			circle3.LineType = lineType;
			circle3.Thickness = 0.4;
			data.Entities.Add(circle3);
			
			// Text.
			data.Entities.Add(new Text(data, "Wout Ware", Vector3D.Zero, 1d));
			
			// Extruded text.
			Text text2 = new Text(data, "Wout Ware", new Vector3D(0d, 2d, 0d), 1d);
			text2.Thickness = 0.4d;
			data.Entities.Add(text2);
			
			// Text with custom text style.
			TextStyle textStyle = new TextStyle("MYSTYLE", "arial.ttf");
			data.TextStyles.Add(textStyle);
			Text text3 = new Text(data, "Wout Ware", new Vector3D(0d, 4d, 0d), 1d);
			text3.Thickness = 0.4d;
			text3.Style = textStyle;
			data.Entities.Add(text3);

			DxfWriter.WriteDxf("DxfWriteTest-R12-ascii.dxf", data, false);
			DxfWriter.WriteDxf("DxfWriteTest-R12-bin.dxf", data, true);
		}
		
		/// <summary>
		/// Example of how to create blocks, inserts, multiline text (DXF release 13).
		/// </summary>
		private static void WriteTestR13() {
			CadData data = new CadData(DxfVersion.Dxf13);

			Layer layer1 = new Layer(data, "LAYER1");
			data.Layers.Add(layer1);
			
			Block block1 = new Block(data, "TESTBLOCK1");

			Text text1 = new Text(data, "Text in block 1", Vector3D.Zero, 0.1d);
			block1.Entities.Add(text1);
			
			Polyline3D polyline1 = new Polyline3D();
			polyline1.Vertices.AddRange(
				new Vertex3D[] {
					new Vertex3D(-0.05d, 0.05d, 0d),
					new Vertex3D(0d, -0.02d, 0d),
					new Vertex3D(1.4d, -0.02d, 0d),
					new Vertex3D(1.45d, 0.05d, 0d),
					new Vertex3D(1.4d, 0.12d, 0d),
					new Vertex3D(0d, 0.12d, 0d)
			    }
			);
			polyline1.Closed = true;
			block1.Entities.Add(polyline1);

			block1.Layer = layer1;
			data.Blocks.Add(block1);
			
			Insert insert1 = new Insert();
			insert1.Layer = layer1;
			insert1.Block = block1;
			
			data.Entities.Add(insert1);
			
			Layer layer2 = new Layer(data, "LAYER2");
			layer2.Color = Color.LawnGreen;
			data.Layers.Add(layer2);
			
			Block block2 = new Block(data, "TESTBLOCK2");

			MText mtext = new MText(
				data, 
				@"Multiline 1 (block color), \P\C001;Multiline 2 (red)", 
				new Vector3D(0d, 0.3d, 0d), 
				0.03d
			);
			block2.Entities.Add(mtext);
			
			Polyline3D polyline2 = new Polyline3D();
			polyline2.Vertices.AddRange(
				new Vertex3D[] {
					new Vertex3D(-0.05d, 0.25d, 0d),
					new Vertex3D(0d, 0.18d, 0d),
					new Vertex3D(1.4d, 0.18d, 0d),
					new Vertex3D(1.45d, 0.25d, 0d),
					new Vertex3D(1.4d, 0.32d, 0d),
					new Vertex3D(0d, 0.32d, 0d)
                }
            );
			polyline2.Closed = true;
			block2.Entities.Add(polyline2);

			block2.Layer = layer2;
			data.Blocks.Add(block2);
			
			Insert insert2 = new Insert();
			insert2.ColorSource = AttributeSource.Layer;
			insert2.Layer = layer2;
			insert2.Block = block2;
			insert2.Rows = 3;
			insert2.RowSpacing = 0.3;
			insert2.Columns = 2;
			insert2.ColumnSpacing = 1.8;
			
			data.Entities.Add(insert2);
			
			DxfWriter.WriteDxf("DxfWriteTest-R13-ascii.dxf", data, false);
			DxfWriter.WriteDxf("DxfWriteTest-R13-bin.dxf", data, true);
		}

		/// <summary>
		/// Example of how to create a hatch (DXF release 14).
		/// </summary>
		private static void WriteTestR14() {
			CadData data = new CadData(DxfVersion.Dxf14);

			// Create an artistic hatch.
			Hatch hatch = new Hatch();
			hatch.Color = Color.Green;
			hatch.ElevationPoint = new Vector3D(0, 0, 2);
			hatch.ZAxis = new Vector3D(-0.707, 0, 0.707);

			// A boundary path bounded by lines.
			BoundaryPath boundaryPath1 = new BoundaryPath();
			boundaryPath1.Type = BoundaryPathType.Default;
			hatch.BoundaryPaths.Add(boundaryPath1);
			boundaryPath1.Edges.Add(new BoundaryPath.LineEdge(new Vector2D(0, 0), new Vector2D(1, 0)));
			boundaryPath1.Edges.Add(new BoundaryPath.LineEdge(new Vector2D(1, 0), new Vector2D(1, 1)));
			boundaryPath1.Edges.Add(new BoundaryPath.LineEdge(new Vector2D(1, 1), new Vector2D(0, 1)));
			boundaryPath1.Edges.Add(new BoundaryPath.LineEdge(new Vector2D(0, 1), new Vector2D(0, 0)));

			// A boundary path bounded by an ellipse.
			BoundaryPath boundaryPath2 = new BoundaryPath();
			boundaryPath2.Type = BoundaryPathType.Default;
			hatch.BoundaryPaths.Add(boundaryPath2);
			BoundaryPath.EllipseEdge edge = new BoundaryPath.EllipseEdge();
			edge.CounterClockWise = true;
			edge.Center = new Vector2D(1, 1);
			edge.MajorAxisEndPoint = new Vector2D(0.4d, -0.2d);
			edge.MinorToMajorRatio = 0.7;
			edge.StartAngle = 0d;
			edge.EndAngle = Math.PI * 2d / 3d;
			boundaryPath2.Edges.Add(edge);
			
			// A boundary path bounded by lines and an arc.
			BoundaryPath boundaryPath3 = new BoundaryPath();
			boundaryPath3.Type = BoundaryPathType.Outermost;
			hatch.BoundaryPaths.Add(boundaryPath3);
			BoundaryPath.ArcEdge arcEdge = new BoundaryPath.ArcEdge();
			arcEdge.Center = new Vector2D(0, 1);
			arcEdge.Radius = 0.5d;
			arcEdge.StartAngle = 0;
			arcEdge.EndAngle = Math.PI / 2d;
			arcEdge.CounterClockWise = true;
			boundaryPath3.Edges.Add(arcEdge);
			boundaryPath3.Edges.Add(new BoundaryPath.LineEdge(new Vector2D(0, 1.5d), new Vector2D(-0.5, 1d)));
			boundaryPath3.Edges.Add(new BoundaryPath.LineEdge(new Vector2D(-0.5, 1d), new Vector2D(0d, 0.5d)));
			boundaryPath3.Edges.Add(new BoundaryPath.LineEdge(new Vector2D(0d, 0.5d), new Vector2D(0.5d, 1d)));
			
			// A boundary path bounded by a polyline.
			BoundaryPath boundaryPath6 = new BoundaryPath();
			boundaryPath6.Type = BoundaryPathType.Polyline;
			hatch.BoundaryPaths.Add(boundaryPath6);
			boundaryPath6.PolylineData =
				new BoundaryPath.Polyline(
					new Vector2DWithBulge[] {
						new Vector2DWithBulge(0.5, -0.5),
						new Vector2DWithBulge(0.5, 0.5),
						new Vector2DWithBulge(1.5, 0.5),
						new Vector2DWithBulge(1.5, 0-.25),
						new Vector2DWithBulge(0.75, -0.25),
						new Vector2DWithBulge(0.75, 0.25),
						new Vector2DWithBulge(1.25, 0.25),
						new Vector2DWithBulge(1.25, -0.5)
					}
				);
			boundaryPath6.PolylineData.Closed = true;

			// Define the hatch fill pattern.
			Pattern pattern = new Pattern();
			PatternLine patternLine = new PatternLine();
			pattern.Lines.Add(patternLine);
			patternLine.Angle = Math.PI / 4d;
			patternLine.Offset = new Vector2D(0.02, -0.01d);
			patternLine.DashLengths.Add(0.02d);
			patternLine.DashLengths.Add(-0.01d);
			patternLine.DashLengths.Add(0d);
			patternLine.DashLengths.Add(-0.01d);
			hatch.Pattern = pattern;
			
			data.Entities.Add(hatch);
			
			DxfWriter.WriteDxf("DxfWriteTest-R14-ascii.dxf", data, false);
			DxfWriter.WriteDxf("DxfWriteTest-R14-bin.dxf", data, true);
		}

		public static void Main(string[] args)
		{
			try {
				WriteTestR12();
				WriteTestR13();
				WriteTestR14();
			} catch (Exception e) {
				Console.Error.WriteLine("Error occurred: " + e.Message);
				Environment.Exit(1);
			}
		}
	}
}
