/*******************  start of original comments  ********************/
/*
 * Written by Douglas Thomson (1989/1990)
 *
 * This source code is released into the public domain.
 */

/*
 * Name:    hardware independent screen IO module
 * Purpose: This file contains the code to interface the rest of the
 *           editor to the display and input hardware.
 * File:    hwind.c
 * Author:  Douglas Thomson
 * System:  this file is intended to be system-independent
 * Date:    October 2, 1989
 * Notes:   This is the only module that is allowed to call the hardware
 *           dependent display IO library.
 *          Typically, functions here check whether any action is
 *           necessary (for example, the cursor may already happen to be
 *           in the required position), call hardware dependent functions
 *           to achieve the required effect, and finally update status
 *           information about the current state of the terminal display.
 *          The idea behind this approach is to keep the hardware
 *           dependent code as small and simple as possible, thus making
 *           porting the code easier.
 */
/*********************  end of original comments   ********************/


/*
 * Some routines were added to display current editor modes in the lite bar
 * at the bottom of the screen. Other routines were rewritten in assembly.
 * I feel the need for speed.
 *
 * New editor name:  tde, the Thomson-Davis Editor.
 * Author:           Frank Davis
 * Date:             June 5, 1991
 *
 * This modification of Douglas Thomson's code is released into the
 * public domain, Frank Davis.  You may distribute it freely.
 */

#include <bios.h>

#include "tdestr.h"
#include "common.h"
#include "tdefunc.h"

/*
 * Name:    xygoto
 * Purpose: To move the cursor to the required column and line.
 * Date:    September 28, 1991
 * Passed:  col:    desired column (0 up to max)
 *          line:   desired line (0 up to max)
 */
void xygoto( int col, int line )
{
union REGS inregs, outregs;

   inregs.h.ah = 2;
   inregs.h.bh = 0;
   inregs.h.dh = line;
   inregs.h.dl = col;
   int86( 0x10, &inregs, &outregs );
}

/*
 * Name:    save_screen_line
 * Purpose: To save the characters and attributes of a line on screen.
 * Date:    June 5, 1991
 * Notes:   No need to redraw entire screen to show a prompt.  Just save
 *          the contents of the line on screen where prompt is to be displayed
 */
void save_screen_line( int col, int line, char *screen_buffer )
{
char far *p;

   p = g_display.display_address + line * 160 + col * 2;
   _fmemcpy( screen_buffer, p, 160 );
}

/*
 * Name:    restore_screen_line
 * Purpose: To restore the characters and attributes of a line on screen.
 * Date:    June 5, 1991
 * Notes:   No need to redraw entire screen to show a prompt.  Just save
 *          the contents of the line on screen where prompt is to be displayed
 */
void restore_screen_line( int col, int line, char *screen_buffer )
{
char far *p;

   p = g_display.display_address + line * 160 + col * 2;
   _fmemcpy( p, screen_buffer, 160 );
}

/*
 * Name:    cls
 * Purpose: clear screen
 * Date:    June 5, 1991
 * Notes:   Call the video BIOS routine to clear the screen.
 */
void cls( void )
{
int line;

   line = g_display.nlines+1;
   _asm {
        xor     ch, ch                  ; starting row in ch = 0
        xor     cl, cl                  ; starting column in cl = 0
        mov     ax, WORD PTR line       ; get ending row
        mov     dh, al                  ; put it in dh
        mov     dl, 79                  ; ending column in dl = 79
        mov     bh, 7                   ; attribute in bh  = 7 (normal)
        mov     al, 0                   ; get number of lines
        mov     ah, 6                   ; get function number
        push    bp                      ; some dos versions wipe out bp
        int     0x10
        pop     bp
   }
}

/*
 * Name:    initialize
 * Purpose: To initialize all the screen status info that is not hardware
 *           dependent, and call the hardware initialization routine to
 *           pick up the hardware dependent stuff.
 * Date:    June 5, 1991
 * Returns: [g_status and g_display]: all set up ready to go
 * Notes:   It is assumed that g_status and g_display are all \0's to begin
 *           with (the default if they use static storage). If this may
 *           not be the case, then clear them explicitly here.
 */
void initialize( void )
{
   /*
    * we do not know where the cursor is yet
    */
   g_display.col = -1;
   g_display.line = -1;

   /*
    * do the hardware initialization first, since this allocates the main
    *  text buffer and sets up other info needed here later.
    */
   hw_initialize( );

   bm.search_defined = ERROR;
   bm.search_case = IGNORE;
   /*
    * the main text buffer must be preceded by a ^Z, so that backward
    *  searches can see the start of the text buffer
    */
   *g_status.start_mem++ = CONTROL_Z;

   /*
    * most of the system's text pointers are safer set to the start
    *  of the text buffer - some of these may not be strictly
    *  necessary.
    */
   g_status.temp_end = g_status.start_mem;
   g_status.end_mem = g_status.start_mem;

   /*
    * set the default modes - may want to read this from a file later
    */
   g_status.marked = FALSE;
   g_status.marked_file = NULL;

   g_status.current_window = NULL;
   g_status.current_file = NULL;
   g_status.window_list = NULL;
   g_status.file_list = NULL;
   g_status.window_count = 0;
   g_status.file_count = 0;
   g_status.undo_head  = -1;
   g_status.next_file_number = 1;

   /*
    * set the number of lines from one page that should still be visible
    *  on the next page after page up or page down.
    */
   g_status.overlap = 1;

   /*
    * clear the screen and show the author's names
    */
   cls( );
   show_credits( );
}


/*
 * Name:    show_modes
 * Purpose: show current editor modes in lite bar at bottom of screen
 * Date:    June 5, 1991
 */
void show_modes( void )
{
char status_line[MAX_COLS+2];
register int color;

   color = g_display.mode_color;
   memset( status_line, ' ', MAX_COLS );
   status_line[MAX_COLS] = '\0';
   s_output( status_line, g_display.mode_line, 0, color );
   s_output( "F =    W = ", g_display.mode_line, 1, color );
   s_output( "memory = ", g_display.mode_line, 16, color );
   show_window_count( g_status.window_count );
   show_file_count( g_status.file_count );
   show_avail_mem( );
   show_indent_mode( );
   show_insert_mode( );
   show_search_case( );
   show_sdelete_mode( );
}


/*
 * Name:    show_file_count
 * Purpose: show number of open files in lite bar at bottom of screen
 * Date:    June 5, 1991
 */
void show_file_count( int fc )
{
char status_line[MAX_COLS+2];
int mode_line, mode_color;

   mode_line = g_display.mode_line;
   mode_color = g_display.mode_color;
   s_output( "  ", mode_line, 5, mode_color );
   s_output( itoa( fc, status_line, 10 ), mode_line, 5, mode_color );
}


/*
 * Name:    show_window_count
 * Purpose: show total number of windows in lite bar at bottom of screen
 * Date:    September 13, 1991
 */
void show_window_count( int wc )
{
char status_line[MAX_COLS+2];
int mode_line, mode_color;

   mode_line = g_display.mode_line;
   mode_color = g_display.mode_color;
   s_output( "  ", mode_line, 12, mode_color );
   s_output( itoa( wc, status_line, 10 ), mode_line, 12, mode_color );
}


/*
 * Name:    show_avail_mem
 * Purpose: show available free memory in lite bar at bottom of screen
 * Date:    June 5, 1991
 */
void show_avail_mem( void )
{
char memory[MAX_COLS+2];
int mode_line, mode_color;
unsigned long avail;

   avail = ptoul( g_status.max_mem ) - ptoul( g_status.end_mem );
   ultoa( avail, memory, 10 );
   mode_line = g_display.mode_line;
   mode_color = g_display.mode_color;
   s_output( "        ", mode_line, 25, mode_color );
   s_output( memory, mode_line, 25, mode_color );
}


/*
 * Name:    show_indent_mode
 * Purpose: show indent mode in lite bar at bottom of screen
 * Date:    June 5, 1991
 */
void show_indent_mode( void )
{
register char *p;
char *indent = "Indent";
char *blank  = "      ";

   if (mode.indent)
      p = indent;
   else
      p = blank;
   s_output( p, g_display.mode_line, 35, g_display.mode_color );
}


/*
 * Name:    show_insert_mode
 * Purpose: show insert mode in lite bar at bottom of screen
 * Date:    June 5, 1991
 */
void show_insert_mode( void )
{
int c;

   c = (mode.insert == TRUE) ? 'i' : 'o';
   c_output( c, 78, g_display.mode_line, g_display.mode_color );
}


/*
 * Name:    show_search_case
 * Purpose: indicate whether to ignore or match case on search strings
 * Date:    June 5, 1991
 */
void show_search_case( void )
{
char *p;
char *ignore = "Ignore";
char *match  = "Match ";

   if (bm.search_case == IGNORE)
      p = ignore;
   else
      p = match;
   s_output( p, g_display.mode_line, 45, g_display.mode_color );
}


/*
 * Name:    show_sdelete_mode
 * Purpose: indicate whether to stream delete with the delete key
 * Date:    June 5, 1991
 */
void show_sdelete_mode( void )
{
char *p;
char *sdel = "Sdel";
char *ndel = "    ";

   if (mode.sdel == TRUE)
      p = sdel;
   else
      p = ndel;
   s_output( p, g_display.mode_line, 55, g_display.mode_color );
}


/*
 * Name:    window_scroll_up
 * Purpose: To scroll all the lines between top and bottom up one line.
 * Date:    June 5, 1991
 * Passed:  top:    top line to be scrolled
 *          bottom: bottom line to be scrolled
 * Notes:   Call the video BIOS routine to scroll window up.
 */
void window_scroll_up( int top, int bottom )
{
int attr;

   attr = g_display.text_color;
   _asm {
        mov     bx, WORD PTR attr
        mov     bh, bl                  ; put attribute in bh
        mov     ax, WORD PTR top        ; get starting row
        mov     ch, al                  ; put it in ch
        xor     cl, cl                  ; starting column in cl = 0
        mov     ax, WORD PTR bottom     ; get ending row
        mov     dh, al                  ; put it in dh
        mov     dl, 79                  ; ending column in dl = 79
        mov     al, 1                   ; get number of lines
        mov     ah, 6                   ; get function number
        push    bp
        int     0x10
        pop     bp
   }
}

/*
 * Name:    window_scroll_down
 * Purpose: To scroll all the lines between top and bottom down one line.
 * Date:    June 5, 1991
 * Passed:  top:    top line to be scrolled
 *          bottom: bottom line to be scrolled
 * Notes:   Call the video BIOS routine to scroll window down.
 */
void window_scroll_down( int top, int bottom )
{
int attr;

   attr = g_display.text_color;
   _asm {
        mov     bx, WORD PTR attr
        mov     bh, bl                  ; put attribute in bh
        mov     ax, WORD PTR top        ; get starting row
        mov     ch, al                  ; put it in ch
        xor     cl, cl                  ; put starting column in cl - 0
        mov     ax, WORD PTR bottom     ; get ending row
        mov     dh, al                  ; put it in dh
        mov     dl, 79                  ; put ending column in dl
        mov     ah, 7                   ; get function number
        mov     al, 1                   ; get number of lines
        push    bp
        int     0x10
        pop     bp
   }
}

/*
 * Name:    window_scroll_up_eof
 * Purpose: To scroll all the lines between top and bottom up one line with
 *          the bottom lines having the 0x07 attribute.
 * Date:    June 5, 1991
 * Passed:  top:    top line to be scrolled
 *          bottom: bottom line to be scrolled
 * Notes:   Call the video BIOS routine to scroll window up.
 */
void window_scroll_up_eof( int top, int bottom )
{

   _asm {
        mov     bh, 7                   ; put attribute in bh
        mov     ax, WORD PTR top        ; get starting row
        mov     ch, al                  ; put it in ch
        xor     cl, cl                  ; starting column in cl = 0
        mov     ax, WORD PTR bottom     ; get ending row
        mov     dh, al                  ; put it in dh
        mov     dl, 79                  ; ending column in dl = 79
        mov     al, 1                   ; get number of lines
        mov     ah, 6                   ; get function number
        push    bp
        int     0x10
        pop     bp
   }
}


/*
 * Name:    combine_strings
 * Purpose: stick 3 strings together
 * Date:    June 5, 1991
 * Passed:  buff:    buffer to hold concatenation of 3 strings
 *          s1:  pointer to string 1
 *          s2:  pointer to string 2
 *          s3:  pointer to string 3
 */
void combine_strings( char *buff, char *s1, char *s2, char *s3 )
{
   strcpy( buff, s1 );
   strcat( buff, s2 );
   strcat( buff, s3 );
}
