#charset "us-ascii"

/* Copyright (c) 2004 M.D. Dollahite.  All Rights Reserved. */
/*
 *   Library Extention: one-way sense connector
 *
 *   Create a single-location one-way sense connector.
 *   Based on work by Steve Breslin.
 */
 
#include <adv3.h>

/*--------------------------------------------------------------------------*/
/*
 *   A OneWaySenseConnector, unlike a normal SenseConnector, transmits sense
 *   data from its location to the objects in its connectionList property.
 *   OneWaySenseConnectors are not MultiLocs, they can be located in their
 *   parent location by any of the normal methods. The locations specified
 *   in the connectionList property can be sensed from the connector's location
 *   according to the transparency returned by the connectorMaterial and
 *   transSensingThrough properties; however, sense data does not go the
 *   other way, from connectionList to location.  This is useful for security
 *   monitors, panoramic vistas, two-way mirrors, and other such objects.
 */
class OneWaySenseConnector: object
    /*
     *   connectionList: the other locations that can be sensed through this
     *   this object.
     */
    connectionList = []

    /*
     *   A OneWaySenseConnector's material by default determines how
     *   senses pass through to the object in connectionList.
     */
    connectorMaterial = adventium

    /*
     *   Determine how senses pass from this connector to the connectionList.
     *   By default, we simply use the material's transparency.
     */
    transSensingThru(sense) { return connectorMaterial.senseThru(sense); }

    /*
     *   Things like windows should transmit light through the sense 
     *   connection, but things like video monitors should not. This
     *   flag controls whether light is transmitted through the connection
     *   (subject to the opacity returned by transSensingThru). Set this
     *   to nil on connectors that should not transmit light to their
     *   connectionList.
     */
    shineThru = true

    /*
     *   Since we can provide a sense connection from this object to the
     *   connectionList, we add the connectionList also.
     */
    addDirectConnections(tab)
    {
        inherited(tab);

        for(local clst = connectionList, local i = 1,
            local len = clst.length() ; i <= len ; ++i)
        {
            if(tab[connectionList[i]] == nil)
                connectionList[i].addDirectConnections(tab);
        }
    }

    /*
     *   Transmit energy from a container onto me.
     */
    shineFromWithout(fromParent, sense, ambient, fill)
    {
        /* if this increases my ambient level, accept the new level */
        if(ambient > tmpAmbient_)
        {
            local levelThru;
            
            /* remember the new level and fill material to this point */
            tmpAmbient_ = ambient;
            tmpAmbientFill_ = fill;

            /* transmit to my contents */
            shineOnContents(sense, ambient, fill);
            
            if(shineThru == true)
            {
                /*
                 *   We must transmit this energy to our connectionList,
                 *   possibly reduced for traversing our connector.
                 *   Calculate the new level after traversing our connector.
                 */
                levelThru = adjustBrightness(ambient, transSensingThru(sense));
    
                /* 
                 *   if there's anything left, transmit it to the connectionList
                 */
                if(levelThru >= 2)
                {
                    for(local clst = connectionList, local i = 1,
                        local len = clst.length() ; i <= len ; ++i)
                    {
                        if(connectionList[i] != fromParent)
                            connectionList[i].shineFromWithin(self, sense, levelThru, fill);
                    }
                }
            }
        }
    }

    /*
     *   Build a sense path from a container to me
     */
    sensePathFromWithout(fromParent, sense, trans, obs, fill)
    {
        /*
         *   if there's better transparency along this path than along any
         *   previous path we've used to visit this item, take this path
         */
        if(transparencyCompare(trans, tmpTrans_) > 0)
        {
            local transThru;
            
            /* remember the new path to this point */
            tmpTrans_ = trans;
            tmpObstructor_ = obs;

            /* we're coming to this object from outside */
            tmpPathIsIn_ = true;

            /* transmit to my contents */
            sensePathToContents(sense, trans, obs, fill);

            /*
             *   We must transmit this energy to each of our connectionList
             *   objects, possibly reduced for traversing our connector.
             *   Calculate the new level after traversing our connector.
             */
            transThru = transparencyAdd(trans, transSensingThru(sense));

            /* if we changed the transparency, we're the obstructor */
            if(transThru != trans)
                obs = self;

            /*
             *   if there's anything left, transmit it to our connectionList
             */
            if(transThru != opaque)
            {
                /* transmit to each connection */
                for(local clst = connectionList, local i = 1,
                    local len = clst.length() ; i <= len ; ++i)
                {
                    if(connectionList[i] != fromParent)
                        connectionList[i].sensePathFromWithin(self, sense, transThru, obs, fill);
                }
            }
        }
    }

    /* 
     *   Check moving an object through me.  This is called when we try to
     *   move an object from one of our containers to another of our
     *   containers through me.  By default, we don't allow it.  
     */
    checkMoveThrough(obj, dest)
    {
        /* return an error - cannot move through <self> */
        return new CheckStatusFailure(&cannotMoveThroughMsg, obj, self);
    }

    /*
     *   Check touching an object through me.  This is called when an
     *   actor tries to reach from one of my containers through me into
     *   another of my containers.  By default, we don't allow it. 
     */
    checkTouchThrough(obj, dest)
    {
        /* return failure - cannot reach through me */
        return new CheckStatusFailure(&cannotReachThroughMsg, dest, self);
    }

    /* check for moving via a path */
    checkMoveViaPath(obj, dest, op)
    {
        /* if moving through us, don't allow it */
        if (op == PathThrough)
            return checkMoveThrough(obj, dest);

        /* if we can inherit, do so */
        if (canInherit())
            return inherited(obj, dest, op);

        /* return success by default */
        return checkStatusSuccess;
    }

    /* check for touching via a path */
    checkTouchViaPath(obj, dest, op)
    {
        /* if moving through us, don't allow it */
        if (op == PathThrough)
            return checkTouchThrough(obj, dest);

        /* if we can inherit, do so */
        if (canInherit())
            return inherited(obj, dest, op);

        /* return success by default */
        return checkStatusSuccess;
    }
    
    /* get objects that are connected to me */
    getConnectedContainers = ([location] + connectionList)
    
    /* call a function for each connected container */
    forEachConnectedContainer(func, [args])
    {
        forEachContainer(func, args...);
    }

    forEachContainer(func, [args])
    {
        foreach(local cur in getConnectedContainers)
            (func)(cur, args...);
    }
;


/*
 *   OneWayDistanceConnector: Does for OneWaySenseConnector what 
 *   DistanceConnector does for SenseConnector.
 *
 *   The normal DistanceConnector class assumes that it's usually
 *   going to be intangible; however since one-way connectors are
 *   more versatile we don't make the same assumption.
 */
class OneWayDistanceConnector: OneWaySenseConnector
    /* all senses are connected through us, but at a distance */
    transSensingThru(sense) { return distant; }

    /* 
     *   When checking for reaching through this connector, specialize the
     *   failure message to indicate that distance is the specific problem.
     *   (Without this specialization, we'd get a generic message when
     *   trying to reach through the connector, such as "you can't reach
     *   that through <self>."  
     */
    checkTouchThrough(obj, dest)
    {
        /* we can't touch through this connector due to the distance */
        return new CheckStatusFailure(&tooDistantMsg, dest);
    }

    /* 
     *   If we're responsible for blocking a thrown object's flight, we
     *   need to provide a custom message.  The default says that we
     *   deflected the object as though we were a physical barrier.
     *   Instead, in our case, the problem isn't deflection but simply
     *   range - so we need to explain that the projectile fell short of
     *   the target.
     *   
     *   By default, we assume that this connector interposes such a great
     *   distance that a character can't throw something all the way to the
     *   other side, which is why we provide this special message.  If you
     *   do want to allow actors to throw things all the way through the
     *   distance connector to the connected location, you can override
     *   checkMoveThrough like this:
     *   
     *   checkMoveThrough(obj, dest) { return checkStatusSuccess; } 
     */
    throwTargetHitWith(projectile, path)
    {
        /* 
         *   figure out where we fall to when we hit this object, then send
         *   the object being thrown to that location 
         */
        getHitFallDestination(projectile, path)
            .receiveDrop(projectile, new DropTypeShortThrow(self, path));
    }
;

