module AR
  module Game
    class Action
      attr_accessor :performer
      attr_accessor :parameters

      def name
        self.class.to_s.split("::").last.gsub(/Action/, "").downcase
      end

      def initialize performer, parameters = []
        @performer = performer 
        @parameters = parameters
      end

      def performable?
        parameters.select{|p| !p.filled? }.empty?
      end

      def engine
        performer.world.events_engine
      end






      def perform
        raise "#{self.class} needs to override perform"
        #return some events
      end

      def self.can_fill_params? performer
        can_fill = true 
        stub_action = self.new performer
        stub_action.parameters.map do |p|
          opts = p.filter performer
          if opts.empty?
            can_fill = false
            break
          else
            p.argument = opts.first
          end
        end 
        can_fill
      end


      def parameter name
        self.parameters.select{|p| p.name == name}.last
      end
      def next_param_to_fill
        self.parameters.select{|p| !p.filled? }.first
      end



      class Parameter
        attr_accessor :name
        attr_accessor :prompt_text
        attr_accessor :argument
        attr_accessor :argument_filter
        attr_accessor :action


        def initialize 
          @name = "param" 
          @action = nil
          @prompt_text = "What/Who/How to do action"
          @argument = nil
          @argument_filter = nil
          yield self
        end

        def filled?
          !argument.nil?
        end

        def filter performer
          return performer.known_objects if argument_filter.nil?
          argument_filter.call performer.known_objects, performer 
        end

      end

    end


    class WaitAction < Action
      def initialize performer,  parameters = [] 
        super performer, parameters
      end

      def perform
      end
    end


    class QuitAction < Action
      def initialize performer,  parameters = [] 
        super performer, parameters
      end

      def perform
        engine.create AR::Events::GameOverEvent
      end
    end


    class JournalAction < Action

      def initialize performer,  parameters = [] 
        super
      end

      def perform
        engine.create AR::Events::GameSaveEvent, {world: performer.world}
      end

    end



    class InspectAction < Action

      def initialize performer,  parameters =[] 
        super performer, parameters
        self.parameters << Parameter.new do |p|
          p.name = "target"
          p.prompt_text = "What do you inspect?"
          p.action = self
        end
      end

      def perform
        #contests can go here
        engine.create AR::Events::DiscoverableInspectedEvent, ({discoverable: parameter("target").argument, inspector: performer})
      end
    end

    class TravelAction < Action
      def initialize performer,  parameters =[] 
        super 
        self.parameters << Parameter.new do |p|
          p.argument_filter = Proc.new do |arg_set|
            arg_set.select{|o| o.behaviors.include? AR::Game::Gateway}
          end
          p.name = "gateway"
          p.prompt_text = "Which path do you take?"
          p.action = self
        end
      end

      def perform
        #contests can go here
        engine.create AR::Events::ARObjectTravelledEvent, ({traveller: performer, gateway: parameter("gateway").argument})
      end
    end


    class TalkAction < Action
      def self.filter target_set, performer
      end

      def initialize performer,  parameters =[] 
        super 
        self.parameters << Parameter.new do |p|
          p.name = "npc"
          p.argument_filter = Proc.new do |arg_set|

            arg_set.select{|o| o.behaviors.include? AR::Game::NpcDialogBehavior}
          end
          p.action = self
          p.prompt_text = "Who do you talk to?"
        end
      end

      def perform
        #contests can go here
        npc = parameter("npc").argument
        engine.create AR::Events::DialogBeganEvent, player: npc.world.player, npc: npc
      end
    end



    
    class GrabAction < Action
      def self.filter target_set, performer
      end

      def initialize performer,  parameters =[] 
        super 
        self.parameters << Parameter.new do |p|
          p.name = "item"
          p.argument_filter = Proc.new do |arg_set|
            initial = arg_set.select{|o| o.behaviors.include? AR::Game::ItemBehavior}
            initial.select{|o| !self.performer.inventory_items.include?(o)}
          end
          p.action = self
          p.prompt_text = "What item do you grab?"
        end
      end

      def perform
        #contests can go here
        engine.create AR::Events::ItemAddedToInventoryEvent, ({inventory_owner: performer, item: parameter("item").argument})
      end
    end

    class PutAction < Action
      def initialize performer,  parameters =[] 
        super 
        self.parameters << Parameter.new do |p|
          p.name = "item"
          p.argument_filter = Proc.new do |arg_set|
            initial = arg_set.select{|o| o.behaviors.include? AR::Game::ItemBehavior}
            initial.select{|o| self.performer.inventory_items.include?(o)}
          end
          p.action = self
          p.prompt_text = "What item do you put down??"
        end
      end

      def perform
        #contests can go here
        engine.create AR::Events::ItemRemovedFromInventoryEvent, ({inventory_owner: performer, item: parameter("item").argument})
      end
    end

    class GiveAction < Action
      def initialize performer,  parameters =[] 
        super 
        self.parameters << Parameter.new do |p|
          p.name = "item"
          p.argument_filter = Proc.new do |arg_set|
            initial = arg_set.select{|o| o.behaviors.include? AR::Game::ItemBehavior}
            initial.select{|o| self.performer.inventory_items.include?(o)}
          end
          p.action = self
          p.prompt_text = "What item do you give?"
        end
        self.parameters << Parameter.new do |p|
          p.name = "receiver"
          p.argument_filter = Proc.new do |arg_set|
            initial = arg_set.select{|o| o.behaviors.include? AR::Game::InventoryBehavior}
            initial - [self.performer]
          end
          p.action = self
          p.prompt_text = "Who do you give it to?"
        end

      end

      def perform
        #contests can go here
        results = []
        results.push engine.create AR::Events::ItemRemovedFromInventoryEvent, ({inventory_owner: performer, item: parameter("item").argument})
        results.push engine.create AR::Events::ItemAddedToInventoryEvent, ({inventory_owner: parameter("receiver").argument, item: parameter("item").argument})
      end
    end

    class AttackAction < Action
      def initialize performer, parameters = []
        super
        self.parameters << Parameter.new do |p|
          p.name = "target"
          p.prompt_text = "Who/What do you attack?"
          p.argument_filter = Proc.new do |arg_set|
            arg_set.select{|o| o.behaviors.include? AR::Game::Health and o != self.performer}
          end
          p.action = self
        end
        self.parameters << Parameter.new do |p|
          p.name = "technique"
          p.prompt_text = "What technique do you use?"
          p.argument_filter = Proc.new do |arg_set|
            self.performer.attack_techniques
          end
          p.action = self
        end
      end
      def perform
        #contests can go here
        #techs have success events and failure events
        #have a way to resolve a contest
        tech = parameter("technique").argument
        target = parameter("target").argument
        results = tech.perform performer: self.performer, target: target
      end
    end

    #just a way to group and easily create new techs 
    class AttackAction::Technique
      
      attr_accessor :when_perform
      attr_accessor :name
      
      def initialize &when_perform
        self.when_perform = when_perform
      end


      #overridden
      def perform params
        self.when_perform.call params
      end
    end

    class Punch <  AttackAction::Technique
      def initialize
        self.name = "punch"
        super do |params|
          performer = params[:performer]
          target = params[:target]
          AR::Events::DamageReceived.new damaged: target, damager: performer, damage_amount: rand(1..4)
        end 
      end
    end

  end
end

