/*
// Abstract:
//	MEMORY---Memory Operations
//
//	The Memory Operations module provides access to the HP 48SX's
//	memory, as stored in the memory dump.
//
// Author:
//	Derek S. Nickel
//
// Creation date:
//	28 October 1990
//
// History:
// V01-001	Derek S. Nickel		28-OCT-1990
//	Original.
//
*/

#include <string.h>
#include <stdlib.h>

#include "index.h"
#include "memory.h"
#include "pager.h"
#include "instruct.h"

static TextFile port0_ca_file;
static TextFile port0_cc_file;
static TextFile port1_ca_file;
static TextFile port1_cc_file;
static TextFile port2_ca_file;
static TextFile port2_cc_file;

static char port0_module[_MAX_PATH];
static char port1_module[_MAX_PATH];
static char port2_module[_MAX_PATH];

port_info_t ports[3] = {
  0, 0, 0x00000, 0x7FFFF, port0_module, NULL, &port0_ca_file, &port0_cc_file,
  0, 0, 0x80000, 0xBFFFF, port1_module, NULL, &port1_ca_file, &port1_cc_file,
  0, 0, 0xC0000, 0xFFFFF, port2_module, NULL, &port2_ca_file, &port2_cc_file
};

const int bad_port = -1;
const bin5_t bad_address = 0x100000;

int WorkPort = -1;
bin5_t WorkPtr = 0;

/***********************************************************************
	Messages.
***********************************************************************/

#define vgr__errsetadr \
"%%VOYAGER-E-ERRACCADR, error accessing address %05lX\n"

#define vgr__pornotloa \
"-VOYAGER-E-PORNOTLOA, port %d has not been loaded\n"

/***********************************************************************
	get_port_number
***********************************************************************/

static int get_port_number(bin5_t adr)
{
	int port_no;

	for (port_no = 0; port_no <= 2; port_no++) {
		if (adr >= ports[port_no].min_adr &&
		    adr <= ports[port_no].max_adr) {
			return port_no;
		}
	}

	return bad_port;
}

/***********************************************************************
	xlate_adr
***********************************************************************/

int xlate_adr(bin5_t adr, int *port_no, bin5_t *port_adr)
{
	int return_status;

	*port_no = get_port_number(adr);

	switch (*port_no) {
	    case 0:
	    case 1:
	    case 2:
		*port_adr = adr - ports[*port_no].min_adr;
		return_status = ports[*port_no].loaded;
		break;

	    default:
		*port_adr = bad_address;
		return_status = 0;
		break;
	}

	return return_status;
}

/***********************************************************************
	SetWorkPtr
***********************************************************************/

void SetWorkPtr(bin5_t adr)
{
	int port_no;
	bin5_t port_adr;

	if (xlate_adr(adr, &port_no, &port_adr)) {
		if (WorkPort != port_no || WorkPtr != adr) {
			WorkPort = port_no;
			WorkPtr = adr;
			fseek(ports[port_no].mem_file, port_adr, SEEK_SET);
		}

	} else {
		pager(0);
		printf(vgr__errsetadr, adr);
		pager(0);
		printf(vgr__pornotloa, port_no);
	}
}

/***********************************************************************
	GetNibble
***********************************************************************/

char GetNibble(void)
{
	int port_no;
	bin5_t port_adr;
	int ans = 0;

	if (xlate_adr(WorkPtr, &port_no, &port_adr)) {
		if (WorkPort != port_no) {
			WorkPort = port_no;
			fseek(ports[port_no].mem_file, port_adr, SEEK_SET);
		}

		ans = getc(ports[port_no].mem_file);
		if (ans != EOF) ++WorkPtr;
	}

	return (char) ans;
}

/***********************************************************************
	GetNNibbles
***********************************************************************/

void GetNNibbles(char *ans, int n)
{
	int port_no;
	bin5_t port_adr;
	int numRead;

	if (xlate_adr(WorkPtr, &port_no, &port_adr)) {
		if (WorkPort != port_no) {
			WorkPort = port_no;
			fseek(ports[port_no].mem_file, port_adr, SEEK_SET);
		}

		numRead = fread(ans, sizeof(char), n, ports[port_no].mem_file);
		ans[numRead] = '\0';
		WorkPtr += numRead;
	}
}

/***********************************************************************
	str2adr
***********************************************************************/

bin5_t str2adr(char *s, char **wp)
{
	bin5_t ans;
	char *junk;

	ans = strtoul(s, (wp) ? wp : &junk, 16);

	if (ans > 0xFFFFF) {
		puts("\n%*** address out of range ***");
		ans = 0;
	}

	return ans;
}

/***********************************************************************
	get_bin5
***********************************************************************/

bin5_t get_bin5(void)
{
	char buf[6];

	GetNNibbles(buf,5);
	strrev(buf);
	return str2adr(buf, 0);
}

/***********************************************************************
	get_address_comment
***********************************************************************/

char *get_address_comment(bin5_t adr)
{
	int port_no;
	bin5_t port_adr;
	static char zero = '\0';

	if (xlate_adr(adr, &port_no, &port_adr))
		return GetTextLine(
			ports[port_no].ca_file, port_adr);
	else
		return &zero;
}

/***********************************************************************
	get_code_comment
***********************************************************************/

char *get_code_comment(bin5_t adr)
{
	int port_no;
	bin5_t port_adr;
	static char zero = '\0';

	if (xlate_adr(adr, &port_no, &port_adr))
		return GetTextLine(
			ports[port_no].cc_file, port_adr);
	else
		return &zero;
}
