/*
// Abstract:
//	BINTOMEM
//
//	Converts a binary file (BIN) to Voyager format (MEM).
//
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define DEBUG 0
#define OVERHEAD 8
#define MAX_IBUF 256
#define MAX_OBUF 513

#define vgr__openin \
"%%VOYAGER-E-OPENIN, error opening %s as input\n"

#define vgr__openout \
"%%VOYAGER-E-OPENOUT, error opening %s as output\n"

#define vgr__usage \
"%%VOYAGER-I-USAGE, usage is: BINTOMEM input[.] [output[.MEM]]\n"

format_data(char *ibuf, size_t length, char *obuf);

FILE *ifile;
FILE *ofile;
char drive[_MAX_DRIVE];
char subdirectory[_MAX_DIR];
char filename[_MAX_FNAME];
char extension[_MAX_EXT];
char ipath[_MAX_PATH];
char opath[_MAX_PATH];
char ibuf[MAX_IBUF];
char obuf[MAX_OBUF];
size_t length;

main(int argc, char *argv[])
{
	/*
	// Check for correct number of parameters.
	*/

	if (argc < 2 || argc > 3) {
		printf(vgr__usage);
		exit(-1);
	}

	/*
	// Get the path for the input file and supply defaults.
	*/

	_splitpath(argv[1],drive,subdirectory,filename,extension);
	if (strcmp(extension,"") == 0) strcpy(extension,".");
	_makepath(ipath,drive,subdirectory,filename,extension);

	/*
	// Get the path for the output file and supply defaults.
	*/

	if (argc == 3) {
		_splitpath(argv[2],drive,subdirectory,filename,extension);
		if (strcmp(extension,"") == 0) strcpy(extension,".MEM");
		_makepath(opath,drive,subdirectory,filename,extension);
	} else {
		strcpy(extension,".MEM");
		_makepath(opath,NULL,NULL,filename,extension);
	}

	/*
	// Convert to uppercase.
	*/

	strupr(ipath);
	strupr(opath);

#if DEBUG
	printf("Input file:   %s\n",ipath);
	printf("Output file:  %s\n",opath);
#endif
	/*
	// Open the input file.
	*/

	ifile = fopen(ipath, "rb");

	if (ifile == NULL) {
		fprintf(stderr, vgr__openin, ipath);
		exit(-1);
	}

	/*
	// Open the output file.
	*/

	ofile = fopen(opath, "wt");

	if (ofile == NULL) {
		fprintf(stderr, vgr__openin, opath);
		exit(-1);
	}

	/*
	// Skip over the header.
	*/

	length = fread(ibuf,sizeof(char),OVERHEAD,ifile);
	ibuf[length] = '\0';
	printf("Header: %s\n", ibuf);

	/*
	// Read from the input and write to the output, reformatting as
	// as we go.
	*/

	while (!feof(ifile)) {
		length = fread(ibuf,sizeof(char),MAX_IBUF,ifile);
		if (length != 0) {
			format_data(ibuf,length,obuf);
			fputs(obuf, ofile);
		}
	}

	/*
	// Close the files.
	*/

	fclose(ifile);
	fclose(ofile);
}

format_data(char *ibuf, size_t length, char *obuf)
{
	int i;
	char tmp[3];

	for (i = 0; i < length; i++) {
		sprintf(tmp,"%02X", (unsigned char) *ibuf);
		*obuf++ = tmp[1];
		*obuf++ = tmp[0];
		ibuf++;
	}

	*obuf = '\0';
}
