/* This file contains a hard disk driver for Trantor MiniSCSI.
 *
 * The driver supports the following operations:
 *
 *	m_type	  DEVICE   PROC_NR	COUNT	 POSITION  ADRRESS
 * ----------------------------------------------------------------
 * |  DISK_READ | device  | proc nr |  bytes  |	 offset | buf ptr |
 * |------------+---------+---------+---------+---------+---------|
 * | DISK_WRITE | device  | proc nr |  bytes  |	 offset | buf ptr |
 * ----------------------------------------------------------------
 * |SCATTERED_IO| device  | proc nr | requests|         | iov ptr |
 * ----------------------------------------------------------------
 * |  TTY_OPEN  | device  | proc nr |open mode|         |         |
 * |------------+---------+---------+---------+---------+---------|
 * |  TTY_CLOSE | device  | proc nr |         |         |         |
 * |------------+---------+---------+---------+---------+---------|
 * |  TTY_IOCTL | device  | proc nr | requests|         | iov ptr |
 * ----------------------------------------------------------------
 *
 * The file contains one entry point:
 *
 *	 scsi_task:		main entry when system is brought up
 *
 */

#include "kernel.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include <minix/partition.h>
#ifndef NULL
#define NULL		(unsigned char *)0
#endif


#define TAPE_SUPPORT	1	/* 1=include tape support, 0=don't */

#if INTEL_32BITS
#define MAX_BLOCKS	10
#else
#define MAX_BLOCKS	1
#endif


#if TAPE_SUPPORT
#include <sys/mtio.h>
#endif

/* Parameters for the disk drive. */
#define MAX_DISKS	2	/* this can be increased if need */
#define DEV_PER_DISK	(1 + NR_PARTITIONS)	/* whole drive & each partn */
#define NR_DISKDEVS	(MAX_DISKS * DEV_PER_DISK)
#define SECTOR_SIZE	512	/* physical sector size in bytes */

/* Parameters for the tape drive. */
#define MAX_TAPES	1	/* this can be increased if need */

/* Retry counts */
#define MAX_TIMEOUT	30000
#define MID_TIMEOUT	250
#define MIN_TIMEOUT	70

/* Constants for MiniSCSI/Pararell port */
#define MAX_PORT	3
#define PARA_PORT1	0x3bc
#define PARA_PORT2	0x378
#define PARA_PORT3	0x278

/* NCR5380 Registers */
#define R_ODATA		0       /* Out Data Register (0) */
#define R_IDATA		0       /* In Data Register (0)
				 * (actually, 5380 data-in register
				 * is 6 but it is accessed via
				 * register 0
				 */

#define R_INITIATOR	1       /* Initiator Command Register (1) */
#define R1_DATA         0x01    /* Assert Data Bus */
#define R1_ATN          0x02    /* Assert ATN */
#define R1_SEL          0x04    /* Assert SEL */
#define R1_BSY          0x08    /* Assert BSY */
#define R1_ACK          0x10    /* Assert ACK */
#define R1_RST          0x80    /* Assert RST */

#define R_MODE		2       /* Mode Register (2) */
#define R2_TARGET       0x40    /* Select Initiator(0)/Target Mode */
#define R2_CHKPE        0x08    /* Enable Parity Check */

#define R_TARGET	3       /* Target Command Register (3) */
#define R3_IO           0x01    /* Assert I/O */
#define R3_CD           0x02    /* Assert C/D */
#define R3_MSG          0x04    /* Assert MSG */
#define R3_REQ          0x08    /* Assert REQ */

#define R_SCBUS         4       /* Current SCSI Bus Status Register (4) */
#define R4_DBP          0x01
#define R4_SEL          0x02
#define R4_IO           0x04
#define R4_CD           0x08
#define R4_MSG          0x10
#define R4_REQ          0x20
#define R4_BSY          0x40
#define R4_RST          0x80

#define R_STAT		5       /* Bus/Status Register (5) */
#define R5_ACK          0x01
#define R5_ATN          0x02
#define R5_PHM          0x08    /* Phase Matched */
#define R5_PE           0x20    /* Parity Error */

#define R_RSTEI		7	/* Reset Parity Error/Interrup Reg. (7) */


/* SCSI bus lines */
#define XP_DATAOUT	0
#define XP_DATAIN	R4_IO
#define XP_COMMAND	R4_CD
#define XP_STAT		(R4_CD|R4_IO)
#define XP_MSGOUT	(R4_MSG|R4_CD)
#define XP_MSGIN	(R4_MSG|R4_CD|R4_IO)
#define XP_BITS		(R4_MSG|R4_CD|R4_IO)
#define XPHASE()	(reg_read(R_SCBUS)&XP_BITS)

/* SCSI ID */
#define MAX_ID		6
#define MAX_LUN		7

/* SCSI status */
#define ST_MASK		0x1e
#define ST_GOOD		0
#define ST_CHKCOND	2

/* SCSI message */
#define MSG_COMPLETE	0	/* Command Complete */

/* SCSI sense key */
#define EXTSENSE	0x70
#define SK_MASK		0x0f
#define SK_NOSENSE	0
#define SK_UNITATN	6
#define SK_ILL		0x20
#define SK_EOM		0x40
#define SK_EOF		0x80


/* SCSI device type code */
#define SC_DISK		0
#define SC_TAPE		1

/* SCSI operation code */
#define SC_READ		0x08
#define SC_WRITE	0x0a


#define b24ul(b)	(((unsigned long)*((b)+0)<<16)|\
			 ((unsigned long)*((b)+1)<<8)|\
			 ((unsigned long)*((b)+2)))

#define b32ul(b)	(((unsigned long)*((b)+0)<<24)|\
			 ((unsigned long)*((b)+1)<<16)|\
			 ((unsigned long)*((b)+2)<<8)|\
			 ((unsigned long)*((b)+3)))


PRIVATE unsigned char buf[BLOCK_SIZE*MAX_BLOCKS] = { 1 };

#if 0
PUBLIC int using_bios = TRUE;	/* this disk driver uses the BIOS */
#endif
PRIVATE message sc_mess;		/* message buffer for in and out */

PRIVATE unsigned int sense_key;		/* SCSI sense key on last command */
PRIVATE unsigned long sense_len;	/* SCSI sense bytes on last command */
PRIVATE unsigned char sense_buf[64];

PRIVATE int mini_installed;	/* 1: MiniSCSI installed */
PRIVATE unsigned int data_port, stat_port, ctrl_port;
PRIVATE int save_pdata, save_pctrl;
PRIVATE int port_tab[MAX_PORT] = {PARA_PORT1, PARA_PORT2, PARA_PORT3};

PRIVATE struct sc_diskdev_t {	/* main drive struct, one entry per drive */
  int opcode;			/* DISK_READ or DISK_WRITE */
  int procnr;			/* which proc wanted this operation? */
  long sector;			/* sector addressed */
  long low;			/* lowest sector of partition */
  long size;			/* size of partition in sectors */
  int count;			/* byte count */
  int drive;			/* unit# 0, 1, ... , MAX_DISKS-1 */
  vir_bytes address;		/* user virtual address */
} sc_diskdev[NR_DISKDEVS];

PRIVATE struct disk_param_t {
  int target_id;		/* SCSI target ID */
  int lun;			/* SCSI lun */
  unsigned long blk_num;	/* number of blocks */
  unsigned long blk_size;	/* block size */
} disk_param[MAX_DISKS];

PRIVATE int nr_disks = 0;

#if TAPE_SUPPORT
PRIVATE struct tape_param_t {
  int target_id;		/* SCSI target ID */
  int lun;			/* SCSI lun */
  unsigned long blk_num;	/* number of blocks */
  unsigned long blk_size;	/* block size */
  int in_use;			/* non-zero if the drive is in use */
  int open_mode;		/* open for reading/writing */
  int at_eof;			/* got EOF mark */
} tape_param[MAX_TAPES];

PRIVATE int nr_tapes = 0;
#endif

PRIVATE int no_phase_timeout = 0;
PRIVATE int curdev = -1;
PRIVATE unsigned char ccdb[12];
PRIVATE unsigned int ccdblen;

/* Functions */
FORWARD void copy_prt();
FORWARD void init_params();
FORWARD void sort();
FORWARD int do_rdwt();
FORWARD int tpio();
FORWARD int tpopen();
FORWARD int tpclose();
FORWARD int tpioctl();

FORWARD void suspend_prt();
FORWARD void resume_prt();
FORWARD void ctrl_out();
FORWARD int reg_read();
FORWARD void reg_write();
FORWARD int reg_setbit();
FORWARD int reg_clrbit();
FORWARD int reg_repbit();
FORWARD int req();
FORWARD int ack_req();
FORWARD int bus_active();
FORWARD int bus_free();
FORWARD int bus_waitphase();
FORWARD int bus_select();
FORWARD void bus_reset();
FORWARD int sc_dout();
FORWARD int sc_din();
FORWARD int sc_rdbyte();
FORWARD int sc_command();
FORWARD int sc_com_dataio_stat();

FORWARD int sc_test_unit_ready();
FORWARD int sc_rewind();
FORWARD int sc_request_sense();
FORWARD int sc_rd_block_limits();
FORWARD int sc_space();
FORWARD int sc_inquiry();
FORWARD int sc_mode_sense();
FORWARD int sc_rw();

FORWARD void dump_cdb();
FORWARD void dump_sense();
FORWARD int mini_check();
FORWARD int mini_find();
FORWARD void mini_reset();

#define set_lun(lun,cdb)	(cdb[1]|=(lun&7)<<5)

/*=========================================================================*
 *				scsi_task				   * 
 *=========================================================================*/
PUBLIC void scsi_task()
{
/* Main program of the scsi disk driver task. */

  int r, caller, proc_nr;

  /* First check devices and initialize the parameters */
  init_params();

  /* Here is the main loop of the disk task.  It waits for a message, carries
   * it out, and sends a reply.
   */

  while (TRUE) {
	/* First wait for a request to read or write a disk block. */
	receive(ANY, &sc_mess);	/* get a request to do some work */

	if (sc_mess.m_source < 0) {
		printf("scsi task got message from %d\n",sc_mess.m_source);
		continue;
	}

	caller = sc_mess.m_source;
	proc_nr = sc_mess.PROC_NR;

	curdev = sc_mess.DEVICE;
	if (!mini_installed) {
	  r = EIO;
#if TAPE_SUPPORT
	} else if (sc_mess.DEVICE & 0x80) {
	  /* tape requests */
	  switch(sc_mess.m_type) {
	  case DISK_READ:
	  case DISK_WRITE:
		r = tpio(&sc_mess);
		break;
	  case SCATTERED_IO:
		r = do_vrdwt(&sc_mess, tpio);
		break;
	  case TTY_OPEN:
		r = tpopen(&sc_mess);
		break;
	  case TTY_CLOSE:
		r = tpclose(&sc_mess);
		break;
	  case TTY_IOCTL:
		r = tpioctl(&sc_mess);
		break;
	  default:
		r = EINVAL;
		break;
	  }
#endif
	} else {
	  /* disk requests */
	  switch(sc_mess.m_type) {
	  case DISK_READ:
	  case DISK_WRITE:
		r = do_rdwt(&sc_mess);
		break;
	  case SCATTERED_IO:
		r = do_vrdwt(&sc_mess, do_rdwt);
		break;
	  case TTY_OPEN:
	  case TTY_CLOSE:
		r = OK;
		break;
	  case TTY_IOCTL:
		r = ENOTTY;
		break;
	  default:
		r = EINVAL;
		break;
	  }
	}

	/* Finally, prepare and send the reply message. */
	sc_mess.m_type = TASK_REPLY;	
	sc_mess.REP_PROC_NR = proc_nr;

	sc_mess.REP_STATUS = r;	/* # of bytes transferred or error code */
	send(caller, &sc_mess);	/* send reply to caller */
  }
}


/*==========================================================================*
 *				do_rdwt					    * 
 *==========================================================================*/
PRIVATE int do_rdwt(m_ptr)
message *m_ptr;			/* pointer to read or write sc_message */
{
/* Carry out a read or write request from the disk. */
  register struct sc_diskdev_t *scp;
  struct disk_param_t *pp;
  int r, device;
  long sector;
  phys_bytes user_phys, buf_phys;

  /* Decode the sc_message parameters. */
  device = m_ptr->DEVICE;	/* minor device #.  1-4 are partitions */
  if (device < 0 || device >= NR_DISKDEVS) return(EIO);
  if (m_ptr->COUNT != BLOCK_SIZE) return(EINVAL);
  scp = &sc_diskdev[device];		/* 'scp' points to entry for this drive */

  /* Set opcode to SC_READ or SC_WRITE. Check for bad starting addr. */
  scp->opcode = (m_ptr->m_type == DISK_WRITE ? SC_WRITE : SC_READ);
  if (m_ptr->POSITION % BLOCK_SIZE != 0) return(EINVAL);

  /* Calculate the physical parameters */
  sector = m_ptr->POSITION/SECTOR_SIZE;	/* relative sector within partition */
  if ((sector+BLOCK_SIZE/SECTOR_SIZE) > scp->size) return(0);
  sector += scp->low;		/* absolute sector number */
  scp->sector = sector;
  scp->count = m_ptr->COUNT;
  scp->address = (vir_bytes) m_ptr->ADDRESS;
  scp->procnr = m_ptr->PROC_NR;

  /* Do the transfer */
  user_phys = numap(scp->procnr, (vir_bytes)scp->address,
		    (vir_bytes)scp->count);
  buf_phys = umap(proc_ptr, D, (vir_bytes)buf, sizeof buf);

  /* Do the transfer */
  pp = disk_param + scp->drive;
  if (scp->opcode == SC_WRITE) {
    phys_copy(user_phys, buf_phys, (long)m_ptr->COUNT);
  }

  /* read or write 2 sectors */
  suspend_prt();
  r = sc_rw(pp->target_id, pp->lun, buf,
	    SECTOR_SIZE, 2L, scp->sector, scp->opcode, SC_DISK);
  resume_prt();

  if (scp->opcode == SC_READ) {
    phys_copy(buf_phys, user_phys, (long)m_ptr->COUNT);
  }

  return(r != ERROR ? BLOCK_SIZE : EIO);
}

#if TAPE_SUPPORT
/*==========================================================================*
 *				tpio					    * 
 *==========================================================================*/
PRIVATE int tpio(m_ptr)
message *m_ptr;			/* pointer to read or write sc_message */
{
/* Carry out a read or write request from the tape. */
  struct tape_param_t *pp;
  int r, tapeno, opcode;
  unsigned int count;
  phys_bytes user_phys, buf_phys;

  /* Decode the sc_message parameters. */
  tapeno = (m_ptr->DEVICE & 0x7f) >> 1;
  if (tapeno < 0 || tapeno >= nr_tapes) return EIO;
  pp = tape_param + tapeno;

  /* Set opcode to SC_READ or SC_WRITE. */
  opcode = (m_ptr->m_type == DISK_WRITE ? SC_WRITE : SC_READ);
  if (opcode == SC_READ && pp->at_eof) return 0;
  if (opcode == SC_WRITE && (m_ptr->COUNT % pp->blk_size)) {
    printf("write: not modulo %d block size\n", (int)pp->blk_size);
  }

  count = (m_ptr->COUNT + pp->blk_size - 1) / pp->blk_size;

  /* Do the transfer */
  user_phys = numap(m_ptr->PROC_NR, (vir_bytes)m_ptr->ADDRESS,
		    (vir_bytes)m_ptr->COUNT);
  buf_phys = umap(proc_ptr, D, (vir_bytes)buf, sizeof buf);

  if (opcode == SC_WRITE) {
    phys_copy(user_phys, buf_phys, (long)m_ptr->COUNT);
  }

  suspend_prt();
  r = sc_rw(pp->target_id, pp->lun,
	    buf, (unsigned int)pp->blk_size, (long)count, 0L, opcode, SC_TAPE);
  resume_prt();

  if (opcode == SC_READ) {
    phys_copy(buf_phys, user_phys, (long)m_ptr->COUNT);
  }

  if (r) {
    if (sense_key && (sense_key & (SK_EOF|SK_EOM|SK_ILL))) {
      pp->at_eof = 1;
      return m_ptr->COUNT - (sense_len * pp->blk_size);
    }
    return EIO;
  }
  return m_ptr->COUNT;
}

/*==========================================================================*
 *				tpopen					    * 
 *==========================================================================*/
PRIVATE int tpopen(m_ptr)
message *m_ptr;			/* pointer to read or write sc_message */
{
  struct tape_param_t *pp;
  int r, tapeno;

  tapeno = (m_ptr->DEVICE & 0x7f) >> 1;
  if (tapeno < 0 || tapeno >= nr_tapes) return EIO;
  pp = tape_param + tapeno;

  if (pp->in_use) return EBUSY;
  suspend_prt();
  r = sc_test_unit_ready(pp->target_id, pp->lun);
  resume_prt();
  if (r && (sense_key & SK_MASK) != SK_UNITATN) return EBUSY;

  pp->in_use = 1;
  pp->at_eof = 0;
  pp->open_mode = m_ptr->COUNT;
  return OK;
}

/*==========================================================================*
 *				tpclose					    * 
 *==========================================================================*/
PRIVATE int tpclose(m_ptr)
message *m_ptr;			/* pointer to read or write sc_message */
{
  struct tape_param_t *pp;
  int r, tapeno;

  tapeno = (m_ptr->DEVICE & 0x7f) >> 1;
  if (tapeno < 0 || tapeno >= nr_tapes) return EIO;
  pp = tape_param + tapeno;

  if (!pp->in_use) return EIO;
  pp->in_use = pp->at_eof = 0;

  suspend_prt();
  /* write filemark for writing */
  if (pp->open_mode & W_BIT) {
    if (sc_write_filemarks(pp->target_id, pp->lun, 1L)) {
      resume_prt();
      return EIO;
    }
  }

  if (m_ptr->DEVICE & 1) {
    if (sc_rewind(pp->target_id, pp->lun, TRUE)) {
      resume_prt();
      return EIO;
    }
  }
  resume_prt();
  return OK;
}

/*==========================================================================*
 *				tpioctl					    * 
 *==========================================================================*/
PRIVATE int tpioctl(m_ptr)
message *m_ptr;			/* pointer to read or write sc_message */
{
  struct tape_param_t *pp;
  struct mtop op;
  int r, tapeno, cond;
  long spclen;
#define MTOPSIZE	(sizeof(struct mtop))

  tapeno = (m_ptr->DEVICE & 0x7f) >> 1;
  if (tapeno < 0 || tapeno >= nr_tapes) return EIO;
  pp = tape_param + tapeno;

  if (m_ptr->TTY_REQUEST != MTIOCTOP) {
    printf("st%d: 0x%x --- bad ioctl request number\n",
	   ((m_ptr->DEVICE & 0x7f) >> 1), m_ptr->TTY_REQUEST);
    return ENOTTY;
  }

  phys_copy(numap(m_ptr->PROC_NR, (vir_bytes)m_ptr->ADDRESS, MTOPSIZE),
	    umap(proc_ptr, S, (vir_bytes)&op, MTOPSIZE), (long)MTOPSIZE);

  suspend_prt();
  switch(op.mt_op) {
  case MTFSF:
  case MTFSR:
  case MTBSF:
  case MTBSR:
    cond = (op.mt_op == MTFSF || op.mt_op == MTBSF);
    spclen = op.mt_count;
    if (op.mt_op == MTBSF || op.mt_op == MTBSR) {
      spclen = -spclen;
    }
    sc_space(pp->target_id, pp->lun, cond, spclen);
    break;
  case MTREW:
    sc_rewind(pp->target_id, pp->lun, TRUE);
    break;
  case MTOFFL:
    sc_load(pp->target_id, pp->lun, FALSE);
    break;
  case MTRETEN:
    sc_load(pp->target_id, pp->lun, TRUE);
    break;
  case MTERASE:
    sc_rewind(pp->target_id, pp->lun, FALSE);
    sc_erase(pp->target_id, pp->lun);
    break;
  case MTWEOF:
  default:
    printf("st%d: 0x%x --- unimplemented ioctl command\n",
	   ((m_ptr->DEVICE & 0x7f) >> 1), op.mt_op);
    break;
  }
  resume_prt();
  r = (sense_key & SK_MASK);
  if (r && r != SK_UNITATN) return EIO;
  return OK;
}
#endif


/*===========================================================================*
 *				init_params				     *
 *===========================================================================*/
PRIVATE void init_params()
{
/* This routine is called at startup to initialize the partition table,
 * the number of drives and the controller
*/
  int i, j, target_id, lun, ret;
  unsigned long blk_num, blk_size;
  struct disk_param_t *pp;
#define INQBUF		(buf)
#define INQBUFSIZ	64
#define MODEBUF		(buf+INQBUFSIZ)
#define MODEBUFSIZ	64

  suspend_prt();
  if (mini_find() != OK) {
    mini_installed = 0;
    resume_prt();
    return;
  }
  mini_reset();
  milli_delay(1000);

  for (target_id = 0; target_id <= MAX_ID; target_id++) {
    for (lun = 0; lun <= MAX_LUN; lun++) {
      if ((ret = sc_inquiry(target_id, lun, INQBUF, INQBUFSIZ)) != ST_GOOD) {
	break;
      }
      if (sc_mode_sense(target_id, lun, MODEBUF, MODEBUFSIZ) != ST_GOOD) {
	continue;
      }

      if (INQBUF[0] == SC_DISK) {
	if (nr_disks + 1 > MAX_DISKS) {
	  goto get_drives_done;
	}
	blk_num = b24ul(MODEBUF+5);
	blk_size = b24ul(MODEBUF+9);
	if (blk_size != SECTOR_SIZE) {
	  continue;
	}

	printf("sd%d: id=%d, lun=%d: %luMB, %.24s\n",
	       nr_disks, target_id, lun,
	       (blk_num*blk_size)/(1024L*1024L), INQBUF+8);
	
	disk_param[nr_disks].target_id = target_id;
	disk_param[nr_disks].lun = lun;
	disk_param[nr_disks].blk_num = blk_num;
	disk_param[nr_disks].blk_size = blk_size;
	++nr_disks;
#if TAPE_SUPPORT
      } else if (INQBUF[0] == SC_TAPE) {
	if (nr_tapes + 1 > MAX_TAPES) {
	  goto get_drives_done;
	}
	blk_num = b24ul(MODEBUF+5);
	blk_size = b24ul(MODEBUF+9);

	if (blk_size != SECTOR_SIZE) {
	  continue;
	}

	printf("st%d: id=%d, lun=%d: bsize=%luB, %.24s\n",
	       nr_tapes, target_id, lun,
	       blk_size, INQBUF+8);
	
	tape_param[nr_tapes].target_id = target_id;
	tape_param[nr_tapes].lun = lun;
	tape_param[nr_tapes].blk_num = blk_num;
	tape_param[nr_tapes].blk_size = blk_size;
	tape_param[nr_tapes].in_use = 0;
	tape_param[nr_tapes].at_eof = 0;
	++nr_tapes;
#endif
      }
    }
  }

 get_drives_done:
  /* Set the parameters in the drive structure */
  for (j = 0; j < MAX_DISKS; j++) {
    for (i = 0; i < DEV_PER_DISK; i++) {
      if (!(i % DEV_PER_DISK)) {
	sc_diskdev[i+DEV_PER_DISK*j].low = 0L;
	sc_diskdev[i+DEV_PER_DISK*j].size = disk_param[j].blk_num;
      }
      sc_diskdev[i].drive = j;
    }
  }

  /* Read the partition table for each drive and save them */
  for (i = 0; i < nr_disks; i++) {
    pp = disk_param + i;
    if (sc_rw(pp->target_id, pp->lun, buf,
	      SECTOR_SIZE, 2L, 0L, SC_READ, SC_DISK) == ERROR) {
      panic("Can't read partition table of scsi disk ", i);
    }
    copy_prt(i * DEV_PER_DISK);
  }
  resume_prt();
#if TAPE_SUPPORT
  mini_installed = (nr_disks > 0 || nr_tapes > 0);
#else
  mini_installed = nr_disks > 0;
#endif
}

/*===========================================================================*
 *				copy_prt				     *
 *===========================================================================*/
PRIVATE void copy_prt(base_dev)
int base_dev;			/* base device for drive */
{
/* This routine copies the partition table for the selected drive to
 * the variables wn_low and wn_size
 */

  register struct part_entry *pe;
  register struct sc_diskdev_t *scp;

  for (pe = (struct part_entry *) &buf[PART_TABLE_OFF],
       scp = &sc_diskdev[base_dev + 1];
       pe < ((struct part_entry *) &buf[PART_TABLE_OFF]) + NR_PARTITIONS;
       ++pe, ++scp) {
	scp->low = pe->lowsec;
	scp->size = pe->size;

	/* Adjust low sector to a multiple of (BLOCK_SIZE/SECTOR_SIZE) for old
	 * Minix partitions only.  We can assume the ratio is 2 and round to
	 * even, which is slightly simpler.
	 */
	if (pe->sysind == OLD_MINIX_PART && scp->low & 1) {
		++scp->low;
		--scp->size;
	}
  }
  sort(&sc_diskdev[base_dev + 1]);
}

/*===========================================================================*
 *				sort					     *
 *===========================================================================*/
PRIVATE void sort(scp)
register struct sc_diskdev_t scp[];
{
  register int i,j;
  struct sc_diskdev_t tmp;

  for (i = 0; i < NR_PARTITIONS; i++)
	for (j = 0; j < NR_PARTITIONS-1; j++)
		if ((scp[j].low == 0 && scp[j+1].low != 0) ||
		    (scp[j].low > scp[j+1].low && scp[j+1].low != 0)) {
			tmp = scp[j];
			scp[j] = scp[j+1];
			scp[j+1] = tmp;
		}
}

/*==========================================================================*
 *				suspend_prt				    *
 *==========================================================================*/
PRIVATE void suspend_prt()
{
  int d;
       
  save_pdata = in_byte(data_port);	/* save data in data port */
  save_pctrl = in_byte(ctrl_port);	/* save data in ctrl port */
  d = (save_pctrl & 0xfb) | 8;		/* set select printer bit */
  lock();
  out_byte(ctrl_port, d);
  out_byte(ctrl_port, d & 0xf7);		/* clear select printer bit */
  unlock();
}
     
/*==========================================================================*
 *				resume_prt				    *
 *==========================================================================*/
PRIVATE void resume_prt()
{
       
  reg_write(R_INITIATOR, 0);
  lock();
  out_byte(data_port, save_pdata);
  out_byte(ctrl_port, save_pctrl | 4);
  unlock();
}

/*==========================================================================*
 *				ctrl_out				    *
 *==========================================================================*/
PRIVATE void ctrl_out(reg, data)
int reg;
int data;
{

  out_byte(data_port, ((data & 0xf8) ^ 0x38) | reg);
  out_byte(ctrl_port, 8);
  out_byte(ctrl_port, 0);
}
     
/*==========================================================================*
 *				reg_read				    *
 *==========================================================================*/
PRIVATE int reg_read(reg)
int reg;
{
  int ret;
       
  ctrl_out(reg, 8);
  out_byte(data_port, 0x80);
  out_byte(ctrl_port, 1);
  ret = (in_byte(stat_port) << 1) & 0xf0;
  out_byte(data_port, 0);
  ret |= (in_byte(stat_port) >> 3) & 0x0f;
  out_byte(ctrl_port, 0);
  return ret;
}

/*==========================================================================*
 *				reg_write				    *
 *==========================================================================*/
PRIVATE void reg_write(reg, data)
int reg;
int data;
{
       
  ctrl_out(reg, 0x10);
  out_byte(data_port, data);
  out_byte(ctrl_port, 1);
  out_byte(ctrl_port, 0);
}

/*==========================================================================*
 *				reg_setbit				    *
 *==========================================================================*/
PRIVATE int reg_setbit(reg, bits)
int reg;
int bits;
{
  int data;
       
  reg_write(reg, data = (reg_read(reg) | bits));
  return data;
}

/*==========================================================================*
 *				reg_clrbit				    *
 *==========================================================================*/
PRIVATE int reg_clrbit(reg, bits)
int reg;
int bits;
{
  int data;
       
  reg_write(reg, data = (reg_read(reg) & ~bits));
  return data;
}

/*==========================================================================*
 *				reg_repbit				    *
 *==========================================================================*/
PRIVATE int reg_repbit(reg, mask, bits)
int reg;
int mask;
int bits;
{
  int data;
       
  reg_write(reg, data = ((reg_read(reg) & mask) | bits));
  return data;
}

/*==========================================================================*
 *				req     				    *
 *==========================================================================*/
PRIVATE int req()
{
  unsigned int n;

  for (n = MAX_TIMEOUT; --n; ) {
    if (reg_read(R_SCBUS) & R4_REQ) { /* wait for req=1 */
      return OK;
    }
/*    milli_delay(1);*/
  }
  return ERROR;
}

/*==========================================================================*
 *				ack_req     				    *
 *==========================================================================*/
PRIVATE int ack_req()
{
  unsigned int n;
  int ret = ERROR;

  reg_setbit(R_INITIATOR, R1_ACK);
  for (n = MAX_TIMEOUT; --n; ) {
    if (!(reg_read(R_SCBUS) & R4_REQ)) { /* wait for req=0 */
      break;
    }
/*    milli_delay(1);*/
  }
  if (n) {
    ret = OK;
  }
  reg_clrbit(R_INITIATOR, R1_ACK);
  return ret;
}

/*==========================================================================*
 *				bus_active				    *
 *==========================================================================*/
PRIVATE int bus_active()
{
  unsigned n;
  int bstat0, bstat1;
	
  for (n = MAX_TIMEOUT; --n; ) {
    bstat0 = reg_read(R_SCBUS);
    if (!(bstat0 & R4_BSY)) {
      return ERROR; /* now in bus free phase */
    }
    if (bstat0 & R4_SEL) {
      continue;
    }
    if (bstat0 & R4_REQ) {
      return OK;
    }
/*    milli_delay(1);*/
  }
  return ERROR;
}


/*==========================================================================*
 *				bus_free				    *
 *==========================================================================*/
PRIVATE int bus_free(n)
unsigned int n;
{
  unsigned int m;
  int bstat0, bstat1;

  /* wait for bus free phase */
  while (n--) {
    for (m = MIN_TIMEOUT; --m; ) {
      bstat0 = reg_read(R_SCBUS);
      if (bstat0 & ~(R4_SEL|R4_BSY)) {
	break;
      }
    }
    if (!m) {
      return OK;
    }
  }
  return ERROR;
}

/*==========================================================================*
 *				bus_waitphase  				    *
 *==========================================================================*/
PRIVATE int bus_waitphase(phase)
int phase;
{

  for (;;) {
    if (bus_active() == OK) {
      break;
    }
    if (!no_phase_timeout) {
      return ERROR;
    }
  }
  return XPHASE() == phase? OK: ERROR;
}


/*==========================================================================*
 *				bus_select				    *
 *==========================================================================*/
PRIVATE int bus_select(target_id)
int target_id;
{
  unsigned n;
  
  if (bus_free(MID_TIMEOUT) != OK) {
    return ERROR;
  }
  
  /* set Initiatore mode (free I/O, C/D, MSG) */
  reg_clrbit(R_MODE, R2_TARGET);
/*  milli_delay(1);*/

  /* clear out data register */
  reg_write(R_ODATA, 0);

  /* syncronize phase */
  reg_repbit(R_TARGET, ~(XP_BITS>>2), XPHASE()>>2);

  /* assert DATA line */
  reg_setbit(R_INITIATOR, R1_DATA);

  /* out Target SCSI ID bits */
  reg_write(R_ODATA, 1<<target_id);

  /* assert SEL */
  reg_setbit(R_INITIATOR, R1_SEL);

  /* wait for selection */
  for (n = MIN_TIMEOUT; --n; ) {
    if (reg_read(R_SCBUS) & R4_BSY) {
      break;
    }
    milli_delay(1);
  }
  if (!n) {
    reg_write(R_ODATA, 0);
    /* another wait for selection */
    for (n = MIN_TIMEOUT; --n; ) {
      if (reg_read(R_SCBUS) & R4_BSY) {
	break;
      }
      milli_delay(1);
    }
    if (!n) {
      reg_write(R_INITIATOR, 0);
      return ERROR;
    }
  }
  reg_clrbit(R_INITIATOR, R1_SEL);
  return OK;
}

/*==========================================================================*
 *				bus_reset				    *
 *==========================================================================*/
PRIVATE void bus_reset()
{
  reg_write(R_INITIATOR, R1_RST);
  milli_delay(1);
  reg_write(R_INITIATOR, 0);
}

/*==========================================================================*
 *				sc_dout					    *
 *==========================================================================*/
PRIVATE int sc_dout(bufp, len)
unsigned char *bufp;
unsigned long len;
{
  unsigned int n;
  int cur_phase;

  if (bus_active() != OK) {
    return ERROR;
  }
  cur_phase = XPHASE();
  /* syncronize phase */
  reg_repbit(R_TARGET, ~(XP_BITS>>2), cur_phase>>2);
  while (len) {
    reg_write(R_ODATA, *bufp);
    if (ack_req() != OK) {
      return ERROR;
    }
    --len;
    ++bufp;
    if (req() != OK)  {
      break;
    }
    if (XPHASE() != cur_phase) {
      break;
    }
  }
  return len;
}

/*==========================================================================*
 *				sc_din					    *
 *==========================================================================*/
PRIVATE int sc_din(bufp, len)
unsigned char *bufp;
unsigned long len;
{
  unsigned int n;
  int cur_phase;

  if (bus_active() != OK) {
    return ERROR;
  }
  cur_phase = XPHASE();
  while (len) {
    *bufp = reg_read(R_IDATA);
    if (ack_req() != OK) {
      return ERROR;
    }
    --len;
    ++bufp;
    if (req() != OK)  {
      break;
    }
    if (XPHASE() != cur_phase) {
      break;
    }
  }
  return len;
}

#define sc_status	sc_rdbyte
#define sc_msgin	sc_rdbyte
/*==========================================================================*
 *				sc_rdbyte				    *
 *==========================================================================*/
PRIVATE int sc_rdbyte()
{
  int data0;

  if (bus_active() != OK) {
    return ERROR;
  }

  data0 = reg_read(R_IDATA);
  ack_req();
  /* return read byte */
  return data0;
}


/*==========================================================================*
 *				sc_command				    *
 *==========================================================================*/
PRIVATE int sc_command(cdb)
unsigned char *cdb;
{
  int len;

  /* return CDB length (depends on operation code) */
  switch((*cdb & 0xe0) >>5) {
  case 0:	/* Group 0 */
    len = 6;
    break;
  case 1:	/* Group 1 */
    len = 10;
    break;
  case 5:	/* Group 5 */
  default:
    len = 12;
    break;
  }

  if (cdb[0] != 3) { /* don't catch 'req sense' */
    ccdblen = len;
    memcpy(ccdb, cdb, ccdblen);
  }
  /* send the CDB */
  return sc_dout(cdb, (unsigned long)len);
}


/*==========================================================================*
 *			sc_com_dataio_stat				    *
 *==========================================================================*/
PRIVATE int sc_com_dataio_stat(cdb, datap, len)
unsigned char *cdb;
unsigned char *datap;
unsigned long len;
{
  int stat0 = ERROR;

  if (bus_waitphase(XP_COMMAND) == OK) {
    if (sc_command(cdb) == 0) {
      switch(XPHASE()) {
      case XP_DATAIN:
	if (sc_din(datap, len) == ERROR) {
	  goto sc_com_error;
	}
	break;
      case XP_DATAOUT:
	if (sc_dout(datap, len) == ERROR) {
	  goto sc_com_error;
	}
	break;
      default:
	break;
      }

      if (bus_waitphase(XP_STAT) == OK) {
	stat0 = sc_status();
	if (bus_waitphase(XP_MSGIN) == OK) {
	  if (sc_msgin() == MSG_COMPLETE) {
	    /* return status */
	    return stat0;
	  }
	}
      }
    }
  }
sc_com_error:
  bus_reset();
  if (curdev >= 0) {
    dump_cdb("timeout");
  }
  return ERROR;
}

/*==========================================================================*
 *			sc_test_unit_ready				    *
 *==========================================================================*/
PRIVATE int sc_test_unit_ready(target_id, lun)
int target_id, lun;
{
  unsigned char cdb[6];
  int ret;

  sense_key = 0;
  memset(cdb, 0, 6);
  /* op code = 0x00 */
  set_lun(lun, cdb);

  if ((ret = bus_select(target_id)) == OK) {
    if ((ret = sc_com_dataio_stat(cdb, NULL, 0L)) != ERROR) {
      switch(ret & ST_MASK) {
      case ST_CHKCOND:
	ret = sc_request_sense(target_id, lun);
	break;
      case ST_GOOD:    
	ret = OK;
	break;
      default:
	ret = ERROR;
	break;
      }
    }
  }
  return ret;
}


/*==========================================================================*
 *				sc_rewind				    *
 *==========================================================================*/
PRIVATE int sc_rewind(target_id, lun, immd)
int target_id, lun, immd;
{
  unsigned char cdb[6];
  int ret, count;

  sense_key = 0;
  count = 2;
retry_rw:
  memset(cdb, 0, 6);
  /* op code = 0x01 */
  cdb[0] = 1;
  cdb[1] = immd; /* immediate flag */
  set_lun(lun, cdb);

  if ((ret = bus_select(target_id)) == OK) {
    if ((ret = sc_com_dataio_stat(cdb, NULL, 0L)) != ERROR) {
      switch(ret & ST_MASK) {
      case ST_CHKCOND:
	switch(sc_request_sense(target_id, lun)) {
	case ST_GOOD:
	  ret = OK;
	  break;
	case SK_UNITATN:
	  if (--count > 0) {
	    goto retry_rw;
	  }
	default:
	  ret = ERROR;
	  break;
	}
	break;
      case ST_GOOD:    
	ret = OK;
	break;
      default:
	ret = ERROR;
	break;
      }
    }
  }
  return ret;
}

/*==========================================================================*
 *			sc_request_sense				    *
 *==========================================================================*/
PRIVATE int sc_request_sense(target_id, lun)
int target_id, lun;
{
  unsigned char cdb[6];
  int ret;

  memset(cdb, 0, 6);
  /* op code = 0x03 */
  cdb[0] = 0x03;
  cdb[4] = sizeof sense_buf;
  set_lun(lun, cdb);

  if ((ret = bus_select(target_id)) == OK) {
    if ((ret = sc_com_dataio_stat(cdb, sense_buf,
				  (unsigned long)(sizeof sense_buf))) == ERROR) {
      return ERROR;
    }
    if(sense_buf[0] != EXTSENSE) {
      return ERROR;
    }
  }
  sense_len = b32ul(sense_buf+3);
  sense_key = sense_buf[2];
  if (curdev >= 0 && (sense_key & SK_MASK) != SK_UNITATN) {
    dump_cdb("failed");
    dump_sense();
    milli_delay(5000);
  }
  return sense_key & (SK_EOF|SK_EOM|SK_ILL|SK_MASK);
}

/*==========================================================================*
 *			sc_rd_block_limits				    *
 *==========================================================================*/
PRIVATE int sc_rd_block_limits(target_id, lun, datap)
int target_id, lun;
unsigned char *datap;
{
  unsigned char cdb[6];
  int ret;

  sense_key = 0;
  memset(cdb, 0, 6);
  /* op code = 0x05 */
  cdb[0] = 0x05;
  set_lun(lun, cdb);

  if ((ret = bus_select(target_id)) == OK) {
    if ((ret = sc_com_dataio_stat(cdb, datap, 6L)) != ERROR) {
      switch(ret & ST_MASK) {
      case ST_GOOD:    
	ret = OK;
	break;
      default:
	ret = ERROR;
	break;
      }
    }
  }
  return ret;
}

/*==========================================================================*
 *				sc_write_filemarks			    *
 *==========================================================================*/
PRIVATE int sc_write_filemarks(target_id, lun, num)
int target_id, lun;
unsigned long num;
{
  unsigned char cdb[6];
  int ret, count;

  sense_key = 0;
  count = 2;
retry_rw:
  memset(cdb, 0, 6);
  /* op code = 0x10 */
  cdb[0] = 0x10;
  cdb[2] = (num & 0xff0000) >> 16;
  cdb[3] = (num & 0xff00) >> 8;
  cdb[4] = num & 0xff;
  set_lun(lun, cdb);

  if ((ret = bus_select(target_id)) == OK) {
    if ((ret = sc_com_dataio_stat(cdb, NULL, 0L)) != ERROR) {
      switch(ret & ST_MASK) {
      case ST_CHKCOND:
	switch(sc_request_sense(target_id, lun)) {
	case ST_GOOD:
	  ret = OK;
	  break;
	case SK_UNITATN:
	  if (--count > 0) {
	    goto retry_rw;
	  }
	default:
	  ret = ERROR;
	  break;
	}
	break;
      case ST_GOOD:    
	ret = OK;
	break;
      default:
	ret = ERROR;
	break;
      }
    }
  }
  return ret;
}

/*==========================================================================*
 *				sc_space				    *
 *==========================================================================*/
PRIVATE int sc_space(target_id, lun, cond, spclen)
int target_id, lun;
int cond;
long spclen;	/* signed */
{
  unsigned char cdb[6];
  int ret, count;

  sense_key = 0;
  count = 2;
retry_rw:
  memset(cdb, 0, 6);
  /* op code = 0x11 */
  cdb[0] = 0x11;
  cdb[1] = cond;
  cdb[2] = (spclen < 0? 0x80: 0)|((spclen & 0x7f0000) >> 16);
  cdb[3] = (spclen & 0xff00) >> 8;
  cdb[4] = spclen & 0xff;
  set_lun(lun, cdb);

  if ((ret = bus_select(target_id)) == OK) {
    no_phase_timeout = 1;	/* this might takes longer time */
    ret = sc_com_dataio_stat(cdb, NULL, 0L);
    no_phase_timeout = 0;
    if (ret != ERROR) {
      switch(ret & ST_MASK) {
      case ST_CHKCOND:
	switch(sc_request_sense(target_id, lun)) {
	case ST_GOOD:
	  ret = OK;
	  break;
	case SK_UNITATN:
	  if (--count > 0) {
	    goto retry_rw;
	  }
	default:
	  ret = ERROR;
	  break;
	}
	break;
      case ST_GOOD:    
	ret = OK;
	break;
      default:
	ret = ERROR;
	break;
      }
    }
  }
  return ret;
}

/*==========================================================================*
 *				sc_inquiry				    *
 *==========================================================================*/
PRIVATE int sc_inquiry(target_id, lun, datap, datalen)
int target_id, lun;
unsigned char *datap;
int datalen;
{
  unsigned char cdb[6];
  int ret;

  sense_key = 0;
  memset(cdb, 0, 6);
  /* op code = 0x12 */
  cdb[0] = 0x12;
  cdb[4] = datalen;
  set_lun(lun, cdb);

  if ((ret = bus_select(target_id)) == OK) {
    if ((ret = sc_com_dataio_stat(cdb, datap,
				  (unsigned long)datalen)) != ERROR) {
      switch(ret & ST_MASK) {
      case ST_GOOD:    
	ret = OK;
	break;
      default:
	ret = ERROR;
	break;
      }
    }
  }
  return ret;
}

/*==========================================================================*
 *				sc_erase				    *
 *==========================================================================*/
PRIVATE int sc_erase(target_id, lun)
int target_id, lun;
{
  unsigned char cdb[6];
  int ret, count;

  sense_key = 0;
  count = 2;
retry_rw:
  memset(cdb, 0, 6);
  /* op code = 0x19 */
  cdb[0] = 0x19;
  cdb[1] = 1;
  set_lun(lun, cdb);

  if ((ret = bus_select(target_id)) == OK) {
    no_phase_timeout = 1;	/* this might takes longer time */
    ret = sc_com_dataio_stat(cdb, NULL, 0L);
    no_phase_timeout = 0;
    if (ret != ERROR) {
      switch(ret & ST_MASK) {
      case ST_CHKCOND:
	switch(sc_request_sense(target_id, lun)) {
	case ST_GOOD:
	  ret = OK;
	  break;
	case SK_UNITATN:
	  if (--count > 0) {
	    goto retry_rw;
	  }
	default:
	  ret = ERROR;
	  break;
	}
	break;
      case ST_GOOD:    
	ret = OK;
	break;
      default:
	ret = ERROR;
	break;
      }
    }
  }
  return ret;
}

/*==========================================================================*
 *				sc_mode_sense				    *
 *==========================================================================*/
PRIVATE int sc_mode_sense(target_id, lun, datap, datalen)
int target_id, lun;
unsigned char *datap;
int datalen;
{
  unsigned char cdb[6];
  int ret;

  sense_key = 0;
  switch(sc_test_unit_ready(target_id, lun)) {
  case SK_UNITATN:
  case ST_GOOD:
    ret = OK;
    break;
  default:
    return ERROR;
  }

  memset(cdb, 0, 6);
  /* op code = 0x1a */
  cdb[0] = 0x1a;
  cdb[4] = datalen;
  set_lun(lun, cdb);

  if ((ret = bus_select(target_id)) == OK) {
    if ((ret = sc_com_dataio_stat(cdb, datap,
				  (unsigned long)datalen)) != ERROR) {
      switch(ret & ST_MASK) {
      case ST_CHKCOND:
	ret = sc_request_sense(target_id, lun);
	break;
      case ST_GOOD:    
	ret = OK;
	break;
      default:
	ret = ERROR;
	break;
      }
    }
  }
  return ret;
}


/*==========================================================================*
 *				sc_load					    *
 *==========================================================================*/
PRIVATE int sc_load(target_id, lun, load)
int target_id, lun, load;
{
  unsigned char cdb[6];
  int ret, count;

  sense_key = 0;
  count = 2;
retry_rw:
  memset(cdb, 0, 6);
  /* op code = 0x1b */
  cdb[0] = 0x1b;
  cdb[1] = 1; /* immediate flag */
  if (load) {
    cdb[4] = 3;	/* retension & load, otherwise unload */
  }
  set_lun(lun, cdb);

  if ((ret = bus_select(target_id)) == OK) {
    if ((ret = sc_com_dataio_stat(cdb, NULL, 0L)) != ERROR) {
      switch(ret & ST_MASK) {
      case ST_CHKCOND:
	switch(sc_request_sense(target_id, lun)) {
	case ST_GOOD:
	  ret = OK;
	  break;
	case SK_UNITATN:
	  if (--count > 0) {
	    goto retry_rw;
	  }
	default:
	  ret = ERROR;
	  break;
	}
	break;
      case ST_GOOD:    
	ret = OK;
	break;
      default:
	ret = ERROR;
	break;
      }
    }
  }
  return ret;
}

/*==========================================================================*
 *				sc_rw					    *
 *==========================================================================*/
PRIVATE int sc_rw(target_id, lun, datap, bsiz, datalen, blockno, op, type)
int target_id, lun;
unsigned char *datap;
unsigned int bsiz;
unsigned long datalen, blockno;
int op;		/* SCSI operation code */
int type;	/* =SC_DISK/SC_TAPE */
{
  unsigned char cdb[6];
  int ret, count;

  sense_key = 0;
  count = 2;
retry_rw:
  memset(cdb, 0, 6);
  /* op code = SC_READ:0x08, SC_WRITE:0x0a */
  cdb[0] = op;
  switch(type) {
  case SC_DISK:
    cdb[1] = (blockno & 0x1f0000) >> 16;
    cdb[2] = (blockno & 0xff00) >> 8;
    cdb[3] = blockno & 0xff;
    cdb[4] = datalen;
    break;
  case SC_TAPE:
    cdb[1] = (bsiz!=1);
    cdb[2] = (datalen & 0xff0000) >> 16;
    cdb[3] = (datalen & 0xff00) >> 8;
    cdb[4] = datalen & 0xff;
    break;
  }
  set_lun(lun, cdb);

  if ((ret = bus_select(target_id)) == OK) {
    if ((ret = sc_com_dataio_stat(cdb, datap, datalen*bsiz)) != ERROR) {
      switch(ret & ST_MASK) {
      case ST_CHKCOND:
	switch(sc_request_sense(target_id, lun)) {
	case ST_GOOD:
	  ret = OK;
	  break;
	case SK_UNITATN:
	  if (--count > 0) {
	    goto retry_rw;
	  }
	default:
	  ret = ERROR;
	  break;
	}
	break;
      case ST_GOOD:    
	ret = OK;
	break;
      default:
	ret = ERROR;
	break;
      }
    }
  }
  return ret;
}


/*==========================================================================*
 *				dump_cdb				    *
 *==========================================================================*/
PRIVATE void dump_cdb(msg)
char *msg;
{
  unsigned char n, len;

  printf("%s%d: %s  cmd =", (curdev&0x80)?"st":"sd", curdev&0x7f, msg);
  for (n = 0; n < ccdblen; n++) {
    printf(" %x", ccdb[n]);
  }
  printf("\n");
}


/*==========================================================================*
 *				dump_sense				    *
 *==========================================================================*/
PRIVATE void dump_sense()
{
  unsigned char n, len;

  printf("%s%d error:  sense key (0x%x)\n    sense =",
	 (curdev&0x80)?"st":"sd", curdev&0x7f, (sense_buf[2]&0x0f));
  len = 8 + sense_buf[7];
  for (n = 0; n < len; n++) {
    printf(" %x", sense_buf[n]);
  }
  printf("\n");
}


/*==========================================================================*
 *				mini_check				    *
 *==========================================================================*/
PRIVATE int mini_check() {
  int data;
  int ret = ERROR;

  ctrl_out(R_ODATA, 0x20);
  ctrl_out(R_ODATA, 0);
  reg_setbit(R_INITIATOR, R1_DATA);
  reg_write(R_ODATA, 0x55);
  if ((data = reg_read(R_IDATA)) == 0x55) {
    reg_write(R_ODATA, 0xaa);
    if ((data = reg_read(R_IDATA)) == 0xaa) {
      reg_clrbit(R_INITIATOR, R1_DATA);
      ret = OK;
    }
  }
  return ret;
}

/*==========================================================================*
 *				mini_find				    *
 *==========================================================================*/
PRIVATE int mini_find() {
  unsigned int port;

  for (port = 0; port <= MAX_PORT; port++) {
    data_port = port_tab[port];
    stat_port = data_port + 1;
    ctrl_port = stat_port + 1;
    if (mini_check() == OK) {
      printf("MiniSCSI host adapter detected at 0x%04x\n", data_port);
      return OK;
    }
  }
  return ERROR;
}

/*==========================================================================*
 *				mini_reset				    *
 *==========================================================================*/
PRIVATE void mini_reset()
{

  bus_reset();
  /* clear error/interrupt registers */
  reg_read(R_RSTEI);
}
