/* sym.c */

#include "const.h"
#include "type.h"
#include "var.h"
#include "/usr/include/a.out.h"		/* full pathname to get right one! */

#define SYMSIZE_MAGIC ((peekoff_t) 2)

#define anticlick( off ) ((off) * CLICK_SIZE)/* good only for unsigned off's */
#define click( off ) ((off) / CLICK_SIZE)
#define clickrest( off ) ((unsigned) (off) % CLICK_SIZE)
#define NSYMPROC 4		/* kernel, mm, fs, init */

extern unsigned sizes[2 * NSYMPROC];

struct symtab_s
{
	char *stname;
	segment_t stseg;
	struct nlist *start;
	struct nlist *end;
	segment_t text;
	segment_t data;
	unsigned nsym;
};

PRIVATE struct symtab_s proc_sym[NSYMPROC] =
{
	{ "kernel" }, { "mm" }, { "fs" }, { "init" },
};

PUBLIC void syminit()
{
	static bool_t already = FALSE;
	segment_t base;
	int p;
	u16_t symoff;
	register struct symtab_s *tp;

	if ( !already )
	{
		already = TRUE;
		info();
		for ( base = codeseg(), tp = &proc_sym[0], p = 0; tp < &proc_sym[NSYMPROC];
		      base += sizes[p] + sizes[p + 1], ++tp, p += 2 )
		{
			if ( (symoff = peekw( base, SYMSIZE_MAGIC )) == 0 )
				/* symbols not loaded - p should be 0 so all tables remain null */
				break;
			tp->text = base;
			tp->stseg = (tp->data = base + sizes[p]) + click( symoff );
			tp->start = (struct nlist *) clickrest( symoff );
			tp->nsym = (anticlick( sizes[p + 1] ) - symoff) / sizeof (struct nlist);
			tp->end = tp->start + tp->nsym;
			/* sort on value only, name search not used much and storage a problem */
			outstr( "Sorting " );
			outh16( tp->nsym );
			outspace();
			outstr( tp->stname );
			outstr( " symbols ... " );
			sortsyms( tp->start, tp->end, tp->stseg );
			outnl();
		}
	}
}

PRIVATE struct nlist *buildsym( sp, tableseg )
struct nlist *sp;
segment_t tableseg;
{
	static struct nlist sym;
	register char *s;
	register char *t;
	char *tend;

	for ( s = sp->n_name, t = sym.n_name, tend = t + sizeof sym;
	      t < tend; ++s, ++t )
		*t = peekb( tableseg, (peekboff_t) s );
	return &sym;
}

PUBLIC struct nlist *findsname( name, where, allflag )
char *name;
int where;
bool_pt allflag;
{
	struct nlist *matchsp;
	char *s;
	unsigned char sclass;
	int schar;
	char *send;
	register struct nlist *sp;
	segment_t tableseg;
	register struct symtab_s *tp;

	for ( tp = &proc_sym[0]; tp < &proc_sym[NSYMPROC]; ++tp )
		/* find process to use */
		if ( tp->text == uptr.seg )
		{
			if ( allflag )
			{
				/* find and print all matching symbols */
				for ( sp = tp->start, tableseg = tp->stseg; sp < tp->end; ++sp )
				{
					if ( symprefix( name, sp, tableseg ) )
					{
						matchsp = buildsym( sp, tableseg );
						for ( s = matchsp->n_name, send = s + sizeof matchsp->n_name;
						      *s != 0 && s < send; ++s )
							outbyte( *s );
						for ( ; s <= send; ++s )
							outspace();
						switch( matchsp->n_sclass & N_SECT )
						{
							case N_ABS: schar = 'a'; break;
							case N_TEXT: schar = 't'; break;
							case N_DATA: schar = 'd'; break;
							case N_BSS: schar = 'b'; break;
							default: schar = '?'; break;
						}
						if ( (matchsp->n_sclass & N_CLASS) == C_EXT && schar != '?' )
							schar += 'A' - 'a';
						outbyte( schar );
						outspace();
						outh16( (u16_t) matchsp->n_value );
						if ( !outnl() )
							break;
					}
				}
				return NULL;
			}
			else
			{
				/* find symbol by dumb linear search */
				for ( sp = tp->start, tableseg = tp->stseg; sp < tp->end; ++sp )
				{
					sclass = peekw( tableseg, (peekoff_t) &sp->n_sclass ) & N_SECT;
					if ( (where == CSEG && sclass == N_TEXT ||
					      where != CSEG && (sclass == N_DATA || sclass == N_BSS)) &&
							 symeq( name, sp, tableseg ) )
						return buildsym( sp, tableseg );
				}
				return NULL;
			}
		}
	return NULL;
}

PUBLIC struct nlist *findsval( value, where )
offset_t value;
int where;
{
	int left;
	int middle;
	int right;
	unsigned char sclass;
	register struct nlist *sp;
	segment_t tableseg;
	register struct symtab_s *tp;

	for ( tp = &proc_sym[0]; tp < &proc_sym[NSYMPROC]; ++tp )
		/* find process to use */
		if ( tp->text == uptr.seg )
		{
			prompt = *tp->stname;
			/* find last symbol with value <= desired one by binary search */
			for ( left = 0, right = tp->nsym - 1, tableseg = tp->stseg;
			      left <= right; )
			{
				middle = (left + right) / 2;
				sp = tp->start + middle;
				if ( value < peekw( tableseg, (peekoff_t) &sp->n_value ) )
					right = middle - 1;
				else
					left = middle + 1;
			}
			if ( right >= 0 )
				/* otherwise tp->start + right may wrap around to > tp->start !! */
				for ( sp = tp->start + right; sp >= tp->start; --sp )
				{
					sclass = peekw( tableseg, (peekoff_t) &sp->n_sclass ) & N_SECT;
					if ( (where == CSEG && sclass == N_TEXT ||
								where != CSEG && (sclass == N_DATA || sclass == N_BSS)) )
					return buildsym( sp, tableseg );
				}
			return NULL;
		}
	if ( where == CSEG )
		prompt = '?';
	return NULL;
}

PUBLIC void outsym( sp, off )
struct nlist *sp;
offset_t off;
{
	register char *s;
	char *send;

	for ( s = sp->n_name, send = s + sizeof sp->n_name; *s != 0 && s < send; ++s )
		outbyte( *s );
	if ( (off -= sp->n_value) != 0 )
	{
		outbyte( '+' );
		if ( off >= 65536L )
			outh32( off );
		else if ( off >= 256 )
			outh16( (u16_t) off );
		else
			outh8( (u8_pt) off );
	}
}

PUBLIC void setproc( c, pdptr, pmptr )
char_pt c;
struct adr *pdptr;
struct adr *pmptr;
{
	register struct symtab_s *tp;

	for ( tp = &proc_sym[0]; tp < &proc_sym[NSYMPROC]; ++tp )
		if ( tp->stname[0] == c )
		{
			uptr.seg = tp->text;
			pmptr->seg = pdptr->seg = tp->data;
			prompt = c;
			return;
		}
}

/* shell sort symbols on value */

PRIVATE void sortsyms( array, top, tableseg )
struct nlist *array;
struct nlist *top;
segment_t tableseg;
{
	int gap;
	int i;
	int j;
	register struct nlist *left;
	register struct nlist *right;
	int size;

	size = top - array;
	/* choose gaps according to Knuth V3 p95 */
	for ( gap = 1, i = 4; (j = 3 * i + 1) < size; gap = i, i = j )
		;
	do
	{
		for ( j = gap; j < size; ++j )
			for ( i = j - gap; i >= 0; i -= gap )
			{
				left = array + i; 
				right = array + (i + gap);
				if ( (unsigned) peekw( tableseg, (peekoff_t) &left->n_value )
				     <= peekw( tableseg, (peekoff_t) &right->n_value ) )
					break;
				symswap( left, right, tableseg, sizeof *left );
			}
	}
	while ( (gap /= 3) != 0 );
}

PRIVATE bool_pt symeq( t, sp, tableseg )
register char *t;
struct nlist *sp;
segment_t tableseg;
{
	register char *s;
	char *send;

	for ( s = sp->n_name, send = s + sizeof sp->n_name; s < send; ++s, ++t )
		if ( *t != peekb( tableseg, (peekboff_t) s ) )
			return FALSE;
	return TRUE;
}

PRIVATE bool_pt symprefix( t, sp, tableseg )
register char *t;
struct nlist *sp;
segment_t tableseg;
{
	register char *s;
	char *send;

	for ( ; *t == '_'; ++t )
		;
	for ( s = sp->n_name, send = s + sizeof sp->n_name;
	      s < send && peekb( tableseg, (peekboff_t) s ) == '_'; ++s )
		;
	for ( ; *t != 0 && s < send; ++s, ++t )
		if ( *t != peekb( tableseg, (peekboff_t) s ) )
			return FALSE;
	return TRUE;
}

#ifdef LINT

/* prototypes for assembler functions */

/* ARGSUSED */
PRIVATE void symswap( left, right, tableseg, length )
struct nlist *left;
struct nlist *right;
segment_t tableseg;
unsigned length;        /* must be even */
{
}

#endif
