/* keyboard.c */

#include "const.h"
#include "keyboard.h"

#define KEYBD 0x60
#define KBIT  0x80
#define PORTB 0x61

#define FIRST_RELEASE_CODE 0x80 /* codes below this are keypresses */
                                /* from here to 0xff are releases */
#define PREFIX0      0xe0       /* prefix for single scan code */
#define PREFIX1      0xe1       /* prefix for double scan code */
#define N_SCAN_CODES 0x80
#define RELEASE_CODE_BIT   0x80

/* states, 0 is normal */

#define S_PRE0  1               /* just seen PREFIX0 */
#define S_PRE1  2               /* just seen PREFIX1 */
#define S_PRE1A 3               /* just seen PREFIX1-LEFT_CTRL */
#define S_PRE1B 4               /* just seen PREFIX1-LEFT_SHIFT */

/*
  Ordinary scan codes candidates for re-mapping only.
  The other scan codes are declared in a more localized way in our tables.
*/

#define SC_DEL          0x53
#define SC_DN           0x50
#define SC_END          0x4f
#define SC_ENTER        0x1c
#define SC_HOME         0x47
#define SC_INS          0x52
#define SC_LEFT         0x4b
#define SC_LEFT_ALT     0x38
#define SC_LEFT_CTRL    0x1d
#define SC_LEFT_SHIFT   0x2a
#define SC_NUM_SHIFT    0x45
#define SC_PGDN         0x51
#define SC_PGUP         0x49
#define SC_PRTSC        0x37
#define SC_RIGHT        0x4d
#define SC_SLASH        0x35
#define SC_UP           0x48

/*
  Re-mapped scan codes from enhanced keyboard.
  The new codes can be any unique codes < 0x80.
  Our area map, shift map and key maps depend on the choices made.
*/

#define SC_CPAD_DEL     0x79
#define SC_CPAD_DN      0x76
#define SC_CPAD_END     0x75
#define SC_CPAD_HOME    0x70
#define SC_CPAD_LEFT    0x73
#define SC_CPAD_INS     0x78
#define SC_CPAD_PGDN    0x77
#define SC_CPAD_PGUP    0x72
#define SC_CPAD_RIGHT   0x74
#define SC_CPAD_UP      0x71
#define SC_NPAD_ENTER   0x7f
#define SC_NPAD_SLASH   0x7e
#define SC_PAUSE        0x7d  /* 0xe1-LeftCtrl-NumLock-0xe1-<release codes> */
#define SC_PRINTSCREEN  0x7c  /* 0xe1-LeftShift-0x37-0xe0-<release codes> */
#define SC_RIGHT_ALT    0x7a
#define SC_RIGHT_CTRL   0x7b

/*
  Map of keyboard areas.
    Cursor Pad  0x70 to 0x79, 0x7e to 0x7f (our re-mapping).
    F1-F10      0x3b to 0x64
    F11-F12     0x57 to 0x58
    Numeric     0x47 to 0x53
    Other       0x54 to 0x7f not C/F/N or Right Alt/Ctrl or (prefix & 0x7f)
*/
PRIVATE area_t area_map[N_SCAN_CODES] =
{
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, FKEY, FKEY, FKEY, FKEY, FKEY,
  FKEY, FKEY, FKEY, FKEY, FKEY, 0, 0, NKEY,
  NKEY, NKEY, NKEY, NKEY, NKEY, NKEY, NKEY, NKEY,
  NKEY, NKEY, NKEY, NKEY, OKEY, OKEY, OKEY, FKEY,
  FKEY, OKEY, OKEY, OKEY, OKEY, OKEY, OKEY, OKEY,
  0, 0, OKEY, OKEY, OKEY, OKEY, OKEY, OKEY,
  OKEY, OKEY, OKEY, OKEY, OKEY, OKEY, OKEY, OKEY,
  CKEY, CKEY, CKEY, CKEY, CKEY, CKEY, CKEY, CKEY,
  CKEY, CKEY, 0, 0, OKEY, OKEY, NKEY, NKEY,
};

/*
  Map of shift scan codes.
    Caps Lock    0x3a
    Ins          0x52
    Ins          0x52 preceded by 0xe0 (state kept is inadequate)
    Left Alt     0x38
    Left Ctrl    0x1d
    Left Shift   0x2a
    Num Lock     0x45
    Right Alt    0x38 preceded by 0xe0
    Right Ctrl   0x1d preceded by 0xe0
    Right Shift  0x36
    Scroll Lock  0x46
*/
PRIVATE shift_t shift_map[N_SCAN_CODES] =
{
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, LEFT_CTRL, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, LEFT_SHIFT, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, RIGHT_SHIFT, 0,
  LEFT_ALT, 0, CAPS_SHIFT, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, NUM_SHIFT, SCROLL_SHIFT, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, INS_SHIFT, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  INS_SHIFT, 0, RIGHT_ALT, RIGHT_CTRL, 0, 0, 0, 0,
};

/*
  Scan codes to ASCII.
  \000 means a special key which is handled by another table.
  The numeric keypad '.' or DEL key is always mapped to DEL.
  The other keypad keys are always mapped to numbers.
  A higher level routine can remap the keypad as it likes using the
  shift state.
*/

/* Scan codes to ASCII for unshifted keys for PC through enhanced AT */
PRIVATE unsigned char unshifted[N_SCAN_CODES] =
{
     0,  27, '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '-', '=','\b',
  '\t', 'q', 'w', 'e', 'r', 't', 'y', 'u', 'i', 'o', 'p', '[', ']','\r',
     0, 'a', 's', 'd', 'f', 'g', 'h', 'j', 'k', 'l', ';','\'', '`',
     0,'\\', 'z', 'x', 'c', 'v', 'b', 'n', 'm', ',', '.', '/',   0, '*',
     0, ' ',   0,
   'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J',
     0,   0,
   '7', '8', '9', '-',
   '4', '5', '6', '+',
   '1', '2', '3',
   '0', 127,
   212, 213, 214,
   'K', 'L',
   217, 218, 219, 220, 221, 222, 223,
   224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236,237,238,239,
   '7', '8', '9',
   '4', '6',
   '1', '2', '3',
   '0', 127,
     0,   0, '*', 253, '/', '\r',
};
/*
  The first 5 rows correspond to rows on the keyboard.
  Function keys return 1234567890 plus their shift flags.
  The numeric keypad returns 789-456+1230..
  Our mapping of enhanced AT codes in disorganized order is at the end.
*/

/* Scan codes to ASCII for shifted keys for PC through enhanced AT */
PRIVATE char shifted[N_SCAN_CODES + 1] =
{
     0,  27, '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+','\b',
  '\t', 'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P', '{', '}','\r',
     0, 'A', 'S', 'D', 'F', 'G', 'H', 'J', 'K', 'L', ':','\"', '~',
     0, '|', 'Z', 'X', 'C', 'V', 'B', 'N', 'M', '<', '>', '?',   0, '*',
     0, ' ',   0,
   'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J',
     0,   0,
   '7', '8', '9', '-',
   '4', '5', '6', '+',
   '1', '2', '3',
   '0', 127,
   212, 213, 214,
   'K', 'L',
   217, 218, 219, 220, 221, 222, 223,
   224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236,237,238,239,
   '7', '8', '9',
   '4', '6',
   '1', '2', '3',
   '0', 127,
     0,   0, '*', 253, '/', '\r',
};

/*
  Translate a scan code from a PC-style keyboard for a given shift state.
  All "normal" keys return an ASCII code packed with the shift state.
*/

PUBLIC void parsecode();

#ifdef DEBUG_KEYBOARD

main()
{
  static struct echar_s ech;
  scancode_pt code;
  scancode_pt oldcode;

  oldcode = inportb( KEYBD );
  while ( 1 )
    if ( (code = inportb( KEYBD )) != oldcode )
    {
      parsecode( code, &ech );
      if ( ech.area != RKEY )
      {
        if ( ech.ch == 3 )
          exit( 0 );
        printf( "%c %2d %04x %02x\n",
                ech.ch >= 32 ? ech.ch : 'x', ech.area, ech.shift, ech.ch );
      }
      oldcode = code;
    }
}

#endif /* DEBUG_KEYBOARD */

PUBLIC void parsecode( code, eptr )
scancode_pt code;
register struct echar_s *eptr;
{
  scancode_pt newcode;
  shift_pt newshift;

  eptr->area = RKEY;        /* prepare to return early */
  if ( eptr->state != 0 )
  {
    /*
      Re-map known special keys so rest of routine can be table-driven.
      Hardware does it the *wrong* way for backward compatibility.
    */
    newcode = code & ~RELEASE_CODE_BIT;
    switch( eptr->state )
    {
    case S_PRE0:
      switch( newcode )
      {
      case SC_DEL: newcode = SC_CPAD_DEL; break;
      case SC_DN: newcode = SC_CPAD_DN; break;
      case SC_END: newcode = SC_CPAD_END; break;
      case SC_ENTER: newcode = SC_NPAD_ENTER; break;
      case SC_HOME: newcode = SC_CPAD_HOME; break;
      case SC_INS: newcode = SC_CPAD_INS; break;
      case SC_LEFT: newcode = SC_CPAD_LEFT; break;
      case SC_LEFT_ALT: newcode = SC_RIGHT_ALT; break;
      case SC_LEFT_CTRL: newcode = SC_RIGHT_CTRL; break;
      case SC_PGDN: newcode = SC_CPAD_PGDN; break;
      case SC_PGUP: newcode = SC_CPAD_PGUP; break;
      case SC_RIGHT: newcode = SC_CPAD_RIGHT; break;
      case SC_SLASH: newcode = SC_NPAD_SLASH; break;
      case SC_UP: newcode = SC_CPAD_UP; break;
      }
      break;
    case S_PRE1:
      switch( newcode )
      {
      case SC_LEFT_CTRL: eptr->state = S_PRE1A; return;
      case SC_LEFT_SHIFT: eptr->state = S_PRE1B; return;
      }
      break;
    case S_PRE1A:
      if ( newcode == SC_NUM_SHIFT )
        newcode = SC_PAUSE;
      break;
    case S_PRE1B:
      if ( newcode == SC_PRTSC )
        newcode = SC_PRINTSCREEN;
      break;
    }
    if ( code >= FIRST_RELEASE_CODE )
      newcode |= RELEASE_CODE_BIT;
    code = newcode;
    eptr->state = 0;
  }

  if ( code == PREFIX0 )
  {
    eptr->state = S_PRE0;
    return;
  }
  if ( code == PREFIX1 )
  {
    eptr->state = S_PRE1;
    return;
  }

  if ( code >= FIRST_RELEASE_CODE )
  {
    /* key release, drop shift state if this is a shift key */
    if ( (newshift = (shift_map - FIRST_RELEASE_CODE)[code]) != 0 )
      eptr->shift &= ~newshift;
    return;
  }

  /* key press */
  if ( (newshift = shift_map[code]) != 0 &&
       (code != SC_INS || !(eptr->shift & NUM_LOCK)) )
  {
    /* toggle if lock and keypress is new, record shift */
    if ( newshift & (CAPS_SHIFT | INS_SHIFT | NUM_SHIFT | SCROLL_SHIFT) &&
         !(newshift & eptr->shift) )
      /* same ratio for all */
      eptr->shift ^= newshift / (CAPS_SHIFT / CAPS_LOCK);
    eptr->shift |= newshift;
    if ( newshift != INS_SHIFT )
    {
      eptr->area = RKEY;
      return;
    }
  }
  if ( (eptr->area = area_map[code]) != 0 )
  {
    /*
      Non-ASCII key.
      Return it as unique ASCII with nonzero area code.
      Caller can decode it further if required.
      Function keys are converted to letters.
      Numeric keypad keys are converted to numbers and punctuation.
    */
    eptr->ch = unshifted[code];
    return;
  }
  if ( eptr->shift & (LEFT_CTRL | RIGHT_CTRL) &&
       (eptr->ch = shifted[code] - '@') < 0x20 )
    return;
  if ( eptr->shift & (LEFT_SHIFT | RIGHT_SHIFT) )
    eptr->ch = shifted[code];
  else
    eptr->ch = unshifted[code];
  if ( eptr->shift & CAPS_LOCK &&
       (eptr->ch & ~('a' - 'A')) - 'A' <= 'Z' -'A' )
    eptr->ch ^= 'a' - 'A';
}

PUBLIC scancode_pt scan_keyboard()
{
  /* fetch character from keyboard hardware and acknowledge it */

  scancode_pt code;
  unsigned status;

  code = inportb( KEYBD );        /* scan code */
  status = inportb( PORTB );      /* strobe keyboard to acknowledge */
  oportb( PORTB, status | KBIT ); /* strobe high */
  oportb( PORTB, status );        /* strobe low */
  return code;
}
