/*
			Minix Second Stage Boot Loader

In this program, all the elements of the Minix system are loaded into memory
from binary files on a filesystem.  All parts of the system are loaded
like any other Minix process, with Text, Data and Stack areas and command
options on a "command line" (which is actually in a binary format).

The routines in this file turn a raw disk into a filesystem.  The primary
routines are:
	inode = open_file(filename)
	a.out = load_file(inode, starting segment)
	count = read_file(inode, buffer)

12/2/88 Glen/B
	Construct this from the reminants of the old (CS548/CS497) boot
	program.
*/

/* include files to get inode, super block structs*/
#include <minix/const.h>
#include <minix/type.h>
#include <fs/const.h>
#include <fs/type.h>
#include <fs/inode.h>
#include <fs/super.h>
#include <stdio.h>
#include <errno.h>
#include <a.out.h>
#include "type.h"
#include "partition.h"
/*#include "hdparam.h"*/

/*#undef printf		/* defined in fs/const.h as 'printk' */
#undef getchar		/* don't want the macros */
#undef putchar
#undef putc
#undef getc

extern	int	errno;
	int	def_dev;			/* default (boot) device */
	char	glb_block[BLOCK_SIZE];		/* misc block buffer */
	int	cylsiz, tracksiz;		/* for low diskio.. */
	int	sizes[10];			/* kernel, mm, fs, init size */
	int	kds, ksp;			/* kernel data seg, sp */

/*#define TRACE*/
/*#define DEBUG*/
/*#define DEBUG1*/

/* Function Types */
int ll_Block_Read(/* int device, block; char *buffer */);
int read_block(/* struct super_block *super; int block; char *buffer; */);
struct super_block *read_super(/* int device; */);
struct part_entry *Read_Part(/* int device; */);
struct param *read_rom(/* int device; */);
struct inode *read_inode(/*struct super_block *sb; int inum;*/);
int read_file(/*struct inode *inode; char *buffer;*/);
int lload_blocks(/*struct super_block *super; char *buffer;
		int *list, nlist;*/);
struct inode *open_file(/*char *filename;*/);
char *device_map(/*char *filename; int *device;*/);
struct inode *follow_path(/*struct super_block *super; char *name*/);
next_name(/*char *from, *name;*/);
int search_dir(/*struct inode *inode; char *name;*/);

#define	READ	0
#define	WRITE	1

/*
 * Filesystem parameters which are not supplied in FS' header files
 */
#define NR_DIRECT_ZONES	NR_ZONE_NUMS-2	/* number of DIRECT zones */
#define IND_ZONE_NUM	7	/* i_zone offset of indirect zone */
#define DIND_ZONE_NUM	8	/* i_zone offset of double indirect block */
#define SECTOR_SIZE	512	/* sector size (bytes) */
#define HEAD_SIZE	32	/* a.out header size (stripped) */

/*
 * Hardware Parameters
 */
#define WINI_DISK	0x40	/* first device number of wini */
#define PART_PER_DRIVE	5	/* how many partitions per wini drive */

#define NSUPERCACHE	10	/* size of super block "cache" */
#define NINODECACHE	20	/* size of inode "cache" */
#define NEXECCACHE	10	/* size of a.out header "cache" */ 

/*
	Intra-Segment Copy

    copy(destination, source, length)
*/
copy(d,s,l)
char *d, *s;
int l;
{
	while(l--)	*d++ = *s++;
}

/*
	Zero Memory	(for clearing the bss segment)

    zero_mem(segment, offset, length)
*/
zero_mem(segment, offset, length)
unsigned int segment, offset, length;
{
	static char zero[1024];	/* this will get put in BSS, which is zeroed */
	int count;

#ifdef DEBUG
	printf("zero_mem: 0x%x:0x%x 0x%x\n", segment, offset, length);
#endif
	count = length / 1024 + 1; /* caveat! trashes to an even 1K boundary */
				/* (no prob for sequential process loading */
	while(count--) {
#ifdef DEBUG
	printf("zero_mem: 0x%x:0x%x #%d\n", segment, offset, count);
#endif
		lcopy(segment, offset, ds(), zero, 1024);
		segment += 64;
	}
}
/*
	Clicks

    calculate the number of clicks from bytes (a click is 16 bytes).
    Compensate for partial clicks by rounding UP.
*/
clicks(bytes)
long bytes;
{
	return((bytes%16) ? (bytes>>CLICK_SHIFT)+1 : (bytes>>CLICK_SHIFT));
}

/*
	Low-Level Block Read

	procedure low_level_block_read
		in device, block number, buffer address
		returns Error_Status

Performs the lowest level data read by calling the disk I/O services
supplied by the Operating System or ROM Support Routines.  No knowledge
of hard-disk partitions is exhibited.

*/
int ll_Block_Read(device, block, buffer)
int device; unsigned int block; char *buffer;
{
	int status;
#ifdef DEBUG1
	printf("ll_Block_Read: device=0x%x block=0x%x buffer=0x%x\n", device, block, buffer);
#endif
#ifdef DEBUG
	fgetc();
#endif
	status = diskio(READ, block*2, buffer, 1, device);
	if(status > 255) {
		errno = EIO;
		printf("Disk Error reading sector 1 of block %d\n",block);
		return(ERROR);
	}
	status = diskio(READ, block*2+1, buffer+SECTOR_SIZE, 1, device);
	if(status > 255) {
		errno = EIO;
		printf("Disk Error reading sector 2 of block %d\n",block);
		return(ERROR);
	}
	return(OK);
}

/*
	Intermediate-Level Block Read

	procedure read_block
		in Super_Block, block number, buffer address
		returns Error_Status

Reads a block from a disk by calling the Low-Level Block Read.  If
the disk is paritioned, the partition information (stored in the super block)
is consulted to determine the offset into the disk of the partition.

*/

int read_block(super, block, buffer)
SUPERBLOCK super;
int block;
char *buffer;
{	int device;
	unsigned int real_block;
#ifdef DEBUG
	printf("Block_Read..%u %ld\n", block, super->s_time);
#endif
	device = (super->s_dev < WINI_DISK) ? super->s_dev : 
			0x80 + ((super->s_dev - WINI_DISK) / PART_PER_DRIVE);
	real_block = block + (unsigned int) (super->s_time & 0xFFFF);
	return(ll_Block_Read(device, real_block, buffer));
}

/*
	Read Super-Block

	procedure read_super_-_block
		in device
		returns Super_Block_Ptr

Read the super-block off of the specified device, if this has not yet
been done.  A pointer to a super-block for the requested device is
returned.  This structure will NOT be overwritten by subsequent
requests for super-blocks on different devices.  NULL is returned if an
error occurs, and the error code will be stored in the variable
"errno".

If the disk being read from is a partition of a larger physical drive,
it's partition table is read and information about where the partition
lies on the physical device is stored in the super block.

Allocation of Super-Blocks is made out of a pool of pre-allocated
blocks.  A count of how many blocks have been allocated is kept
so that new blocks can be painlessly allocated; since no de-allocation
is allowed, this is sufficent.  Prior to allocating a new blocks,
all allocated blocks are searched for the wanted block.  If it is not
found, then a new one is allocated and loaded.

*/

/*
search sb list for this defive
if ! found
	have to load it!...
	allocate SB
	if partitioned disk (wini)
		read partition table
		build partial super-block
	else
		set non-partitioned parameters in SB
	read SB from disk
return sb

*/

SUPERBLOCK read_super(device)
int device;
{
	static struct super_block s_cache[NSUPERCACHE],
				*last_super = s_cache;
	SUPERBLOCK sb, last;
	struct part_entry *partition;
	long lowsec;		/* low sector of partition */

	Set_Param(device);
#ifdef TRACE
	printf("read_super: device=%d...", device);
#endif
	/* Search the cache for a possibly already loaded super-block
	 */
	for(sb = s_cache;
	    sb < last_super && sb->s_dev != device;
	    sb++) ;

	if (sb >= last_super) { /* load it */
#ifdef TRACE
		printf("loading super block\n");
#endif
		/* The Super-Block is not loaded.  Allocate a new one
		 * and load it.
		 */
		sb = last_super++;	/* allocate new SB */
		sb->s_dirt++;
		sb->s_dev = device;

		/* fix up device number and block number for partitioned
		 * disks (winchesters)
		 */
		if (device >= WINI_DISK) { /* partitioned WINI device */
			partition = Read_Part(device);
			sb->s_time = partition->lowsec/2;
			lowsec = partition->lowsec/2;
		} else {
			sb->s_time = 0;
			lowsec = 0;
		}

		if(read_block(sb, SUPER_BLOCK, glb_block) == OK) {
			copy(sb, glb_block, SUPER_SIZE);
			sb->s_dev = device;
			sb->s_time = lowsec;
			sb->s_dirt = 1;
#ifdef DEBUG
			dump_super(sb);
#endif
		} else {
			sb = NULL;
		}
	} else {
#ifdef TRACE
		printf("super block already loaded\n");
#endif
	}
	return(sb);
}

/*
	Set Parameters

	procedure Set_Param
		in device

Read and set the disk track and cylinder size parameters appropriately.
If the device is a hard disk, read the hard disk parameters from their
ROM tables.
*/

Set_Param(device)
int device;
{
/*X	struct param *param; X*/

	if(device < WINI_DISK) {
		tracksiz = 9;
		cylsiz = 18;
	} else {
		hd_param(0x80 + ((device - WINI_DISK) / PART_PER_DRIVE));
/*		param = read_rom((device - WINI_DISK) / PART_PER_DRIVE);
/*		tracksiz = 0x11;			/* not in my rom */
/*		cylsiz = tracksiz * param->nr_heads;*/
/*		cylsiz = tracksiz * 6;
 */	}
}

/*
	Read Partition Table

	procedure Read_Part
		in device
		return Ptr to partition entry

Read the partition table in sector 1 of a hard disk and return a pointer
to the partition entry for the device we're using.

*/
struct part_entry *Read_Part(device)
int device;
{
	int real_dev, partition;

	if(device >= WINI_DISK) {
		real_dev = 0x80 + ((device - WINI_DISK) / PART_PER_DRIVE);
		partition = ((device - WINI_DISK) % PART_PER_DRIVE) -1;
	} else {
		real_dev = device;
		partition = 0;
	}

#ifdef TRACE
	printf("Read_Part: device=0x%x partition=0x%x\n", real_dev, partition);
#endif
	ll_Block_Read(real_dev, 0, glb_block);
#ifdef DEBUG
	dump_prt(glb_block + (partition * sizeof(struct part_entry)) + TABLEOFFSET);
#endif
	return(glb_block + (partition * sizeof(struct part_entry)) + TABLEOFFSET);
}

/*
	Read i-node

	procedure read_i_-_node
		in super-block, i-node number
		returns Inode_Ptr

Read one i-node from the specified device.  If the inode has already
been read into the internal cache, it is not reread.  A pointer to an
internal i-node is returned (this holds more information than an
ordinary disk inode does).  NULL is returned if an error occurs, and
the error code will be stored in the variable "errno".

*/

INODE read_inode(sb, inum)
SUPERBLOCK sb;
int inum;
{
	static struct inode i_cache[NINODECACHE],
				*last_inode = i_cache;
	INODE i, last;

	int blk_off, i_off;

#ifdef TRACE
	printf("read_inode: inum=%d...",inum);
#endif
	/* Search the cache for a possibly already loaded i-node
	 */
	for(i = i_cache;
	    i < last_inode && (i->i_num != inum || i->i_dev != sb->s_dev);
	    i++) ;

	if(i >= last_inode ) {
#ifdef TRACE
		printf("loading inode\n");
#endif
		/* The i-node is not loaded.  Allocate a new one and load it
		 */
		i = last_inode++;	/* error check allocating too many */
		i->i_dirt++;
		i->i_count = 1;
		i->i_num = inum;
		i->i_dev = sb->s_dev;
	
		/* calculate the block and where in the block the inode is */
		blk_off = (sb->s_imap_blocks + sb->s_zmap_blocks +2) + 
			  (inum - 1)/INODES_PER_BLOCK;	/* fs/inode.c:8558 */
#ifdef DEBUG
		printf("iblock: i=0x%x 0x%x z=0x%x 0x%x\n", sb->s_imap_blocks, sb->s_zmap_blocks, (inum -1)/INODES_PER_BLOCK, INODES_PER_BLOCK);
#endif

		i_off = ((inum - 1) % INODES_PER_BLOCK);

#ifdef DEBUG
		printf("Reading Inode i=%d blk=%u off=0x%x\n",inum, blk_off, i_off);
#endif
		if(read_block(sb, blk_off, glb_block) == OK) {
#ifdef DEBUG
			printf("copy: 0x%x 0x%x 0x%x 0x%x\n",i, glb_block + i_off * INODE_SIZE, INODE_SIZE, i_off * INODE_SIZE);
#endif
			copy(i, glb_block + i_off * INODE_SIZE, INODE_SIZE);
		} else {
			i = NULL;
		}
	}
#ifdef TRACE
	else printf("inode already loaded\n");
#endif
	return(i);
}

/*
	Read File

	procedure read_file
		in Inode_Ptr, buffer address
		returns bytes read

Reads a disk file into the current data segment.  If the returned byte count
is zero, an error has occured and an error code is in the global variable 
"errno".

*/

int read_file(inode, buffer)
INODE inode;
char *buffer;
{
	int count, blocks;
	SUPERBLOCK super;
	static int indirect[BLOCK_SIZE/2];	/* buffer for indirect block */

#ifdef TRACE
	printf("Read_File: inum=%d\n", inode->i_num);
#endif
	blocks = 0;
	super = read_super(inode->i_dev);
	count = lload_blocks(super, buffer, inode->i_zone, NR_DZONE_NUM);
	buffer += count*BLOCK_SIZE;
	blocks += count;
	if(inode->i_zone[NR_DZONE_NUM] != 0) {
		read_block(super, inode->i_zone[NR_DZONE_NUM], indirect);
		count = lload_blocks(super, buffer, indirect, BLOCK_SIZE/2);
		blocks += count;
	}
	return(blocks);
}

/*
	Local Load Blocks

	procedure Local_Load_Blocks
		in Super-Block_Ptr, buffer address, list address, list length
		returns blocks read

Loads disk blocks listed in the "list" into the current data segment.
A count of blocks read is returned; if this count is zero, an error has
occured and an error code is in the global variable "errno".
*/

int lload_blocks(super, buffer, list, nlist)
SUPERBLOCK super;
char *buffer;
unsigned int *list;
int nlist;
{
	int count = 0;
#ifdef DEBUG
	printf("Loading file's data blocks\n");
#endif
	while(nlist > 0 && *list != 0) {
		if(read_block(super, *list, buffer) != OK) {
			count = 0;	/* Indicate error */
			break;		/* get outa here! */
		}
		buffer += BLOCK_SIZE;
		list++;
		nlist--;
		count++;
	}
	return(count);
}

/*
	Load File

	procedure  load_file
		in Inode_Ptr, buffer address (segment)
		returns a.out header

Load a Minix format binary object file into any location in memory (the
segment being given in the buffer address).  An "a.out" header
structure is returned.  This contains information on the text, data,
bss and stack sizes.  The text and data segments will have been loaded,
and the bss and stack segments zeroed at this point, so only writing
the program's arguments to the stack segment remains to be done.  If an
error occurs, NULL is returned and an error code is in the global
variable "errno".

*/
AOUT load_file(inode, buffer)
INODE inode;
char *buffer;
{
	static struct exec e_cache[NEXECCACHE],
				*last_exec = e_cache,
				*e;

	int count;
	SUPERBLOCK super;
	static int indirect[BLOCK_SIZE/2];	/* buffer for indirect block */

#ifdef TRACE
	printf("Load_File: inum=%d\n", inode->i_num);
#endif
	super = read_super(inode->i_dev);

	/* first block requires special pampering because it has the
	   a.out header in it */
	read_block(super, inode->i_zone[0], glb_block);
	e = last_exec++;	/* allocate a.out structure */
	copy(e, glb_block, A_MINHDR);	/* the file *must* be stripped */
	lcopy(buffer, 0, ds(), glb_block+A_MINHDR, BLOCK_SIZE - A_MINHDR);
	buffer += (BLOCK_SIZE / 16) - (A_MINHDR / 16);

	count = fload_blocks(super, buffer, inode->i_zone+1, NR_DZONE_NUM - 1);
	buffer += count * BLOCK_SIZE / 16;

	if(inode->i_zone[NR_DZONE_NUM] != 0) {
		read_block(super, inode->i_zone[NR_DZONE_NUM], indirect);
		count = fload_blocks(super, buffer, indirect, BLOCK_SIZE/2);
	}
	return(e);
}

/*
	Far Load Blocks

	procedure Far_Load_Blocks
		in Super-Block_Ptr, buffer address, list address, list length
		returns blocks read

Loads disk blocks listed in the "list" into memory starting at the
segment specified in "buffer address".  A count of blocks read is
returned; if this count is zero, an error has occured and an error code
is in the global variable "errno".

*/
int fload_blocks(super, buffer, list, nlist)
SUPERBLOCK super;
char *buffer;
unsigned int *list;
int nlist;
{
	int count = 0;
#ifdef DEBUG
	printf("Loading file's data blocks: glb_block= 0x%x:0x%x\n", ds(), glb_block);
#endif
	while(nlist > 0 && *list != 0) {
		if(read_block(super, *list, glb_block) != OK) {
			count = 0;	/* Indicate error */
			break;		/* get outa here! */
		}
#ifdef DEBUG
		printf("copy block %u to 0x%x:0x%x\n", *list, buffer, 0);
#endif
		lcopy(buffer, 0, ds(), glb_block, BLOCK_SIZE);
		buffer += BLOCK_SIZE / 16;
		list++;
		nlist--;
		count++;
	}
	return(count);
}

/*
	Open File

	procecure open_file
		in filename
		returns Inode_Ptr

Attempt to open a file with a name of the format "device:path".  The device,
if given, specifies which physical drive and partition to load the file
from.  If an error occurs durring processing, a NULL Inode pointer is returned
and an error codes is put in the global variable "errno".

Notes:
1)	///usr//ast == /usr/ast
2)	the last element is ALWAYS a file, so the case:
		/usr/ast/
	cannot occur.
*/
INODE open_file(filename)
char *filename;
{
	int device;
	char *path;
	struct super_block *super;
	struct inode *inode;

#ifdef TRACE
	printf("Open File %s\n",filename);
#endif
	path = device_map(filename,&device);
			/* how to error check? */
	if((super = read_super(device)) == NULL) {
		errno = ENODEV;
		return(NULL);
	}

	/* follow the pathname down the directory tree */
	inode = follow_path(super, path);

	/* We now have a proposed file to open.  Make sure it's a
	 * "regular" file (as opposed to directory or device).
	 */
	if(inode != NULL && (inode->i_mode & I_TYPE) != I_REGULAR) {
		errno = EISDIR;
		return(NULL);
	} else {
		return(inode);
	}
}

/*
	Device Calculation

	procedure device_map
		in filename
		out device number
		returns path

Checks the filename for a device:path form, and if there is a device
it is looked up in a table of device names and the given device number 
is returned, along with the part of the string that contains only the
path.  If no device name exists, the default device (the one booted from)
is returned as the device, and the entire filename (which is just the path)
is returned.

If a invalid device name is given, a device number of -1 is returned, and
the pathname is set as if the device was OK.  This is a feature.

*/

char *device_map(filename, device)
char *filename;
int *device;
{
	char devicename[80];
	char *s, *d;
	static struct {
		char *name;
		int dev;
	} map[] = {		/* list of existing devices */
		"fd0", 0,	"fd1", 1,	"fd2", 2,	"fd3", 3,
		"hd0.0", 0x40,	"hd0.1", 0x41,	"hd0.2", 0x42,
				"hd0.3", 0x43,	"hd0.4", 0x44,
		"hd1.0", 0x45,	"hd1.1", 0x46,	"hd1.2", 0x47,
				"hd1.3", 0x48,	"hd1.4", 0x49,
		"", -1
	}, *mp;


	for(s=filename, d=devicename; *s != '\0' && *s != ':'; ) {
		*d++ = *s++;
	}
	if(*s == ':') s++;
	*d = '\0';

	if(*s == '\0') {
		/* no ":" found, so there's no device name here */
		*device = def_dev;
		return(filename);
	} else {
		for(mp = map; mp->dev != -1 && strcmp(mp->name, devicename);
		    mp++) ;
		*device = mp->dev;
		return(s);
	}
}


/*
	Follow Path

	procedure follow_path
		in Pathname, Super_Block
		returns Inode_Ptr

The file given in _pathname_ is searched for on the device which the
super-block is on.  If it exists, an i-node for it is returned; otherwise
NULL is returned and an error code is placed in the global variable "errno".

*/
INODE follow_path(super, path)
SUPERBLOCK super;
char *path;
{
	INODE inode;
	char name[80];
	int i_num;

	i_num = ROOT_INODE;

	while((path = next_name(path, name)) != NULL) {
		/* there are more directories to descend.  First make sure
		 * that our current directory is in fact a directory before
		 * loading and searching it.
		 */
#ifdef DEBUG
		printf("follow_path: looking for %s in %d\n", name, i_num);
#endif
		inode = read_inode(super, i_num);
			/* obvious problem if an error occurs and read_inode
			   returns NULL */
		if((inode->i_mode & I_TYPE) != I_DIRECTORY) {
			errno = ENOTDIR;
			printf("ENOTDIR in follow_path\n");
			return((struct inode *)NULL);
		}

		/* so far, so good.  Search this directory for the next
		 * path element.
		 */ /* inum=zero is an error situation; zero is root */
		if((i_num = search_dir(inode, name)) == 0) {
			errno = EINVAL;
			printf("can't find %s!\n", name);
			return((struct inode *)NULL);
		}
	}
	inode = read_inode(super, i_num);
		/* no real problem here if an error occurs and read_inode
		   returns NULL... it will be detected 1 level up */
	return(inode);
}

/*
	Next Pathname Element

	procedure next_name
		in pathname
		out next path element
		returns remaining path after the current element

Moves down the pathname for the next file name, and returns that and
the remaining path after the next file name.  NULL is returned when
the path is exhausted.

*/

next_name(path, name)
char *path, *name;
{	char *from, *to;

	for(from = path; *from == '/'; from++) ;	/* skip all '/' */

	for(to = name; *from != '/' && *from != '\0';) {
		*to++ = *from++;
	}
	
	*to = '\0';
	if(to == name)
		return(NULL);
	else
		return(from);
}

/*
	Search Directory

	procedure search_dir
		in inode, name to search for
		returns new inode number

Search the directory described by _inode_ for the file name in _name_.
If found, return the inode number of the found file; otherwise return 0.

A static array is used as the destination for the directory load.  An
unchecked error condition exists if the directory is larger than this!

*/

int search_dir(inode, name)
INODE inode;
char *name;
{
	dir_struct *dir, *lastdir;
	static dir_struct directory[NR_DIR_ENTRIES * 8];
		/* directory buffer 8 blocks long */
	int count;	/* directory length in blocks */

#ifdef DEBUG
	printf("search_dir: inum=%d, name=%s.\n", inode->i_num, name);
#endif
	count = read_file(inode, directory);
	lastdir = directory + NR_DIR_ENTRIES * count + 1;
#ifdef DEBUG
	printf("search_dir: count=%d dir=0x%x lastdir=0x%x\n", count, directory, lastdir);
#endif
	for(dir = directory; 
	    dir <= lastdir && strcmp(dir->d_name, name) != 0;
	    dir++)
#ifdef DEBUG
		printf("search_dir: checking %s\n",dir->d_name);
#endif
		;
#ifdef DEBUG
		printf("search_dir: found %s\n",dir->d_name);
#endif
	return(dir > lastdir ? NULL : dir->d_inum);
}

#ifdef DEBUG
/* Dump Super-Block contents
*/
dump_super(super)
SUPBERBLOCK super;
{
	printf("\nSuper-Block Contents\n");
	printf(" s_ninodes=%u      ", super->s_ninodes);
	printf(" s_nzones=%u       ", super->s_nzones);		fputc('\n');
	printf(" s_imap_blocks=%u  ", super->s_imap_blocks);
	printf(" s_zmap_blocks=%u  ", super->s_zmap_blocks);
	printf(" s_firstdatazone=%u", super->s_firstdatazone);	fputc('\n');
	printf(" s_log_zone_size=%u", super->s_log_zone_size);
	printf(" s_max_size=%u     ", super->s_max_size);
	printf(" s_magic=%u        ", super->s_magic);		fputc('\n');

	printf(" s_dev=%u          ", super->s_dev);
	printf(" s_time=%ld        ", super->s_time);		fputc('\n');
	fputc('\n');
}
/* dump partition table contents
*/
dump_prt(part)
struct part_entry *part;
{
	printf("\nPartition Table Contents\n");
	printf(" at 0x%x       ", part);			fputc('\n');
	printf(" bootind=0x%x  ", part->bootind);
	printf(" start_head=%d ", part->start_head);		fputc('\n');
	printf(" start_sec=%d  ", part->start_sec);
	printf(" start_cyl=%d  ", part->start_cyl);		fputc('\n');
	printf(" sysind=0x%x   ", part->sysind);
	printf(" last_head=%d  ", part->last_head);		fputc('\n');
	printf(" last_sec=%d   ", part->last_sec);
	printf(" last_cyl=%d   ", part->last_cyl);		fputc('\n');
	printf(" lowsec=%ld    ", part->lowsec);
	printf(" size=%ld      ", part->size);			fputc('\n');
	fputc('\n');
}

/* dump Fixed Disk parameter table
 */
dump_param(dest)
struct param *dest;
{
	printf("\nParameter Table Contents\n");
	printf(" nr_cyl=0x%x     ",  dest->nr_cyl);
	printf(" nr_heads=0x%x   ", dest->nr_heads);		fputc('\n');
	printf(" reduced_wr=0x%x ", dest->reduced_wr);
	printf(" wr_precomp=0x%x ", dest->wr_precomp);		fputc('\n');
	printf(" max_ecc=0x%x    ", dest->max_ecc);
	printf(" ctrl_byte=0x%x  ", dest->ctrl_byte);		fputc('\n');
	printf(" sectors=0x%x    ", dest->sectors);
	fputc('\n');
}
#endif

/* dump EXEC structure
 */
dump_exec(exec)
AOUT exec;
{
	printf("\nEXEC Structure Contents\n");
	printf(" magic= 0x%x, 0x%x", exec->a_magic[0], exec->a_magic[1]);
	printf(" flags= 0x%x    ", exec->a_flags);		fputc('\n');
	printf(" cpu= 0x%x      ", exec->a_cpu);
	printf(" hdrlen= 0x%x   ", exec->a_hdrlen);
	printf(" version= 0x%x  ", exec->a_version);		fputc('\n');
	printf(" text size= %ld ", exec->a_text);
	printf(" data size= %ld ", exec->a_data);
	printf(" bss size= %ld  ", exec->a_bss);
	printf(" total size= %ld", exec->a_total);		fputc('\n');
	printf(" syms size= %ld ", exec->a_syms);
	printf(" entry point= %ld", exec->a_no_entry);		fputc('\n');
	fputc('\n');
}
