/*
 * Point - find out the z-value of a point inside a triangle/rectangle
 */

#include <math.h>
#include <stdio.h>
#include <malloc.h>
#include "CNplot.h"

void CNprobe_dataset();
int  CNpoint_in_region();
static int  point_in_mesh();
static int  point_in_tria();
static int  point_in_rect();
 

/*
 * Probe a dataset
 */
void CNprobe_dataset(Dptr, x, y, probehead, probetail, verbose)
CNdatasetptr Dptr;                    /* Dataset to be probed */
double       x, y;                    /* Probe coordinates    */
CNprobeptr   *probehead, *probetail;  /* Results              */
int          verbose;
{
   /* Probe the dataset mesh */
   (void) point_in_mesh(x,y,
                        Dptr->triahead, Dptr->triatail,
                        Dptr->recthead, Dptr->recttail,
                        Dptr->regionhead, Dptr->regiontail,
                        probehead, probetail, 
                        Dptr->data_pr.logx,
                        Dptr->data_pr.logy,
                        Dptr->data_pr.logz,
                        (int) Dptr->ID, 
                        verbose);

   if (verbose) CNprint_probe_list(*probehead, *probetail);
}


/*
 * Check to see if a point in the mesh coincides with a region
 */
int CNpoint_in_region(Dptr,x,y,regID,verbose)
CNdatasetptr Dptr;                    /* Dataset to be probed */
double       x, y;                    /* Probe coordinates    */
int          regID;
int          verbose;
{
   CNprobeptr probehead=NULL, probetail=NULL, P;
   int        MATCHFOUND=CN_FALSE;

   /* Probe the mesh */
   (void) CNprobe_dataset(Dptr, x, y, &probehead, &probetail, verbose);

   /* Check the probe-points to see if the regionID coincides */
   for (P=probehead; P!=NULL && !MATCHFOUND; P=P->next)
      if ((P->T != NULL) && (P->T->region == regID))
         MATCHFOUND = CN_TRUE;

   /* Delete the list */
   CNdelete_probe_list(&probehead, &probetail);

   /* return */
   return(MATCHFOUND);
}


/*
 * Search thru the mesh for an intersection with a line ax+by+d=0 (const z)
 */
/*ARGSUSED*/
static int point_in_mesh(x,y,
                         triahead,triatail,
                         recthead,recttail,
                         regionhead,regiontail,
                         probehead, probetail, 
                         logx, logy, logz,
                         dataID, verbose)
double       x, y;                    /* Probe coordinates        */
CNtriaptr    triahead, triatail;      /* Triangle  list           */
CNrectptr    recthead, recttail;      /* Rectangle list           */
CNregionptr  regionhead, regiontail;  /* Region list              */
CNprobeptr   *probehead, *probetail;  /* Results                  */
short        logx, logy, logz;        /* Log/Linear interpolation */
int          dataID;
int          verbose;
{
   CNpointptr  pointhead=NULL, pointtail=NULL, P;
   CNtriaptr   T;
   CNrectptr   R;
   CNregionptr reg, Treg=NULL;
   int         probe_count, new_probe_points;
   int         TREG_FOUND;

   /* Count the number of items in the probelist */
   probe_count = CNcount_probes(*probehead, *probetail);

   /* Go thru the triangles first */
   for (T=triahead; T!=NULL; T=T->next) {
      /* Search for point-intersections */
      (void) point_in_tria(T,x,y,&pointhead,&pointtail,
                           logx, logy, logz, verbose);

      /* Find the region */
      TREG_FOUND = CN_FALSE;
      for (reg=regionhead; reg!=NULL; reg=reg->next) 
         if (reg->ID == T->region) {
            TREG_FOUND = CN_TRUE;
            Treg       = reg;
         }
      if (!TREG_FOUND) Treg = NULL;

      /* Place the results in a probe-list */
      for (P=pointhead; P!=NULL; P=P->next) 
         (void) CNinsert_probe(probehead, probetail,
                               P->x,P->y,P->z,T,(CNrectptr)NULL,Treg,dataID);

      /* Remove everything in the pointlist */
      CNdelete_point_list(&pointhead, &pointtail);
   }

   /* Go thru the rectangles next */
   Treg = NULL;
   for (R=recthead; R!=NULL; R=R->next) {
      /* Search for point-intersections */
      (void) point_in_rect(R,x,y,&pointhead,&pointtail,
                           logx, logy, logz, verbose);

      /* Place the results in a probe-list */
      for (P=pointhead; P!=NULL; P=P->next) 
         (void) CNinsert_probe(probehead, probetail,
                               P->x,P->y,P->z,(CNtriaptr)NULL,R,Treg,dataID);
 
      /* Remove everything in the pointlist */
      CNdelete_point_list(&pointhead, &pointtail);
   }

   /* Now count the number of probe-points */
   new_probe_points = CNcount_probes(*probehead, *probetail) - probe_count;
   
   if (verbose) 
      (void) fprintf(stdout,"Found %d probe-points\n",new_probe_points);

   /* Return */
   return(new_probe_points);
}

/*
 * Find the z-value of a point inside a triangle
 * Store the result in a pointlist, for compatibilty with the
 * point-rectangle routine.
 */
/*ARGSUSED*/
static int point_in_tria(T,x,y,pointhead,pointtail,
                         logx, logy, logz, verbose)
CNtriaptr  T;
double     x, y;                    /* The x-y coordinates      */
CNpointptr *pointhead, *pointtail;  /* Point list               */
short      logx, logy, logz;        /* Log/Linear interpolation */
int        verbose;
{
   double     xmax, xmin, ymax, ymin, dx, t, z, px, py;
   CNpoint    pt1, pt2, pt3, ptmin, ptmax;
   int        signz;
   int        xlogmode, ylogmode, zlogmode;
   int        intsct;

   /* Check the triangle */
   if (T==NULL) return(0);

   /* Check the boundaries of the triangle first */
   CNget_tria_xmaxmin(T,&xmin,&xmax);
   CNget_tria_ymaxmin(T,&ymin,&ymax);
   if (x < xmin || x > xmax) return(0);
   if (y < ymin || y > ymax) return(0);

   /* Copy the point coordinates */
   pt1 = *(T->n1->coord);   pt1.z = T->n1->t;
   pt2 = *(T->n2->coord);   pt2.z = T->n2->t;
   pt3 = *(T->n3->coord);   pt3.z = T->n3->t;

   /*
    * Convert the numbers to log if necessary then interpolate
    */
   px = x;
   py = y;
   if (logx) {
      xlogmode = CNlogmode3(pt1.x, pt2.x, pt3.x);
      pt1.x = CNlogabs(pt1.x, xlogmode);
      pt2.x = CNlogabs(pt2.x, xlogmode);
      pt3.x = CNlogabs(pt3.x, xlogmode);
      px    = CNlogabs(x    , xlogmode);
   }
   if (logy) {
      ylogmode = CNlogmode3(pt1.y, pt2.y, pt3.y);
      pt1.y = CNlogabs(pt1.y, ylogmode);
      pt2.y = CNlogabs(pt2.y, ylogmode);
      pt3.y = CNlogabs(pt3.y, ylogmode);
      py    = CNlogabs(y    , ylogmode);
   }
   if (logz) {
      signz    = CNsign(pt1.z);
      zlogmode = CNlogmode3(pt1.z, pt2.z, pt3.z);
      pt1.z = CNlogabs(pt1.z, zlogmode);
      pt2.z = CNlogabs(pt2.z, zlogmode);
      pt3.z = CNlogabs(pt3.z, zlogmode);
   }

   /* Now check for an intersection with the segments */
   intsct = CNpoly3_intsct_plane(0.0,1.0,0.0,-py,
                                 &pt1,&pt2,&pt3,&ptmin,&ptmax,
                                 verbose);

   /* This should give me a line parallel to the x-axis */
   if (intsct != 1) return(0);

   /* Reorder ptmin and ptmax in x */
   if (ptmin.x > ptmax.x) {
      pt1   = ptmax;
      ptmax = ptmin;
      ptmin = pt1;
   }

   /* if the point is outside the x-limits of the intersection, return */
   if (px < ptmin.x || x > ptmax.x) return(0);

   /* Now get the intersection */
   dx = ptmax.x - ptmin.x;
   if (fabs(dx) < CN_SMALLER) t = 0.0;
   else                       t = (px - ptmin.x) / dx;
   z = ptmin.z + t*(ptmax.z - ptmin.z);

   /* Reconvert the results */
   if (logz) z = CNinvlogabs(z,signz,zlogmode);

   /* Put the coords inside a point-list */
   (void) CNinsert_point(pointhead, pointtail, x, y, z, 0);

   /* Return */
   return(1);
}

/*
 * Find the z-value of a point inside a rectangle
 * There could be more than 1 point, so store the results in a pointlist
 */
/*ARGSUSED*/
static int point_in_rect(R,x,y,pointhead,pointtail,
                         logx, logy, logz, verbose)
CNrectptr  R;
double     x, y;                    /* The x-y coordinates      */
CNpointptr *pointhead, *pointtail;  /* Point list               */
short      logx, logy, logz;        /* Log/Linear interpolation */
int        verbose;
{
   double     xmax, xmin, ymax, ymin, dx, t, z, px, py;
   CNpoint    pt1, pt2, pt3, pt4, newpt;
   CNpoint    points[10], ptmin, ptmax;
   CNpointptr P;
   int        signz;
   int        xlogmode, ylogmode, zlogmode;
   int        i, npts=0, intsct, FOUND=CN_FALSE, PTFOUND=CN_FALSE;

   /* Check the rectangle */
   if (R==NULL) return(0);

   /* Check the boundaries of the rectangle first */
   CNget_rect_xmaxmin(R,&xmin,&xmax);
   CNget_rect_ymaxmin(R,&ymin,&ymax);
   if (x < xmin || x > xmax) return(0);
   if (y < ymin || y > ymax) return(0);

   /* Copy the point coordinates */
   pt1 = *(R->n1->coord);   pt1.z = R->n1->t;
   pt2 = *(R->n2->coord);   pt2.z = R->n2->t;
   pt3 = *(R->n3->coord);   pt3.z = R->n3->t;
   pt4 = *(R->n4->coord);   pt4.z = R->n4->t;

   /*
    * Convert the numbers to log if necessary then interpolate
    */
   px = x;
   py = y;
   if (logx) {
      xlogmode = CNlogmode4(pt1.x, pt2.x, pt3.x, pt4.x);
      pt1.x = CNlogabs(pt1.x, xlogmode);
      pt2.x = CNlogabs(pt2.x, xlogmode);
      pt3.x = CNlogabs(pt3.x, xlogmode);
      pt4.x = CNlogabs(pt4.x, xlogmode);
      px    = CNlogabs(x    , xlogmode);
   }
   if (logy) {
      ylogmode = CNlogmode4(pt1.y, pt2.y, pt3.y, pt4.y);
      pt1.y = CNlogabs(pt1.y, ylogmode);
      pt2.y = CNlogabs(pt2.y, ylogmode);
      pt3.y = CNlogabs(pt3.y, ylogmode);
      pt4.y = CNlogabs(pt4.y, ylogmode);
      py    = CNlogabs(y    , ylogmode);
   }
   if (logz) {
      signz    = CNsign(pt1.z);
      zlogmode = CNlogmode4(pt1.z, pt2.z, pt3.z, pt4.z);
      pt1.z = CNlogabs(pt1.z, zlogmode);
      pt2.z = CNlogabs(pt2.z, zlogmode);
      pt3.z = CNlogabs(pt3.z, zlogmode);
      pt4.z = CNlogabs(pt4.z, zlogmode);
   }

   /* Now check for an intersection with the segments */
   intsct = CNpoly4_intsct_plane(0.0,1.0,0.0,-py,
                                 &pt1,&pt2,&pt3,&pt4,
                                 points, &npts, 
                                 verbose);

   /* This should give me lines parallel to the x-axis */
   if (intsct) {
      for (i=0; i<npts; i=i+2) {
         /* Skip if the line is too short */
         if (!CNlongline(&(points[i]),&(points[i+1]),1.0e-5)) continue;

         /* skip if the point is outside the x-limits of the intersection */
         if (points[i].x > points[i+1].x) {
            ptmin = points[i+1];
            ptmax = points[i];
         } else {
            ptmin = points[i];
            ptmax = points[i+1];
         }
         if (px < ptmin.x || px > ptmax.x) continue; 

         /* Now get the intersection */
         dx = ptmax.x - ptmin.x;
         if (fabs(dx) < CN_SMALLER) t = 0.0;
         else                       t = (px - ptmin.x) / dx;
         z = ptmin.z + t*(ptmax.z - ptmin.z);

         /* Reconvert the results */
         if (logz) z = CNinvlogabs(z,signz,zlogmode);

         /* Put the coords inside a point-structure for comparison */
         newpt.x = x;
         newpt.y = y;
         newpt.z = z;
         
         /* Find out if the point is already in the list */
         PTFOUND = CN_FALSE;
         for (P=(*pointhead); P!=NULL && !PTFOUND; P=P->next) 
             if (!CNlongline(P,&newpt,1.0e-5)) PTFOUND = CN_TRUE; 

         /* Put the coords inside a point-list */
         if (!PTFOUND) {
            (void) CNinsert_point(pointhead, pointtail, x, y, z, 0);
            FOUND = CN_TRUE;
         }
      }
   }
   return(FOUND);
}


/*
 * Linked list procedures
 */

/*
 * Allocate room for a probe
 */
static CNprobeptr make_probe(x,y,z,T,R,reg,ID)
double      x,y,z;
CNtriaptr   T;
CNrectptr   R;
CNregionptr reg;
int         ID;
{
   CNprobeptr newptr;
   unsigned int size = sizeof(CNprobe);
 
   if ((newptr = (CNprobeptr)malloc(size))!=NULL) {
      newptr->ID   = ID;
      newptr->x    = x;
      newptr->y    = y;
      newptr->z    = z;
      newptr->T    = T;
      newptr->R    = R;
      newptr->reg  = reg;
      newptr->next = NULL;
      newptr->prev = NULL;
   }
   return(newptr);
}

/*
 * Insert a probe at the tail of the current probe list
 */
CNprobeptr CNinsert_probe(probe_listhead, probe_listtail, 
                          x, y, z, T, R, reg, ID)
CNprobeptr *probe_listhead, *probe_listtail;
double      x, y, z;
CNtriaptr   T; 
CNrectptr   R; 
CNregionptr reg;
int         ID;
{
   static CNprobeptr make_probe();
   CNprobeptr next,A,B;
 
   A = *probe_listtail;
   if ((B=make_probe(x,y,z,T,R,reg,ID))!=NULL) {
      if (A==NULL) {
         *probe_listhead = B;
         *probe_listtail = B;
      } else {
         next = A->next;
         B->next = next;
         B->prev = A;
         A->next = B;
         if (next != NULL) next->prev = B;
         if (B->next == NULL) *probe_listtail = B;
      }
   }
   return(B);
}


/*
 * Delete probe at address L
 */
void CNdelete_probe(probe_listhead, probe_listtail, L)
CNprobeptr *probe_listhead, *probe_listtail;
CNprobeptr L;
{
   CNprobeptr prev,next;
 
   prev = L->prev;
   next = L->next;
   if (prev!=NULL) prev->next = next;
   if (next!=NULL) next->prev = prev;
   if (L== *probe_listhead) *probe_listhead = next;
   if (L== *probe_listtail) *probe_listtail = prev;
   free ((char*)L);
   L = NULL;
}
 
 
/*
 * Delete all the probes in the list
 */
void CNdelete_probe_list(probe_listhead, probe_listtail)
CNprobeptr *probe_listhead, *probe_listtail;
{
   CNprobeptr P;
   while ((P = *probe_listhead) != NULL)
      CNdelete_probe(probe_listhead, probe_listtail, P);
}
 
 
/*
 * Print out the list of probes
 */
/*ARGSUSED*/
void CNprint_probe_list(probe_listhead, probe_listtail)
CNprobeptr probe_listhead, probe_listtail;
{
   static void print_probe();
   CNprobeptr P;
 
   (void) fprintf(stdout,"   Found %d intersections :\n",
           CNcount_probes(probe_listhead, probe_listtail));

   for (P=probe_listhead; P!=NULL; P=P->next)
      print_probe(P);
}
 
 
/*
 * print the coordinates of a probe
 */
static void print_probe(pt)
CNprobeptr pt;
{
   (void) fprintf(stdout,"Probe Coordinates : (%g %g %g)",pt->x,pt->y,pt->z);
   if (pt->T != NULL)
   (void) fprintf(stdout,"   Tria ID=%d  Reg ID=%d",pt->T->ID,pt->T->region);
   if (pt->T != NULL && pt->reg != NULL)
   (void) fprintf(stdout,"   Mat Name=\"%s\"",pt->reg->matname);
   if (pt->R != NULL)
   (void) fprintf(stdout,"   Rect ID=%d",pt->R->ID);
   (void) fprintf(stdout,"\n");
}
 
 
/*
 * Count the number of probes in the list
 */
/*ARGSUSED*/
int CNcount_probes(probe_listhead, probe_listtail)
CNprobeptr probe_listhead, probe_listtail;
{
   CNprobeptr P;
   int        count = 0;
 
   for (P=probe_listhead; P!=NULL; P=P->next) count++;
 
   return(count);
}

