#   Licensed under the Apache License, Version 2.0 (the "License"); you may
#   not use this file except in compliance with the License. You may obtain
#   a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#   WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#   License for the specific language governing permissions and limitations
#   under the License.
#

"""Test Timing pseudo-command"""

import datetime

from keystoneauth1 import session

from osc_lib.command import timing
from osc_lib.tests import fakes
from osc_lib.tests import utils


timing_method = 'GET'
timing_url = 'http://localhost:5000'
timing_elapsed = 0.872809


class FakeGenericClient(object):
    def __init__(self, **kwargs):
        self.auth_token = kwargs['token']
        self.management_url = kwargs['endpoint']


class TestTiming(utils.TestCommand):
    columns = (
        'URL',
        'Seconds',
    )

    def setUp(self):
        super(TestTiming, self).setUp()

        self.app.timing_data = []

        self.app.client_manager.compute = FakeGenericClient(
            endpoint=fakes.AUTH_URL,
            token=fakes.AUTH_TOKEN,
        )

        self.app.client_manager.volume = FakeGenericClient(
            endpoint=fakes.AUTH_URL,
            token=fakes.AUTH_TOKEN,
        )

        # Get the command object to test
        self.cmd = timing.Timing(self.app, None)

    def test_timing_list_no_data(self):
        arglist = []
        verifylist = []
        parsed_args = self.check_parser(self.cmd, arglist, verifylist)

        # In base command class Lister in cliff, abstract method take_action()
        # returns a tuple containing the column names and an iterable
        # containing the data to be listed.
        columns, data = self.cmd.take_action(parsed_args)

        self.assertEqual(self.columns, columns)
        datalist = [
            (
                'Total',
                0.0,
            )
        ]
        self.assertEqual(datalist, data)

    def test_timing_list(self):
        self.app.timing_data = [
            session.RequestTiming(
                method=timing_method,
                url=timing_url,
                elapsed=datetime.timedelta(
                    microseconds=timing_elapsed * 1000000
                ),
            )
        ]

        arglist = []
        verifylist = []
        parsed_args = self.check_parser(self.cmd, arglist, verifylist)

        # In base command class Lister in cliff, abstract method take_action()
        # returns a tuple containing the column names and an iterable
        # containing the data to be listed.
        columns, data = self.cmd.take_action(parsed_args)
        self.assertEqual(self.columns, columns)
        datalist = [
            (timing_method + ' ' + timing_url, timing_elapsed),
            ('Total', timing_elapsed),
        ]
        self.assertEqual(datalist, data)
