/*
 * Copyright (c) 2007, intarsys consulting GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * - Neither the name of intarsys nor the names of its contributors may be used
 *   to endorse or promote products derived from this software without specific
 *   prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package de.intarsys.tools.logging;

import java.io.IOException;
import java.util.List;
import java.util.logging.FileHandler;
import java.util.logging.Handler;
import java.util.logging.Logger;

/**
 * Tool methods to ease life with java.util.logging.
 * 
 */
public class LogTools {

	private static IDumpObject DumpObject;

	public static final String INDENT = "    "; //$NON-NLS-1$

	static {
		DumpObject = new CommonDumpObject();
	}

	/**
	 * Closes a fileLogger.
	 */
	public static void closeFileLogger(Logger logger) {
		Handler[] handlers = logger.getHandlers();
		for (int i = 0; i < handlers.length; i++) {
			handlers[i].close();
		}
	}

	/**
	 * Creates a fileLogger.
	 * 
	 * @return A Logger named with 'id'.
	 * @throws IOException
	 * @throws SecurityException
	 */
	public static Logger createFileLogger(String id, String filePattern,
			boolean append) throws SecurityException, IOException {
		Logger logger = Logger.getLogger(id);
		Handler handler = new FileHandler(filePattern, append);
		logger.addHandler(handler);
		return logger;
	}

	static public List<String> dumpObject(String prefix, Object object) {
		return dumpObject(prefix, object, DumpObject);
	}

	static public List<String> dumpObject(String prefix, Object object,
			IDumpObject details) {
		return DumpObject.dump(prefix, object, details);
	}

	/**
	 * Get a logger that corresponds to <code>clazz</code>.
	 * 
	 * @param clazz
	 *            The class object we want a logger for.
	 * 
	 * @return A Logger that corresponds to clazz.
	 */
	public static Logger getLogger(Class<?> clazz) {
		Logger logger = Logger.getLogger(toLoggerName(clazz));
		LogConfigurator.get().configure(logger);
		return logger;
	}

	/**
	 * Get a logger for name <code>name</code>.
	 * 
	 * @param name
	 *            The logger name/category
	 * 
	 * @return Get a logger for name <code>name</code>.
	 */
	public static Logger getLogger(String name) {
		Logger logger = Logger.getLogger(name);
		LogConfigurator.get().configure(logger);
		return logger;
	}

	/**
	 * The name we want to use for a logger used in <code>clazz</code>. This
	 * implementation uses the package name.
	 * 
	 * @param clazz
	 *            The class object we want a name for.
	 * 
	 * @return The name we want to use for a logger used in <code>clazz</code>.
	 */
	public static String toLoggerName(Class<?> clazz) {
		String result = clazz.getName();
		int index = result.lastIndexOf('.');
		if (index > -1) {
			result = result.substring(0, index);
		}
		return result;
	}

	/**
	 * 
	 */
	private LogTools() {
		// tool class
	}

}
