#ifndef __LINUX_USB_H
#define __LINUX_USB_H

#include <linux/types.h>

/*
 * USB device number allocation bitmap. There's one bitmap
 * per USB tree.
 */
struct usb_devmap {
	unsigned long devicemap[128 / (8*sizeof(unsigned long))];
};

/* Device descriptor information */
struct usb_dev_descriptor {
	__u8  bLength;
	__u8  bDescriptorType;
	__u16 bcdUSB;
	__u8  bDeviceClass;
	__u8  bDeviceSubClass;
	__u8  bDeviceProtocol;
	__u8  bMaxPacketSize0;
	__u16 idVendor;
	__u16 idProduct;
	__u16 bcdDevice;
	__u8  iManufacturer;
	__u8  iProduct;
	__u8  iSerialNumber;
	__u8  bNumConfigurations;
} usb_dev;

/*
 * This is a USB device descriptor.
 *
 * Right now we only keep track of the topology,
 * add the full descriptions later..
 */
struct usb_device {
	int devnum;
	int slow;

	/* USB device information */
	struct usb_dev_descriptor descriptor;

	/* USB maintenance information. */
	struct usb_devmap *devmap;
	struct usb_device *parent;
	struct usb_device *children[8];
};

extern void usb_connect(struct usb_device *dev, struct usb_device *parent, unsigned int port);
extern void usb_disconnect(struct usb_device *dev);

/*
 * Calling this entity a "pipe" is glorifying it. A USB pipe
 * is something embarrassingly simple: it basically consists
 * of the following information:
 *  - device number (7 bits)
 *  - endpoint number (4 bits)
 *  - current Data0/1 state (1 bit)
 *  - direction (1 bit)
 *  - speed (1 bit)
 *  - max packet size (2 bits: 8, 16, 32 or 64)
 *  - pipe type (2 bits: control, interrupt, bulk, isochronous)
 *
 * That's 18 bits. Really. Nothing more. And the USB people have
 * documented these eighteen bits as some kind of glorious
 * virtual data structure.
 *
 * Let's not fall in that trap. We'll just encode it as a simple
 * unsigned int. The encoding is:
 *
 *  - device:		bits 8-14
 *  - endpoint:		bits 15-18
 *  - Data0/1:		bit 19
 *  - direction:	bit 7		(0 = Host-to-Device, 1 = Device-to-Host)
 *  - speed:		bit 26		(0 = High, 1 = Low Speed)
 *  - max size:		bits 0-1	(00 = 8, 01 = 16, 10 = 32, 11 = 64)
 *  - pipe type:	bits 30-31	(00 = isochronous, 01 = interrupt, 10 = control, 11 = bulk)
 *
 * Why? Because it's arbitrary, and whatever encoding we select is really
 * up to us. This one happens to share a lot of bit positions with the UCHI
 * specification, so that much of the uhci driver can just mask the bits
 * appropriately.
 */

#define usb_maxpacket(pipe)	(8 << ((pipe) & 3))
#define usb_packetid(pipe)	(((pipe) & 0x80) ? 0x69 : 0xE1)

#define usb_pipedevice(pipe)	(((pipe) >> 8) & 0x7f)
#define usb_pipeendpoint(pipe)	(((pipe) >> 15) & 0xf)
#define usb_pipedata(pipe)	(((pipe) >> 19) & 1)
#define usb_pipeout(pipe)	(((pipe) & 0x80) == 0)
#define usb_pipeslow(pipe)	(((pipe) >> 26) & 1)
#define usb_pipetype(pipe)	(((pipe) >> 30) & 3)

static inline unsigned int __create_pipe(struct usb_device *dev, unsigned int endpoint)
{
	return (dev->devnum << 8) | (endpoint << 15) | (dev->slow << 26);
}

static inline unsigned int __default_pipe(struct usb_device *dev)
{
	return (dev->slow << 26);
}

/* Create control pipes.. */
#define usb_sndctrlpipe(dev,endpoint)	((2 << 30) | __create_pipe(dev,endpoint))
#define usb_rcvctrlpipe(dev,endpoint)	((2 << 30) | __create_pipe(dev,endpoint) | 0x80)
#define usb_snddefctrl(dev)		((2 << 30) | __default_pipe(dev))
#define usb_rcvdefctrl(dev)		((2 << 30) | __default_pipe(dev) | 0x80)

/* Create .. */

#endif
