(* Copyright (c) 1991 by Carnegie Mellon University *)

(* 7/19/90 (DRT) Modified to be loaded after pervasives are loaded but before
   they are opened.

   7/19/90 (DRT) Rewrote loader to execute each object in the datalist
   in the order in which they are found on the datalist.

   7/20/90 (DRT) Removed messages saying "Loading ..." and "Executing ..."

   9/26/90 (DRT) Replaced use of System.Unsafe.boot with a cast.  We don't
   need to add 8 because we don't have code strings in the datalist.
*)

signature Startup =
 sig 
     val core : Initial.System.Unsafe.object
     val initial : Initial.System.Unsafe.object
     val math : Initial.System.Unsafe.object
 end

functor Loader ( S : Startup ) : sig end =
struct

 open Initial
 open Ref List String IO General System System.Unsafe

 val  boot : string ->
	  unit ->
	 ((object list -> (object * ByteArray.bytearray array)) * string list)
	  = fn x => System.Unsafe.cast(ref x) (* a "closure" *)

 val dict : (string*object) list ref = 
	ref [("Initial",S.initial),("Core",S.core),("Math",S.math)]

 val _ = pstruct := {core=S.core,math=S.math,initial=S.initial}

 exception Notfound_Loader

 fun lookup s =
    let fun f ((s1,stru)::r) = if s=s1 then stru else f r
	  | f nil = raise Notfound_Loader
     in f (!dict)
    end

 fun enter pair = dict := pair::(!dict)

 val say = outputc std_out

 structure CleanUp = System.Unsafe.CleanUp

 val _ =
     let fun f DATANIL = ()
	   | f (DATACONS(name,object,rest)) =
	    let val s = substring(name,3,Integer.-(String.length name,6))
	        val _ = lookup s handle Notfound_Loader =>
	         let (*  val _ = (say "[Loading "; say s; say "]\n") *)
	            val (exec,sl) = boot object ()
	            val saver = ref exec  (* save a pointer for the garbage
					     collector *)
                    val structs = map lookup sl
                 (* val _ = (say "[Executing "; say ; say "]\n")  *)
	            val (str,profile) = exec structs
	         in enter (s,str);
		    System.Control.ProfileInternals.add profile;
	            saver := !saver;    (* trickery for the g.c. *)
		    str
		 end
            in f rest
	    end
     in (f System.Unsafe.datalist;
	 CleanUp.cleanup CleanUp.CleanForQuit)
	    (* this is the global exception handler of the sml system *)
	    handle Io s =>
		     (say "uncaught Io exception (Loader): ";
		      say s;
		      say "\n";
		      CleanUp.cleanup CleanUp.CleanForQuit)
		 | exn =>
		     (say "uncaught exception: ";
		      say (exn_name exn);
		      say "\n";
		      CleanUp.cleanup CleanUp.CleanForQuit)
     end
end (* functor Loader *)

