(* Copyright (c) 1991 by Carnegie Mellon University *)

(* Import signature for sml2c.  The function getModule takes an
   environment, a "compiler" function, an "entry" function, and a file name
   as arguments.   It returns a list of the information returned by the
   compiler function.

   The "compiler" function takes an abstract syntax declaration and
   a destination file, compiles the abstract syntax into the destination
   file, and returns a list of lvars and their associated entry point
   names.

   The "entry" function allows getModule to enter the entry point names
   for lvars. *)

signature LINK =
   sig
     type info
     val project : info -> Access.lvar
     val rename : Access.lvar * info -> info
     val save : info -> unit
end

signature CIMPORTER =
   sig
      exception Import
      structure Link : LINK
      val getModule : Modules.env *
	              (Absyn.dec * Modules.env * string -> Link.info list) *
		      string ->
		         Modules.env * Link.info list * string list
   end

(* modified 2/7/91 by David Tarditi *)
(* bug 1 fixed 8/12/92 by David Tarditi *)

functor CImporter(structure FilePaths: FILEPATHS
		 val fileExtension: string
		 structure Link : LINK) : CIMPORTER =
struct
  structure Link = Link
  type info = Link.info

  open PrintUtil Access Modules NewParse

  fun all pred list = fold (fn (this, res) => pred this andalso res) list true

(* uniq: remove duplicate entries from a list that are towards the end.
   It is important to preserve order.*)

val uniq : ''a list -> ''a list =
    let fun f (seen, h :: t) =
	  if exists (fn a => a=h) seen then f(seen,t)
          else h :: f (h :: seen,t)
          | f (_,nil) = nil
    in fn x => f (nil,x)
    end

(* 
  val uniq : string list -> string list = 
      fn nil => nil 
       | h :: t =>
	     let fun scan (a,nil) = nil
                   | scan (a,h::t) = if a=h then scan(a,t) else h :: scan(h,t)
             in h :: scan(h,Sort.sort String.> t)
	     end
*)
  val DEBUG = false
  val debug = if DEBUG then fn str => outputc std_out ("<" ^ str ^ ">\n")
                       else fn _ => ()

  exception Import
     (* A single exception for any failure to import (barring compiler bugs).
        compSource requires a protective coating so that it doesn't leave
	the global static environment in a funny state. *)

  (* Feedback messages. If anybody's interested, files which may
     cause failures, or may cause nested reads, are done as:
       [reading fred.sml]
       [closing fred.sml]

     Ones which shouldn't (eg. reading from a binary) produce:
       [reading fred.bin... done]
   *)

  fun reading(file, indent) =
      (tab indent; print("[reading " ^ file ^ "]\n"))
  fun reading1(file, indent) =
      (tab indent; print("[reading " ^ file ^ "... "); flush_out std_out)
  fun writing(file, indent) =
      (tab indent; print("[writing " ^ file ^ "]\n"))
  fun writing1(file, indent) =
      (tab indent; print("[writing " ^ file ^ "... "); flush_out std_out)
  fun closing(file, indent) =
      (tab indent; print("[closing " ^ file ^ "]\n"))
  fun done() = print "done]\n"

  fun fail msg = (print("import: " ^ msg ^ "\n"); raise Import)

 (* impliedPath: derived from FilePaths.impliedPath, but catches
    ImpliedPath if a "~"-filename fails to translate. *)

  fun impliedPath(oldPath, oldName) =
      FilePaths.impliedPath(oldPath, oldName)
      handle FilePaths.ImpliedPath =>
	fail("couldn't translate path in: " ^ oldName)

  datatype statModule =
      STATmodule of {env: Modules.env, linkage : Link.info list}

    (* Rename the lvars of the static module.
       Only signature and functor bindings are accepted.
       For each functor binding, a fresh lvar will be chosen; hence
       at run-time, several imports of the same functor will presumably
       lead to a new copy of the code of that functor *)

  fun importModule (STATmodule{env,linkage}) : Modules.env * Link.info list = 
      let val newlinkage = map (fn s => Link.rename(mkLvar(),s)) linkage
          val lvars = map Link.project linkage
          val newlvars = map Link.project newlinkage
	  fun lookup x =
	      let fun f(a::ar, b::br) = if a=x then b else f(ar,br)
		    | f _ = ErrorMsg.impossible "importModule 1"
	       in f(lvars,newlvars)
	      end
	  fun adjustBinding (FCTbind(FCTvar{name,access=PATH[lvar],binding}))=
 	         FCTbind(FCTvar{name = name, access= PATH[lookup lvar],
				binding = binding})
            | adjustBinding (b as SIGbind _) = b
	    | adjustBinding _ = ErrorMsg.impossible "importModule 2"
          val newenv = Env.map adjustBinding env
      in app Link.save newlinkage;
	 (newenv,newlinkage)
      end

  type BinFormat = {statModule: statModule,
		    imports : string list}

  val blastRead: instream -> BinFormat = System.Unsafe.blast_read
  val blastWrite: (outstream * BinFormat) -> unit = System.Unsafe.blast_write

  val blastWrite =    (* Silent version. *)
      fn (stream, obj) =>
	 let val gcmessages = System.Control.Runtime.gcmessages 
	     val oldmsgs = !gcmessages
	  in gcmessages := 0;
	     (blastWrite(stream, obj); gcmessages := oldmsgs)
	     handle e => (gcmessages := oldmsgs; raise e)
	 end

  val BIN_VERSION = System.version ^ " " ^ fileExtension^ "\n"
       (* This is stored as the first line of the binary file. *)

  structure SysIO = System.Unsafe.SysIO
  fun timeFile filename =
      let val System.Timer.TIME{sec, ...} = SysIO.mtime(SysIO.PATH filename)
       in SOME sec
      end
      handle _ => NONE

 (* We must do a syntactic check that the source declarations in a module
    are just functor and signature declarations (or sequences thereof),
    otherwise the renaming routines will fall over later. Importer is the
    place to do it, where we still have a fighting chance of a putting
    out a decent diagnostic message. We don't allow IMPORT - that should
    have been dealt with earlier. *)

  val rec kosherModuleDecl =
	fn BareAbsyn.FCTdec _ => ()
	 | BareAbsyn.SIGdec _ => ()
	 | BareAbsyn.MARKdec(dec,_,_) => kosherModuleDecl dec
	 | BareAbsyn.SEQdec decs => app kosherModuleDecl decs
	 | _ => fail "expecting SIGNATURE/FUNCTOR/IMPORTS"

 (* failure codes for uptodate *)

 datatype cond = OK of BinFormat   (* ok, return contents of bin file *)
               | FINE              (* ok, but don't read whole bin file *)
	       | BIN | OBJ | VERSION | NOBIN | NOOBJ  

  (* check whether a file is up to date.  Optionally returns the contents
     of the associated bin file if the file is up to date.  A file is up to
     date iff a .bin and .o file exist and they are newer than any source
     file.  Note that we do not check whether imports of the file are up to
     date *)

  fun uptodate (path,readbin) name : cond =
       let val {newPath, validName} = impliedPath(path, name)
	   val checkVersion : unit -> cond =
	       fn () =>
	         let val binary = open_in (validName ^ ".bin")
		     val binVersion = input_line binary
	         in if binVersion <> BIN_VERSION
		    then (close_in binary; VERSION)
		    else (if readbin then
			   (OK (blastRead binary))
			 else FINE) before close_in binary
	         end
       in case (timeFile(validName ^ ".sml"), timeFile(validName ^ ".bin"),
		 timeFile(validName ^ ".sml.o"))
	    of (SOME srcTime, SOME binTime,SOME otime) => 
	        (if srcTime >= binTime then BIN
		 else if srcTime >= otime then OBJ
		 else checkVersion ()
		   handle (Io s) => (fail s handle Import => (); NOBIN))
            | (SOME _,SOME _,NONE) => NOOBJ (* No .o file, recompile *)
	    | (SOME _,NONE, _) => NOBIN  (* No bin: force recompile *)
	    | (NONE, SOME _,SOME _) => (* No source: trust for now... *)
		       checkVersion ()
            | (NONE, SOME _,NONE) => fail("cannot open " ^ validName)
	    | (NONE, NONE,_) => fail("cannot find source or binary of module "
				   ^ validName)
       end (* uptodate *)

 (* getModule: returns a static module for a file and the list of all
    files in  the transitive closure of the "import" relation, including
    the file *)

 fun getModule(pervasives, parents, indent, path, name,compiler) :
     statModule * string list =
     (* "parents" is a depth-first list of filenames used for
	a circularity check. "indent" is for cosmetic purposes. *)
    let val {validName as filename, newPath as path} = impliedPath(path, name)
	val _ = if exists (fn x  => x = filename) parents
		then fail("self-referential import of " ^ validName)
		else ()
	val parents = filename :: parents
	val _ = debug("getModule(name=" ^ name ^ ")")
	val srcName = filename ^ ".sml" and binName = filename ^ ".bin"

	fun compSource0(source: instream, name:string) : statModule * string list =
	  let val inputSource = ErrorMsg.newSource(name,source,false,std_out,
						   NONE)
	      val parser = NewParse.parse inputSource
	      fun loop(EOF,  env, lvars, imports) = (env,lvars, imports)
		| loop(ABORT,_,_,_) = fail "syntax error"
		| loop(ERROR,_,_,_) = fail "syntax or semantic error"
		| loop(PARSE(BareAbsyn.IMPORTdec(name::rest),_),
		       env,lvars, imports) =
		       let val (stat,subimports) =
			          getModule(pervasives,parents,indent+2,
					    path,name,compiler)
			   val (newEnv,newLvars) = importModule stat
		       in loop(PARSE(BareAbsyn.IMPORTdec rest,Env.empty),
			       Env.atop(newEnv,env),
			       lvars @ newLvars,imports @ subimports)
		      end
		| loop(PARSE(BareAbsyn.IMPORTdec nil,_), env, l, i) =
			loop(parser(Env.atop(env,pervasives)), env, l, i)
		| loop(PARSE(absyn,env'), env, lvars, imports) =
		     (* normal program *)
		      let val _ = kosherModuleDecl absyn
			  val _ = tab indent
			  val env'' = Env.atop(env',env)
			  val newLvars = compiler(absyn,env'',srcName)
		       in loop(parser(Env.atop(env'',pervasives)),env'',
			       lvars @ newLvars, imports)
		      end

	      val (env, lvars, imports) = 
		  loop(parser pervasives, Env.empty, [], [])
		  handle Import  => raise Import
		       | Io x => fail("unexpected: Io("^x^")")
		       | exn =>  fail("compile-time exception: "
				      ^ System.exn_name exn)
              val imports = uniq imports
	      val _ = debug("imports: "^fold (fn (a,b)=>a^" "^b) imports "")
	      val statModule= STATmodule{env=Env.consolidate env,
					 linkage=lvars}
      	   in let val outstream = open_out binName handle Io s => fail s
               in writing1(binName, indent);
	          outputc outstream BIN_VERSION;
	          blastWrite(outstream,
	  	             {statModule=statModule, imports=imports});
	          close_out outstream;
	          done()
              end handle Import => (); (* make failed writes nonfatal *)
	      (statModule,imports)
	  end  (* fun compSource0 *)

	fun compSource () = let
	      val _ = debug(filename ^ ": source only")
	      val source = open_in srcName
	      fun cleanup () = (closing(srcName, indent);
				close_in source)
	       in reading(srcName, indent);
		  let val (stat,imports) = compSource0(source, srcName)
                  in (stat,imports @ [name]) before cleanup()
                  end handle exp => (cleanup(); raise exp)
	      end

     in reading1(binName, indent);
        case (uptodate(path,true) filename handle e => (print "]\n";
						   closing (binName,indent);
						   raise e))
        of OK {statModule,imports} =>
	         (debug(filename^": checking imports");
	         if all (fn s => case uptodate(path,false) s
		               of FINE => true
				| _ => false) imports
		     then (debug(filename ^ ": all up to date");
			   done();
			   (statModule,name :: imports))
                     else (print "]\n"; tab indent;
			      print("[import(s) of " ^ filename
				^ " are out of date; recompiling]\n");
			   closing(binName,indent);
			   compSource()))
         | BIN => (print "]\n";
		   tab indent;
		   print("[" ^ filename ^ ".bin is out of date; \
			 \recompiling]\n");
		   closing(binName,indent);
		   compSource())
         | OBJ => (print "]\n";
		   print("[" ^ filename ^ ".sml.o is out of date; \
			 \recompiling]\n");
		   closing(binName,indent);
		   compSource())
         | VERSION => (tab indent;
		       print("[" ^ binName
			     ^ " is the wrong format; recompiling\n");
		       closing(binName, indent);
		       compSource())
         | _ => compSource()

    end (* getModule *)

 fun getModule'(pervasives,compiler,filename) =
    if !System.Control.interp then fail "interpreter cannot import"
    else
      let val (statModule as STATmodule{env, ...},srcfiles) =
	      getModule(pervasives,[],0,FilePaths.defaultPath,filename,
	                compiler)
	 (* redo the static bindings of the module *)
	  val (newenv,newlvars) = importModule statModule
       in PrintDec.printBindingTbl env;
	  (newenv,newlvars, srcfiles)
      end

  val getModule = getModule'

end (* functor Importer *)
