signature List' =
sig

(* HISTORY

   Created by:	Dave Berry, LFCS, University of Edinburgh
		db@lfcs.ed.ac.uk
   Date:	3 Oct 1989

   Maintenance:	Author


   DESCRIPTION

   Functions on the built-in type 'a list.

   Many of these functions use an orthoganal naming scheme.  They can be
   grouped as follows:

   f;
     if f is "last", return the last element.
     if f is "sub", return the (n-1)th element.
     if f is "first", return the first element that satisfies a predicate p.
     if f is "all", return the list of elements that satisfy p.
     if f is "prefix", return the prefix of elements that satisfy p.

   dropF ... l; l with (f l) removed.
   splitF ... l; a pair of (f l, dropF l) in the order they appear in l.
   updateF ... l; l with (f l) replaced with a value.
   changeF ... l; l with (f l) replaced with (g (f l)).
   spliceF ... l; l with (f l) replaced with the elements in another list.
   insertF ... l; l with the elements in another list inserted before (f l).
   appendF ... l; l with the elements in another list inserted after (f l).

   The following combinations not provided, because the changeF etc. functions
   operate on an element of a list and the ...Prefix functions return sublists: 
     changePrefix, updatePrefix, insertPrefix, appendPrefix, splicePrefix.
   
   The "last" functions raise "Empty" if l is empty.
   The "sub" functions raise "Sub" if not (0 <= n <= size l - 1).
   The "first" functions (ecept dropFirst) raise "First" if there aren't
	any elements that satisfy p.


   SEE ALSO

   List


   NOTES

   Possibly there should be a dropExtract (aka delete?) function.
*)

  val version: real
   (* version = 0.1 *)


(* PERVASIVES *)

  val map: ('a -> 'b) -> 'a list -> 'b list
  val rev: 'a list -> 'a list
  (* infixr 5 @ *)
  val @ : 'a list * 'a list -> 'a list


(* CREATORS *)

  val create: Nat -> 'a -> 'a list
   (* create n e; create a list of size n, each element being e. *)

  val generate: Nat -> (int -> 'a) -> 'a list
  (* generate n f; create a list of size n, sych that the element with index i
     is initialised to (f i). *)

  val generate': Nat -> ('b -> 'a * 'b) -> 'b -> 'a list
  (* generate' n f base; create a list l of size n, with (l sub 0) initialised
     to (# 1 (f base)) and (l sub i (i > 0)) initialised to (# 1 (f (# 2 f_i))),
     where f_i is the result of the i^th application of f. *)


(* CONVERTORS *)

  val stringSep: string -> string -> string ->
                 ('a -> string) -> 'a list -> string
   (* stringSep start finish sep p l; returns the string representation of l,
      beginning with start, ending with finish, and with the elements
      separated by sep. *)

  val string: ('a -> string) -> 'a list -> string
   (* string p l; returns the canonical string representation of l. *)


(* OBSERVERS *)

  val size: 'a list -> Nat
   (* size l; returns the number of elements in l *)

  val exists: ('a -> bool) -> 'a list -> bool
   (* exists p l; true if there exists an element of l satisfying p *)

  val forAll: ('a -> bool) -> 'a list -> bool
   (* forAll p l; true if every element of l satisfies p *)

  val member: ''a -> ''a list -> bool
   (* member a l; true if a is an element of l *)

  val prefixes: ''a list -> ''a list -> bool
   (* prefixes l1 l2; returns true if l1 is a prefix of l2. *)

  val lt : ('a -> 'a -> bool) -> 'a list -> 'a list -> bool
   (* lt p x y; returns true if (p (x sub i) (y sub i)) for
      0 <= i <= size x or if x is empty and y is not empty. *)
      
  val le : ('a -> 'a -> bool) -> 'a list -> 'a list -> bool
   (* le p x y; returns true if (p (x sub i) (y sub i)) for
      0 <= i <= size x or if x is empty. *)
      
  val gt : ('a -> 'a -> bool) -> 'a list -> 'a list -> bool
   (* gt p x y; returns true if (p (x sub i) (y sub i)) for
      0 <= i <= size y or if y is empty and x is not empty. *)
      
  val ge : ('a -> 'a -> bool) -> 'a list -> 'a list -> bool
   (* lt p x y; returns true if (p (x sub i) (y sub i)) for
      0 <= i <= size x or if y is empty. *)

  val eq: ('a -> 'a -> bool) -> 'a list -> 'a list -> bool
   (* eq x y; returns true if (size x = size y) and for all i,
      0 <= i <= size x, (p (x sub i) (y sub i)). *)

  val ne: ('a -> 'a -> bool) -> 'a list -> 'a list -> bool
   (* ne x y; returns true if (size x <> size y) or for all i,
      0 <= i <= size x, (p (x sub i) (y sub i)). *)

  val eq': ''a list -> ''a list -> bool
   (* eq x y; returns true if (size x = size y) and for all i,
      0 <= i <= size x, ((x sub i) = (y sub i)). *)

  val ne': ''a list -> ''a list -> bool
   (* ne x y; returns true if (size x <> size y) or for all i,
      0 <= i <= size x, ((x sub i) <> (y sub i)). *)

  val index: ('a -> bool) -> 'a list -> (int, unit) Result
   (* index p l; returns the position in l of the first element
      of l satisfying p. *)


(* MANIPULATING THE LAST ELEMENT *)

  exception Empty of string

  val last: 'a list -> 'a
   (* last l; returns the last element of l.  Raises (Empty "last") if l
      is empty. *)

  val dropLast: 'a list -> 'a list
   (* dropLast l; returns l without its last element.
      Raises (Empty "dropLast") if l is empty. *)

  val splitLast: 'a list -> ('a list * 'a)
   (* splitLast l; returns (dropLast l, getLast l).
      Raises (Empty "splitLast") if l is empty. *)

  val updateLast: 'a -> 'a list -> 'a list
   (* updateLast v l; returns l with its last element replaced by v.
      Raises (Empty "updateLast") if l is empty. *)

  val changeLast: ('a -> 'a) -> 'a list -> 'a list
   (* changeLast f l; returns l with (last l) replaced by (f (last l)).
      Raises (Empty "changeLast") if l is empty. *)

  val insertLast: 'a list -> 'a list -> 'a list
   (* insertLast l' l; returns l with the elements of l' inserted before
      (last l).  Raises (Empty "insertLast") if l is empty. *)

  val appendLast: 'a list -> 'a list -> 'a list
   (* appendLast l' l; returns l with the elements of l' appended after
      (last l).  Raises (Empty "insertLast") if l is empty. *)

  val spliceLast: 'a list -> 'a list -> 'a list
   (* spliceLast l' l; returns l with (last l) replaced by the elements of l'.
      Raises (Empty "spliceLast") if l is empty. *)


(* MANIPULATING THE NTH ELEMENT *)

  exception Sub of string * int

  (* infix 9 sub *)
  val sub: 'a list * int -> 'a
   (* l sub n; returns the n-1'th element of l.
      Raises (Sub ("sub", n)) if not (0 <= n <= size l - 1). *)

  val nth: int -> 'a list -> 'a
   (* nth n l; returns the n-1'th element of l.
      Raises (Sub ("nth", n)) if not (0 <= n <= size l - 1). *)

  val splitSub: int -> 'a list -> ('a list * 'a list)
   (* splitSub l n; returns ([l sub 0, ... l sub (n-1)], [l sub n, ... last l]).
      Raises (Sub ("splitSub", n)) if not (0 <= n <= size l - 1).
      If n = (size l - 1) then the second element of the pair is nil. *)

  val dropSub: int -> 'a list -> 'a list
   (* dropSub l n; returns l without (l sub n).
      Raises (Sub ("dropSub", n)) if not (0 <= n <= size l - 1). *)

  val updateSub: int -> 'a -> 'a list -> 'a list
   (* updateSub n v l; returns l with (l sub n) replaced by v.
      Raises (Sub ("updateSub", n)) if not (0 <= n <= size l - 1). *)

  val changeSub: int -> ('a -> 'a) -> 'a list -> 'a list
   (* changeSub n f l; returns l with (l sub n) replaced by (f (l sub n)).
      Raises (Sub ("changeSub", n)) if not (0 <= n <= size l - 1). *)

  val insertSub: int -> 'a list -> 'a list -> 'a list
   (* insertSub n l' l; returns l with the elements of l' inserted before
      (l sub n).
      Raises (Sub ("insertSub", n)) if not (0 <= n <= size l - 1). *)

  val appendSub: int -> 'a list -> 'a list -> 'a list
   (* appendSub n l' l; returns l with the elements of l' appended after
      (l sub n).
      Raises (Sub ("insertSub", n)) if not (0 <= n <= size l - 1). *)

  val spliceSub: int -> 'a list -> 'a list -> 'a list
   (* spliceSub n l' l; returns l with (l sub n) replaced by the elements of
      l'.  Raises (Sub ("spliceSub", n)) if not (0 <= n <= size l - 1). *)


(* ACCESSING A RANGE OF ELEMENTS *)

  exception ExtractLast of int
  val extractLast: int -> 'a list -> 'a list
   (* extractLast l start ; returns the elements of l from (l sub start) to
      last l.  Raises (ExtractLast start) if not (0 <= start <= size l - 1). *)

  exception Extract of int * int
  val extract: int -> int -> 'a list -> 'a list
   (* extract start finish l; returns the elements of l from (l sub start) to
      (l sub (finish - 1)).  Returns [] if (start = finish).  Raises
      (Extract (start, finish)) if not (0 <= start <= finish <= size l - 1). *)


(* MANIPULATING THE FIRST ELEMENT THAT SATISFIES A PREDICATE *)

  exception First of string

  val first: ('a -> bool) -> 'a list -> 'a
   (* first p l; returns the first element in l satisfying p.
      Raises (First "first") if p doesn't hold for any element of l. *)

  val dropFirst: ('a -> bool) -> 'a list -> 'a list
   (* dropFirst p l; returns l without the first of its elements (if any)
      that satisfy p. *)

  val splitFirst: ('a -> bool) -> 'a list -> ('a list * 'a list)
   (* splitFirst p l; returns ([l sub 0 .. l sub (n-1)], extractLast n l),
      where (l sub n) is the first element of l that satisfies p.
      Raises (First "splitFirst") if p doesn't hold for any element of l. *)

  val updateFirst: ('a -> bool) -> 'a -> 'a list -> 'a list
   (* updateFirst p v l; returns l with (first p l) replaced by v.
      Raises (First "updateFirst") if there is no (first p l). *)

  val changeFirst: ('a -> bool) -> ('a -> 'a) -> 'a list -> 'a list
   (* changeFirst p f l; returns l with (first p l) replaced by
      (f (first p l)).  Raises (First "changeFirst") if there is no
      (first p l). *)

  val insertFirst: ('a -> bool) -> 'a list -> 'a list -> 'a list
   (* insertFirst p l' l; returns l with the elements of l' inserted before
      (first p l).  Raises (First "insertFirst") if there is no (first p l). *)

  val appendFirst: ('a -> bool) -> 'a list -> 'a list -> 'a list
   (* appendFirst p l' l; returns l with the elements of l' appended after
      (first p l).  Raises (First "insertFirst") if there is no (first p l). *)

  val spliceFirst: ('a -> bool) -> 'a list -> 'a list -> 'a list
   (* spliceFirst p l' l; returns l with (first p l) replaced by the elements
      of l'.  Raises (First "spliceFirst") if there is no (first p l). *)


(* TAKING A PREFIX OF ELEMENTS THAT SATISFY A PREDICATE *)

  val prefix: ('a -> bool) -> 'a list -> 'a list
   (* prefix p l; returns the largest prefix of l each of whose
      elements satisfies p *)

  val dropPrefix: ('a -> bool) -> 'a list -> 'a list
   (* dropPrefix p l; returns l without the largest prefix of l
      each of whose elements satisfies p *)

  val splitPrefix: ('a -> bool) -> 'a list -> ('a list * 'a list)
   (* splitPrefix p l; returns (prefix p l, dropPrefix p l). *)


(* MANIPULATING ALL ELEMENTS THAT SATISFY A PREDICATE *)

  val all: ('a -> bool) -> 'a list -> 'a list
   (* all p l: returns a list of the elements of l that satisfy p. *)

  val dropAll: ('a -> bool) -> 'a list -> 'a list
   (* dropAll p l: returns a list of the elements of l that don't satisfy p. *)

  val splitAll: ('a -> bool) -> 'a list -> ('a list * 'a list)
   (* splitAll p l; returns (all p l, dropAll p l). *)

  val updateAll: ('a -> bool) -> 'a -> 'a list -> 'a list
   (* updateAll p v l; returns l with each element that satisfies p
      replaced by v. *)

  val changeAll: ('a -> bool) -> ('a -> 'a) -> 'a list -> 'a list
   (* changeAll p f l; returns l with each element e that satisfies p
      replaced by (f e). *)

  val insertAll: ('a -> bool) -> 'a list -> 'a list -> 'a list
   (* insertAll p l' l; returns l with the elements of l' inserted before each
      element of l that satisfies p. *)

  val appendAll: ('a -> bool) -> 'a list -> 'a list -> 'a list
   (* appendAll p l' l; returns l with the elements of l' appended after each
      element of l that satisfies p. *)

  val spliceAll: ('a -> bool) -> 'a list -> 'a list -> 'a list
   (* spliceAll p l' l; returns l with each element that satisfies p
      replaced by the elements of l'. *)


(* OTHER MANIPULATORS *)

  val appendIfAll: ('a -> bool) -> 'a list -> 'a list -> 'a list
   (* appendIfAll p l' l; appends l' at the end of l if every element of
      l satisfies p. *)

  val perms: 'a list -> 'a list list
   (* perms l; returns a list whose elements are all the permutations of l*)

  val sort: ('a -> 'a -> bool) -> 'a list -> 'a list
   (* sort p l; returns l sorted by p. *)

  val dropRepeats: ('a -> 'a -> bool) -> 'a list -> 'a list
   (* dropRepeats p l; removes duplicated elements from l.  The result is
      undefined if p is not commutative. *)

  val dropRepeats': ''a list -> ''a list
   (* dropRepeats' l; removes duplicated elements from l. *)


(* ITERATORS *)

(* map is pervasive
  val map: ('a -> 'b) -> 'a list -> 'b list
   (* map f l; builds a list by applying f to each element of l. *)
*)

  val mapAll: ('a -> bool) -> ('a -> 'b) -> 'a list -> 'b list
   (* mapAll p f l; builds a list by applying f to each element of l that
      satisfies p. *)

  val iterate: ('a * int -> 'b) -> 'a list -> 'b list
   (* iterate f l; builds a list by applying f to each element of l paired
      with its index. *)

  val apply: ('a -> unit) -> 'a list -> unit
   (* apply f l; applies f to each element of l. *)

  val applyAll: ('a -> bool) -> ('a -> unit) -> 'a list -> unit
   (* applyAll p f l; applies f to each element of l that satisfies p. *)

  val iterateApply: ('a * int -> unit) -> 'a list -> unit
   (* iterateApply f l; applies f to each element of l paired
      with its index. *)


(* REDUCERS *)

  val foldR: ('a -> 'b -> 'b) -> 'b -> 'a list -> 'b
   (* foldR f base l; folds using f associating to the right over the
      base element.
      foldR f [a1,a2,...,an] base = f a1 (f a2 ... (op an base)...). *)

  val foldL: ('a -> 'b -> 'b) -> 'b -> 'a list -> 'b
   (* foldL f l base; folds using f associating to the left over the
      base element.
      foldL f [a1,a2,...,an] base = f an ... (f a2 (f a1 base))... . *)

  val foldR': ('a -> 'a -> 'a) -> 'a list -> 'a
   (* foldR' f l; folds using f associating to the right over the
      last element of l.  Raises (Empty "foldR'") if l is empty. *)

  val foldL': ('a -> 'a -> 'a) -> 'a list -> 'a
   (* foldL' f l; folds using f associating to the left over the
      first element of l.  Raises (Empty "foldL'") if l is empty. *)

  val pairwise: ('a -> 'a -> bool) -> 'a list -> bool
   (* pairwise f l; true if (f (l sub i) (l sub (i + 1))) is true for all
      0 <= i < size l, or if l is empty. *)
end

