signature InStream =
sig

(* INPUT STREAMS

   Created by:  Dave Berry, LFCS, University of Edinburgh
                db@lfcs.ed.ac.uk
   Date:        12 Nov 1989

   Maintenance: Author


   DESCRIPTION

   Functions on input streams.


   SEE ALSO

   OutStream, StreamPair


   NOTES

   Possibly these system functions should raise a new exception when called
   on a closed stream.

   All counterparts of functions in the initial basis have been given both
   the original names (for backwards compatibility) and names that fit the
   library conventions.

   The third argument of the General.Io exception is a system-specific
   message.

   The input1 function is included to handle the common case of reading a
   single character; this function will remain valid even if input is
   changed to be curried.
*)

  val version: real
   (* version = 0.1 *)


(* TYPES *)

  type InStream
  type instream
    sharing type instream = InStream


(* SYSTEM *)

  val openString: string -> InStream
   (* openString s; returns an InStream.  The characters read from this
      stream will be those in s, in order, with the end of s being read as
      an end of file. *)

  val canInput: InStream -> Nat -> bool
   (* canInput i n; returns true if n characters can be read from i
      without blocking. *)

  val reset: InStream -> bool
   (* reset i; if i can be reset to the beginning, in some sense,
     this is done and true is returned.  Otherwise false is returned. *)

  val interactive: InStream -> bool
   (* interactive i; returns true if i is associated with an interactive
      device. *)

  val input': InStream -> Nat -> string
   (* input' i n; returns the empty string if there are no characters
      remaining in i before an end of file.  Otherwise it returns a string
      of at least 1 and not more than n characters, whatever input is
      available from i. *)


(* MANIPULATORS *)

  val stdIn: InStream
   (* stdin; the standard input stream.  By default, connected to the
      terminal. *)

  val std_in: InStream
   (* std_in = stdIn *)

  val openIn: string -> InStream
   (* openIn s; opens the file s for reading.
      Raises (General.Io ("openIn", s, "")) if s does not exist or cannot
      be opened for reading. *)

  val open_in: string -> InStream
   (* open_in = openIn *)

  val closeIn: InStream -> unit
   (* closeIn i; empties and terminates i. *)

  val close_in: InStream -> unit
   (* close_in = closeIn *)

  val input: InStream * Nat -> string
   (* input (i, n); returns a string containing the first n characters of i,
      removing them from i.  If less than n characters can be read, the
      read blocks until either n characters can be read or an end of file
      is read.  Reading an end of file does not necessarily terminate the
      stream. *)

  val input1: InStream -> string
   (* input1 i; returns the first character on the InStream i, or the empty
      string if an end of file is read.  Blocks if there is no character
      or end of file to be read. *)

  val lookahead: InStream -> string
   (* lookahead i; returns the next character available on i, without removing
      it from i.  If no character is available the call blocks until one
      becomes available or an end of file is read.  If an end of file is read,
      lookahead returns "" and consumes the end of file.  Reading an end of
      file does not necessarily terminate the stream. *)

  val end_of_stream: InStream -> bool
   (* end_of_stream i; returns true if the last call to input or lookahead on
      i read an end of file. *)

  val eof: InStream -> bool
   (* eof = end_of_stream *)

  val readString: InStream -> string -> (unit, string) Result
   (* readString i s; returns OK () if reading from i gives the characters
      in s.  Returns (Fail s') as soon as the end of file is reached or a
      character is read that doesn't match the corresponding one in s, where
      s' is the characters read so far.  In either case all characters read
      from i are lost.  *)

  val skip: (string -> bool) -> InStream -> unit
   (* skip p i; reads all characters from i that satisfy p.  Leaves the first
      character that doesn't satisfy p to be read.  *)

  val inputLine : InStream -> string
   (* inputLine i; returns a strings consisting of characters read from i
      up to and including the next end of line character.  If the end of the
      file is reached first, all characters up to the end of file are returned
      (without a new line character). *)
end

