(* Gene Rollins rollins@cs.cmu.edu
   School of Computer Science, Carnegie-Mellon Univ., Pittsburgh, PA 15213
DESCRIPTION
  The module GnuTags provides functions to generate TAGS files that can be used
  with the gnu-emacs Tags package.  That package provides a set of functions
  for finding the definitions within source files of identifiers (called tags).
  The SML compiler silently generates index files as it compiles source files.
  After the index files are generated, one can build a TAGS file from them
  using the GnuTags functions. *)

functor GnuTagsFun 
 (structure DirFile :DIRFILE
  structure Pathname :PATHNAME
  structure StringXtra :STRINGXTRA
 ) = struct

type tag'info = {last'chr :string, left'lnum :string,
                 left'pos :string, source'text :string}

val index'stream = ref(std_in)
val source'stream = ref(std_in)
val tags'stream = ref(std_out)
val tags'list :tag'info list ref = ref []
val source'line :string ref = ref ""
val source'pos :int ref = ref 0
val source'linenum :int ref = ref 0

fun get'next'source'line () =
 (source'line := input_line (!source'stream);
  source'linenum := (!source'linenum) + 1;
  source'pos := 0)

fun goto'source'line (n :int) =
  while (!source'linenum) < n do get'next'source'line()

fun ith (str,i) = if i >= (size str) then " " else chr(ordof(str,i))

datatype ScanState = START | STRING | COMMENT | IDENT | MATCH

fun bump i =
  if (i+1) >= size (!source'line)
    then (get'next'source'line (); 0)
    else i+1

fun bump2 i =
  if (i+2) >= size (!source'line)
    then (get'next'source'line (); 0)
    else i+2

fun scanIdentifier ident source'position (line'limit:int) =
  let val id'len = String.size ident
      fun scan i id state =
        let val next'src = ith (!source'line, i)
            val second'src = ith (!source'line, i+1) in 
          if (!source'linenum) > line'limit then 0 else
            case state of
              START =>
                if next'src = "\""
                    then scan (bump i) 0 STRING
                  else if next'src = "(" andalso (second'src = "*")
                    then scan (bump2 i) 0 COMMENT
                  else if next'src = (ith (ident, 0))
                    then scan (bump i) 1 MATCH
                  else if StringXtra.isIdChr next'src
                    then scan (bump i) 0 IDENT
                    else scan (bump i) 0 START |
              IDENT =>
                if StringXtra.isIdChr next'src
                  then scan (bump i) 0 IDENT
                  else scan (bump i) 0 START |
              STRING =>
                if next'src = "\\"
                    then scan (bump2 i) 0 STRING
                  else if next'src = "\""
                    then scan (bump i) 0 START
                    else scan (bump i) 0 STRING |
              COMMENT =>
                if next'src = "*" andalso (second'src = ")")
                  then scan (bump2 i) 0 START
                  else scan (bump i) 0 COMMENT |
              MATCH =>
                if id = id'len
                  then if StringXtra.isIdChr next'src
                    then scan (bump i) 0 IDENT
                    else i
                  else if next'src = (ith (ident, id))
                    then scan (bump i) (id+1) MATCH
                    else if StringXtra.isIdChr next'src
                      then scan (bump i) 0 IDENT
                      else scan (bump i) 0 START
        end
  in
    if id'len = 0 then 0 else scan source'position 0 START
  end

fun get'next'index'line () =
  if end_of_stream (!index'stream) then (0, "")
    else
      let val line = input_line(!index'stream)
          val pos = StringXtra.skipBlanks (0, line)
      in
        if pos = (size line)
          then get'next'index'line()
          else (pos, line)
      end

fun next'index'line () =
  let val (l0, line) = get'next'index'line () in 
    if line = ""
      then ("", 0, 0, false, "")
      else
        let fun skipBlanks (pos) = (StringXtra.skipBlanks (pos, line), line)
	    val (l1, tag'name) = StringXtra.getWord (l0, line)
            val (l2, left'lnum) = StringXtra.getInt (skipBlanks l1)
            val (l3, left'pos) = StringXtra.getWord (skipBlanks l2)
            val (l4, and'chr) = StringXtra.getWord (skipBlanks l3)
            val (_, right'lnum) = StringXtra.getInt (skipBlanks l4)
            val andp = (ith (and'chr, 0)) = "A"
        in (tag'name, left'lnum, right'lnum, andp, left'pos)
        end
  end

fun do'index'file (result :tag'info list) =
  let val (name,left'lnum,right'lnum,andp,left'pos) = next'index'line() in
    if name = "" then result
      else
        (goto'source'line left'lnum;
	 if andp
	   then source'pos := scanIdentifier "and" (!source'pos) right'lnum
	   else ();
         let val id'pos = scanIdentifier name (!source'pos) right'lnum
         in if id'pos = 0
              then do'index'file result
              else 
                let val last'chr =
		      if (id'pos+1) < (size (!source'line))
		        then ith (!source'line,id'pos) else ""
                    val tag'line =
                          {left'lnum = makestring (!source'linenum),
                           left'pos = left'pos,
                           source'text = substring (!source'line, 0, id'pos),
                           last'chr = last'chr}
                in do'index'file (tag'line::result)
                end
         end)
  end;

fun text'length ({left'lnum,left'pos,source'text,last'chr} :tag'info) =
  (size source'text) + (size last'chr) + (size left'lnum) + (size left'pos) + 2

fun tags'length [] = 0
  | tags'length (head::tail) = (text'length head) + 1 + tags'length tail

fun output'tag'line ({left'lnum,left'pos,source'text,last'chr} :tag'info) =
  let val pr = outputc (!tags'stream) in
    pr source'text;
    pr last'chr; pr (chr 127);
    pr left'lnum; pr ",";
    pr left'pos; pr "\n"
  end

fun output'tags [] = ()
  | output'tags (head::tail) = (output'tag'line head; output'tags tail)

exception ignore'index'file;

fun open'in'ignoring'errors (filename :string) (indexname :string) =
  (open_in filename handle e as Io s =>
     (print ("% "^ s ^ "\n  Index file " ^ indexname ^ " will be ignored\n");
      raise ignore'index'file))

fun do'tagging (indexname :string) (sourcename :string) =
 ((source'stream := open'in'ignoring'errors sourcename indexname;
   index'stream := open'in'ignoring'errors indexname indexname;
   source'linenum := 0;
   tags'list := do'index'file [];
   outputc (!tags'stream) "\n"; 
   outputc (!tags'stream)
     (sourcename ^ "," ^ (makestring (tags'length (!tags'list))) ^"\n");
   output'tags (rev (!tags'list))) handle ignore'index'file => ();
  close_in (!source'stream);
  close_in (!index'stream))
 
fun tag'file dirname filename filetype =
  if ((size filename) > 3) andalso (substring (filename, 0, 3) = ".i.")
    then do'tagging
           (Pathname.mergeDirFile dirname filename)
           (Pathname.mergeDirFile dirname
             (substring (filename, 3, (size filename)-3)))
    else ()
  
fun clean'file dirname filename filetype =
  if ((size filename) > 3) andalso (substring (filename, 0, 3) = ".i.")
    then
      System.Unsafe.SysIO.unlink (Pathname.mergeDirFile dirname filename)
    else ()

fun cleandir (recursive :bool, follow :bool) (dirname :string)  :unit =
  let val recoption = if recursive then [DirFile.RECURSIVE] else []
      val options =
	    if follow then (DirFile.FOLLOWDIRS::recoption) else recoption
  in DirFile.scan clean'file options dirname
  end

fun tag'directory  (recursive:bool, follow:bool, dirname:string) =
  let val recoption = if recursive then [DirFile.RECURSIVE] else []
      val options =
	    if follow then (DirFile.FOLLOWDIRS::recoption) else recoption
  in DirFile.scan tag'file (DirFile.ALPHA::options) dirname
  end

fun tagdir (recursive:bool, follow:bool) (dirname:string)  :unit =
 (tags'stream := open_out (Pathname.mergeDirFile dirname "TAGS");
  tag'directory (recursive, follow, dirname);
  close_out (!tags'stream))

fun tagFile dname filename filetype =
  let val dirname = if dname = "" then "." else dname
      val tagfilename =  (Pathname.mergeDirFile dirname (".t."^filename)) in
    tags'stream := open_out tagfilename;
    do'tagging (Pathname.mergeDirFile dirname (".i." ^ filename))
               (Pathname.mergeDirFile dirname filename);
    close_out (!tags'stream)
  end

fun b (flag:bool)  = if flag then "T" else "F"

fun operate (doCleaning:bool, recursive :bool, follow :bool)
            (pathname:string) =
  let val (dirname, filename) = Pathname.splitDirFile pathname
      val ((newdirname, newfilename), kind, accessible) = 
            DirFile.checkFile dirname filename follow false
  in
    case kind of
       DirFile.DIR => (if doCleaning then cleandir else tagdir)
                (recursive, follow)
                (Pathname.mergeDirFile newdirname newfilename)
     | _ => (if doCleaning then clean'file else tagFile)
              newdirname newfilename kind
  end

fun err (msg:string) =
  (output (std_err, "? sml-tags: "^msg^"\n");
   System.Unsafe.CInterface.exit 1)

fun switch (c,f,r,switches,[]:string list) = (c,f,r,[])
  | switch (c,f,r,false,head::rest) =
      if substring(head,0,1) = "-" 
        then err "All switches must be listed first"
        else let val (c',f',r',filenames) = switch(c,f,r,false,rest)
             in (c,f,r,head::filenames) end
  | switch (c,f,r,true,head::rest) =
      if not (substring(head,0,1) = "-") then switch (c,f,r,false,head::rest)
        else if head = "-r" then switch (c,f,true,true,rest)
        else if head = "-f" then switch (c,true,r,true,rest)
        else if head = "-c" then switch (true,f,r,true,rest)
        else err ("Unknown switch " ^ head)

fun dispatch ([] :string list,_:string list) = err "internal error - no argv"
  | dispatch (name::[],_) = tagdir (false, false) "."
  | dispatch (name::args,_) =
      let val (c,f,r,filenames) = switch (false,false,false,true,args) in
        if filenames = []
          then (if c then cleandir (r,f) "." else tagdir (r,f) ".")
          else (map (operate (c,r,f)) filenames; ())
      end

end
