;;; -*- Mode: LISP; Syntax: Common-Lisp; Package: KR; Base: 10 -*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;         The Garnet User Interface Development Environment.      ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; This code was written as part of the Garnet project at          ;;;
;;; Carnegie Mellon University, and has been placed in the public   ;;;
;;; domain.  If you are using this code or any part of Garnet,      ;;;
;;; please contact garnet@cs.cmu.edu to be put on the mailing list. ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; This file contains a number of changes to kr that are required to
;;; make lapidary work. These changes will probably be incorporated in
;;; the next release of kr.

(in-package 'kr)

(defun copy-to-all-instances (schema a-slot value &optional (is-first T))
  (s-value schema a-slot value)
  ;; Do not create copies of formulas, but set things up for inheritance
  (if is-first
    (if (formula-p value)
      (setf value *no-value*)))
  (dolist (inverse *inheritance-inverse-relations*)
    (let ((children (if (eq inverse :IS-A-INV) ; for efficiency
                      (slot-accessor schema :IS-A-INV)
                      (get-local-value schema inverse))))
      (if (not (eq children *no-value*))
        (dolist (child children)
          ;; force new inheritance
	  (when (not (formula-p (get-value child a-slot)))
		(copy-to-all-instances child a-slot value NIL)))))))


;;;; GET-DEPENDENTS
;;;
;;; RETURNS: the formulas which depend on the <slot> of the <schema>.
;;;
(let ((list-of-one (list nil)))

(defun get-dependents (schema slot)
  (multiple-value-bind (value position)
      (slot-accessor schema slot)	; set accessors
    (declare (ignore value))
    ;; access the dependent formulas.
    (when position
      (let ((dependents (last-slot-dependents (schema-slots schema) position)))
        (when dependents
	      (if (listp dependents)
		  dependents
	          (progn
		    (setf (car list-of-one) dependents)
		    list-of-one))))))))


;;;; CHANGE-FORMULA
;;; 
;;; Modify the function associated with a formula.  Several possible
;;; combinations exist:
;;; - If the function is local and there are no children, just go ahead and
;;;   invalidate the formula.
;;; - if the function is local and there are children, invalidate all the
;;;   children formulas as well.
;;; - if the function used to be inherited, replace it and eliminate the
;;;   link with the parent formula.
;;; 
(defun change-formula (schema slot form)
  "Modifies the formula at position 0 in the <slot> of the <schema> to have
  <form> as its new function.  Inherited formulas are treated appropriately."
  (let ((formula (get-value schema slot)))
    (when (formula-p formula)
      (when (a-formula-is-a formula)
	;; This function was inherited.  Cut the IS-A link.
	(let ((parent (a-formula-is-a formula)))
	  (setf (a-formula-is-a-inv parent)
		(delete formula (a-formula-is-a-inv parent))))
	(setf (a-formula-is-a formula) NIL))

      ;; If this formula has children, we need to invalidate them as well.
      (dolist (f-child (a-formula-is-a-inv formula))
	#-EAGER
	(set-cache-is-valid f-child nil)
	#-EAGER
	(mark-as-changed (on-schema f-child) (on-slot f-child))
	#+EAGER
	;; If this formula has children, we need to place them on the
	;; evaluation queue
	(setf *eval-queue* (insert-pq f-child *eval-queue*)))
      #-EAGER
      ;; Invalidate the formula itself.
      (set-cache-is-valid formula nil)
      #-EAGER
      (mark-as-changed schema slot)
      #+EAGER
      ;; Add the formula itself to the evaluation queue
      (setf *eval-queue* (insert-pq formula *eval-queue*))

      ;; Record the new function.
      (setf (a-formula-function formula) `(lambda () ,form))
      ;; store the new form in the lambda slot of the formula
      (setf (a-formula-lambda formula) form))))

;;;; DESTROY-CONSTRAINT
;;; Replaces the formula in the <slot> with its value, physically eliminating
;;; the constraint.  If there is no formula, <schema> is unchanged.
;;; 
(defun destroy-constraint (schema slot)
  "If the value in the <slot> of the <schema> is a formula, replace it with
  the current value of the formula and eliminate the formula.  This
  effectively eliminates the constraint on the value."
;; only do something if the schema has not been destroyed--this should be
;; a temporary fix--a formula should not be left in a destroyed object
  (when (and (schema-p schema) slot)
  (multiple-value-bind (value position)
      (slot-accessor schema slot)
    (let ((formula (or value (g-value-inherit-values schema slot T position))))
      (when (and formula (formula-p formula)
		 (a-formula-slots formula))	; not already deleted
	(let ((value (g-cached-value schema slot)))
	  ;; All children formulas are eliminated as well.
	  (dolist (child (a-formula-is-a-inv formula))
	    (when (a-formula-slots child) ; do nothing if already deleted.
;	      (g-value (on-schema child) (on-slot child))	; get value
	      (destroy-constraint (on-schema child) (on-slot child))))
	  (delete-formula formula NIL)
	  #+EAGER
	  ;; set all dependency edges to be non-cycle edges since this
	  ;; position can no longer be part of a cycle
	  (dolist-test (dependent (get-formula-dependents formula)
				  (and (cycle-edge-p dependent)
				       (valid-dependency-p dependent)))
		       (set-cycle-edge-bit dependent nil))
	  ;; Replace formula with its cached value.
	  (set-slot-accessor schema slot value *local-mask*)
	  NIL))))))
