/* Random.c -- implementation of pseudo-random number generator

	THIS SOFTWARE FITS THE DESCRIPTION IN THE U.S. COPYRIGHT ACT OF A
	"UNITED STATES GOVERNMENT WORK".  IT WAS WRITTEN AS A PART OF THE
	AUTHOR'S OFFICIAL DUTIES AS A GOVERNMENT EMPLOYEE.  THIS MEANS IT
	CANNOT BE COPYRIGHTED.  THIS SOFTWARE IS FREELY AVAILABLE TO THE
	PUBLIC FOR USE WITHOUT A COPYRIGHT NOTICE, AND THERE ARE NO
	RESTRICTIONS ON ITS USE, NOW OR SUBSEQUENTLY.

Author:
	K. E. Gorlen
	Bg. 12A, Rm. 2033
	Computer Systems Laboratory
	Division of Computer Research and Technology
	National Institutes of Health
	Bethesda, Maryland 20892
	Phone: (301) 496-1111
	uucp: uunet!nih-csl!keith
	Internet: keith@alw.nih.gov
	December, 1985

Function:
	
A psuedo-random number generator.  The function next() returns random
numbers uniformly distributed over the interval (0.0,1.0).

Reference:

Pierre L'ecuyer, "Efficient and Portable Combined Random Number
Generators", Commun. ACM 31, 6 (June 1988), 742-749.

$Log:	Random.c,v $
 * Revision 2.204  89/10/07  23:21:05  keith
 * Pre-release
 * 
 * Revision 2.203  89/08/08  15:23:46  keith
 * Pre-release
 * 
 * Revision 2.202.1.1  89/07/01  21:55:34  keith
 * Base revision for R2.00 MI version
 * 
 * Revision 2.202  89/06/22  20:55:32  keith
 * Base revision for AT&T C++ R2.0 release (Cycle 20)
 * 
 * Revision 2.201.1.2  89/06/21  11:12:47  keith
 * Add Object::desc() to VIRTUAL_BASE_CLASSES.
 * Add explicit base class names to constructor initializer lists.
 * 
 * Revision 2.201.1.1  89/06/13  22:53:17  keith
 * Base revision for Cycle 16.1.
 * 
 * Revision 2.201  89/05/12  11:19:47  keith
 * Release for R2.0 Beta test.
 * 
 * Revision 2.200.1.3  89/05/12  11:00:32  keith
 * Revised Object I/O.
 * 
 * Revision 2.200.1.2  89/05/03  23:09:31  keith
 * Utilize abstract classes.
 * 
 * Revision 2.200.1.1  89/04/24  17:17:16  keith
 * Working revision for R2.0 Beta 6++
 * 
 * Revision 2.200  89/04/17  23:30:44  keith
 * Base revision for R2.0 Beta 6.
 * 
 * Revision 2.121  89/02/16  11:08:16  keith
 * Base revision for C++ R1.2.1 compatible version.
 * 
 */

#include "Random.h"
#include <time.h>
#include "nihclIO.h"

#define	THIS	Random
#define	BASE	Object
#define BASE_CLASSES BASE::desc()
#define MEMBER_CLASSES
#define VIRTUAL_BASE_CLASSES Object::desc()

DEFINE_CLASS(Random,2,"$Header: Random.c,v 2.204 89/10/07 23:21:05 keith Stab $",NULL,NULL);

const long s1max = 2147483562;
const long s2max = 2147483398;

void Random::checkSeeds()
{
	if (sizeof(long) != 4) {
		cerr << "\nRandom only works on 32-bit machines\n";
		exit(1);
	}
	if (s1 < 1 || s1 > s1max) s1 = (ABS(s1) % s1max) + 1;
	if (s2 < 1 || s2 > s2max) s2 = (ABS(s2) % s2max) + 1;
}

Random::Random()
{
	time(&s1);
	s2 = (long)this;
	checkSeeds();
}

Random::Random(long seed1, long seed2)
{
	s1 = seed1;  s2 = seed2;
	checkSeeds();
}

float Random::next()
{
	long Z,k;

	k = s1/53668;
	s1 = 40014 * (s1 - k * 53668) - k * 12211;
	if (s1 < 0) s1 += s1max + 1;

	k = s2/52774;
	s2 = 40692 * (s2 - k * 52774) - k * 3791;
	if (s2 < 0) s2 += s2max + 1;

	Z = s1 - s2;
	if (Z < 1) Z += s1max;

	return Z * 4.656613E-10;
}

void Random::deepenShallowCopy()	{}

unsigned Random::hash() const	{ return (unsigned)this; }

bool Random::isEqual(const Object& ob) const   { return isSame(ob); }

void Random::printOn(ostream& strm) const
{
	strm << s1 << ' ' << s2;
}

Random::Random(OIOin& strm)
	: BASE(strm)
{
	strm >> s1 >> s2;
}

void Random::storer(OIOout& strm) const
{
	BASE::storer(strm);
	strm << s1 << s2;
}

Random::Random(OIOifd& fd)
	: BASE(fd)
{
	fd >> s1 >> s2;
}

void Random::storer(OIOofd& fd) const
{
	BASE::storer(fd);
	fd << s1 << s2;
}

int Random::compare(const Object&) const
{
	shouldNotImplement("compare");
	return 0;
}
