#ifndef FDSET_H
#define FDSET_H

/*$Header: FDSet.h,v 2.204 89/10/07 23:19:23 keith Stab $*/

/* FDSet.h -- Class for manipulating fd_set objects used by select(2)

	THIS SOFTWARE FITS THE DESCRIPTION IN THE U.S. COPYRIGHT ACT OF A
	"UNITED STATES GOVERNMENT WORK".  IT WAS WRITTEN AS A PART OF THE
	AUTHOR'S OFFICIAL DUTIES AS A GOVERNMENT EMPLOYEE.  THIS MEANS IT
	CANNOT BE COPYRIGHTED.  THIS SOFTWARE IS FREELY AVAILABLE TO THE
	PUBLIC FOR USE WITHOUT A COPYRIGHT NOTICE, AND THERE ARE NO
	RESTRICTIONS ON ITS USE, NOW OR SUBSEQUENTLY.

Author:
	K. E. Gorlen
	Computer Systems Laboratory, DCRT
	National Institutes of Health
	Bethesda, MD 20892

Modification History:

$Log:	FDSet.h,v $
 * Revision 2.204  89/10/07  23:19:23  keith
 * Pre-release
 * 
 * Revision 2.203  89/08/08  15:21:55  keith
 * Pre-release
 * 
 * Revision 2.202.1.1  89/07/01  21:53:53  keith
 * Base revision for R2.00 MI version
 * 
 * Revision 2.202  89/06/22  20:53:54  keith
 * Base revision for AT&T C++ R2.0 release (Cycle 20)
 * 
 * Revision 2.201.1.2  89/06/21  22:05:23  keith
 * Add private copy constructor to private classes.
 * 
 * Revision 2.201.1.1  89/06/01  23:15:28  keith
 * Remove base class argument from DECLARE_MEMBERS.
 * 
 * Revision 2.201  89/05/30  23:44:26  keith
 * Initial revision.
 * 

*/

#include "Object.h"
#include <libc.h>
#include <stream.h>
#include <sys/types.h>

#ifndef	FD_SETSIZE

const unsigned FD_SETSIZE = sizeof(fd_set) * 8;
inline void FD_SET(int	fd, fd_set* p)	{ *(long*)p |= 1 << (fd % FD_SETSIZE); }
inline void FD_CLR(int	fd, fd_set* p)	{ *(long*)p &= ~(1 << (fd % FD_SETSIZE)); }
inline bool FD_ISSET(int fd, fd_set* p)	{ return *(long*)p & (1 << (fd % FD_SETSIZE)); }
inline void FD_ZERO(fd_set* p)		{ *(long*)p = 0; }

#endif

class FDSet;

class FDSetRef: public NIHCL {
	fd_set*	p;	// pointer to word containing bit
	int fd;		// file descriptor
	FDSetRef(fd_set& fdset, int fdesc) {
		p = &fdset; fd = fdesc;
	}
	FDSetRef(const FDSetRef& s) {
		p = s.p; fd = s.fd;
	}
	friend FDSet;
public:
	operator bool() const	{ return FD_ISSET(fd,p); }
	bool operator=(bool b) {
		if (b) FD_SET(fd,p);
		else FD_CLR(fd,p);
		return b;
	}
	void operator&=(bool b) {
		if (b && FD_ISSET(fd,p)) FD_SET(fd,p);
		else FD_CLR(fd,p);
	}
	void operator|=(bool b) {
		if (b || FD_ISSET(fd,p)) FD_SET(fd,p);
		else FD_CLR(fd,p);
	}
	void operator^=(bool b) {
		if ((b != 0) != (FD_ISSET(fd,p) !=0 )) FD_SET(fd,p);
		else FD_CLR(fd,p);
	}
};

class FDSet: public VIRTUAL Object {
	DECLARE_MEMBERS(FDSet);
	static unsigned	dtablesz;   // size of file descriptor table
public:
	static unsigned dtablesize() { return dtablesz; }
private:
	fd_set fs;		    // system-defined file descriptor set
protected:		// storer() functions for object I/O
	virtual void storer(OIOofd&) const;
	virtual void storer(OIOout&) const;
public:
	FDSet()				{ FD_ZERO(&fs); }
	operator fd_set&()		{ return fs; }
	operator const fd_set&() const	{ return fs; }
	operator fd_set*()		{ return &fs; }
	operator const fd_set*() const	{ return &fs; }
	FDSetRef operator[](int	fd)		{ return FDSetRef(fs,fd); }
	const FDSetRef operator[](int fd) const	{ return FDSetRef(fs,fd); }
	FDSet operator-(const FDSet&) const;
	FDSet operator&(const FDSet&) const;
	FDSet operator|(const FDSet&) const;
	FDSet operator^(const FDSet&) const;
	bool operator==(const FDSet& a) const	{ return !bcmp(&fs,&a.fs,sizeof(fs)); }
	bool operator!=(const FDSet& a)	const	{ return !(*this==a); }
	void operator-=(const FDSet&);
	void operator&=(const FDSet&);
	void operator|=(const FDSet&);
	void operator^=(const FDSet&);
	void clr(int fd)	{ FD_CLR(fd,&fs); }
	bool includes(unsigned fd) const { return FD_ISSET(fd,&fs); }
	bool isSet(int fd) const	{ return FD_ISSET(fd,&fs); }
	void set(int fd)	{ FD_SET(fd,&fs); }
	void zero()		{ FD_ZERO(&fs); }
	virtual	unsigned capacity() const;  // returns descriptor table size
	virtual	int compare(const Object&) const;
	virtual	void deepenShallowCopy();   // {}
	virtual unsigned hash() const;
	virtual bool isEmpty() const;
	virtual bool isEqual(const Object&) const;
	virtual void printOn(ostream& strm =cout) const;
	virtual	unsigned size()	const;	    // returns # of fd bits set
	virtual const Class* species() const;
};

#endif
