INTERFACE SIO; (*Simple Input/Output 13.04.94. LB*)

(*SIO provides a Reader and a Writer type.
  A reader or a writer is a stream of typed data.
  Data can be read from a reader by "Get" procedures,
  and can be written by "Put" procedures onto a writer.
  The "Get" and "Put" procedures advance a (hidden) position
  over the reader resp. writer stream.
  The procedure "LookAhead" returns the next character,
  without advancing the reader position.

  All "Put" procedures flush automtically on stdout.
  GetText, GetInt, GetReal, GetBool terminate with any white space character,
  leading whitespaces are ignored. 
  (White spaces are: new line, tab, space, form feed and carriage return.)
  The terminating character is removed from the reader,
  and can be retrieved by the TermChar function.
  "Error" exception is raised only for readers, not connected to Stdio.stdin.
  For Stdio.stdin, the user is prompted to type in a new value.

  SIO provides some additional functions,
  such as positioning (seeking) in readers,
  length and end of readers, flushing of writers etc.
  Writers are strictly sequential, positioning is not supported.
  The default value of the reader or writer parameter is always NIL,
  with the effect, selecting the appropriate standard device.
  Standard reader is Stdio.stdin, which is normally the keyboard.
  Standard writer is Stdio.stdout, which is normally the screen.*)

  IMPORT Rd, Wr;

  EXCEPTION Error;

  TYPE
	 Reader = Rd.T;  (*A Reader is an Rd.T*)
	 Writer = Wr.T;  (*A Writer is a Wr.T*)

  (*--------  Basic procedures  --------*)  

  PROCEDURE GetChar(rd: Reader := NIL): CHAR
	 RAISES {Error};
  (*Returns the next char.*)

  PROCEDURE PutChar(ch: CHAR; wr: Writer := NIL);
  (*Outputs a single character.*)

  PROCEDURE GetText(rd: Reader := NIL): TEXT
	 RAISES {Error};
  (*Reads a text. The terminating character is not appended.*)

  PROCEDURE PutText(t: TEXT; wr: Writer := NIL);
  (*Outputs all characters in t.*)

  PROCEDURE GetInt(rd: Reader := NIL): INTEGER
	 RAISES {Error};
  (*Reads a decimal integer.*)

  PROCEDURE PutInt(i: INTEGER; length := 3; wr: Writer := NIL);
  (*Outputs an integer number. 
    The number is right-aligned in a field of length "length", i.e.
    leading blanks are output if the number of digits < "length".*)

  PROCEDURE GetReal(rd: Reader := NIL): REAL
	 RAISES {Error};
  (*Reads a real number.*)

  PROCEDURE PutReal(r: REAL; wr: Writer := NIL);
  (*Outputs a real number. *)

  PROCEDURE GetLongReal(rd: Reader := NIL): LONGREAL
	 RAISES {Error};
  (*Reads a longreal number.*)

  PROCEDURE PutLongReal(r: LONGREAL; wr: Writer := NIL);
  (*Outputs a longreal number. *)

  PROCEDURE GetBool(rd: Reader := NIL): BOOLEAN
	 RAISES {Error};
  (*Reads a Boolean constant, terminated by any character in "terminate".
	 Legal values are: TRUE, True, true, T, t resp. FALSE, False, false, F, f.*)
  (*!!! Currently accepts only T, t, F, f due to a bug in "Lex".*)

  PROCEDURE PutBool(b: BOOLEAN; wr: Writer := NIL);
  (*Outputs a Boolean value.*)

  (*--------  Additional procedures  --------*)  

  PROCEDURE LookAhead(rd: Reader := NIL): CHAR
	 RAISES {Error};
  (*Returns the next character, without removing it from the reader.*)

  PROCEDURE TermChar(rd: Reader := NIL): CHAR    
	 RAISES {Error};       
  (*Returns the last terminating character.*)

  PROCEDURE Nl(wr: Writer := NIL);
  (*Outputs a new line.*)

  PROCEDURE PutUnsigned(i: INTEGER; length := 6; base: [2..16] := 16;
                       wr: Writer := NIL);
  (*Outputs an unsigned number with given base.
    The number is right-aligned in a field of length "length".*)

  PROCEDURE End(rd: Reader := NIL): BOOLEAN;
  (*Returns TRUE iff end of reader reached.
	 On the keyborad CTRL-Z on the PC, and CTRL-D in Unix.*)

  PROCEDURE Flush(wr: Writer := NIL);
  (*Flushes the writer on the file. Not necessary for Stdio.stdout.*)

  PROCEDURE Available(rd: Reader := NIL): BOOLEAN;
  (*Returns TRUE iff some characters are available.*)
  (*!!! Currently doesn't work on the PC !!! - it blocks*)

  PROCEDURE Length(rd: Reader := NIL): CARDINAL;
  (*Returns the length of the reader.
	 Returns 0 if the length cannot be computed.*)
  (*!!! Currently doesn't work on the PC !!!*)

  PROCEDURE Seek(rd: Reader := NIL; position: CARDINAL := 0);
  (*Sets the reader on "position" if it is seekable.
	 Default corresponds to reset a reader.*)
  (*!!! Currently doesn't work on the PC !!!*)

END SIO.
