/*
 This file is part of GNU Taler
 (C) 2021 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AccessToken,
  Duration,
  LoginTokenScope,
  MerchantAuthMethod,
  succeedOrThrow,
  TalerMerchantManagementHttpClient,
} from "@gnu-taler/taler-util";
import {
  ExchangeService,
  GlobalTestState,
  harnessHttpLib,
  MerchantService,
  setupDb,
} from "../harness/harness.js";

/**
 * Do basic checks on instance management and authentication.
 */
export async function runMerchantInstancesUrlsTest(t: GlobalTestState) {
  const db = await setupDb(t);

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    httpPort: 8083,
    database: db.connStr,
  });

  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  const adminPw = "i-am-admin" as AccessToken;
  const clientForAdminInst = new TalerMerchantManagementHttpClient(
    merchant.makeInstanceBaseUrl(),
  );

  const clientForMyInst = new TalerMerchantManagementHttpClient(
    merchant.makeInstanceBaseUrl("myinst"),
  );

  succeedOrThrow(
    await clientForAdminInst.createInstance(adminPw, {
      id: "admin",
      address: {},
      use_stefan: true,
      default_pay_delay: Duration.toTalerProtocolDuration(
        Duration.fromSpec({ seconds: 60 }),
      ),
      default_wire_transfer_delay: Duration.toTalerProtocolDuration(
        Duration.fromSpec({ seconds: 60 }),
      ),
      jurisdiction: {},
      name: "My Admin Instance",
      auth: {
        method: MerchantAuthMethod.TOKEN,
        password: adminPw,
      },
    }),
  );

  const { access_token: adminToken } = succeedOrThrow(
    await clientForAdminInst.createAccessToken("admin", adminPw, {
      scope: LoginTokenScope.All,
    }),
  );

  succeedOrThrow(
    await clientForAdminInst.createInstance(adminToken, {
      id: "myinst",
      address: {},
      default_pay_delay: Duration.toTalerProtocolDuration(
        Duration.fromSpec({ seconds: 60 }),
      ),
      use_stefan: true,
      default_wire_transfer_delay: Duration.toTalerProtocolDuration(
        Duration.fromSpec({ seconds: 60 }),
      ),
      jurisdiction: {},
      name: "My Second Instance",
      auth: {
        method: MerchantAuthMethod.TOKEN,
        password: "i-am-myinst",
      },
    }),
  );

  const { access_token: myInstToken } = succeedOrThrow(
    await clientForMyInst.createAccessToken("myinst", "i-am-myinst", {
      scope: LoginTokenScope.All,
    }),
  );

  async function check(url: string, token: string, expectedStatus: number) {
    const resp = await harnessHttpLib.fetch(url, {
      headers: {
        Authorization: `Bearer ${token}`,
      },
    });
    console.log(
      `checking ${url}, expected ${expectedStatus}, got ${resp.status}`,
    );
    t.assertDeepEqual(resp.status, expectedStatus);
  }

  const adminInstBaseUrl = merchant.makeInstanceBaseUrl();

  await check(
    `${adminInstBaseUrl}private/instances/admin/instances/admin/config`,
    adminToken,
    404,
  );

  // Instance management is only available when accessing the admin instance
  // directly.
  await check(
    `${adminInstBaseUrl}instances/admin/private/instances`,
    "foo",
    404,
  );

  // Non-admin instances don't allow instance management.
  await check(`${adminInstBaseUrl}instances/foo/private/instances`, "foo", 404);
  await check(
    `${adminInstBaseUrl}instances/myinst/private/instances`,
    "foo",
    404,
  );

  await check(`${adminInstBaseUrl}config`, "foo", 200);
  await check(`${adminInstBaseUrl}instances/admin/config`, "foo", 200);
  await check(`${adminInstBaseUrl}instances/myinst/config`, "foo", 200);
  await check(`${adminInstBaseUrl}instances/foo/config`, "foo", 404);
  await check(
    `${adminInstBaseUrl}instances/admin/instances/config`,
    "foo",
    404,
  );

  await check(
    `${adminInstBaseUrl}private/instances/myinst/config`,
    adminToken,
    404,
  );

  await check(
    `${adminInstBaseUrl}instances/myinst/private/orders`,
    adminToken,
    401,
  );

  await check(
    `${adminInstBaseUrl}instances/myinst/private/orders`,
    adminToken,
    401,
  );

  await check(
    `${adminInstBaseUrl}instances/myinst/private/orders`,
    myInstToken,
    200,
  );

  await check(
    `${adminInstBaseUrl}private/instances/myinst/orders`,
    adminToken,
    404,
  );
}

runMerchantInstancesUrlsTest.suites = ["merchant"];
