/******************************************************************/
/* 		Copyright (c) 1989, Intel Corporation

   Intel hereby grants you permission to copy, modify, and 
   distribute this software and its documentation.  Intel grants
   this permission provided that the above copyright notice 
   appears in all copies and that both the copyright notice and
   this permission notice appear in supporting documentation.  In
   addition, Intel grants this permission provided that you
   prominently mark as not part of the original any modifications
   made to this software or documentation, and that the name of 
   Intel Corporation not be used in advertising or publicity 
   pertaining to distribution of the software or the documentation 
   without specific, written prior permission.  

   Intel Corporation does not warrant, guarantee or make any 
   representations regarding the use of, or the results of the use
   of, the software and documentation in terms of correctness, 
   accuracy, reliability, currentness, or otherwise; and you rely
   on the software, documentation and results solely at your own 
   risk.							  */
/******************************************************************/
#include <stdio.h>
#include "defines.h"
#include "globals.h"
#include "regs.h"
#include "version.h"

extern char *strchr();

#ifdef GDB
char gdb_debug = FALSE;	/* See 'd' command in description below */

int gdb_cmd();
static void bintohex();
static void hextobin();
static int fromhex();
static int getpkt();
static int pkt_out();
static int putbinpkt();
static int putpkt();
static int recvmem();
static int recvregs();
static int sendmem();
static int sendregs();
static unsigned int swap_bytes();
static int update_reg();

/* tohex[N] is the ascii for hex digit N */
static char tohex[] = "0123456789abcdef";

/******************************************************************
 * This module implements the commands that can be sent 
 * by a remote host running GDB.
 *
 * All GDB commands, and responses to them, are transferred between
 * nindy and the host in messages of the following format:
 *
 *		<info>#<checksum>
 *
 * where 
 *	#	is a literal character
 *
 *	<info>	is binary data transferred as a series of ASCII
 *		hex digits ('0'-'9' and lowercase 'a'-'f').
 *
 *	<checksum>
 *		is a pair of ASCII hex digits representing
 *		an 8-bit checksum formed by adding together
 *		each of the characters in <info>.
 *
 * The receiver of a message always sends a single 
 * character to the sender to indicate that the checksum 
 * was good ('+') or bad ('-');  the sender re-transmits
 * the entire message over until a '+' is received.
 *
 * GDB commands are indicated by a '^P' prefix.  NINDY
 * always responds by sending back requested data or an 
 * error return of the form "Xnn":
 *
 *	X00 - No errors
 *	X01 - Buffer overflow
 *	X02 - Unknown command
 *	X03 - Wrong amount of data to load register(s)
 *	X04 - Missing command argument(s)
 *	X05 - Odd number of digits sent to load memory
 *	X06 - Unknown register name
 *
 * The legal commands are:
 *
 * c<addr>
 *	Resume user program execution at address <addr>.
 *	If <addr> is omitted, resumes at address in the ip.
 *
 * d	Turn on GDB debug facilities: suppress checksum 
 *	checking (to allow easy entry of GDB commands from
 * 	keyboard);  prompt for command input; allow verbose output.
 *
 * D	Download a COFF executable file.
 *
 * E	Exit GDB mode.
 *
 * m<addr>,<len>
 *	Send contents of <len> bytes of memory starting at 
 *	memory location <addr>, or "Xnn" if there's an error.
 *
 * M<addr>,<data>
 *	Updates memory, starting at memory location <addr>, 
 *	with the hex <data>.
 *
 *	Respond "X00" for success, "Xnn" for error.
 *
 * r	send contents of ALL registers, in the following order:
 *		pfp  sp   rip  r3   r4   r5   r6   r7 
 *		r8   r9   r10  r11  r12  r13  r14  r15 
 *		g0   g1   g2   g3   g4   g5   g6   g7 
 *		g8   g9   g10  g11  g12  g13  g14  fp 
 *		pc   ac   ip   tc   fp0  fp1  fp2  fp3
 *
 *	Floating point registers are converted to, and
 *	transferred as, 8-byte values.  All others are 4-byte.
 *	Bytes for each register are sent from low-order to high-order.
 *
 *	Respond "Xnn" if there's an error ("nn" is error 
 *	number: 2 hex digits).
 *
 * R<regs>
 *	Update the user registers with <regs> (same format
 * 	same as 'r' command).
 *
 *	Respond "Xnn" ("nn" is error number: 2 hex digits, "X00" if OK).
 *
 * s<addr>
 *	Single-step user program at address <addr>.
 *	If <addr> is omitted, resumes at address in the ip.
 *
 * U<regname>,<value>
 *	Update the register whose name is <regname> with the hex
 *	value <value>.
 *
 *	<regname> can be any of the names list under the 'r' command.
 *
 *	<value> must be 16 hex digits for floating point registers,
 *	8 for other registers.  Must be sent low-order byte first.
 *
 *	Respond "Xnn" ("nn" is error number: 2 hex digits, "X00" if OK).
 *
 * v	Send version number, as an ascii string in the form "x.xx".
 *
 * X	Reset the processor, suppressing powerup greeting message.
 *
 * ?	Send status with which user program halted.
 *	Format is "xxyyiiiiiiiiffffffffssssssss" (hex digits) where
 *	    xx       is non-zero if user program exited, 00 otherwise
 *	    yy       is return code (if program exited) or fault code
 *			(See fault.c for legal values).
 *	    iiiiiiii is the ip (address of the next instruction to execute,
 *			if program didn't exit).
 *	    ssssssss is the sp
 *	    ffffffff is the fp
 *
 ******************************************************************/


/* WARNING:
 *	If these change or are added to, update the gdb960 source
 *	file "remote.c".
 */
static char X_OK[]	= "X00"; /* No errors				*/
static char X_BUFFER[]	= "X01"; /* Buffer overflow			*/
static char X_CMD[]	= "X02"; /* Unknown command			*/
static char X_DATLEN[]	= "X03"; /* Wrong amount of data to load register(s) */
static char X_ARG[]	= "X04"; /* Missing command argument(s)		*/
static char X_MEMLD[]	= "X05"; /* Odd number of digits sent to load memory */
static char X_REGNAME[]	= "X06"; /* Unknown register name		*/


/********************************************************
 * 'gdb_cmd' -- process a command from GDB
 *
 * Assume the ^P prefix of a GDB command has been read.  
 * Receive the rest of the command from the host and 
 * process it.
 *
 * Return TRUE if the command is processed.  Return FALSE 
 * if the caller should process the command (which will 
 * have been left in the globals 'cmd' and 'linebuff').
 *
 ********************************************************/
int
gdb_cmd()
{
	char args[ (2*36*REG_SIZE) + (2*4*FP_REG_SIZE) + 30 ];
		/* Command arguments go here; 'args[0]' is command name.
		 * The 'R' command should be the most data we get, so
		 * I've allocated enough to hold its args, plus a little
		 */
	int done;	/* FALSE if we still have to get a gdb command in */
	extern char end;
		/* This is the linker-generated address of the first location
		 * after bss.  Set the byte there to 0x55 on a reset if we want
		 * NINDY to come up again in gdb mode.  (Allows gdb to reset us
		 * without receiving extraneous verbiage it can't handle.)
		 */
	unsigned int ip;


	for ( done = FALSE; !done; ){

		/* Read rest of command from host. */
		if ( !getpkt(args,sizeof(args)) ){
			/* We didn't get a good command */
			return TRUE;
		}

		switch ( args[0] ){
		case 'c':
		case 's':
			/* Command name should be followed by address at which
			 * user program should resume.  Reformat the command
			 * into the normal 'step' or 'go' command as it would
			 * be entered from the keyboard, and let the monitor
			 * handle it as such.
			 */
			strcpy( cmd, (args[0] == 's') ? "st" : "go" );

			/* The following would be simpler with a sprintf,
			 * but sprintf causes too much stuff to get linked in
			 * from libc
			 */
			strcpy( linebuff, cmd );
			strcat( linebuff, " ");
			if ( args[1] != 0 ){
				strcat( linebuff, &args[1] );
			} else {
				/* If no address specified, use value in ip.
				 * Swap bytes first because bintohex works
				 * from an address in memory, the memory image
				 * of a register is stored low-order byte first.
				 */
				ip = swap_bytes( register_set[REG_IP] );
				bintohex( 4, &ip, linebuff+3 );
				linebuff[11] = '\0';
			}
			return FALSE;
		case 'd':
			gdb_debug = TRUE;
			done = TRUE;
			break;
		case 'D':
			/* Reformat the command into the normal 'do' command as
			 * it would be entered from the keyboard, and let the
			 * monitor handle it as such.
			 */
			strcpy( cmd, "do" );
			strcpy( linebuff, cmd );
			return FALSE;
		case 'E':
			gdb = FALSE;
			done = TRUE;
			break;
		case 'm':
			done = sendmem( &args[1] );
			break;
		case 'M':
			done = recvmem( &args[1] );
			break;
		case 'r':
			done = sendregs();
			break;
		case 'R':
			done = recvregs( &args[1] );
			break;
		case 'U':
			done = update_reg( &args[1] );
			break;
		case 'v':
			done = putpkt( VERSION );
			break;
		case 'X':
			/* Reset -- set a magic flag so we come up again in GDB
			 * mode (see comment on declaration of 'end', above).
			 * Kill time to allow the last ACK to get out of the
			 * UART: if we reset too soon it gets lost and the
			 * host gets out of sync.
			 */
			end = 0x55;
			eat_time(1000);
			reset();	/* No return */
			break;
		case '?':
			/* Send reason why user program last stopped, as well
			 * as values of ip, fp, sp with which it stopped.
			 */
			bintohex( 1, &stop_exit, args );
			bintohex( 1, &stop_code, args+2 );
			bintohex( 4, &register_set[REG_IP], args+4 );
			bintohex( 4, &register_set[REG_FP], args+12 );
			bintohex( 4, &register_set[REG_SP], args+20 );
			args[28] = '\0';
			done = putpkt( args );
			break;
		default:
			done = putpkt( X_CMD );
			break;
		}
	}
	return TRUE;
}


/********************************************************
 * recvregs()
 *	Update our copy of the user registers with the 
 * 	hex values in the passed buffer.  Format of info
 *	in the buffer is the same as we send in the 
 *	sendregs routine.
 *
 *	Return TRUE if we finished execution; FALSE if
 *	we gave up because we received the start of a
 *	new GDB command from the host, and need to 
 *	re-sync with it.
 *
 * WARNING:
 *	Assumes order of registers in 'register_set' array is:
 *		r0-r15, g0-g15, pc,  ac,  ip,  tc
 ********************************************************/
static int
recvregs( buf )
    char *buf;	/* New user register contents */
{
	if ( strlen(buf) != (2*36*REG_SIZE + 2*4*FP_REG_SIZE) ){
		/* Incorrect amount of data in buffer */
		return putpkt( X_DATLEN );
	}

	/* Pick up local, global, and control registers */
	hextobin( 36*REG_SIZE, buf, register_set );

	/* Pick up fp registers */
	hextobin( 4*FP_REG_SIZE, &buf[2*36*REG_SIZE], fp_register_set );

	return putpkt( X_OK );
}


/********************************************************
 * update_reg()
 *	Update our copy of the user registers for the
 * 	single register whose name is in the argument
 *	buffer.
 *
 *	Return TRUE if we finished execution; FALSE if
 *	we gave up because we received the start of a
 *	new GDB command from the host, and need to 
 *	re-sync with it.
 *
 ********************************************************/
static int
update_reg( buf )
    char *buf;	/* Should contain "<regname>,<value>"	*/
{
	char *value;	/* Start of <value> in buffer			*/
	char *dest;	/* Where to put binary equivalent of <value>	*/
	int regnum;	/* Number of register whose name is <regname>	*/
	int regsize;	/* Start of <value> in buffer			*/

	value = strchr( buf, ',' );
	if ( value == NULL ){
		return putpkt( X_ARG );	/* Missing argument */
	}
	*value++ = '\0';

	if ( (regnum = get_regnum(buf)) == ERROR ){
		return putpkt( X_REGNAME );	/* Bad register name */
	} else if ( regnum < NUM_REGS ){
		regsize = 4;
		dest = (char *) &register_set[regnum];
	} else {
		/* FLOATING POINT REGISTER */
		regsize = 8;
		dest = (char *) &fp_register_set[regnum-NUM_REGS];
	}

	if ( strlen(value) != 2*regsize ){
		/* Incorrect amount of data in buffer */
		return putpkt( X_DATLEN );
	}

	hextobin( regsize, value, dest );

	return putpkt( X_OK );
}


/********************************************************
 * sendregs()
 *	Send contents of ALL registers.
 *
 *	Return TRUE if we finished execution; FALSE if 
 *	we gave up because we received the start of a
 *	new GDB command from the host, and need to 
 *	re-sync with it.
 *
 * WARNING:
 *	Assumes order of registers in 'register_set' array is:
 *		r0-r15, g0-g15, pc,  ac,  ip,  tc
 ********************************************************/
static int
sendregs()
{
	char buf[ (2*(36*REG_SIZE+4*FP_REG_SIZE)) + 1 ];
		/* Enough storage for all registers (at 2 ASCII characters
		 * per byte of binary data) plus string terminator.
		 */


	/* Local, global, and control registers */
	bintohex( 36*REG_SIZE, register_set, &buf[0] );

	/* Then fp registers */
	bintohex( 4*FP_REG_SIZE, fp_register_set, &buf[2*36*REG_SIZE] );

	/* Terminate string and send to host */
	buf[ (2*36*REG_SIZE) + (2*4*FP_REG_SIZE) ] = '\0';
	return putpkt( buf );
}


/********************************************************
 * 'recvmem()' 
 *
 *	Buffer should contain request of the form: 
 *		"<addr>,<len>,<data>".
 *	Update <len> bytes of memory, starting at memory
 *	location <addr>, with the hex <data>.
 *
 *	Return TRUE if we finished execution; FALSE if 
 *	we gave up because we received the start of a 
 *	new GDB command from the host, and need to 
 *	re-sync with it.
 ********************************************************/

static int
recvmem( buf )
    char *buf;	/* Command arguments */
{
	long addr;	/* Beginning location in memory to update */
	int len;	/* Number of digits in buf		*/

	/* Read starting address out of command buffer */
	addr = 0;
	for ( ; *buf != ','; buf++ ){
		if ( *buf == '\0' ){
			/* Missing command arguments */
			return putpkt( X_ARG );
		}
		addr = (addr << 4) + fromhex(*buf);
	}
	buf++;	/* Skip comma */

	/* There better be an even number of characters left,
	 * 2 hex digits per byte of memory.
	 */
	len = strlen(buf);
	if ( len & 1 ){
		return putpkt( X_MEMLD );
	}

	/* Patch memory */
	hextobin( len/2, buf, (char*)addr );
	return putpkt( X_OK );
}


/********************************************************
 * 'sendmem' routine
 *
 *	Buffer should contain a request of the form 
 *		"<addr>,<len>".
 *	Send contents of <len> bytes of memory starting
 *	at memory location <addr>, or "Xnn" if there's
 *	an error.
 *
 *	Return TRUE if we finished execution; FALSE if 
 *	we gave up because we received the start of a 
 *	new GDB command from the host, and need to 
 *	re-sync with it.
 ********************************************************/
static int
sendmem( buf )
    char *buf;
{
	long addr;		/* Address of memory to be sent	*/
	int len;		/* # of binary bytes to be sent	*/

	/* Pick starting address out of command buffer */
	addr = 0;
	for ( ; *buf != ','; buf++ ){
		if ( *buf == '\0' ){
			/* Missing command arguments */
			return putpkt( X_ARG );
		}
		addr = (addr << 4) + fromhex(*buf);
	}
	buf++;		/* Advance past comma */

	/* Pick length out of command buffer */
	len = 0;
	for ( ; *buf != '\0'; buf++ ){
		len = (len << 4) + fromhex(*buf);
	}

	return putbinpkt( addr, len );
}


/********************************************************
 * putpkt -- checksum ascii message and send to host
 *
 *	Send message over and over to host until positive
 *	acknowledgment received.  If we get a DLE instead
 *	of a ack, we somehow got out of sync with host --
 *	give up and let the caller figure out how to handle
 *	things.
 *
 *	Return TRUE if positive ack received, FALSE otherwise
 *
 *	There are 2 entry points:
 *	    putpkt(): accepts a string of ascii data.
 *	    putbinpkt():  accepts a string of binary data,
 *			converting it to ascii as it goes.
 *				
 *	Actual work in both cases is done by the 'pkt_out()'
 *	subroutine.
 ********************************************************/
static int
putbinpkt( buf, len )
    char *buf;	/* Packet of ascii info to be sent */
    int len;	/* Number of bytes of info in 'buf' */
{
	return pkt_out( buf, len, TRUE );
}

static int
putpkt( buf )
    char *buf;	/* '\0'-terminated ascii string to be sent */
{
	return pkt_out( buf, strlen(buf), FALSE );
}

static int
pkt_out( buf, len, binary )
    char *buf;	/* Packet of info to be sent */
    int len;	/* Number of bytes of info in buffer */
    int binary;	/* TRUE => buffer contains binary data.
		 * FALSE => buffer contains ascii data.
		 */
{
	char csum;	/* Checksum			*/
	char ack;	/* Ack character from host	*/
	int i;


	/* Send message over and over until we get a positive ack */

	do {
		csum = 0;

		if ( binary ){
			for ( i = 0; i < len; i++ ){
				out_hex( buf[i], BYTE, TRUE );
				csum += tohex[ (buf[i] >> 4) & 0xf ];
				csum += tohex[ buf[i] & 0xf ];
			}
		} else {
			for ( i = 0; i < len; i++ ){
				csum += buf[i];	/* Add character to checksum */
				co( buf[i] );	/* Send character to host */
			}
		}

		/* Send checksum */
		co( '#' );
		out_hex( csum, BYTE, TRUE );

		/* Get ACK from host, unless debugging the gdb commands */
		if ( gdb_debug ){
			return TRUE;
		}
		ack = ci();

	} while ( (ack != '+') && (ack != DLE) );

	return (ack == '+' ? TRUE : FALSE );
}


/********************************************************
 * getpkt -- get ascii message from host;  
 *	     validate and strip off checksum
 *
 *	Restart (to resync with host) if a DLE is 
 *	received at any time.  Return TRUE if successful,
 *	FALSE if buffer overflows
 *
 * NOTE!:  assumes the leading DLE of a GDB message has
 * already been read in.
 ********************************************************/
static int
getpkt( buf, buflen )
    char *buf;		/* Where to put command */
    int buflen;		/* Number of bytes in buffer 'buf' */
{
	int c, c1, c2;
	int count;
	int csum;


	/* Repeat until checksum ok */
	while ( 1 ){
		csum = DLE;		/* assume DLE already read */

		c = count = 0;
		while ( c != '#' ){
			switch ( c = ci() ){
			case DLE:
				count = 0;
				csum = DLE;
				break;
			case '#':
				/* Read checksum digits */
				if ( ((c1 = ci()) == DLE)
				||   ((c2 = ci()) == DLE) ){
					c = DLE;
					count = 0;
					csum = DLE;
				}
				break;
			default:
				if ( count < buflen-1 ){
					buf[count++] = c;
					csum += c;
				}
				break;
			}
		}

		if ( count >= buflen-1 ){
			/* Buffer Overflow */
			putpkt( X_BUFFER );
			return FALSE;
		}
		buf[count] = 0;

	/* Validate checksum (don't bother if debugging gdb mode) */
		if ( gdb_debug
		||   ((csum & 0xff) == (fromhex(c1) << 4) + fromhex(c2)) ){
			break;	/* All done */
		}

	/* Bad checksum: send NAK and wait for start of next command */
		co( '-' );
		while ( ci() != DLE ){
			;
		}
	}
	co( '+' );
	return TRUE;
}

/********************************************************
 * fromhex -- Convert ASCII hex digit to binary 
 *
 ********************************************************/
static int
fromhex(a)
     int a;
{
	return (a >= '0' && a <= '9') ? a - '0' : a - 'a' + 10;
}

/********************************************************
 * hextobin -- Convert a string of ASCII hex digits to a
 *		string of binary bytes
 *
 ********************************************************/
static void
hextobin( n, hexp, binp )
    int n;		/* Number of bytes to convert (twice this many digits)*/
    char *hexp;		/* Get hex from here		*/
    char *binp;		/* Put binary here		*/
{
	while ( n-- ){
		*binp++ = (fromhex(*hexp) << 4) + fromhex(*(hexp+1));
		hexp += 2;
	}
}


/********************************************************
 * bintohex -- Convert a string of binary bytes to a
 *	 	string of ASCII hex digits 
 *
 ********************************************************/
static void
bintohex( n, binp, hexp )
    int n;		/* Number of bytes to convert	*/
    char *binp;		/* Get binary from here		*/
    char *hexp;		/* Place hex here		*/
{
	while ( n-- ){
		*hexp++ = tohex[ (*binp >> 4) & 0xf ];
		*hexp++ = tohex[ *binp & 0xf ];
		binp++;
	}
}


/********************************************************
 * swap_bytes -- Reverse the byte-ordering of an integer.
 *
 ********************************************************/
static unsigned int
swap_bytes( in )
    unsigned int in;
{
	int i;
	unsigned int out;

	out = 0;
	for ( i = 0; i < sizeof(in); i++ ){
		out <<= 8;
		out |= in & 0xff;
		in >>= 8;
	}
	return out;
}

#endif
