;;;
;;; maclib.scm -- A object-oriented library of Macintosh toolbox functions
;;;		  in SCHEME
;;;
;;; Part of MacSCM 1.0.
;;;
;;; Author: Kevin Scott Kunzelman (kkunzelm@cam.cornell.edu)
;;;
;;; This code is in the public domain.	You can copy it, sell it, modify it,
;;; all without any restrictions.
;;;
;;; An object-oriented style of programming is used in the following file.
;;; An object is created by spawning an environment in which to hold
;;; instance variables (usually called "my-<whatever>").  A
;;; message-dispatching instance variable is then created (usually called
;;; "self").  This message-dispatcher is then returned to the caller.
;;; Because SCHEME has environment retention, the environment that was
;;; spawned can still be accessed by the dispatcher, but not by any other
;;; exterior function.	This method then provides a good way to protect
;;; instance variables as well as pass messages to objects. The key thing
;;; to remember about this method is that after objects are created, one is
;;; not dealing with the objects per se, but with their message
;;; dispatchers. However, once this idea is understood, the methodology is
;;; fairly straightforward.
;;;

(define cons*
  (letrec ((cons*-iter
	    (lambda (x)
	      (cond ((null? x) '())
		    ((null? (cdr x)) (car x))
		    (else (cons (car x) (cons*-iter (cdr x))))))))
    (lambda (. x) (cons*-iter x))))

(define null-object (lambda (message . data) #f))

;;;
;;; widg-ctrl-new
;;;
;;; This object constructor creates a control widget object.
;;;
;;; parent    (obj)	-- the parent object
;;; rect      (rect)	-- the boundary rectangle of the control
;;; title     (string)	-- the title of the control
;;; visible   (boolean) -- whether the control is initially visible
;;; value     (integer) -- the initial value (such as position of scroll-bar)
;;; min-value (integer) -- the minimum value (such as left position of scroll-bar)
;;; max-value (integer) -- the maximum value (such as right position of scroll-bar)
;;; proc      (integer) -- the procedure ID of the control (SEE mac.scm)
;;;

(define (widg-ctrl-new parent rect title visible value min-value max-value proc)
  (letrec
    ((my-ctrl
      (mac:ctrl-new
       (parent 'get-wind) rect title visible value min-value max-value proc))
     (self
      (lambda (message . data)
	(case message
	  ((get-wind) (parent 'get-wind))
	  ((idle)  #f)

	  ((size)      (apply mac:ctrl-size	 (cons my-ctrl data)))
	  ((move)      (apply mac:ctrl-move	 (cons my-ctrl data)))
	  ((hide)      (apply mac:ctrl-hide	 (cons my-ctrl data)))
	  ((show)      (apply mac:ctrl-show	 (cons my-ctrl data)))
	  ((update)    (apply mac:ctrl-draw	 (cons my-ctrl data)))
	  ((dispose)   (apply mac:ctrl-dispose	 (cons my-ctrl data)))
	  ((set-min)   (apply mac:ctrl-set-min	 (cons my-ctrl data)))
	  ((set-max)   (apply mac:ctrl-set-max	 (cons my-ctrl data)))
	  ((set-value) (apply mac:ctrl-set-value (cons my-ctrl data)))
	  ((set-title) (apply mac:ctrl-set-title (cons my-ctrl data)))
	  ((get-min)   (apply mac:ctrl-get-min	 (cons my-ctrl data)))
	  ((get-max)   (apply mac:ctrl-get-max	 (cons my-ctrl data)))
	  ((get-value) (apply mac:ctrl-get-value (cons my-ctrl data)))
	  ((get-title) (apply mac:ctrl-get-title (cons my-ctrl data)))
	  ((evt)
	   (let*
	     ((the-evt (car data))
	      (evt-type (mac:evt-get-what the-evt))
	      (the-pt (mac:global-to-local (mac:evt-get-where the-evt))))
	     (case evt-type
	       ((#.mouseDown)
		(if (< (mac:wind-find-ctrlp (self 'get-wind) the-pt) 128)
		    (begin
		     (set! mac:ctrl-action-proc
			   (lambda (the-ctrl the-part)
			     (case the-part
			       ((#.inButton)	 (parent 'in-button))
			       ((#.inCheckBox)	 (parent 'in-check-box))
			       ((#.inUpButton)	 (parent 'scroll-up))
			       ((#.inDownButton) (parent 'scroll-down))
			       ((#.inPageUp)	 (parent 'page-up))
			       ((#.inPageDown)	 (parent 'page-down)))))
		     (mac:ctrl-track-2 my-ctrl the-pt))
		    (begin
		     (mac:ctrl-track my-ctrl the-pt)
		     (parent 'scroll-to))))
	       (else (error "Invalid event to control widget" data))
	       )))
	  (else	 (error "Invalid message to control widget" message))
	  ))))
    self))


;;;
;;; The following are some definitions for specific control widgets
;;;

(define (widg-scroll-bar-new parent rect visible value min-value max-value)
  (widg-ctrl-new parent rect "" visible value min-value max-value scrollBarProc))

(define (widg-button-new parent rect title visible)
  (widg-ctrl-new parent rect title visible 0 0 0 pushButProc))

(define (widg-check-box-new parent rect title visible)
  (widg-ctrl-new parent rect title visible 0 0 1 checkBoxProc))

(define (widg-radio-button-new parent rect title visible)
  (widg-ctrl-new parent rect title visible 0 0 1 radioButProc))

(define (widg-pop-up-menu-new parent rect title visible)
  (widg-ctrl-new parent rect title visible 0 0 1 popupMenuProc))

;;;
;;; widg-te-new
;;;
;;; This object constructor creates a text-edit widget object.
;;;
;;; parent  (obj)   -- the parent object
;;; rect    (rect)  -- the boundary rectangle of the text-edit
;;;

(define (widg-te-new parent rect)
  (letrec
    ((my-te #f)
     (self
      (lambda (message . data)
	(case message
	  ((get-wind)	  (parent 'get-wind))

	  ((set-dest-rect)    (apply mac:te-set-dest-rect    (cons my-te data)))
	  ((set-view-rect)    (apply mac:te-set-view-rect    (cons my-te data)))

	  ((set-text)	      (apply mac:te-set-text	     (cons my-te data)))
	  ((set-just)	      (apply mac:te-set-justify	     (cons my-te data)))
	  ((set-select)	      (apply mac:te-set-select	     (cons my-te data)))

	  ((set-font)	      (apply mac:te-set-font	     (cons my-te data)))
	  ((set-face)	      (apply mac:te-set-face	     (cons my-te data)))
	  ((set-mode)	      (apply mac:te-set-mode	     (cons my-te data)))
	  ((set-size)	      (apply mac:te-set-size	     (cons my-te data)))
	  ((set-line-height)  (apply mac:te-set-line-height  (cons my-te data)))
	  ((set-font-ascent)  (apply mac:te-set-font-ascent  (cons my-te data)))

	  ((get-dest-rect)    (apply mac:te-get-dest-rect    (cons my-te data)))
	  ((get-view-rect)    (apply mac:te-get-view-rect    (cons my-te data)))

	  ((get-text)	      (apply mac:te-get-text	     (cons my-te data)))
	  ((get-just)	      (apply mac:te-get-just	     (cons my-te data)))
	  ((get-select-pt)    (apply mac:te-get-select-pt    (cons my-te data)))
	  ((get-select-start) (apply mac:te-get-select-start (cons my-te data)))
	  ((get-select-end)   (apply mac:te-get-select-end   (cons my-te data)))

	  ((get-font)	      (apply mac:te-get-font	     (cons my-te data)))
	  ((get-face)	      (apply mac:te-get-face	     (cons my-te data)))
	  ((get-mode)	      (apply mac:te-get-mode	     (cons my-te data)))
	  ((get-size)	      (apply mac:te-get-size	     (cons my-te data)))
	  ((get-line-height)  (apply mac:te-get-line-height  (cons my-te data)))
	  ((get-font-ascent)  (apply mac:te-get-font-ascent  (cons my-te data)))

	  ((get-lines)	      (apply mac:te-get-lines	     (cons my-te data)))

	  ((find-pt)	      (apply mac:te-find-pt	     (cons my-te data)))
	  ((find-offset)      (apply mac:te-find-offset	     (cons my-te data)))

	  ((idle)	      (apply mac:te-idle	     (cons my-te data)))
	  ((activate)	      (apply mac:te-activate	     (cons my-te data)))
	  ((deactivate)	      (apply mac:te-deactivate	     (cons my-te data)))
	  ((update)	      (apply mac:te-update (list my-te (self 'get-view-rect))))
	  ((scroll)	      (apply mac:te-scroll	     (cons my-te data)))

	  ((cut)	      (apply mac:te-cut		     (cons my-te data)))
	  ((copy)	      (apply mac:te-copy	     (cons my-te data)))
	  ((paste)	      (apply mac:te-paste	     (cons my-te data)))
	  ((delete)	      (apply mac:te-delete	     (cons my-te data)))

	  ((dispose)	      (apply mac:te-dispose	     (cons my-te data)))
	  ((insert)	      (apply mac:te-insert	     (cons my-te data)))
	  ((evt)
	   (let*
	     ((the-evt (car data))
	      (evt-type (mac:evt-get-what the-evt))
	      (shift-down (mac:mods-get-shift (mac:evt-get-mods the-evt)))
	      (the-message (mac:evt-get-msg-int the-evt))
	      (the-pt (mac:global-to-local (mac:evt-get-where the-evt))))
	     (case evt-type
	       ((#.mouseDown)
		(mac:te-click my-te the-pt shift-down))

	       ((#.keyDown #.autoKey)
		(mac:te-key my-te the-message))

	       (else (error "Invalid event to te widget" data))
	       )))

	  (else (error "Invalid message to te widget" message))
	  ))))

    (let ((old-wind (mac:get-port)))
      (mac:set-port (parent 'get-wind))
      (set! my-te (mac:te-new rect rect))
      (mac:set-port old-wind))
    self))


;;;
;;; widg-te-scroll-new
;;;
;;; This object constructor creates a text-edit widget object with scroll-bar.
;;;
;;; parent  (obj)   -- the parent object
;;; rect    (rect)  -- the boundary rectangle of the text-edit
;;;

(define (widg-te-scroll-new parent rect)
  (letrec
    ((scroll-rect #f)
     (frame-rect #f)
     (text-rect #f)
     (my-te #f)
     (my-scroll-bar #f)
     (self
      (lambda (message . data)
	(case message
	  ((get-wind) (parent 'get-wind))

	  ((update)
	   (mac:rect-frame frame-rect)
	   (apply my-te (cons message data))
	   (apply my-scroll-bar (cons message data)))

	  ((dispose)
	   (apply my-te (cons message data))
	   (apply my-scroll-bar (cons message data)))

	  ((scroll) #f)

	  ((scroll-up)
	   (my-scroll-bar 'set-value
			  (max (my-scroll-bar 'get-min)
			       (- (my-scroll-bar 'get-value) 1)))
	   (self 'scroll-to))

	  ((scroll-down)
	   (my-scroll-bar 'set-value
			  (min (my-scroll-bar 'get-max)
			       (+ (my-scroll-bar 'get-value) 1)))
	   (self 'scroll-to))

	  ((page-up)
	   (my-scroll-bar 'set-value
			  (min (my-scroll-bar 'get-max)
			       (- (my-scroll-bar 'get-value)
				  (quotient
				   (- (mac:rect-get-bottom (my-te 'get-view-rect))
				      (mac:rect-get-top	   (my-te 'get-view-rect)))
				   (my-te 'get-line-height)))))
	   (self 'scroll-to))

	  ((page-down)
	   (my-scroll-bar 'set-value
			  (min (my-scroll-bar 'get-max)
			       (+ (my-scroll-bar 'get-value)
				  (quotient
				   (- (mac:rect-get-bottom (my-te 'get-view-rect))
				      (mac:rect-get-top	   (my-te 'get-view-rect)))
				   (my-te 'get-line-height)))))
	   (self 'scroll-to))

	  ((scroll-to)
	   (let ((view-rect (my-te 'get-view-rect))
		 (offset (* (my-scroll-bar 'get-value)
			    (my-te 'get-line-height))))
	     (my-te 'set-dest-rect
		    (mac:rect-new (mac:rect-get-left view-rect)
				  (- (mac:rect-get-top view-rect) offset)
				  (mac:rect-get-right view-rect)
				  (- (mac:rect-get-bottom view-rect) offset)))
	     (mac:rect-erase view-rect)
	     (my-te 'update view-rect)))

	  ((evt)
	   (let*
	     ((the-evt (car data))
	      (evt-type (mac:evt-get-what the-evt))
	      (pt (mac:global-to-local (mac:evt-get-where the-evt))))
	     (case evt-type
	       ((#.mouseDown)
		(cond ((mac:pt-in-rect? pt scroll-rect)
		       (apply my-scroll-bar (cons message data)))

		      (else (apply my-te (cons message data)))))

	       ((#.keyDown #.autoKey)
		(apply my-te (cons message data)) ; relay message
		(my-scroll-bar 'set-max	  (-1+ (my-te 'get-lines)))
		(my-scroll-bar 'set-value
			       (quotient
				(- (mac:rect-get-top (my-te 'get-view-rect))
				   (mac:rect-get-top (my-te 'get-dest-rect)))
				(my-te 'get-line-height)))
		(let*
		  ((offset (my-te 'get-select-end))
		   (pt (my-te 'find-pt offset))
		   (v (mac:pt-get-v pt))
		   (top (mac:rect-get-top text-rect))
		   (bottom (mac:rect-get-bottom text-rect))
		   (tippy-top (+ top (my-te 'get-line-height))))
		  (cond ((<= v tippy-top)
			 (my-te 'scroll 0 (- tippy-top v)))
			((> v bottom)
			 (my-te 'scroll 0 (- bottom v)))))
		))))

	  ((idle activate deactivate scroll
	    cut copy paste delete insert
	    set-text set-just set-select
	    set-font set-face set-mode set-size set-line-height set-font-ascent
	    get-text get-just get-select-start get-select-end get-select-pt
	    get-font get-face get-mode get-size get-line-height get-font-ascent
	    get-lines get-length
	    find-pt find-offset)
	   (apply my-te (cons message data)))

	  (else (error "Invalid message to te-scroll widget" message))
	  ))))

    (set! scroll-rect (mac:rect-new (- (mac:rect-get-right rect) 16)
				    (mac:rect-get-top rect)
				    (mac:rect-get-right rect)
				    (mac:rect-get-bottom rect)))
    (set! frame-rect  (mac:rect-new (mac:rect-get-left rect)
				    (mac:rect-get-top rect)
				    (- (mac:rect-get-right rect) 15)
				    (mac:rect-get-bottom rect)))
    (set! text-rect   (mac:rect-inset frame-rect 2 2))
    (let
      ((old-wind (mac:get-port)))
      (mac:set-port (parent 'get-wind))
      (set! my-te (widg-te-new self text-rect))
      (set! my-scroll-bar (widg-scroll-bar-new self scroll-rect #t 0 0 0))
      (mac:rect-frame frame-rect)
      (mac:set-port old-wind))
    self))

;;;
;;; wind-new
;;;
;;; This object constructor creates a wind object.
;;;
;;; rect     (rect)    -- the boundary rectangle of the window.
;;; title    (string)  -- the title of the window.
;;; visible  (boolean) -- whether the window is initially visible.
;;; proc     (int)     -- the window proc ID.
;;; go-away  (boolean) -- whether the window has a close box.
;;;

(define (wind-new rect title visible proc go-away)
  (letrec
    ((my-wind #f)
     (my-menu-bar #f)
     (my-widg #f)
     (self
      (lambda (message . data)
	(case message
	  ((get-wind)	 my-wind)
	  ((get-widg)	 my-widg)
	  ((set-widg)	 (set! my-widg (car data)))
	  ((do-menu-select) (my-menu-bar 'choice (menu-bar-select (car data))))
	  ((do-menu-key) (my-menu-bar 'choice (menu-bar-key (car data))))
	  ((get-menu-bar)   my-menu-bar)
	  ((set-menu-bar)   (set! my-menu-bar (car data)))

	  ((idle)	 (apply my-widg		   (cons message data)))
	  ((size)	 (apply mac:wind-size	   (cons my-wind data)))
	  ((move)	 (apply mac:wind-move	   (cons my-wind data)))
	  ((front)	 (apply mac:bring-to-front (cons my-wind data)))
	  ((behind)	 (apply mac:send-behind	   (cons my-wind data)))
	  ((hilite)	 (apply mac:wind-hilite	   (cons my-wind data)))
	  ((zoom)	 (apply mac:wind-zoom	   (cons my-wind data))
			 (self 'update))
	  ((activate)	 (apply my-widg		   (cons message data))
			 (my-menu-bar 'activate)
			 (mac:set-port my-wind))
	  ((deactivate)	 (apply my-widg		   (cons message data)))
	  ((select)	 (apply mac:wind-select	   (cons my-wind data))
			 (self 'activate))

	  ((update)	 (let ((old-port (mac:get-port)))
			   (mac:set-port my-wind)
			   (mac:wind-begin-update my-wind)
			   (my-widg 'update)
			   (mac:wind-end-update my-wind)
			   (mac:set-port old-port)))
	  ((dispose)	 (apply my-widg		   (cons message data))
			 (apply mac:wind-dispose   (cons my-wind data)))
	  ((hide)	 (apply mac:wind-hide	   (cons my-wind data)))
	  ((show)	 (apply mac:wind-show	   (cons my-wind data)))
	  ((set-title)	 (apply mac:wind-set-title (cons my-wind data)))
	  ((get-title)	 (apply mac:wind-get-title (cons my-wind data)))
	  ((evt)	 (apply my-widg		   (cons message data)))
	  (else	 (error "Invalid message to window" message))
	  ))))
    (set! my-wind (mac:wind-new rect title visible proc go-away))
    (mac:wind-set-ref my-wind self)
    (set! my-widg null-object)
    self))

(define (wind-doc-new rect title visible go-away)
  (wind-new rect title visible documentProc go-away))

(define (wind-zoom-doc-new rect title visible go-away)
  (wind-new rect title visible zoomDocProc go-away))

(define (wind-no-grow-doc-new rect title visible go-away)
  (wind-new rect title visible noGrowDocProc go-away))

;;;
;;; menu-bar-explode
;;;
;;; Turn a menu into a list of strings representing each item.
;;;

(define menu-bar-explode
  (lambda ()
    (letrec
      ((menu-bar-explode-iter
	(lambda (menu-num menu-list)
	  (if (= menu-num 0)
	      menu-list
	      (let ((menu (mac:get-menu-handle menu-num)))
		(menu-bar-explode-iter
		 (-1+ menu-num)
		 (if menu
		     (cons (list (mac:menu-get-title menu)
				 (menu-explode menu))
			   menu-list)
		     menu-list)))))))
      (menu-bar-explode-iter 15 '()))))

;;;
;;; menu-explode
;;;
;;; Turn a menu into a list of strings representing each item.
;;;

(define menu-explode
  (lambda (menu)
    (letrec
      ((menu-explode-iter
	(lambda (item-num item-list)
	  (if (= item-num 0)
	      item-list
	      (menu-explode-iter (-1+ item-num)
				 (cons (menu-item-explode menu item-num)
				       item-list))))))
      (menu-explode-iter (mac:menu-count-items menu) '()))))

;;;
;;; menu-item-explode
;;;
;;; Turn a menu item into a list of descriptions.
;;;

(define menu-item-explode
  (lambda (menu item-num)
    (list (mac:menu-item-get menu item-num)
	  (integer->char (mac:menu-item-get-cmd menu item-num))
	  (mac:menu-item-get-mark menu item-num)
	  (mac:menu-item-get-icon menu item-num)
	  (mac:menu-item-get-style menu item-num))))

;;;
;;; menu-new
;;;
;;; Create a new menu from a list of items.
;;;

(define menu-new
  (lambda (name num item-list)
    (letrec
      ((new-menu (mac:menu-new num name))
       (menu-new-iter
	(lambda (item-list)
	  (if (not (null? item-list))
	      (begin
	       (mac:menu-append new-menu (caar item-list))
	       (menu-new-iter (cdr item-list)))))))
      (begin
       (menu-new-iter item-list)
       new-menu))))

;;;
;;; menu-bar-new
;;;
;;; Create a new menu from a list of items.
;;;

(define (menu-bar-choice the-choice)
  (let*
    ((the-menu-id (quotient the-choice 65536))
     (the-menu (mac:get-menu-handle the-menu-id))
     (the-menu-item-id (remainder the-choice 65536))
     )
;    (if (not (= 0 the-menu-id))
;	 (begin
;	  (display (mac:menu-get-title the-menu))
;	  (newline)
;	  (display (mac:menu-item-get the-menu the-menu-item-id))
;	  (newline)
;	  ))
    (if (and (= 1 the-menu-id)
	     (< 1 the-menu-item-id))
	(begin
	 (mac:menu-open-desk-acc the-menu the-menu-item-id)
	 (mac:menu-hilite 0)
	 '(0 0))
	(begin
	 (mac:menu-hilite 0)
	 (list the-menu-id the-menu-item-id)))))

(define (menu-bar-select the-pt)
  (menu-bar-choice (mac:menu-select the-pt)))

(define (menu-bar-key the-char)
  (menu-bar-choice (mac:menu-key the-char)))

(define (menu-list-get-action menu-list menu-id menu-item-id)
  (cadr (list-ref (list-ref menu-list (+ -2 menu-id)) menu-item-id)))

(define (menu-bar-get-action menu-list the-choice)
  (let*
    ((the-menu-id (car the-choice))
     (the-menu (mac:get-menu-handle the-menu-id))
     (the-menu-item-id (cadr the-choice))
     )
    (if (= 0 the-menu-id)
	(lambda () #f)
	(begin
	 (menu-list-get-action menu-list the-menu-id the-menu-item-id)))))

(define (menu-bar-clear)
  (letrec
    ((menu-bar-clear-iter
	(lambda (menu-num)
	  (if (> menu-num 1)
	      (begin
	       (mac:menu-delete menu-num)
	       (menu-bar-clear-iter (-1+ menu-num)))))))
    (menu-bar-clear-iter 16)))

(define (menu-bar-set-up menu-list)
  (letrec
    ((menu-bar-set-up-iter
	(lambda (menu-list menu-num)
	  (if (not (null? menu-list))
	      (begin
	       (let*
		 ((menu (car menu-list))
		  (menu-new (menu-new (car menu) menu-num (cdr menu))))
		 (mac:menu-insert menu-new (1+ menu-num)))
	       (menu-bar-set-up-iter (cdr menu-list) (1+ menu-num)))))))
    (menu-bar-clear)
    (menu-bar-set-up-iter menu-list 2)
    (mac:menu-bar-draw)))

(define (menu-bar-new)
  (letrec
    ((my-menu-list #f)
     (my-menu-bar #f)
     (self
       (lambda (message . data)
	 (case message
	   ((set-menus)
	    (let*
	      ((the-menu-list (car data))
	       (old-menu-bar (mac:menu-bar-get)))
	      (set! my-menu-list the-menu-list)
	      (menu-bar-clear)
	      (menu-bar-set-up the-menu-list)
	      (set! my-menu-bar (mac:menu-bar-get))
	      (mac:menu-bar-set old-menu-bar)))

	   ((activate)
	    (mac:menu-bar-set my-menu-bar)
	    (mac:menu-bar-draw))

	   ((choice)
	    (apply (menu-bar-get-action my-menu-list (car data)) '()))
	  ))))
    self))

;;;
;;; remove-first
;;; remove-all
;;;
;;; These functions take an item and a list and remove the first occurrence of the
;;; item or all occurrences, respectively.
;;;

(define (remove-first item list)
  (letrec
    ((remove-first-iter
      (lambda (item list-a list-b)
	(if (or (null? list-b)
		(eq? item (car list-b)))
	    (append  (reverse list-a) (cdr list-b))
	    (remove-first-iter item (cons (car list-b) list-a) (cdr list-b))))))

    (remove-first-iter item '() list)))

(define (remove-all item list)
  (letrec
    ((remove-all-iter
      (lambda (item list-a list-b)
	(if (null? list-b)
	    (reverse list-a)
	    (remove-all-iter item
			     (if (eq? item (car list-b))
				 list-a
				 (cons (car list-b) list-a))
			     (cdr list-b))))))

    (remove-all-iter item '() list)))

;;;
;;; set-new
;;;
;;; This object constructor creates a set object.
;;;

(define (set-new)
  (letrec
    ((my-list '())
     (self
      (lambda (message . data)
	(case message
	  ((get-list) my-list)
	  ((iter)  (map (car data) my-list) #f) ; should return unspecified.
	  ((empty)  (eq? my-list '()))
	  ((member) (if (memq (car data) my-list) #t #f))
	  ((add)  (if (not (self 'member (car data)))
		      (set! my-list (cons (car data) my-list))))
	  ((remove) (set! my-list (remove-all (car data) my-list)))))))
    self))


;;;
;;; app-new
;;;
;;; This object constructor creates an application object (there should only be one).
;;;

(define global-menu-bar (mac:menu-bar-get))

(define (app-new)
  (letrec
    ((my-wind-set (set-new))
     (self
      (lambda (message . data)
	(case message
	  ((get-wind-list) (my-wind-set 'get-list))
	  ((own-wind)	   (my-wind-set 'member (car data)))
	  ((add-wind)	   (my-wind-set 'add	(car data)))
	  ((remove-wind)   (my-wind-set 'remove (car data)))
	  ((dispose-wind)  (let ((the-wind (car data)))
			     (self 'remove-wind the-wind)
			     (the-wind 'dispose)))
	  ((idle dispose update)
	   (my-wind-set 'iter (lambda (x) (apply x (cons message data)))))

	  ((evt)
	   (let*
	     ((the-evt (car data))
	      (event-type (mac:evt-get-what the-evt)))
	     (case event-type

	       ((#.mouseDown #.mouseUp)
		(let*
		  ((the-pt (mac:evt-get-where the-evt))
		   (part-code (mac:find-wind-part the-pt))
		   (the-wind (mac:find-wind (mac:evt-get-where the-evt)))
		   (the-wind-obj (if the-wind
				     (mac:wind-get-ref the-wind)
				     #f))
		   (the-front-wind (mac:front-wind))
		   (the-front-wind-obj (if the-front-wind
					   (mac:wind-get-ref the-front-wind)
					   #f)))
		  (case part-code
		    ((#.inDesk)
		     #f)

		    ((#.inMenuBar)
		     (if (self 'own-wind the-front-wind-obj)
			 (the-front-wind-obj 'do-menu-select the-pt)
			 (menu-bar-select the-pt)))

		    ((#.inSysWindow)
		     (mac:system-click the-evt the-wind)
		     (display "inSysWindow")
		     (newline)
		     )

		    ((#.inContent)
		     ; this needs to be a little more robust
		     ; basically, we need to make sure the window is activated
		     ; before we send it a message.
		     (the-wind-obj 'select)
		     (the-wind-obj 'evt the-evt))

		    ((#.inDrag)
		     (the-wind-obj 'select)
		     (mac:wind-drag the-wind the-pt))

		    ((#.inGrow)
		     (display "inGrow")
		     (newline)
		     )

		    ((#.inGoAway)
		     (if (mac:wind-track-go-away the-wind the-pt)
			 (self 'dispose-wind the-wind-obj)))

		    ((#.inZoomIn #.inZoomOut)
		     (if (mac:wind-track-box the-wind the-pt part-code)
			 (the-wind-obj 'zoom part-code #f)))

		    (else (error "window part code error" part-code)))))

	       ((#.keyDown #.keyUp #.autoKey)
		(let*
		  ((the-front-wind (mac:front-wind))
		   (the-front-wind-obj (if the-front-wind
					   (mac:wind-get-ref the-front-wind)
					   #f)))
		  (if (self 'own-wind the-front-wind-obj)
		      (if (mac:mods-get-command (mac:evt-get-mods the-evt))
			  (the-front-wind-obj 'do-menu-key
					      (mac:evt-get-msg-int the-evt))
			  (the-front-wind-obj 'evt the-evt)))))

	       ((#.updateEvt)
		(let*
		  ((the-wind (mac:evt-get-msg-wind the-evt))
		   (the-wind-obj (if the-wind
				     (mac:wind-get-ref the-wind)
				     #f)))
		  (if (self 'own-wind the-wind-obj)
		      (the-wind-obj 'update))))

	       ((#.diskEvt)
		(display "disk-event")
		(newline)
		)

	       ((#.activateEvt)
		(let*
		  ((the-wind (mac:evt-get-msg-wind the-evt))
		   (the-wind-obj (if the-wind
				     (mac:wind-get-ref the-wind)
				     #f)))
		  (if (self 'own-wind the-wind-obj)
		      (if (mac:mods-get-active (mac:evt-get-mods the-evt))
			  (the-wind-obj 'activate)
			  (the-wind-obj 'deactivate))
		      (if (mac:mods-get-active (mac:evt-get-mods the-evt))
			  (begin
			   (mac:menu-bar-set global-menu-bar)
			   (mac:menu-bar-draw))))
		  ))

	       (else
		(display "other event: ")
		(display event-type)
		(newline)
		))))

	  (else	 (error "Invalid message to application" message))
	  ))))
    self))
