/* Conversion of files between different charsets and usages.
   Copyright (C) 1990 Free Software Foundation, Inc.
   Francois Pinard <pinard@iro.umontreal.ca>, 1988.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#define STEP	latin1_latex
#include <stdio.h>
#include "common.h"

extern int diacritics_only;

struct translation
{
  int code;			/* code being translated */
  char *string;			/* translation string */
};

struct translation diacritic_translations [] =
{
  {0300, "\\`A"},		/* capital A with grave accent */
  {0301, "\\'A"},		/* capital A with acute accent */
  {0302, "\\^A"},		/* capital A with circumflex accent */
  {0303, "\\~A"},		/* capital A with tilde */
  {0304, "\\\"A"},		/* capital A diaeresis */
  {0305, "\\AA{}"},		/* capital A with ring above */
  {0306, "\\AE{}"},		/* capital diphthong A with E */
  {0307, "\\c{C}"},		/* capital C with cedilla */
  {0310, "\\`E"},		/* capital E with grave accent */
  {0311, "\\'E"},		/* capital E with acute accent */
  {0312, "\\^E"},		/* capital E with circumflex accent */
  {0313, "\\\"E"},		/* capital E with diaeresis */
  {0314, "\\`I"},		/* capital I with grave accent */
  {0315, "\\'I"},		/* capital I with acute accent */
  {0316, "\\^I"},		/* capital I with circumflex accent */
  {0317, "\\\"I"},		/* capital I with diaeresis */
  {0321, "\\~N"},		/* capital N with tilde */
  {0322, "\\`O"},		/* capital O with grave accent */
  {0323, "\\'O"},		/* capital O with acute accent */
  {0324, "\\^O"},		/* capital O with circumflex accent */
  {0325, "\\~O"},		/* capital O with tilde */
  {0326, "\\\"O"},		/* capital O with diaeresis */
  {0330, "\\O{}"},		/* capital O with oblique stroke */
  {0331, "\\`U"},		/* capital U with grave accent */
  {0332, "\\'U"},		/* capital U with acute accent */
  {0333, "\\^U"},		/* capital U with circumflex accent */
  {0334, "\\\"U"},		/* capital U with diaeresis */
  {0335, "\\'Y"},		/* capital Y with acute accent */
  {0337, "\\ss{}"},		/* small german sharp s */
  {0340, "\\`a"},		/* small a with grave accent */
  {0341, "\\'a"},		/* small a with acute accent */
  {0342, "\\^a"},		/* small a with circumflex accent */
  {0343, "\\~a"},		/* small a with tilde */
  {0344, "\\\"a"},		/* small a with diaeresis */
  {0345, "\\aa{}"},		/* small a with ring above */
  {0346, "\\ae{}"},		/* small diphthong a with e */
  {0347, "\\c{c}"},		/* small c with cedilla */
  {0350, "\\`e"},		/* small e with grave accent */
  {0351, "\\'e"},		/* small e with acute accent */
  {0352, "\\^e"},		/* small e with circumflex accent */
  {0353, "\\\"e"},		/* small e with diaeresis */
  {0354, "\\`\\i{}"},		/* small i with grave accent */
  {0355, "\\'\\i{}"},		/* small i with acute accent */
  {0356, "\\^\\i{}"},		/* small i with circumflex accent */
  {0357, "\\\"\\i{}"},		/* small i with diaeresis */
  {0361, "\\~n"},		/* small n with tilde */
  {0362, "\\`o"},		/* small o with grave accent */
  {0363, "\\'o"},		/* small o with acute accent */
  {0364, "\\^o"},		/* small o with circumflex accent */
  {0365, "\\~o"},		/* small o with tilde */
  {0366, "\\\"o"},		/* small o with diaeresis */
  {0370, "\\o{}"},		/* small o with oblique stroke */
  {0371, "\\`u"},		/* small u with grave accent */
  {0372, "\\'u"},		/* small u with acute accent */
  {0373, "\\^u"},		/* small u with circumflex accent */
  {0374, "\\\"u"},		/* small u with diaeresis */
  {0375, "\\'y"},		/* small y with acute accent */
  {0377, "\\\"y"},		/* small y with diaeresis */
  {0, NULL}
};

struct translation other_translations [] =
{
  {0043, "\\#"},
  {0044, "\\$"},
  {0045, "\\%"},
  {0046, "\\&"},
  {0134, "\\backslash{}"},
  {0137, "\\_"},
  {0173, "\\{"},
  {0175, "\\}"},
  {0240, "\\ "},		/* no-break space */
  {0241, "!'"},			/* inverted exclamation mark */
  {0247, "\\S{}"},		/* paragraph sign, section sign */
  {0250, "\\\"{}"},		/* diaeresis */
  {0253, "``"},			/* left angle quotation mark */
  {0254, "\\neg{}"},		/* not sign */
  {0255, "\\-"},		/* soft hyphen */
  {0260, "\\mbox{$^\\circ$}"},	/* degree sign */
  {0261, "\\mbox{$\\pm$}"},	/* plus-minus sign */
  {0262, "\\mbox{$^2$}"},	/* superscript two */
  {0263, "\\mbox{$^3$}"},	/* superscript three */
  {0264, "\\'{}"},		/* acute accent */
  {0265, "\\mbox{$\\mu$}"},	/* small greek mu, micro sign */
  {0267, "\\cdotp"},		/* middle dot */
  {0270, "\\,{}"},		/* cedilla */
  {0271, "\\mbox{$^1$}"},	/* superscript one */
  {0273, "''"},			/* right angle quotation mark */
  {0274, "\\frac1/4{}"},	/* vulgar fraction one quarter */
  {0275, "\\frac1/2{}"},	/* vulgar fraction one half */
  {0276, "\\frac3/4{}"},	/* vulgar fraction three quarters */
  {0277, "?'"},			/* inverted question mark */
  {0, NULL}
};

void
STEP (FILE *input_file, FILE *output_file)
{
  char *translation_table[256];	/* one string for each character */
  char character_pool[256];	/* for some strings of translation_table */
  struct translation *cursor;	/* cursor in translation arrays */
  int input_char;		/* current character */
  const char *output_string;	/* translated characters */

  for (input_char = 0; input_char < 128; input_char++)
    {
      translation_table[input_char] = character_pool + 2*input_char;
      translation_table[input_char][0] = input_char;
      translation_table[input_char][1] = '\0';
    }
  for (input_char = 128; input_char < 256; input_char++)
    translation_table[input_char] = NULL;

  for (cursor = diacritic_translations; cursor->code; cursor++)
    translation_table[cursor->code] = cursor->string;
  if (!diacritics_only)
    for (cursor = other_translations; cursor->code; cursor++)
      translation_table[cursor->code] = cursor->string;

  while (input_char = getc (input_file), input_char != EOF)
    if (output_string = translation_table[input_char], output_string)
      do
	{
	  putc (*output_string, output_file);
	  output_string++;
	}
      while (*output_string);
}
