#define GSR_OWNER
#include "gsr.h"

/*-
   gnuplot Graphics Support Routines: gsr.c

   This module provides graphics routines similar to those used to
   implement gnuplot.  These routines are based on (and require) the
   gnuplot Graphics Terminal library (gterm.c).
-*/

void plot_impulses();
void plot_lines();
void plot_points();
void plot_dots();
void edge_intersect();
TBOOLEAN two_edge_intersect();


#define inrange(z,min,max) \
((min<max) ? ((z>=min)&&(z<=max)) : ((z>=max)&&(z<=min)) )

/* (DFK) Watch for cancellation error near zero on axes labels */
#define SIGNIF (0.01)		/* less than one hundredth of a tic mark */
#define CheckZero(x,tic) (fabs(x) < ((tic) * SIGNIF) ? 0.0 : (x))
#define NearlyEqual(x,y,tic) (fabs((x)-(y)) < ((tic) * SIGNIF))

/* (DFK) For some reason, the Sun386i compiler screws up with the CheckLog 
 * macro, so I write it as a function on that machine.
 */
#ifndef sun386
/* (DFK) Use 10^x if logscale is in effect, else x */
#define CheckLog(log, x) ((log) ? pow(10., (x)) : (x))
#else
static double CheckLog(log, x)
     TBOOLEAN log;
     double x;
{
  if (log)
    return(pow(10., x));
  else
    return(x);
}
#endif /* sun386 */


static double LogScale(coord, islog, what, axis)
	double coord;			/* the value */
	TBOOLEAN islog;			/* is this axis in logscale? */
	char *what;			/* what is the coord for? */
	char *axis;			/* which axis is this for ("x" or "y")? */
{
    if (islog) {
	   if (coord <= 0.0) {
		  char errbuf[100];		/* place to write error message */
		(void) sprintf(errbuf,"%s has %s coord of %g; must be above 0 for log scale!",
				what, axis, coord);
		  (*GTterm_tbl[GTterm].text)();
		  (void) fflush(GSRoutfile);
		  int_error(errbuf, NO_CARET);
	   } else
		return(log10(coord));
    } else {
	   return(coord);
    }
	return((double) 0.0); /* shut lint up */
}

/* borders of plotting area */
/* computed once on every call to do_plot */
static boundary(scaling)
	TBOOLEAN scaling;		/* TRUE if terminal is doing the scaling */
{
    register struct termentry *t = &GTterm_tbl[GTterm];
    GSRxleft = (t->h_char)*12;
    GSRxright = (scaling ? 1 : GTxsize) * (t->xmax) - (t->h_char)*2 - (t->h_tic);
    GSRybot = (t->v_char)*5/2 + 1;
    GSRytop = (scaling ? 1 : GTysize) * (t->ymax) - (t->v_char)*3/2 - 1;
}


static double dbl_raise(x,y)
double x;
int y;
{
register int i;
double val;

	val = 1.0;
	for (i=0; i < abs(y); i++)
		val *= x;
	if (y < 0 ) return (1.0/val);
	return(val);
}


static double make_tics(tmin,tmax,logscale)
double tmin,tmax;
TBOOLEAN logscale;
{
register double xr,xnorm,tics,tic,l10;

	xr = fabs(tmin-tmax);
	
	l10 = log10(xr);
	if (logscale) {
		tic = dbl_raise(10.0,(l10 >= 0.0 ) ? (int)l10 : ((int)l10-1));
		if (tic < 1.0)
			tic = 1.0;
	} else {
		xnorm = pow(10.0,l10-(double)((l10 >= 0.0 ) ? (int)l10 : ((int)l10-1)));
		if (xnorm <= 2)
			tics = 0.2;
		else if (xnorm <= 5)
			tics = 0.5;
		else tics = 1.0;	
		tic = tics * dbl_raise(10.0,(l10 >= 0.0 ) ? (int)l10 : ((int)l10-1));
	}
	return(tic);
}


gsr_init (fp, min_x, max_x, min_y, max_y, auto_x, auto_y, log_x, log_y)
FILE *fp;
double min_x, max_x;
double min_y, max_y;
TBOOLEAN auto_x, auto_y, log_x, log_y;
/*-
   Initialize the internal variables used by the Gnuplot graphics support
   routines.  Note that this routine should be called at least once before
   any plot and at least once after changing the terminal type.

   Parameters:

           fp: Output file pointer.  This is stored as GSRoutfile (which is
               the same as GToutfile) and is the stream where all the 
               output for the plot is written.  If fp is different than 
               GSRoutfile then the old GSRoutfile is replaced with the
               new fp.  Note that the old fp is *not* closed, this is the
               responsibility of the caller.  If fp is NULL the old 
               GSRoutfile continues to be used.

        min_x:  Minimum x value that you intend to plot (in your own units).

        max_x:  Maximum x value that you intend to plot (in your own units).

        min_y:  Minimum y value that you intend to plot (in your own units).

        max_y:  Maximum y value that you intend to plot (in your own units).

       auto_x:  Automatically adjust the x tics to a sensible round-off point.

       auto_y:  Automatically adjust the y tics to a sensible round-off point.

        log_x:  Prepare for a logarithmic scale in the x dimension.

        log_y:  Prepare for a logarithmic scale in the y dimension.

-*/
{
register struct termentry *t = &GTterm_tbl[GTterm];
register int xl, yl;
double xtemp, ytemp;
struct text_label *this_label;
struct arrow_def *this_arrow;
TBOOLEAN scaling;

/* Make sure a terminal has been selected. */
   if (GTterm == 0) {
      gt_init_terminal();
      if (GTterm == 0) {
	int_error ("GTterm is 0--have you defined the variable GNUTERM?");
      }
   }

/* Take care of GSRoutfile. */
   if (fp == NULL) {
      if (GSRoutfile == NULL) {
	 int_error("GSRoutfile NULL--gsr_init needs a file pointer!",NO_CARET);
      }
   }
   else {
      GSRoutfile = fp;
   }

/* store these in variables global to this file */
/* otherwise, we have to pass them around a lot */
	GSRxmin = min_x;
	GSRxmax = max_x; 
  	GSRymin = min_y;
	GSRymax = max_y;
        GSRlogx = log_x;
        GSRlogy = log_y;

/* SETUP RANGES, SCALES AND TIC PLACES */
	GSRytic = make_tics(GSRymin,GSRymax,GSRlogy);
    
   if (auto_y) {
	  if (GSRymin < GSRymax) {
		 GSRymin = GSRytic * floor(GSRymin/GSRytic);       
		 GSRymax = GSRytic * ceil(GSRymax/GSRytic);
	  }
	  else {			/* reverse axis */
		 GSRymin = GSRytic * ceil(GSRymin/GSRytic);       
		 GSRymax = GSRytic * floor(GSRymax/GSRytic);
	   }
    }

	GSRxtic = make_tics(GSRxmin,GSRxmax,GSRlogx);
	   
	if (auto_x) {
		if (GSRxmin < GSRxmax) {
			GSRxmin = GSRxtic * floor(GSRxmin/GSRxtic);	
			GSRxmax = GSRxtic * ceil(GSRxmax/GSRxtic);
	  } else {
			GSRxmin = GSRxtic * ceil(GSRxmin/GSRxtic);
			GSRxmax = GSRxtic * floor(GSRxmax/GSRxtic);	
	   }
    }

/*	This was GSRxmax == GSRxmin, but that caused an infinite loop once. */
	if (fabs(GSRxmax - GSRxmin) < ZERO)
		int_error("GSRxmin should not equal GSRxmax!",NO_CARET);
	if (fabs(GSRymax - GSRymin) < ZERO)
		int_error("GSRymin should not equal GSRymax!",NO_CARET);

/* INITIALIZE TERMINAL */
	if (!GTterm_init) {
		(*t->init)();
		GTterm_init = TRUE;
	}
	scaling = (*t->scale)(GTxsize, GTysize);

     /* now compute boundary for plot (GSRxleft, GSRxright, GSRytop, GSRybot) */
     boundary(scaling);

/* SCALE FACTORS  (gsr_map_x & y will now be ready to use) */
	GSRyscale = (GSRytop - GSRybot)/(GSRymax - GSRymin);
	GSRxscale = (GSRxright - GSRxleft)/(GSRxmax - GSRxmin);
}
	
gsr_graphics()
/*-
   Routine to call the proper terminal t->graphics() routine.  This
   should be called prior to trying to draw anything on the graphics
   terminal.
-*/
{
register struct termentry *t = &GTterm_tbl[GTterm];

	(*t->graphics)();
}


gsr_reset_terminal ()
/*-
   Do what has to be done to reset the terminal after we are all
   done plotting.  (Mostly just call (*t->reset).)
-*/
{
   struct termentry *t = &GTterm_tbl[GTterm];

   (*t->reset)();
#ifdef VMS
   vms_reset();
#endif
}


gsr_draw_axis (xloc, yloc, visible)
double xloc, yloc;
int visible;
/*-
   Routine to draw axis lines (with linetype -1).  A horizontal line
   will be drawn at xloc and a vertical line at yloc.  Normal usage
   is for xloc and yloc to be 0.0  Note that the yloc value is also
   used as the line to draw impulses on for plotting IMPULSE style
   curves.  If ``visible'' is 0 then the lines won't be drawn, but
   GSRxaxis_y will be computed and saved for use with IMPULSE plots.

   Parameters:

         xloc:  Location (in user coordinates) of x axis line.

         yloc:  Location (in user coordinates) of y axis line.

      visible:  0 - neither axis is drawn, 
                1 - x axis only is drawn,
                2 - y axis only is drawn,
                3 - both axis are drawn.

-*/
{
/* DRAW AXES */
register struct termentry *t = &GTterm_tbl[GTterm];
register int xaxis_y, yaxis_x;

/* Do we need to store linetype between each of these? */
	xaxis_y = gsr_map_y(xloc);
	yaxis_x = gsr_map_x(yloc); 

	if (visible) 
		(*t->linetype)(-1);	/* axis line type */

	if (xaxis_y < GSRybot)
		xaxis_y = GSRybot;				/* save for impulse plotting */
	else if (xaxis_y >= GSRytop)
		xaxis_y = GSRytop ;
	else if ((visible & 1) && !GSRlogy) {
		(*t->move)(GSRxleft,xaxis_y);
		(*t->vector)(GSRxright,xaxis_y);
	}

	if ((visible & 2) && !GSRlogx && 
				yaxis_x >= GSRxleft && yaxis_x < GSRxright ) {
		(*t->move)(yaxis_x,GSRybot);
		(*t->vector)(yaxis_x,GSRytop);
	}
	GSRxaxis_y = xaxis_y;
}


#ifdef RLAB
gsr_set_tic_format (xformat, yformat)
char *xformat, *yformat;
/*-
   Routine to change the x and y tic mark formats (GSRxformat and GSRyformat)
   the default for both is "%g".

   Parameter:

     xformat:  Pointer to new sprintf GSRxformat string;

     yformat:  Pointer to new sprintf GSRyformat string;
-*/
{
   GSRxformat = xformat;
   GSRyformat = yformat;
}
#endif


gsr_draw_tics (xtics, xgrid, xtic_in, ytics, ygrid, ytic_in)
int xtics, xgrid, xtic_in, ytics, ygrid, ytic_in;
/*-
   Routine to draw and label the tic marks.

   Parameters:

        xtics:  If 1 draw x axis tics, if 0 don't.

        xgrid:  If 1 draw an x grid, if 0 don't.

      xtic_in:  If 1 draw the x tics inward, if 0 don't.

        ytics:  If 1 draw y axis tics, if 0 don't.

        ygrid:  If 1 draw a y grid, if 0 don't.

      ytic_in:  If 1 draw the y tics inward, if 0 don't.
-*/
{
#define TIC_COMPUTED 1
register struct termentry *t = &GTterm_tbl[GTterm];
int type=TIC_COMPUTED;

/* 
   This routine can be improved by allowing for TIC_SERIES and TIC_USER
   as is done in the main gnuplot package.
*/
   
/* DRAW TICS */
	(*t->linetype)(-2); /* border linetype */

    /* label y axis tics */
     if (ytics) {
	    switch (type) {
		   case TIC_COMPUTED: {
 			  if (GSRymin < GSRymax)
			    draw_ytics(GSRytic * floor(GSRymin/GSRytic),
						GSRytic,
						GSRytic * ceil(GSRymax/GSRytic), ygrid, ytic_in);
			  else
			    draw_ytics(GSRytic * floor(GSRymax/GSRytic),
						GSRytic,
						GSRytic * ceil(GSRymin/GSRytic), ygrid, ytic_in);

			  break;
		   }
#ifdef NOTUSED_YET
		   case TIC_SERIES: {
			  draw_ytics(yticdef.def.series.start, 
					  yticdef.def.series.incr, 
					  yticdef.def.series.end, ygrid, ytic_in);
			  break;
		   }
		   case TIC_USER: {
			  draw_user_ytics(yticdef.def.user, ygrid, ytic_in);
			  break;
		   }
#endif
   default: {
		  (*t->text)();
		  (void) fflush(GSRoutfile);
	  int_error("unknown tic type in yticdef in do_plot", NO_CARET);
	  break;		/* NOTREACHED */
		   }
	    }
	}

    /* label x axis tics */
     if (xtics) {
	    switch (type) {
		   case TIC_COMPUTED: {
 			  if (GSRxmin < GSRxmax)
			    draw_xtics(GSRxtic * floor(GSRxmin/GSRxtic),
						GSRxtic,
						GSRxtic * ceil(GSRxmax/GSRxtic), xgrid, xtic_in);
			  else
			    draw_xtics(GSRxtic * floor(GSRxmax/GSRxtic),
						GSRxtic,
						GSRxtic * ceil(GSRxmin/GSRxtic), xgrid, xtic_in);

			  break;
		   }
#ifdef NOTUSED_YET
		   case TIC_SERIES: {
			  draw_xtics(xticdef.def.series.start, 
					  xticdef.def.series.incr, 
					  xticdef.def.series.end, xgrid, xtic_in);
			  break;
		   }
		   case TIC_USER: {
			  draw_user_xtics(xticdef.def.user, xgrid, xtic_in);
			  break;
		   }
#endif
		   default: {
			  (*t->text)();
			  (void) fflush(GSRoutfile);
			  int_error("unknown tic type in xticdef in do_plot", NO_CARET);
			  break;		/* NOTREACHED */
		   }
	    }
	}
}


gsr_boundary ()
/*-
   Routine to draw the boundary (border around the plot area).
-*/
{

/* DRAW PLOT BORDER */
register struct termentry *t = &GTterm_tbl[GTterm];
	(*t->linetype)(-2); /* border linetype */
	(*t->move)(GSRxleft,GSRybot);	
	(*t->vector)(GSRxright,GSRybot);	
	(*t->vector)(GSRxright,GSRytop);	
	(*t->vector)(GSRxleft,GSRytop);	
	(*t->vector)(GSRxleft,GSRybot);
}

gsr_ylabel (ylabel)
char *ylabel;
/*-
   Routine to write a text lable for the y-axis.  Some terminal
   drivers will write this as vertically oriented text.  LaTeX
   will center the horizontal text on the left side of the plot.

   Parameter:

      ylabel:  Text string to place as the y axis label.
-*/
{
/* PLACE YLABEL */
register struct termentry *t = &GTterm_tbl[GTterm];
    if (*ylabel != (char)NULL) {
	   if ((*t->text_angle)(1)) { 
		  if ((*t->justify_text)(CENTRE)) { 
			 (*t->put_text)((t->v_char),
						 (GSRytop+GSRybot)/2, ylabel);
		  }
		  else {
			 (*t->put_text)((t->v_char),
						 (GSRytop+GSRybot)/2-(t->h_char)*strlen(ylabel)/2, 
						 ylabel);
		  }
	   }
	   else {
		  (void)(*t->justify_text)(LEFT);
		  (*t->put_text)(0,GSRytop+(t->v_char), ylabel);
	   }
	   (void)(*t->text_angle)(0);
    }
}


gsr_xlabel (xlabel)
char *xlabel;
/*-
   Routine to write a text label for the x-axis.  

   Parameter:

      xlabel:  Text string to place as the x axis label.
-*/
{

/* PLACE XLABEL */
register struct termentry *t = &GTterm_tbl[GTterm];
    if (*xlabel != (char)NULL) {
	   if ((*t->justify_text)(CENTRE)) 
		(*t->put_text)( (GSRxleft+GSRxright)/2,
					GSRybot-2*(t->v_char), xlabel);
	   else
		(*t->put_text)( (GSRxleft+GSRxright)/2 - strlen(xlabel)*(t->h_char)/2,
					GSRybot-2*(t->v_char), xlabel);
    }
}

gsr_title (title)
char *title;
/*-
   Routine to write a title at the top of the plot.  

   Parameter:

       title:  Text string to place as the title for the plot.
-*/
{
/* PLACE TITLE */
register struct termentry *t = &GTterm_tbl[GTterm];
    if (*title != (char)NULL) {
	   if ((*t->justify_text)(CENTRE)) 
		(*t->put_text)( (GSRxleft+GSRxright)/2, 
					GSRytop+(t->v_char), title);
	   else
		(*t->put_text)( (GSRxleft+GSRxright)/2 - strlen(title)*(t->h_char)/2,
					GSRytop+(t->v_char), title);
    }
}


gsr_label (x, y, text, justify, position)
double x,y;
char *text;
TBOOLEAN justify;
int position;
/*-
    Routine to place a label (text string) at (x,y) where (x,y) is in user
    coordinates.

    Parameters:

             x: user coordinate x location for text string.

             y: user coordinate y location for text string.

          text: Text to plot at this location.

       justify: Flag indicating if text is to be justified in place:
                0 - don't justify
                1 - justify (if terminal device can do so).

      position: From gtplot.h, one of LEFT, CENTRE, RIGHT 
-*/
{
/* PLACE LABELS */
double xtemp, ytemp;
register struct termentry *t = &GTterm_tbl[GTterm];
    xtemp = LogScale(x, GSRlogx, "label", "x");
    ytemp = LogScale(y, GSRlogy, "label", "y");
	if ((*t->justify_text)(position)) {
		(*t->put_text)(gsr_map_x(xtemp),gsr_map_y(ytemp),text);
	}
	else {
		switch(position) {
			case  LEFT:
				(*t->put_text)(gsr_map_x(xtemp),gsr_map_y(ytemp), text);
			break;
			case CENTRE:
				(*t->put_text)(gsr_map_x(xtemp)-(t->h_char)*strlen(text)/2,
					gsr_map_y(ytemp), text);
			break;
			case RIGHT:
				(*t->put_text)(gsr_map_x(xtemp)-(t->h_char)*strlen(text),
						gsr_map_y(ytemp), text);
			break;
		 }
	 }
}


gsr_arrow (start_x, start_y, end_x, end_y)
double start_x, start_y, end_x, end_y;
/*-
    Draw an arrow from (start_x,start_y) to (end_x,end_y).

    Parameters:

       start_x:  starting x coordinate in user coordinates.

       start_y:  starting y coordinate in user coordinates.

         end_x:  ending x coordinate in user coordinates.

         end_y:  ending y coordinate in user coordinates.
-*/
{
/* PLACE ARROWS */
register struct termentry *t = &GTterm_tbl[GTterm];
	   int sx = gsr_map_x(LogScale(start_x, GSRlogx, "arrow", "x"));
	   int sy = gsr_map_y(LogScale(start_y, GSRlogy, "arrow", "y"));
	   int ex = gsr_map_x(LogScale(end_x, GSRlogx, "arrow", "x"));
	   int ey = gsr_map_y(LogScale(end_y, GSRlogy, "arrow", "y"));

    (*t->linetype)(0);	/* arrow line type */
	   
	   (*t->arrow)(sx, sy, ex, ey);
}


gsr_curve (style, points, npoints)
int style, npoints;
struct coordinate *points;
/*-
   Routine to draw the curve located in the array points to the device.

   Parameters:

        style: One of IMPULSES, LINES, POINTSTYLE, LINESPOINTS, or DOTS from
               gtplot.h

       points: An array of npoints coordinates, where struct coordinate
               is defined in gtplot.h.

      npoints: Number of points to plot out of the points array.
-*/
{
register struct termentry *t = &GTterm_tbl[GTterm];
/* DRAW CURVES */
		switch(style) {
		    case IMPULSES: 
			   plot_impulses(points, npoints, GSRxaxis_y);
		   break;
		    case LINES: 
			   plot_lines(points, npoints);
			break;
		    case POINTSTYLE: 
			   plot_points(points, npoints);
			   break;
		    case LINESPOINTS: 
			   /* put lines */
			   plot_lines(points, npoints);

			   /* put points */
			   plot_points(points, npoints);
			break;
		    case DOTS: 
			   plot_dots(points, npoints);
		    break;
			default:
			  (*t->text)();
			  (void) fflush(GSRoutfile);
			  int_error("unknown style in gsr_points", NO_CARET);
			  break;		/* NOTREACHED */
		}
}


gsr_text()
/*- 
   Routine to call the appropriate terminal _text() routine.  This works
   together with gsr_graphics() to go in and out of graphics mode: 

      gsr_graphics(); ... plot routines ...  gsr_text();

   On many terminal devices this must be called before any of the plot 
   will show up on the device.
-*/
{
register struct termentry *t = &GTterm_tbl[GTterm];

	(*t->text)();
	(void) fflush(GSRoutfile);
}


gsr_line_point_type (line_type, point_type)
/*-
   Routine to select (ahead of gsr_curve()) the line_type or point_type
   that you wish to use.

   Parameters:

    line_type:  The desired line type (from -2 - 12), the exact range
                depends upon the term device in use.

   point_type:  The desired point type (from -1 - 6) the exact range
                depends upon the term device in use.
-*/
{
   struct termentry *t = &GTterm_tbl[GTterm];

/* Record both requests. */
   GSRpoint_type = point_type;
   GSRline_type = line_type;

/* Actually set the line_type request. */
   (*t->linetype)(line_type);
}
   

#define OUTRANGE   0
#define INRANGE    1
#define UNDEFINED -1

/* plot_impulses:
 * Plot the curves in IMPULSES style
 */
static void
plot_impulses(points, npoints, xaxis_y)
	struct coordinate *points;
	int xaxis_y;
{
    int i;
    int x,y;
	int type;
    struct termentry *t = &GTterm_tbl[GTterm];

    for (i = 0; i < npoints; i++) {
	   type = inrange (points[i].x, GSRxmin, GSRxmax) &&
	                                inrange (points[i].y, GSRymin, GSRymax);
	   switch (type) {
		  case INRANGE: {
			 x = gsr_map_x(points[i].x);
			 y = gsr_map_y(points[i].y);
			 break;
		  }
		  case OUTRANGE: {
			 if (!inrange(points[i].x, GSRxmin,GSRxmax))
			   continue;
			 x = gsr_map_x(points[i].x);
			 if ((GSRymin < GSRymax && points[i].y < GSRymin)
				|| (GSRymax < GSRymin && points[i].y > GSRymin))
			   y = gsr_map_y(GSRymin);
			 if ((GSRymin < GSRymax && points[i].y > GSRymax)
				|| (GSRymax<GSRymin && points[i].y < GSRymax))
			   y = gsr_map_y(GSRymax);
			 break;
		  }
		  default:		/* just a safety */
		  case UNDEFINED: {
			 continue;
		  }
	   }
				    
	   (*t->move)(x,xaxis_y);
	   (*t->vector)(x,y);
    }

}

/* plot_lines:
 * Plot the curves in LINES style
 */
static void
plot_lines(points, npoints)
	struct coordinate *points;
	int npoints;
{
    int i;				/* point index */
    int x,y;				/* point in terminal coordinates */
    int type;
	int clip_lines1 = 0,  /* Not used yet. */
	    clip_lines2 = 0;
    struct termentry *t = &GTterm_tbl[GTterm];
    enum coord_type prev = UNDEFINED; /* type of previous point */
    double ex, ey;			/* an edge point */
    double lx[2], ly[2];		/* two edge points */

    for (i = 0; i < npoints; i++) {
	   type = inrange (points[i].x, GSRxmin, GSRxmax) &&
	                                inrange (points[i].y, GSRymin, GSRymax);
	   switch (type) {
		  case INRANGE: {
			 x = gsr_map_x(points[i].x);
			 y = gsr_map_y(points[i].y);

			 if (prev == INRANGE) {
				(*t->vector)(x,y);
			 } else if (prev == OUTRANGE) {
				/* from outrange to inrange */
				if (!clip_lines1) {
				    (*t->move)(x,y);
				} else {
				    edge_intersect(points, i, &ex, &ey);
				    (*t->move)(gsr_map_x(ex), gsr_map_y(ey));
				    (*t->vector)(x,y);
				}
			 } else {		/* prev == UNDEFINED */
				(*t->move)(x,y);
				(*t->vector)(x,y);
			 }
				    
			 break;
		  }
		  case OUTRANGE: {
			 if (prev == INRANGE) {
				/* from inrange to outrange */
				if (clip_lines1) {
				    edge_intersect(points, i, &ex, &ey);
				    (*t->vector)(gsr_map_x(ex), gsr_map_y(ey));
				}
			 } else if (prev == OUTRANGE) {
				/* from outrange to outrange */
				if (clip_lines2) {
				    if (two_edge_intersect(points, i, lx, ly)) {
					   (*t->move)(gsr_map_x(lx[0]), gsr_map_y(ly[0]));
					   (*t->vector)(gsr_map_x(lx[1]), gsr_map_y(ly[1]));
				    }
				}
			 }
			 break;
		  }
		  default:		/* just a safety */
		  case UNDEFINED: {
			 break;
		  }
	   }
	   prev = type;
    }
}

/* plot_points:
 * Plot the curves in POINTS style
 */
static void
plot_points(points, npoints)
	struct coordinate *points;
	int npoints;
{
    int i;
    int x,y;
	int type;
    struct termentry *t = &GTterm_tbl[GTterm];

    for (i = 0; i < npoints; i++) {
	   type = inrange (points[i].x, GSRxmin, GSRxmax) &&
	                                inrange (points[i].y, GSRymin, GSRymax);
	   if (type == INRANGE) {
		  x = gsr_map_x(points[i].x);
		  y = gsr_map_y(points[i].y);
		  /* do clipping (was ...if necessary...) */
		  if (   x >= GSRxleft + t->h_tic  && y >= GSRybot + t->v_tic 
			  && x <= GSRxright - t->h_tic && y <= GSRytop - t->v_tic)
		    (*t->point)(x,y, GSRpoint_type);
	   }
    }
}

/* plot_dots:
 * Plot the curves in DOTS style
 */
static void
plot_dots(points, npoints)
	struct coordinate *points;
	int npoints;
{
    int i;
    int x,y;
	int type;
    struct termentry *t = &GTterm_tbl[GTterm];

    for (i = 0; i < npoints; i++) {
	   type = inrange (points[i].x, GSRxmin, GSRxmax) &&
	                                inrange (points[i].y, GSRymin, GSRymax);
	   if (type == INRANGE) {
		  x = gsr_map_x(points[i].x);
		  y = gsr_map_y(points[i].y);
		  /* point type -1 is a dot */
		  (*t->point)(x,y, -1);
	   }
    }
}


/* single edge intersection algorithm */
/* Given two points, one inside and one outside the plot, return
 * the point where an edge of the plot intersects the line segment defined 
 * by the two points.
 */
static void
edge_intersect(points, i, ex, ey)
	struct coordinate *points; /* the points array */
	int i;				/* line segment from point i-1 to point i */
	double *ex, *ey;		/* the point where it crosses an edge */
{
    /* global xmin, xmax, ymin, xmax */
    double ax = points[i-1].x;
    double ay = points[i-1].y;
    double bx = points[i].x;
    double by = points[i].y;
    double x, y;			/* possible intersection point */

    if (by == ay) {
	   /* horizontal line */
	   /* assume inrange(by, GSRymin, GSRymax) */
	   *ey = by;		/* == ay */

	   if (inrange(GSRxmax, ax, bx))
		*ex = GSRxmax;
	   else if (inrange(GSRxmin, ax, bx))
		*ex = GSRxmin;
	   else {
		(*GTterm_tbl[GTterm].text)();
	    (void) fflush(GSRoutfile);
		int_error("error in edge_intersect", NO_CARET);
	   }
	   return;
    } else if (bx == ax) {
	   /* vertical line */
	   /* assume inrange(bx, GSRxmin, GSRxmax) */
	   *ex = bx;		/* == ax */

	   if (inrange(GSRymax, ay, by))
		*ey = GSRymax;
	   else if (inrange(GSRymin, ay, by))
		*ey = GSRymin;
	   else {
		(*GTterm_tbl[GTterm].text)();
	    (void) fflush(GSRoutfile);
		int_error("error in edge_intersect", NO_CARET);
	   }
	   return;
    }

    /* slanted line of some kind */

    /* does it intersect ymin edge */
    if (inrange(GSRymin, ay, by) && GSRymin != ay && GSRymin != by) {
	   x = ax + (GSRymin-ay) * ((bx-ax) / (by-ay));
	   if (inrange(x, GSRxmin, GSRxmax)) {
		  *ex = x;
		  *ey = GSRymin;
		  return;			/* yes */
	   }
    }
    
    /* does it intersect ymax edge */
    if (inrange(GSRymax, ay, by) && GSRymax != ay && GSRymax != by) {
	   x = ax + (GSRymax-ay) * ((bx-ax) / (by-ay));
	   if (inrange(x, GSRxmin, GSRxmax)) {
		  *ex = x;
		  *ey = GSRymax;
		  return;			/* yes */
	   }
    }

    /* does it intersect xmin edge */
    if (inrange(GSRxmin, ax, bx) && GSRxmin != ax && GSRxmin != bx) {
	   y = ay + (GSRxmin-ax) * ((by-ay) / (bx-ax));
	   if (inrange(y, GSRymin, GSRymax)) {
		  *ex = GSRxmin;
		  *ey = y;
		  return;
	   }
    }

    /* does it intersect xmax edge */
    if (inrange(GSRxmax, ax, bx) && GSRxmax != ax && GSRxmax != bx) {
	   y = ay + (GSRxmax-ax) * ((by-ay) / (bx-ax));
	   if (inrange(y, GSRymin, GSRymax)) {
		  *ex = GSRxmax;
		  *ey = y;
		  return;
	   }
    }

    /* It is possible for one or two of the [ab][xy] values to be -VERYLARGE.
	* If ax=bx=-VERYLARGE or ay=by=-VERYLARGE we have already returned 
	* FALSE above. Otherwise we fall through all the tests above. 
	* If two are -VERYLARGE, it is ax=ay=-VERYLARGE or bx=by=-VERYLARGE 
	* since either a or b must be INRANGE. 
	* Note that for ax=ay=-VERYLARGE or bx=by=-VERYLARGE we can do nothing.
	* Handle them carefully here. As yet we have no way for them to be 
	* +VERYLARGE.
	*/
    if (ax == -VERYLARGE) {
	   if (ay != -VERYLARGE) {
		  *ex = min(GSRxmin, GSRxmax);
		  *ey = by;
		  return;
	   }
    } else if (bx == -VERYLARGE) {
	   if (by != -VERYLARGE) {
		  *ex = min(GSRxmin, GSRxmax);
		  *ey = ay;
		  return;
	   }
    } else if (ay == -VERYLARGE) {
	   /* note we know ax != -VERYLARGE */
	   *ex = bx;
	   *ey = min(GSRymin, GSRymax);
	   return;
    } else if (by == -VERYLARGE) {
	   /* note we know bx != -VERYLARGE */
	   *ex = ax;
	   *ey = min(GSRymin, GSRymax);
	   return;
    }

    /* If we reach here, then either one point is (-VERYLARGE,-VERYLARGE), 
	* or the inrange point is on the edge, and
     * the line segment from the outrange point does not cross any 
	* other edges to get there. In either case, we return the inrange 
	* point as the 'edge' intersection point. This will basically draw
	* line.
	*/
/* Had to change this--just guessing this is right.  JDC */
    if (points[i].x != -VERYLARGE) {
	   *ex = bx; 
	   *ey = by;
    } else {
	   *ex = ax; 
	   *ey = ay;
    }
    return;
}

/* double edge intersection algorithm */
/* Given two points, both outside the plot, return
 * the points where an edge of the plot intersects the line segment defined 
 * by the two points. There may be zero, one, two, or an infinite number
 * of intersection points. (One means an intersection at a corner, infinite
 * means overlaying the edge itself). We return FALSE when there is nothing
 * to draw (zero intersections), and TRUE when there is something to 
 * draw (the one-point case is a degenerate of the two-point case and we do 
 * not distinguish it - we draw it anyway).
 */
static TBOOLEAN				/* any intersection? */
two_edge_intersect(points, i, lx, ly)
	struct coordinate *points; /* the points array */
	int i;				/* line segment from point i-1 to point i */
	double *lx, *ly;		/* lx[2], ly[2]: points where it crosses edges */
{
    /* global xmin, xmax, ymin, xmax */
    double ax = points[i-1].x;
    double ay = points[i-1].y;
    double bx = points[i].x;
    double by = points[i].y;
    double x, y;			/* possible intersection point */
    TBOOLEAN intersect = FALSE;

    if (by == ay) {
	   /* horizontal line */
	   /* y coord must be in range, and line must span both xmin and xmax */
	   /* note that spanning GSRxmin implies spanning GSRxmax */
	   if (inrange(by, GSRymin, GSRymax) && inrange(GSRxmin, ax, bx)) {
		  *lx++ = GSRxmin;
		  *ly++ = by;
		  *lx++ = GSRxmax;
		  *ly++ = by;
		  return(TRUE);
	   } else
		return(FALSE);
    } else if (bx == ax) {
	   /* vertical line */
	   /* x coord must be in range, and line must span both GSRymin and GSRymax */
	   /* note that spanning GSRymin implies spanning GSRymax */
	   if (inrange(bx, GSRxmin, GSRxmax) && inrange(GSRymin, ay, by)) {
		  *lx++ = bx;
		  *ly++ = GSRymin;
		  *lx++ = bx;
		  *ly++ = GSRymax;
		  return(TRUE);
	   } else
		return(FALSE);
    }

    /* slanted line of some kind */
    /* there can be only zero or two intersections below */

    /* does it intersect GSRymin edge */
    if (inrange(GSRymin, ay, by)) {
	   x = ax + (GSRymin-ay) * ((bx-ax) / (by-ay));
	   if (inrange(x, GSRxmin, GSRxmax)) {
		  *lx++ = x;
		  *ly++ = GSRymin;
		  intersect = TRUE;
	   }
    }
    
    /* does it intersect GSRymax edge */
    if (inrange(GSRymax, ay, by)) {
	   x = ax + (GSRymax-ay) * ((bx-ax) / (by-ay));
	   if (inrange(x, GSRxmin, GSRxmax)) {
		  *lx++ = x;
		  *ly++ = GSRymax;
		  intersect = TRUE;
	   }
    }

    /* does it intersect GSRxmin edge */
    if (inrange(GSRxmin, ax, bx)) {
	   y = ay + (GSRxmin-ax) * ((by-ay) / (bx-ax));
	   if (inrange(y, GSRymin, GSRymax)) {
		  *lx++ = GSRxmin;
		  *ly++ = y;
		  intersect = TRUE;
	   }
    }

    /* does it intersect GSRxmax edge */
    if (inrange(GSRxmax, ax, bx)) {
	   y = ay + (GSRxmax-ax) * ((by-ay) / (bx-ax));
	   if (inrange(y, GSRymin, GSRymax)) {
		  *lx++ = GSRxmax;
		  *ly++ = y;
		  intersect = TRUE;
	   }
    }

    if (intersect)
	 return(TRUE);

    /* It is possible for one or more of the [ab][xy] values to be -VERYLARGE.
	* If ax=bx=-VERYLARGE or ay=by=-VERYLARGE we have already returned
	* FALSE above.
	* Note that for ax=ay=-VERYLARGE or bx=by=-VERYLARGE we can do nothing.
	* Otherwise we fall through all the tests above. 
	* Handle them carefully here. As yet we have no way for them to be +VERYLARGE.
	*/
    if (ax == -VERYLARGE) {
	   if (ay != -VERYLARGE
		  && inrange(by, GSRymin, GSRymax) && inrange(GSRxmax, ax, bx)) {
		  *lx++ = GSRxmin;
		  *ly = by;
		  *lx++ = GSRxmax;
		  *ly = by;
		  intersect = TRUE;
	   }
    } else if (bx == -VERYLARGE) {
	   if (by != -VERYLARGE
		  && inrange(ay, GSRymin, GSRymax) && inrange(GSRxmax, ax, bx)) {
		  *lx++ = GSRxmin;
		  *ly = ay;
		  *lx++ = GSRxmax;
		  *ly = ay;
		  intersect = TRUE;
	   }
    } else if (ay == -VERYLARGE) {
	   /* note we know ax != -VERYLARGE */
	   if (inrange(bx, GSRxmin, GSRxmax) && inrange(GSRymax, ay, by)) {
		  *lx++ = bx;
		  *ly = GSRymin;
		  *lx++ = bx;
		  *ly = GSRymax;
		  intersect = TRUE;
	   }
    } else if (by == -VERYLARGE) {
	   /* note we know bx != -VERYLARGE */
	   if (inrange(ax, GSRxmin, GSRxmax) && inrange(GSRymax, ay, by)) {
		  *lx++ = ax;
		  *ly = GSRymin;
		  *lx++ = ax;
		  *ly = GSRymax;
		  intersect = TRUE;
	   }
    }

    return(intersect);
}

/* DRAW_YTICS: draw a regular tic series, y axis */
static draw_ytics(start, incr, end, ygrid, ytic_in)
double start, incr, end; /* tic series definition */
int ygrid, ytic_in;
		/* assume start < end, incr > 0 */
{
	double ticplace;
	int ltic;			/* for mini log tics */
	double lticplace;	/* for mini log tics */
	double ticmin, ticmax;	/* for checking if tic is almost inrange */

	if (end == VERYLARGE)            /* for user-def series */
		end = max(GSRymin,GSRymax);

	/* limit to right side of plot */
	end = min(end, max(GSRymin,GSRymax));

	/* to allow for rounding errors */
	ticmin = min(GSRymin,GSRymax) - SIGNIF*incr;
	ticmax = max(GSRymin,GSRymax) + SIGNIF*incr;
	end = end + SIGNIF*incr; 

	for (ticplace = start; ticplace <= end; ticplace +=incr) {
		if ( inrange(ticplace,ticmin,ticmax) )
			ytick(ticplace, GSRyformat, incr, 1.0, ygrid, ytic_in);
		if (GSRlogy && incr == 1.0) {
			/* add mini-ticks to log scale ticmarks */
			for (ltic = 2; ltic <= 9; ltic++) {
				lticplace = ticplace+log10((double)ltic);
				if ( inrange(lticplace,ticmin,ticmax) )
					ytick(lticplace, (char *)NULL, incr, 0.5, ygrid, ytic_in);
			}
		}
	}
}


/* DRAW_XTICS: draw a regular tic series, x axis */
static draw_xtics(start, incr, end, xgrid, xtic_in)
double start, incr, end; /* tic series definition */
int xgrid, xtic_in;
		/* assume start < end, incr > 0 */
{
	double ticplace;
	int ltic;			/* for mini log tics */
	double lticplace;	/* for mini log tics */
	double ticmin, ticmax;	/* for checking if tic is almost inrange */

	if (end == VERYLARGE)            /* for user-def series */
		end = max(GSRxmin,GSRxmax);

	/* limit to right side of plot */
	end = min(end, max(GSRxmin,GSRxmax));

	/* to allow for rounding errors */
	ticmin = min(GSRxmin,GSRxmax) - SIGNIF*incr;
	ticmax = max(GSRxmin,GSRxmax) + SIGNIF*incr;
	end = end + SIGNIF*incr; 

	for (ticplace = start; ticplace <= end; ticplace +=incr) {
		if ( inrange(ticplace,ticmin,ticmax) )
			xtick(ticplace, GSRxformat, incr, 1.0, xgrid, xtic_in);
		if (GSRlogx && incr == 1.0) {
			/* add mini-ticks to log scale ticmarks */
			for (ltic = 2; ltic <= 9; ltic++) {
				lticplace = ticplace+log10((double)ltic);
				if ( inrange(lticplace,ticmin,ticmax) )
					xtick(lticplace, (char *)NULL, incr, 0.5, xgrid, xtic_in);
			}
		}
	}
}

#ifdef NOTUSED_YET
/* DRAW_USER_YTICS: draw a user tic series, y axis */
static draw_user_ytics(list, ygrid, ytic_in)
struct ticmark *list;	/* list of tic marks */
int ygrid, ytic_in;
{
    double ticplace;
    double incr = (GSRymax - GSRymin) / 10;
    /* global GSRxmin, GSRxmax, xscale, GSRymin, GSRymax, yscale */

    while (list != NULL) {
	   ticplace = list->position;
	   if ( inrange(ticplace, GSRymin, GSRymax) 		/* in range */
		  || NearlyEqual(ticplace, GSRymin, incr)	/* == GSRymin */
		  || NearlyEqual(ticplace, GSRymax, incr))	/* == GSRymax */
		ytick(ticplace, list->label, incr, 1.0, ygrid, ytic_in);

	   list = list->next;
    }
}
#endif

#ifdef NOTUSED_YET
/* DRAW_USER_XTICS: draw a user tic series, x axis */
static draw_user_xtics(list, xgrid, xtic_in)
struct ticmark *list;	/* list of tic marks */
int xgrid, xtic_in;
{
    double ticplace;
    double incr = (GSRxmax - GSRxmin) / 10;
    /* global GSRxmin, GSRxmax, xscale, GSRymin, GSRymax, yscale */

    while (list != NULL) {
	   ticplace = list->position;
	   if ( inrange(ticplace, GSRxmin, GSRxmax) 		/* in range */
		  || NearlyEqual(ticplace, GSRxmin, incr)	/* == GSRxmin */
		  || NearlyEqual(ticplace, GSRxmax, incr))	/* == GSRxmax */
		xtick(ticplace, list->label, incr, 1.0, xgrid, xtic_in);

	   list = list->next;
    }
}
#endif

/* draw and label a y-axis ticmark */
static ytick(place, text, spacing, ticscale, ygrid, ytic_in)
double place;                   /* where on axis to put it */
char *text;                     /* optional text label */
double spacing;         /* something to use with checkzero */
float ticscale;         /* scale factor for tic mark (0..1] */
int ygrid, ytic_in;
{
    register struct termentry *t = &GTterm_tbl[GTterm];
    char ticlabel[101];
    int ticsize = (int)((t->h_tic) * ticscale);

	place = CheckZero(place,spacing); /* to fix rounding error near zero */
    if (ygrid) {
           (*t->linetype)(-1);  /* axis line type */
           (*t->move)(GSRxleft, gsr_map_y(place));
           (*t->vector)(GSRxright, gsr_map_y(place));
           (*t->linetype)(-2); /* border linetype */
    }
    if (ytic_in) {
           (*t->move)(GSRxleft, gsr_map_y(place));
           (*t->vector)(GSRxleft + ticsize, gsr_map_y(place));
           (*t->move)(GSRxright, gsr_map_y(place));
           (*t->vector)(GSRxright - ticsize, gsr_map_y(place));
    } else {
           (*t->move)(GSRxleft, gsr_map_y(place));
           (*t->vector)(GSRxleft - ticsize, gsr_map_y(place));
    }

    /* label the ticmark */
	if (text) {
	    (void) sprintf(ticlabel, text, CheckLog(GSRlogy, place));
	    if ((*t->justify_text)(RIGHT)) {
		   (*t->put_text)(GSRxleft-(t->h_char),
					   gsr_map_y(place), ticlabel);
	    } else {
		   (*t->put_text)(GSRxleft-(t->h_char)*(strlen(ticlabel)+1),
					   gsr_map_y(place), ticlabel);
	    }
	}
}

/* draw and label an x-axis ticmark */
static xtick(place, text, spacing, ticscale, xgrid, xtic_in)
double place;                   /* where on axis to put it */
char *text;                     /* optional text label */
double spacing;         /* something to use with checkzero */
float ticscale;         /* scale factor for tic mark (0..1] */
int xgrid, xtic_in;
{
    register struct termentry *t = &GTterm_tbl[GTterm];
    char ticlabel[101];
    int ticsize = (int)((t->v_tic) * ticscale);

	place = CheckZero(place,spacing); /* to fix rounding error near zero */
    if (xgrid) {
           (*t->linetype)(-1);  /* axis line type */
           (*t->move)(gsr_map_x(place), GSRybot);
           (*t->vector)(gsr_map_x(place), GSRytop);
           (*t->linetype)(-2); /* border linetype */
    }
    if (xtic_in) {
           (*t->move)(gsr_map_x(place), GSRybot);
           (*t->vector)(gsr_map_x(place), GSRybot + ticsize);
           (*t->move)(gsr_map_x(place), GSRytop);
           (*t->vector)(gsr_map_x(place), GSRytop - ticsize);
    } else {
           (*t->move)(gsr_map_x(place), GSRybot);
           (*t->vector)(gsr_map_x(place), GSRybot - ticsize);
    }

    /* label the ticmark */
    if (text) {
	   (void) sprintf(ticlabel, text, CheckLog(GSRlogx, place));
	   if ((*t->justify_text)(CENTRE)) {
		  (*t->put_text)(gsr_map_x(place),
					  GSRybot-(t->v_char), ticlabel);
	   } else {
		  (*t->put_text)(gsr_map_x(place)-(t->h_char)*strlen(ticlabel)/2,
					  GSRybot-(t->v_char), ticlabel);
	   }
    }
}

static int_error(str,t_num)
char str[];
int t_num;
/*
   Routine to print an error and die.  This is modified from the more
   complex gnuplot version which longjumps back to the gnuplot prompt.
   Note: NO_CARET is kept in some of the code above just to make these
   routines look more like the originals.  It isn't used at all here, 
   however.
*/
{
/* t_num is the token number, we don't have tokens... */
     fprintf(stderr,"%s\n\n", str);

#ifdef MAYBE_LATER
	longjmp(env, TRUE);	/* bail out to command line */
#else
	exit(1);
#endif
}
