/* Copyright (C) 1992 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gxcmap.c */
/* Color mapping and conversion for Ghostscript */
#include "gx.h"
#include "gserrors.h"
#include "gscspace.h"
#include "gxfrac.h"
#include "gxlum.h"
#include "gxcolor.h"
#include "gxdevice.h"
#include "gzcolor.h"
#include "gzstate.h"

/* Convert a frac to a gx_color_value. */
/* This is needed because map_rgb_color still uses gx_color_value. */
#define _cv_bits (sizeof(gx_color_value) * 8)
#define frac2cv(fr)\
  ( ((fr) << (_cv_bits - frac_bits)) +\
    ((fr) >> (frac_bits * 2 - _cv_bits)) )
#define cv2frac(cv) ((frac)((cv) >> (_cv_bits - frac_bits)))

/* Note: the color model conversion algorithms are taken from */
/* Rogers, Procedural Elements for Computer Graphics, pp. 401-403. */

/* ------ Conversion between HSB and RGB ------ */

/* Convert RGB to HSB. */
void
color_rgb_to_hsb(floatp r, floatp g, floatp b, float hsb[3])
{	frac red = float2frac(r), green = float2frac(g), blue = float2frac(b);
#define rhue hsb[0]
#define rsat hsb[1]
#define rbri hsb[2]
	if ( red == green && green == blue )
	   {	rhue = 0;	/* arbitrary */
		rsat = 0;
		rbri = r;	/* pick any one */
	   }
	else
	   {	/* Convert rgb to hsb */
		frac V, Temp;
		long diff, H;
		float fV;
		V = (red > green ? red : green);
		if ( blue > V ) V = blue;
		Temp = (red > green ? green : red);
		if ( blue < Temp ) Temp = blue;
		diff = V - Temp;
		if ( V == red )
		    H = (green - blue) * frac_1_long / diff;
		else if ( V == green )
		    H = (blue - red) * frac_1_long / diff + 2 * frac_1_long;
		else /* V == blue */
		    H = (red - green) * frac_1_long / diff + 4 * frac_1_long;
		if ( H < 0 ) H += 6 * frac_1_long;
		fV = frac2float(V);
		rhue = H / 6.0;
		rsat = diff / fV;
		rbri = fV;
	   }
#undef rhue
#undef rsat
#undef rbri
}

/* Convert HSB to RGB. */
void
color_hsb_to_rgb(floatp hue, floatp saturation, floatp brightness, float rgb[3])
{	if ( saturation == 0 )
	   {	rgb[0] = rgb[1] = rgb[2] = brightness;
	   }
	else
	   {	/* Convert hsb to rgb. */
		/* We rely on the fact that the product of two */
		/* fracs fits into an unsigned long. */
		floatp h6 = hue * 6;
		ulong V = float2frac(brightness);	/* force arithmetic to long */
		frac S = float2frac(saturation);
	        int I = (int)h6;
		ulong F = float2frac(h6 - I);		/* ditto */
		/* M = V*(1-S), N = V*(1-S*F), K = V*(1-S*(1-F)) = M-N+V */
		frac M = V * (frac_1_long - S) / frac_1_long;
		frac N = V * (frac_1_long - S * F / frac_1_long) / frac_1_long;
		frac K = M - N + V;
		frac R, G, B;
		switch ( I )
		   {
		default: R = V; G = K; B = M; break;
		case 1: R = N; G = V; B = M; break;
		case 2: R = M; G = V; B = K; break;
		case 3: R = M; G = N; B = V; break;
		case 4: R = K; G = M; B = V; break;
		case 5: R = V; G = M; B = N; break;
		   }
		rgb[0] = frac2float(R);
		rgb[1] = frac2float(G);
		rgb[2] = frac2float(B);
#ifdef DEBUG
if ( debug_c('c') )
{		dprintf7("[c]hsb(%g,%g,%g)->VSFI(%ld,%d,%ld,%d)->\n",
			 hue, saturation, brightness, V, S, F, I);
		dprintf6("   RGB(%d,%d,%d)->rgb(%g,%g,%g)\n",
			 R, G, B, rgb[0], rgb[1], rgb[2]);
}
#endif
	   }
}

/* ------ Color space conversion ------ */

/* Only 4 of the 6 conversions are implemented here; */
/* the other 2 (Gray to RGB/CMYK) are trivial. */
/* The CMYK to RGB algorithms specified by Adobe are, e.g., */
/*	R = 1.0 - min(1.0, C + K)	*/
/* but we get much better results with */
/*	R = (1.0 - C) * (1.0 - K)	*/

/* Convert RGB to Gray. */
frac
color_rgb_to_gray(frac r, frac g, frac b, const gs_state *pgs)
{	return (r * (unsigned long)lum_red_weight +
		g * (unsigned long)lum_green_weight +
		b * (unsigned long)lum_blue_weight +
		(lum_all_weights / 2))
	    / lum_all_weights;
}

/* Convert RGB to CMYK. */
/* Note that this involves black generation and undercolor removal. */
void
color_rgb_to_cmyk(frac r, frac g, frac b, const gs_state *pgs,
  frac cmyk[4])
{	frac c = frac_1 - r, m = frac_1 - g, y = frac_1 - b;
	frac k = (c < m ? min(c, y) : min(m, y));
	floatp fk = frac2float(k);
	/* The default UCR and BG functions are pretty arbitrary.... */
	float fucr =
		(pgs->undercolor_removal == NULL ? 0.0 :
		 (*pgs->undercolor_removal)(pgs, fk));
	float fbg =
		(pgs->black_generation == NULL ? 0.0 :
		 (*pgs->black_generation)(pgs, fk));
	signed_frac ucr =
	  (fucr < -1.0 ? -frac_1 : fucr > 1.0 ? frac_1 : float2frac(fucr));
	frac bg =
	  (fbg < 0.0 ? frac_0 : fbg > 1.0 ? frac_1 : float2frac(fbg));
	cmyk[0] = (c < ucr ? frac_0 : ucr < c - frac_1 ? frac_1 : c - ucr);
	cmyk[1] = (m < ucr ? frac_0 : ucr < m - frac_1 ? frac_1 : m - ucr);
	cmyk[2] = (y < ucr ? frac_0 : ucr < y - frac_1 ? frac_1 : y - ucr);
	cmyk[3] = bg;
}

/* Convert CMYK to Gray. */
frac
color_cmyk_to_gray(frac c, frac m, frac y, frac k, const gs_state *pgs)
{	frac not_gray = color_rgb_to_gray(c, m, y, pgs);
	return (not_gray > frac_1 - k ?		/* gray + k > 1.0 */
		frac_0 : frac_1 - (not_gray + k));
}

/* Convert CMYK to RGB. */
void
color_cmyk_to_rgb(frac c, frac m, frac y, frac k, const gs_state *pgs,
  frac rgb[3])
{	switch ( k )
	{
	case frac_0:
		rgb[0] = frac_1 - c;
		rgb[1] = frac_1 - m;
		rgb[2] = frac_1 - y;
		break;
	case frac_1:
		rgb[0] = rgb[1] = rgb[2] = frac_0;
		break;
	default:
	{	ulong not_k = frac_1 - k;
		/* Compute not_k * (frac_1 - v) / frac_1 efficiently. */
		ulong prod;
#define deduct_black(v)\
  (prod = (frac_1 - (v)) * not_k,\
   (prod + (prod >> frac_bits) + 1) >> frac_bits)
		rgb[0] = deduct_black(c);
		rgb[1] = deduct_black(m);
		rgb[2] = deduct_black(y);
	}
	}
}

/* ------ Device color rendering ------ */

private cmap_proc_gray(cmap_gray_halftoned);
private cmap_proc_gray(cmap_gray_direct);
private cmap_proc_gray(cmap_gray_to_rgb);
private cmap_proc_gray(cmap_gray_to_cmyk);
private cmap_proc_rgb(cmap_rgb_halftoned);
private cmap_proc_rgb(cmap_rgb_direct);
private cmap_proc_rgb(cmap_rgb_to_gray);
private cmap_proc_rgb(cmap_rgb_to_cmyk);
private cmap_proc_cmyk(cmap_cmyk_halftoned);
private cmap_proc_cmyk(cmap_cmyk_direct);
private cmap_proc_cmyk(cmap_cmyk_to_gray);
private cmap_proc_cmyk(cmap_cmyk_to_rgb);

private const gx_color_map_procs
	cmap_gray_few =
		{ cmap_gray_halftoned, cmap_rgb_to_gray, cmap_cmyk_to_gray },
	cmap_gray_many =
		{ cmap_gray_direct, cmap_rgb_to_gray, cmap_cmyk_to_gray },
	cmap_rgb_few =
		{ cmap_gray_to_rgb, cmap_rgb_halftoned, cmap_cmyk_to_rgb },
	cmap_rgb_many =
		{ cmap_gray_to_rgb, cmap_rgb_direct, cmap_cmyk_to_rgb },
	cmap_cmyk_few =
		{ cmap_gray_to_cmyk, cmap_rgb_to_cmyk, cmap_cmyk_halftoned },
	cmap_cmyk_many =
		{ cmap_gray_to_cmyk, cmap_rgb_to_cmyk, cmap_cmyk_direct };

const gx_color_map_procs *cmap_procs_default = &cmap_gray_many;

private const gx_color_map_procs _ds *cmap_few[] = {
	0, &cmap_gray_few, 0, &cmap_rgb_few, &cmap_cmyk_few
};

private const gx_color_map_procs _ds *cmap_many[] = {
	0, &cmap_gray_many, 0, &cmap_rgb_many, &cmap_cmyk_many
};

/* Set the color mapping procedures in the graphics state. */
void
gx_set_cmap_procs(gs_state *pgs)
{	gx_device *dev = gs_currentdevice(pgs);
	pgs->cmap_procs =
		((gx_device_has_color(dev) ? dev->color_info.max_rgb :
		  dev->color_info.max_gray) >= 31 ? cmap_many : cmap_few)
		 [dev->color_info.num_components];
}

/* Remap the color in the graphics state. */
int
gx_remap_color(gs_state *pgs)
{	const gs_color_space *pcs = pgs->color_space;
	return (*pcs->type->remap_color)(pgs->ccolor, pcs,
				pgs->dev_color, pgs);
}
/* Color remappers for the standard color spaces. */
#define unit_frac(v)\
  (ftemp = (v),\
   (ftemp < 0.0 ? frac_0 : ftemp > 1.0 ? frac_1 : float2frac(ftemp)))
int
gx_remap_DeviceGray(const gs_client_color *pc, const gs_color_space *pcs,
  gx_device_color *pdc, gs_state *pgs)
{	float ftemp;
	(*pgs->cmap_procs->map_gray)
		(unit_frac(pc->paint.values[0]),
		 pdc, pgs);
	return 0;
}
int
gx_remap_DeviceRGB(const gs_client_color *pc, const gs_color_space *pcs,
  gx_device_color *pdc, gs_state *pgs)
{	float ftemp;
	(*pgs->cmap_procs->map_rgb)
		(unit_frac(pc->paint.values[0]),
		 unit_frac(pc->paint.values[1]),
		 unit_frac(pc->paint.values[2]),
		 pdc, pgs);
	return 0;
}
int
gx_remap_DeviceCMYK(const gs_client_color *pc, const gs_color_space *pcs,
  gx_device_color *pdc, gs_state *pgs)
{	float ftemp;
	(*pgs->cmap_procs->map_cmyk)
		(unit_frac(pc->paint.values[0]),
		 unit_frac(pc->paint.values[1]),
		 unit_frac(pc->paint.values[2]),
		 unit_frac(pc->paint.values[3]),
		 pdc, pgs);
	return 0;
}
#undef unit_frac

/* Render Gray color. */

private void
cmap_gray_direct(frac gray, gx_device_color *pdc, const gs_state *pgs)
{	gx_device *dev = gs_currentdevice(pgs);
	frac mgray = gx_map_color_frac(pgs, gray, gray);
	gx_color_value cv_gray = frac2cv(mgray);
	gx_color_index color =
		(*dev->procs->map_rgb_color)(dev, cv_gray, cv_gray, cv_gray);
	if ( color == gx_no_color_index )
	{	gx_render_gray(mgray, pdc, pgs);
		return;
	}
	pdc->color1 = pdc->color2 = color;
	pdc->halftone_level = 0;
}

private void
cmap_gray_halftoned(frac gray, gx_device_color *pdc, const gs_state *pgs)
{	gx_render_gray(gx_map_color_frac(pgs, gray, gray), pdc, pgs);
}

private void
cmap_gray_to_rgb(frac gray, gx_device_color *pdc, const gs_state *pgs)
{	(*pgs->cmap_procs->map_rgb)(gray, gray, gray, pdc, pgs);
}

private void
cmap_gray_to_cmyk(frac gray, gx_device_color *pdc, const gs_state *pgs)
{	(*pgs->cmap_procs->map_cmyk)(frac_0, frac_0, frac_0, gray, pdc, pgs);
}

/* Render RGB color. */

private void
cmap_rgb_direct(frac r, frac g, frac b, gx_device_color *pdc,
  const gs_state *pgs)
{	if ( r == g && g == b )
	{	cmap_gray_direct(r, pdc, pgs);
		return;
	}
	{	gx_device *dev = gs_currentdevice(pgs);
		frac mred = gx_map_color_frac(pgs, r, red);
		gx_color_value cv_red = frac2cv(mred);
		frac mgreen = gx_map_color_frac(pgs, g, green);
		gx_color_value cv_green = frac2cv(mgreen);
		frac mblue = gx_map_color_frac(pgs, b, blue);
		gx_color_value cv_blue = frac2cv(mblue);
		gx_color_index color =
			(*dev->procs->map_rgb_color)(dev,
				cv_red, cv_green, cv_blue);
		if ( color == gx_no_color_index )
		{	gx_render_rgb(mred, mgreen, mblue, pdc, pgs);
			return;
		}
		pdc->color1 = pdc->color2 = color;
		pdc->halftone_level = 0;
	}
}

private void
cmap_rgb_halftoned(frac r, frac g, frac b, gx_device_color *pdc,
  const gs_state *pgs)
{	if ( r == g && g == b )
	{	cmap_gray_halftoned(r, pdc, pgs);	/* pick any one */
		return;
	}
	gx_render_rgb(gx_map_color_frac(pgs, r, red),
		      gx_map_color_frac(pgs, g, green),
		      gx_map_color_frac(pgs, b, blue),
		      pdc, pgs);
}

private void
cmap_rgb_to_gray(frac r, frac g, frac b, gx_device_color *pdc,
  const gs_state *pgs)
{	(*pgs->cmap_procs->map_gray)(color_rgb_to_gray(r, g, b, pgs), pdc, pgs);
}

private void
cmap_rgb_to_cmyk(frac r, frac g, frac b, gx_device_color *pdc,
  const gs_state *pgs)
{	frac cmyk[4];
	color_rgb_to_cmyk(r, g, b, pgs, cmyk);
	(*pgs->cmap_procs->map_cmyk)(cmyk[0], cmyk[1], cmyk[2], cmyk[3], pdc, pgs);
}

/* Render CMYK color. */

private void
cmap_cmyk_direct(frac c, frac m, frac y, frac k, gx_device_color *pdc,
  const gs_state *pgs)
{	if ( c == m && m == y )
	{	cmap_gray_direct(color_cmyk_to_gray(c, m, y, k, pgs), pdc, pgs);
		return;
	}
	{	gx_device *dev = gs_currentdevice(pgs);
		frac mcyan = gx_map_color_frac(pgs, c, red);
		gx_color_value cv_cyan = frac2cv(mcyan);
		frac mmagenta = gx_map_color_frac(pgs, m, green);
		gx_color_value cv_magenta = frac2cv(mmagenta);
		frac myellow = gx_map_color_frac(pgs, y, blue);
		gx_color_value cv_yellow = frac2cv(myellow);
		frac mblack = gx_map_color_frac(pgs, k, gray);
		gx_color_value cv_black = frac2cv(mblack);
		gx_color_index color =
			(*dev->procs->map_cmyk_color)(dev,
				cv_cyan, cv_magenta, cv_yellow, cv_black);
		if ( color == gx_no_color_index )
		{	cmap_cmyk_halftoned(c, m, y, k, pdc, pgs);
			return;
		}
		pdc->color1 = pdc->color2 = color;
		pdc->halftone_level = 0;
	}
}

private void
cmap_cmyk_halftoned(frac c, frac m, frac y, frac k, gx_device_color *pdc,
  const gs_state *pgs)
{	/* CMYK halftones are not implemented yet. */
	frac rgb[3];
	color_cmyk_to_rgb(c, m, y, k, pgs, rgb);
	cmap_rgb_halftoned(rgb[0], rgb[1], rgb[2], pdc, pgs);
}

private void
cmap_cmyk_to_gray(frac c, frac m, frac y, frac k, gx_device_color *pdc, const gs_state *pgs)
{	(*pgs->cmap_procs->map_gray)(color_cmyk_to_gray(c, m, y, k, pgs), pdc, pgs);
}

private void
cmap_cmyk_to_rgb(frac c, frac m, frac y, frac k, gx_device_color *pdc, const gs_state *pgs)
{	frac rgb[3];
	color_cmyk_to_rgb(c, m, y, k, pgs, rgb);
	(*pgs->cmap_procs->map_rgb)(rgb[0], rgb[1], rgb[2], pdc, pgs);
}

/* ------ Transfer function mapping ------ */

/* Map a color fraction through a transfer map. */
frac
gx_color_frac_map(frac cv, const frac *values)
{
#define cp_frac_bits (frac_bits - log2_transfer_map_size)
	int cmi = cv >> cp_frac_bits;
	frac mv = values[cmi];
	int rem, mdv;
	/* Interpolate between two adjacent values if needed. */
	rem = (cv & ((1 << cp_frac_bits) - 1)) - (cv >> (frac_bits - cp_frac_bits));
	if ( rem == 0 ) return mv;
	else if ( rem > 0 ) mdv = values[cmi + 1] - mv;
	else mdv = mv - values[cmi - 1];
#if arch_ints_are_short
	/* Only use long multiplication if necessary. */
	if ( mdv > 1 << (16 - cp_frac_bits) )
		return mv + (uint)(((ulong)rem * mdv) >> cp_frac_bits);
#endif
	return mv + ((rem * mdv) >> cp_frac_bits);
#undef cp_frac_bits
}

/* ------ Default device color mapping ------ */

gx_color_index
gx_default_map_rgb_color(gx_device *dev,
  gx_color_value r, gx_color_value g, gx_color_value b)
{	/* Map values >= 1/2 to 1, < 1/2 to 0. */
	return ((r | g | b) > gx_max_color_value / 2 ?
		(gx_color_index)1 : (gx_color_index)0);
}

int
gx_default_map_color_rgb(gx_device *dev, gx_color_index color,
  gx_color_value prgb[3])
{	/* Map 1 to max_value, 0 to 0. */
	prgb[0] = prgb[1] = prgb[2] = -(gx_color_value)color;
	return 0;
}

gx_color_index
gx_default_map_cmyk_color(gx_device *dev,
  gx_color_value c, gx_color_value m, gx_color_value y, gx_color_value k)
{	/* Convert to RGB */
	frac rgb[3];
	color_cmyk_to_rgb(cv2frac(c), cv2frac(m), cv2frac(y), cv2frac(k),
			  NULL, rgb);
	return (*dev->procs->map_rgb_color)(dev,
		frac2cv(rgb[0]), frac2cv(rgb[1]), frac2cv(rgb[2]));
}
