/* Copyright (C) 1989, 1990, 1991 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gdevsun.c */
/* SunView driver for Ghostscript */
#include <suntool/sunview.h>
#include <suntool/canvas.h>
#include <stdio.h>
/* Hack to get around the fact that something in the SunView library */
/* defines uint and ushort.... */
#  define uint _uint
#  define ushort _ushort
#include "gx.h"			/* for gx_bitmap; includes std.h */
#  undef _uint
#  undef _ushort
#include "gsmatrix.h"			/* needed for gxdevice.h */
#include "gxdevice.h"

#ifndef DEFAULT_DPI
#  define DEFAULT_DPI 75		/* Sun standard monitor */
#endif

#define COLOR 1				/* choose color or monochrome */

/* Procedures */
dev_proc_open_device(sun_open);
dev_proc_sync_output(sun_sync);
dev_proc_close_device(sun_close);
dev_proc_map_rgb_color(sun_map_rgb_color);
dev_proc_map_color_rgb(sun_map_color_rgb);
dev_proc_fill_rectangle(sun_fill_rectangle);
dev_proc_copy_mono(sun_copy_mono);
dev_proc_copy_color(sun_copy_color);
dev_proc_draw_line(sun_draw_line);

/* The device descriptor */
private gx_device_procs sun_procs = {
	sun_open,
	gx_default_get_initial_matrix,
	sun_sync,
	gx_default_output_page,
	sun_close,
	sun_map_rgb_color,
	sun_map_color_rgb,
	sun_fill_rectangle,
	gx_default_tile_rectangle,
	sun_copy_mono,
	sun_copy_color,
	sun_draw_line,
	gx_default_get_bits,
	gx_default_get_props,
	gx_default_put_props
};

/* Define the SunView device */
typedef struct gx_device_sun {
	gx_device_common;
	Frame frame;
	Canvas canvas;
	Pixwin *pw;
	struct mpr_data mpr;
	Pixrect	pr;
#if !arch_is_big_endian			/* need to swap bits & bytes */
#  define BUF_WIDTH_BYTES (((int)(8.5*DEFAULT_DPI)+15)/16*2)
	byte swap_buf[BUF_WIDTH_BYTES];
#endif
} gx_device_sun;

#if !arch_is_big_endian
/* Define a table for reversing bit order. */
static byte reverse_bits[256] = {
  0, 128, 64, 192, 32, 160, 96, 224, 16, 144, 80, 208, 48, 176, 112, 240,
  8, 136, 72, 200, 40, 168, 104, 232, 24, 152, 88, 216, 56, 184, 120, 248,
  4, 132, 68, 196, 36, 164, 100, 228, 20, 148, 84, 212, 52, 180, 116, 244,
  12, 140, 76, 204, 44, 172, 108, 236, 28, 156, 92, 220, 60, 188, 124, 252,
  2, 130, 66, 194, 34, 162, 98, 226, 18, 146, 82, 210, 50, 178, 114, 242,
  10, 138, 74, 202, 42, 170, 106, 234, 26, 154, 90, 218, 58, 186, 122, 250,
  6, 134, 70, 198, 38, 166, 102, 230, 22, 150, 86, 214, 54, 182, 118, 246,
  14, 142, 78, 206, 46, 174, 110, 238, 30, 158, 94, 222, 62, 190, 126, 254,
  1, 129, 65, 193, 33, 161, 97, 225, 17, 145, 81, 209, 49, 177, 113, 241,
  9, 137, 73, 201, 41, 169, 105, 233, 25, 153, 89, 217, 57, 185, 121, 249,
  5, 133, 69, 197, 37, 165, 101, 229, 21, 149, 85, 213, 53, 181, 117, 245,
  13, 141, 77, 205, 45, 173, 109, 237, 29, 157, 93, 221, 61, 189, 125, 253,
  3, 131, 67, 195, 35, 163, 99, 227, 19, 147, 83, 211, 51, 179, 115, 243,
  11, 139, 75, 203, 43, 171, 107, 235, 27, 155, 91, 219, 59, 187, 123, 251,
  7, 135, 71, 199, 39, 167, 103, 231, 23, 151, 87, 215, 55, 183, 119, 247,
  15, 143, 79, 207, 47, 175, 111, 239, 31, 159, 95, 223, 63, 191, 127, 255
};
#endif

/* The instance is public. */
#if COLOR				/* ****** */
gx_device_sun gs_sunview_device = {
	sizeof(gx_device_sun),
	&sun_procs,
	"sunview",
 	(int)(8.5*DEFAULT_DPI), (int)(11*DEFAULT_DPI),	/* x and y extent */
 	DEFAULT_DPI, DEFAULT_DPI,	/* x and y density */
		/* Following parameters are initialized for color */
	1,			/* has color */
	3,			/* max r-g-b value */
	8,			/* bits per color pixel */
 	0,			/* connection not initialized */
		/* End of monochrome/color parameters */
};
#else					/* ****** */
gx_device_sun gs_sunview_device = {
	sizeof(gx_device_sun),
	&sun_procs,
	"sunview",
 	(int)(8.5*DEFAULT_DPI), (int)(11*DEFAULT_DPI),	/* x and y extent */
 	DEFAULT_DPI, DEFAULT_DPI,	/* x and y density */
		/* Following parameters are initialized for monochrome */
	0,			/* has color */
	1,			/* max r-g-b value */
	1,			/* bits per color pixel */
 	0,			/* connection not initialized */
		/* End of monochrome/color parameters */
};
#endif					/* ****** */

/* Macro for casting gx_device argument */
#define xdev ((gx_device_sun *)dev)

/* Macro to validate arguments */
#define check_rect()\
	if ( w <= 0 || h <= 0 ) return 0;\
	if ( x < 0 || x > xdev->width - w || y < 0 || y > xdev->height - h )\
		return -1

/* Initialize the device. */
private Notify_value destroy_func();
int
sun_open(register gx_device *dev)
{
#ifdef gs_DEBUG
if ( gs_debug['X'] )
	{ extern int _Xdebug;
	  _Xdebug = 1;
	}
#endif
	if (xdev->frame == (Frame)0)
	    xdev->frame =
		window_create(NULL, FRAME, FRAME_LABEL, "ghostscript",
			WIN_HEIGHT, xdev->width + 20,
			WIN_HEIGHT, xdev->height + 40,	0);
	xdev->canvas = window_create(xdev->frame, CANVAS,
			CANVAS_AUTO_EXPAND,		FALSE,
			CANVAS_AUTO_SHRINK,		FALSE,
			CANVAS_WIDTH,			xdev->width,
			CANVAS_HEIGHT,			xdev->height,
			WIN_VERTICAL_SCROLLBAR,		scrollbar_create(0),
			WIN_HORIZONTAL_SCROLLBAR,	scrollbar_create(0),
		0);
	xdev->pw = canvas_pixwin(xdev->canvas);
	if ( xdev->has_color )
	   {	byte Red[64], Green[64], Blue[64];
		int j;
		for ( j = 0; j < 64; j++ )
		   {	Red[63 - j] = ((j & 0x30) >> 4) * (255 / 3);
			Green[63 - j] = ((j & 0xc) >> 2) * (255 / 3);
			Blue[63 - j] = ((j & 0x3)) * (255 / 3);
		   }
		pw_setcmsname(xdev->pw, "abc");
		pw_putcolormap(xdev->pw, 0, 64, Red, Green, Blue);
	   }
	window_set(xdev->frame, WIN_SHOW, TRUE, 0);
	/* Interpose a destroy function to keep Ghostscript from */
	/* getting confused if the user closes the window. */
	notify_interpose_destroy_func(xdev->frame, destroy_func);
	(void) notify_do_dispatch();
	(void) notify_dispatch();
	return 0;
}
/* Prevent the user from closing the window. */
private Notify_value
destroy_func(frame,status)
    Frame frame;
    Destroy_status status;
{	if ( status == DESTROY_CHECKING )
	   {	notify_veto_destroy(frame);
		return (NOTIFY_DONE);
	   }
	return (notify_next_destroy_func(frame, status));
}

/* Close the device. */
int
sun_close(gx_device *dev)
{	window_destroy(xdev->frame);
	xdev->frame = (Frame)0;
	xdev->canvas = (Canvas)0;
	xdev->pw = (Pixwin *)0;
	return 0;
}

/* Synchronize the display with the commands already given */
int
sun_sync(register gx_device *dev)
{	(void) notify_dispatch();
	return 0;
}

/* Map RGB to color number - simple 64 color system for 8 bit displays */
gx_color_index
sun_map_rgb_color(gx_device *dev, unsigned short red,
	unsigned short green, unsigned short blue)
{	if ( !dev->has_color)
		return gx_default_map_rgb_color(dev, red, green, blue);
	return (63 - ((red << 4) + (green << 2) + blue));
}

/* Map color number back to RGB values  - see sun_map_rgb_color(), above */
int
sun_map_color_rgb(gx_device *dev, gx_color_index color,
	unsigned short rgb[3])
{	if ( !dev->has_color )
		return gx_default_map_color_rgb(dev, color, rgb);
	color = 63 - color;
	rgb[0] = (unsigned short)((color & 0x30) >> 4);
	rgb[1] = (unsigned short)((color & 0x0c) >> 2);
	rgb[2] = (unsigned short)((color & 0x03));
	return 0;
}

/* Fill a rectangle with a color. */
int
sun_fill_rectangle(register gx_device *dev,
  int x, int y, int w, int h, gx_color_index color)
{	check_rect();
	pw_write(xdev->pw, x, y, w, h, PIX_SRC | PIX_COLOR((int)(color)),
		 (Pixrect *)0, 0, 0);
	(void) notify_dispatch();
	return 0;
}

/* Copy a monochrome bitmap. */
int
sun_copy_mono(register gx_device *dev,
  byte *base, int sourcex, int raster, gx_bitmap_id id,
  int x, int y, int w, int h, gx_color_index zero, gx_color_index one)
{
	register int i;
	int nbytes = h * raster;
	extern struct pixrectops mem_ops;
#if !arch_is_big_endian			/* need to swap bits & bytes */
#  define BUF_WIDTH_BYTES (((int)(8.5*DEFAULT_DPI)+15)/16*2)
	byte swap_buf[BUF_WIDTH_BYTES];
#endif

	check_rect();

	xdev->pr.pr_ops = &mem_ops;
	xdev->pr.pr_width = w + sourcex + 8;
	xdev->pr.pr_height = h;
	xdev->pr.pr_depth = 1;
	xdev->pr.pr_data = (caddr_t)&(xdev->mpr);
	xdev->mpr.md_linebytes = raster;
	xdev->mpr.md_image = (short *)((ulong)base & ~1);
#if !arch_is_big_endian
	/* Reverse the bit order in each byte. */
	for ( i = 0; i < nbytes; i++ ) base[i] = reverse_bits[base[i]];
#endif
	pw_batch_on(xdev->pw);
	if (one != gx_no_color_index)
	{	pw_stencil(xdev->pw, x, y, w, h,
			PIX_SRC | PIX_COLOR(one), &(xdev->pr),
			((int)base & 1) ? sourcex + 8 : sourcex, 0,
			(Pixrect *)0, 0, 0);
	}
	if (zero != gx_no_color_index)
	{	for (i = 0; i < nbytes; i++) base[i] = ~base[i];
		pw_stencil(xdev->pw, x, y, w, h,
			PIX_SRC | PIX_COLOR(zero), &(xdev->pr),
			((int)base & 1) ? sourcex + 8 : sourcex, 0,
			(Pixrect *)0, 0, 0);
		for (i = 0; i < nbytes; i++) base[i] = ~base[i];
	}
	pw_batch_off(xdev->pw);
#if !arch_is_big_endian
	/* Reverse the bits back again. */
	for ( i = 0; i < nbytes; i++ ) base[i] = reverse_bits[base[i]];
#endif
	(void) notify_dispatch();
	return 0;
}

/* Copy a color bitmap. */
int
sun_copy_color(register gx_device *dev,
  byte *base, int sourcex, int raster, gx_bitmap_id id,
  int x, int y, int w, int h)
{
	register int i;
	extern struct pixrectops mem_ops;

	if ( !dev->has_color )
		return sun_copy_mono(dev, base, sourcex, raster, id,
				     x, y, w, h,
				     (gx_color_index)0, (gx_color_index)1);

	check_rect();

	xdev->pr.pr_ops = &mem_ops;
	xdev->pr.pr_width = w + sourcex + 8;
	xdev->pr.pr_height = h;
	xdev->pr.pr_depth = 8;
	xdev->pr.pr_data = (caddr_t)&(xdev->mpr);
	xdev->mpr.md_linebytes = raster;
	xdev->mpr.md_image = (short *)((ulong)base & ~1);
	pw_write(xdev->pw, x, y, w, h,
		 PIX_SRC, &(xdev->pr),
		 (((int)base & 1) ? sourcex + 8 : sourcex), 0);
	(void) notify_dispatch();
	return 0;
}

/* Draw a line */
int
sun_draw_line(register gx_device *dev,
  int x0, int y0, int x1, int y1, gx_color_index color)
{	pw_vector(xdev->pw, x0, y0, x1, y1, PIX_SRC, color);
	(void) notify_dispatch();
	return 0;
}
