
/*

FUNCTION
        <<tan>>, <<tanf>>---tangent

INDEX
tan
INDEX
tanf

ANSI_SYNOPSIS
        #include <math.h>
        double tan(double <[x]>);
        float tanf(float <[x]>);

TRAD_SYNOPSIS
        #include <math.h>
        double tan(<[x]>)
        double <[x]>;

        float tanf(<[x]>)
        float <[x]>;


DESCRIPTION
<<tan>> computes the tangent of the argument <[x]>.  
Angles are specified in radians.  

<<tanf>> is identical, save that it takes and returns <<float>> values.

RETURNS
The tangent of <[x]> is returned. 

If <[x]> is NaN (not a number), <<errno>> is set to <<EDOM>>.

You can modify error handling for these functions using <<matherr>>.

PORTABILITY
<<tan>> is ANSI. <<tanf>> is an extension.
*/



#include "mathimpl.h"


#define P1 -0.13338350006421960681e+0
#define P2  0.34248878235890589960e-2
#define P3 -0.17861707342254426711e-4
#define Q0  1.0
#define Q1 -0.46671683339755294240e+0
#define Q2  0.25663832289440112864e-1
#define Q3 -0.31181531907010027307e-3
#define Q4  0.49819433993786512270e-6

#define P(f,g) (((P3*(g)+P2)*(g)+P1)*(g)*(f)+(f))
#define Q(g) ((((Q4*(g)+Q3)*(g)+Q2)*(g)+Q1)*(g)+Q0)

#define YMAX 6.74652e09

#ifdef DOUBLE
#define STAN_NAME "tan"
#define SCOTAN_NAME "cotan"
#else
#define STAN_NAME "tanf"
#define SCOTAN_NAME "cotanf"
#endif


static TYPE_RET
_DEFUN(tansub,(x, y, flag),
       TYPE_ARG x _AND
       TYPE_ARG y _AND
       int flag)
{
  TYPE f, g, xn;
  TYPE xnum, xden;
	
  if (y > YMAX) 
  {
    return __matherror(STAN_NAME, x, y, OVERFLOW, 0.0);
  }
  if (fabs(modf(x*0.63661977236758134308, &xn)) >= 0.5)
   xn += (x < 0.0) ? -1.0 : 1.0;
  f = modf(x, &g);
  f = ((g - xn*(3217.0/2048)) + f) - xn*-4.454455103380768678308e-6;
  if (fabs(f) < 2.33e-10) 
  {
    xnum = f;
    xden = 1.0;
  } else 
  {
    g = f*f;
    xnum = P(f,g);
    xden = Q(g);
  }
  flag |= ((int)xn & 1);
  switch (flag) 
  {
  case 1:			/* A: TAN_NAME, xn odd */
    xnum = -xnum;
  case 2:			/* B: COTAN_NAME, xn even */
    return xden/xnum;
		
  case 3:			/* C: COTAN_NAME, xn odd */
    xnum = -xnum;
  case 0:			/* D: TAN_NAME, xn even */
    return xnum/xden;
  }
  return 0.0;
}



TYPE_RET
_DEFUN(cotan,(x),
TYPE_ARG x)
{
  TYPE y;
  y = fabs(x);
  if (y < TYPE_MIN)
  {
    return __matherror(SCOTAN_NAME, x, 0.0, OVERFLOW, TYPE_MAX);
  }
  return tansub(x,y,2);
}

TYPE_RET
_DEFUN(tan,(x),
       TYPE_ARG x)
{
  return tansub(x, fabs(x), 0);
}

