/*
FUNCTION
        <<sin>>, <<sinf>>, <<cos>>, <<cosf>>---sine or cosine
INDEX
sin
INDEX
sinf
INDEX
cos
INDEX
cosf
ANSI_SYNOPSIS
        #include <math.h>
        double sin(double <[x]>);
        float  sinf(float <[x]>);
        double cos(double <[x]>);
        float cosf(float <[x]>);

TRAD_SYNOPSIS
        #include <math.h>
        double sin(<[x]>)
        double <[x]>;
        float  sinf(<[x]>)
        float <[x]>;

        double cos(<[x]>)
        double <[x]>;
        float cosf(<[x]>)
        float <[x]>;

DESCRIPTION
	<<sin>> and <<cos>> compute (respectively) the sine and cosine
	of the argument <[x]>.  Angles are specified in radians. 

	<<sinf>> and <<cosf>> are identical, save that they take and
	return <<float>> values. 


RETURNS
	The sine or cosine of <[x]> is returned. If <[x]> is NaN (not
	a number), <<errno>> is set to <<EDOM>>. 

	You can modify error handling for these functions using <<matherr>>.

PORTABILITY
	<<sin>> and <<cos>> are ANSI C. 
	<<sinf>> and <<cosf>> are extensions.

QUICKREF
	sin ansi pure
	sinf - pure
*/


									       
#include "mathimpl.h"									       
#ifdef FLOAT
#define SNAME "sinf"
#define COSNAME "cosf"
#else 
#define SNAME "sin"
#define COSNAME "cos"
#endif


#if defined(FLOAT) && __STDC__
#define P7	 0.27204790957888846175e-14F
#define P6	-0.76429178068910467734e-12F
#define P5	 0.16058936490371589114e-9F
#define P4	-0.25052106798274584544e-7F
#define P3	 0.27557319210152756119e-5F
#define P2	-0.19841269841201840457e-3F
#define P1	 0.83333333333331650314e-2F
#define P0	-0.16666666666666665052e0F
#else
#define P7	 0.27204790957888846175e-14
#define P6	-0.76429178068910467734e-12
#define P5	 0.16058936490371589114e-9
#define P4	-0.25052106798274584544e-7
#define P3	 0.27557319210152756119e-5
#define P2	-0.19841269841201840457e-3
#define P1	 0.83333333333331650314e-2
#define P0	-0.16666666666666665052e0
#endif


static TYPE_RET
_DEFUN(SIN_POLY,(a),
       TYPE_ARG a)
{
  TYPE z = a*a;
  TYPE r;

  r= ((z*(P0+z*(P1+z*(P2+z*(P3+z*(P4+z*(P5+z*(P6+z*P7)))))))));
  return a+a *r;
}



TYPE_RET
_DEFUN(sin,(x),
TYPE_ARG x)

{

  if (!finite(x))
  {
    return __matherror(SNAME, x, 0.0, DOMAIN, x);
  }
  
  if (x < 0.0) 
  {
    x = M_TWOPI - (fmod (- x, M_TWOPI));
  }
  
  else if (x >= M_PI * 2.0) 
  {
    x = fmod (x, M_TWOPI);
  }
  

  if (x < M_PI_2)
   return SIN_POLY (x);
  else if (x < M_PI) 
  {
    x = M_PI - x;
	  
    return SIN_POLY (x);
  }
	
  else if (x < 3.0 * M_PI_2) 
  {
    x = x - M_PI;
    return (- SIN_POLY (x));
  }
	
  else 
  {
    x = M_TWOPI - x;
	  
    return (- SIN_POLY (x));
  }
	
}




TYPE_RET
_DEFUN(cos, (x), 
       TYPE_ARG x)
{

  if (!finite(x)) 
  {
    return __matherror(COSNAME, x, 0.0, DOMAIN, x);
  }
  
  return sin(x+M_PI_2);
}

