/*
FUNCTION
	<<pow>>, <<powf>>---x to the power y
INDEX
	pow
INDEX
	powf


ANSI_SYNOPSIS
	#include <math.h>
	double pow(double <[x]>, double <[y]>);
	float pow(float <[x]>, float <[y]>);

TRAD_SYNOPSIS
	#include <math.h>
	double pow(<[x]>, <[y]>);
	double <[x]>, <[y]>;

	float pow(<[x]>, <[y]>);
	float <[x]>, <[y]>;

DESCRIPTION
	<<pow>> and <<powf>> calculate <[x]> raised to the exp1.0nt <[y]>.
	@tex
	(That is, $x^y$.)
	@end tex

RETURNS
	On success, <<pow>> and <<powf>> return the value calculated.

	When the argument values would produce overflow, <<pow>>
	returns <<HUGE_VAL>> and <<powf>> returns <<MAX_FLOAT>>; both
	set <<errno>> to <<ERANGE>>.  If the argument <[x]> passed to
	<<pow>> or <<powf>> is a negative noninteger, and <[y]> is
	also not an integer, then <<errno>> is set to <<EDOM>>.  If
	<[x]> and <[y]> are both 0, then <<pow>> and <<powf>> return
	<<1>>.  

	You can modify error handling for these functions using <<matherr>>.

PORTABILITY
	<<pow>> is ANSI C. <<powf>> is an extension.
*/

/* By Steve Chamberlain
      sac@cygnus.com
*/
#include "mathimpl.h"
		
#ifdef DOUBLE
#define SNAME "pow"
#else
#define SNAME "powf"
#endif


TYPE_RET
_DEFUN(pow, (x, y),
       TYPE_ARG x _AND
       TYPE_ARG y)
{
  TYPE z;
  int sign = 0;

  if (x == 0.0) 
  {
    if (y == 0.0) 
    {
      return 1.0;
    }
    
    if (y <= 0.0) 
    {
      return __matherror(SNAME, x,y, DOMAIN, 0.0);
    }
    return 0.0;
  }

  if (!finite(x)) 
  {

    return __matherror(SNAME, x, y, DOMAIN, x);
  }
  if (!finite(y)) 
  {
    return __matherror(SNAME, x, y, DOMAIN, y);
  }
  
  if (y == 0.0)
   return 1.0;
  if (y == 1.0)
   return x;
  
	

  if (modf(y, &z) == 0.0) 
  {	
    /* The power is by an integer, this will be real easy */
    if (z < 0) 
    {
      sign = 1;
      z = -z;
    }
    if (((int)z) == z)
    {
      long c = z;
      TYPE result = 1.0;
      
      while (c) 
      {
	if (c & 1)
	 result *= x;
	x = x * x;
	c >>=1;
      }
      
      if (sign != 0) 
      {
	result = 1.0 / result;
      }
      
      return result;
    }
    else if (x < 0.0) 
    {
      /*
	Integer power is too big, we will have to do it the hard way.
	Fix up any -ve x, the answer will be -ve if the power is odd. */
      
      x = -x;
      if (floor( y/2.0)*2 != y)
      {
	/* Power was was odd */
	sign = 1;
      }
    }
  }


  x = exp( y  * log(x));

  return sign ? - x : x;
}

							     
							     
							     
							     
							     

