
/*
 * Code for Hitachi H8/300.
 * Copyright (C) 1992 Free Software Foundation, Inc.
 * 
 * This file is part of GNU CC.
 * 
 * GNU CC is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2, or (at your option) any later version.
 * 
 * GNU CC is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU CC; see the file COPYING.  If not, write to the Free Software
 * Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Contributed by Steve Chamberlain
   		  sac@cygnus.com

 */

#include <stdio.h>


#include "config.h"
#include "rtl.h"
#include "regs.h"
#include "hard-reg-set.h"
#include "real.h"
#include "flags.h"
#include "insn-config.h"
#include "conditions.h"
#include "insn-flags.h"
#include "output.h"
#include "insn-attr.h"
#include "tree.h"
char *strchr();

/* 

The H8/300 is an 8 bit micro controller, we support floating point by
inventing a floating point instruction set and generating calls to it.
The FP machine has N registers, and instruction encoding which looks like:

 opcode
 amode0.

amodes
NNNN NNNN NNNN NNN0			addr = N
XXXX XXXX RRRR 0001   			addr = (@R+X)
0000 0000 RRRR 0011 XXXX XXXX XXXX XXXX addr = (@R+X)
0000 0000 RRRR 0101       		addr = (-@R)
0000 0000 RRRR 0111                     addr = REG
*/


/* H8/300 Calling convention:

   Regs R2 and R3 are preserved over a call. Doubles are returned as
   structs, structs are returned through a pointer to struct as the
   1st secret argument to a function.

	the quickest return code for a framefull function looks like:

	   	mov.w	fp,sp
		pop	fp
		pop	r3	
		pop	r2
		rts

	so we build the stack thus:

		push	r2	
		push	r3
		push	fp
		mov.w	#-frame_size,rn
		add.w	rn,sp

	so  after the prologue code the stack looks like:

		argn
		..
	pap->	arg0
		prev pc
		r2
		r3
	fp->	fp
		auton
	sp->	auto0

	It is very hard to get GCC to take into account how many of
	the saved regs (r2,r3) are saved in the frame, to vary the fp
	offset.  So, in framefull functions, we allways save them
	both.


	for a function without frame, the code looks like:

		argn
		..
	pap->	arg0
		prev pc
		r2		; when necessary
	pfp->	r3
		auton
	sp->	auto0


	push	r2	
	push 	r3
	adjust  framefp+extra

	the extra is necess

*/

/* Set when we've seen a floating point operand in a function */

static int had_float_operand;
static int had_any_float_operand;


int             pragma_interrupt;
int             pragma_saveall;
char           *pragma_context;

int fp_arg_offset ;


char           *
rev_cond_name(op)
    rtx             op;
{
    switch (GET_CODE(op)) {
    case EQ:
	return "neq";
    case NE:
	return "eql";
    case LT:
	return "geq";
    case LE:
	return "gtr";
    case GT:
	return "leq";
    case GE:
	return "lss";
    case LTU:
	return "gequ";
    case LEU:
	return "gtru";
    case GTU:
	return "lequ";
    case GEU:
	return "lssu";

    default:
	abort();
    }
}
extern int      frame_pointer_needed;


typedef struct {
char qi,hi,si,sf,df;
char *name_small;
char *name_big;
int even;
int fp;
int class;
char *name_float;

} reginfo_type;
#define FP_REGS GENERAL_REGS

/* If I keep them in h/l order, I'm fucked with extendqihi */
/* Keep the registers in big endian order */
reginfo_type regs_stuff[] =
{
  1,0,0,0,0,    "r0l",	"r0",	0,0,GENERAL_REGS, "bad",
  1,0,0,0,0,    "r1l",	"r1",	0,0,GENERAL_REGS, "bad",
  1,0,0,0,0,    "r2l",	"r2",	0,0,GENERAL_REGS, "bad",
  1,0,0,0,0,    "r3l",	"r3",	0,0,GENERAL_REGS, "bad",
  1,0,0,0,0,    "r4l",	"r4",	0,0,GENERAL_REGS, "bad",
  1,0,0,0,0,    "r5l",	"r5",	0,0,GENERAL_REGS, "bad",
  1,0,0,0,0,    "r6l",	"r6",	0,0,GENERAL_REGS, "bad",
  0,0,0,0,0,    "r7l",	"r7",	0,0,GENERAL_REGS, "bad",

  0,0,0,1,1,"#__fpreg8","#__fpreg8", 1,1,FP_REGS,"fake1",
  0,0,0,1,1,"#__fpreg9","#__fpreg9", 1,1,FP_REGS,"fake1",
  0,0,0,1,1,"#__fpreg10","#__fpreg10", 1,1,FP_REGS,"fake1",
  0,0,0,1,1,"#__fpreg11","#__fpreg11", 1,1,FP_REGS,"fake1",
  0,0,0,1,1,"#__fpreg12","#__fpreg12", 1,1,FP_REGS,"fake1",
  0,0,0,1,1,"#__fpreg13","#__fpreg13", 1,1,FP_REGS,"fake1",
  0,0,0,1,1,"#__fpreg14","#__fpreg14", 1,1,FP_REGS,"fake1",
  0,0,0,1,1,"#__fpreg15","#__fpreg15", 1,1,FP_REGS,"fake1",
};



	  



function_arg_pass_by_reference(cum, mode, type, named)
int cum;
enum machine_mode mode;
tree type;
int named;
  {

return 0;


  }


static void dosize(file,op,size)
FILE *file;
char *op;
unsigned int size;
{
  while (size) 
  {
    switch (size) 
    {
    case 1:
    case 2:
      fprintf(file, "\t%ss\t#%d,sp\n",op,size);      
      size = 0;
      break;
    case 3:
    case 4:
      fprintf(file, "\t%ss\t#%d,sp\n",op, 2);      	
      size -=2;
      break;
    default:	
      fprintf(file, "\tmov.w\t#%d,r5\n\t%s.w\tr5,sp\n", size, op);
      size = 0;
    }

  }
}
#define REG_USED(regno) \
((( pragma_interrupt  || pragma_saveall  || (regs_ever_live[regno] && !call_used_regs[regno])))?2:0)


/* H8/300 Calling convention:

	the quickest return code for a framefull function looks like:

	   	mov.w	fp,sp
		pop	fp
		pop	r3	
		pop	r2
		rts

	so we build the stack thus:

		push	r2	
		push	r3
		push	fp
		mov.w	#-frame_size,rn
		add.w	rn,sp

	so  after the prologue code the stack looks like:

		argn
		..
	pap->	arg0
		prev pc
		r2
		r3
	fp->	fp
		auton
	sp->	auto0

	It is very hard to get GCC to take into account how many of
	the saved regs (r2,r3) are saved in the frame, to vary the fp
	offset.  So, in framefull functions, we allways save them
	both.


	for a function without frame, the code looks like:

		argn
		..
	pap->	arg0
		prev pc
		r2		; when necessary
	pfp->	r3
		auton
	sp->	auto0


	push	r2	
	push 	r3
	adjust  framefp+extra

	the extra is necess

*/
int prev_linenum;

void
function_prologue(file, size)
    FILE           *file;
    int             size;
{
  register int    regno;
  register int    mask = 0;
  extern char     call_used_regs[];
  int             i;

  int fsize = (size+ 1)&-2;
  prev_linenum = 0;
  
  if (frame_pointer_needed) 
  {
	
    fprintf(file, "\tpush\tr2\n");
    fprintf(file, "\tpush\tr3\n");
    fprintf(file, "\tpush\tr6\n");
    fprintf(file, "\tmov.w\tr7,r6\n");
    dosize(file,"sub",fsize);    
  }
  else {

      for (regno = 0; regno < 6 ; regno ++) 
      {
	if (REG_USED(regno)) 
	{
	  fprintf(file, "\tpush\t%s\n", regs_stuff[regno].name_big);
	}
      }
       dosize(file,"sub",fsize);    

    }
}

void
function_epilogue(file, size)
    FILE           *file;
    int             size;
{
  register int    regno;
  register int    mask = 0;
  extern char     call_used_regs[];
  int fsize = (size +1) & -2;
  int nregs;
  int offset;  
  rtx insn = get_last_insn ();
  /* If the last insn was a BARRIER, we don't have to write any code.  */
  if (GET_CODE (insn) == NOTE)
   insn = prev_nonnote_insn (insn);
  if (insn && GET_CODE (insn) == BARRIER)
   return;
  nregs = 0;


  if (frame_pointer_needed) 
  {
    fprintf(file, "\tmov.w\tr6,r7\n");
    fprintf(file, "\tpop\tr6\n");
    fprintf(file, "\tpop\tr3\n");
    fprintf(file, "\tpop\tr2\n");
  }
  else  {
       dosize(file, "add", fsize);   
      for (regno = 5; regno >= 0; regno--) 
      {
	if (REG_USED(regno))
	 fprintf(file, "\tpop\t%s\n", regs_stuff[regno].name_big);
      }



    }
  if (pragma_interrupt) 
  {
    fprintf(file, "\trte\n");
  }
  else 
  {
    fprintf(file, "\trts\n");
  }
  pragma_interrupt = 0;
  pragma_saveall = 0;
  had_float_operand = 0;

  
}




int             part;

/*
 * Note that this contains a kludge that knows that the only reason we have
 * an address (plus (label_ref...) (reg...)) is in the insn before a
 * tablejump, and we know that m68k.md generates a label LInnn: on such an
 * insn.
 */


asm_file_start(file)
    FILE           *file;
{
    extern int      optimize;
    extern char *main_input_filename;
    
    fprintf(file, ";\tGCC Hitachi H8/300\n");
    fprintf(file, ";\tCygnus Support\n");
    if (optimize)
     fprintf(file,"; -O%d\n", optimize);
    fprintf(file, "\n\n");
    output_file_directive(file, main_input_filename);
    

}


emit_ashift(operands, func1)
rtx             operands[];
rtx(*func1) ();
{
  rtx             copy_to_mode_reg();
  rtx             gen_label_rtx();
  rtx             convert_to_mode();
  rtx             dst = operands[0];
  rtx             src = operands[1];
  rtx             count = operands[2];
  emit(gen_move_insn(dst, src));
  if (GET_CODE(count) == CONST_INT) {
      unsigned int    x = INTVAL(count);

      if (x < 8) {
	  while (x--) {
	      emit_insn(func1(dst, dst, const1_rtx));
	    }
	} else {
	    rtx             counter =
	     copy_to_mode_reg(QImode, convert_to_mode(QImode, count, 0));
	    rtx             loop_label = gen_label_rtx();
	    emit(loop_label);
	    emit(func1(dst, dst, const1_rtx));
	    emit(gen_subqi3(counter, counter, const1_rtx));
	    emit(gen_cmpqi(counter, const0_rtx));
	    emit(gen_bne(loop_label));
	  }
    } else {
	/* Copy from int and test at top of loop */
	rtx             counter =
	 copy_to_mode_reg(QImode, convert_to_mode(QImode, count, 0));
	rtx             loop_top = gen_label_rtx();
	rtx             loop_bottom = gen_label_rtx();
	emit(loop_top);
	emit(gen_subqi3(counter, counter, const1_rtx));
	emit(gen_cmpqi(counter, const0_rtx));
	emit(gen_bne(loop_bottom));
	emit(func1(dst, dst, const1_rtx));
	emit(gen_jump(loop_top));
	emit_barrier();
	emit(loop_bottom);
      }
  emit(gen_move_insn(dst, dst));
}

char           *
print_cond(file, op)
    FILE           *file;
    int             op;
{
    switch (op) {
    case NE:
	return "ne";
    case EQ:
	return "eq";
    case GE:
	return "ge";
    case GT:
	return "gt";
    case LE:
	return "le";
    case LT:
	return "lt";
    case GEU:
	return "hs";
    case GTU:
	return "hi";
    case LEU:
	return "ls";
    case LTU:
	return "lo";
    }
}


incok(x, strict,mode)
rtx             x;
int strict;
enum machine_mode mode;
{
  
  if (GET_CODE(x) == PRE_DEC || GET_CODE(x) == POST_INC) 
  {
      
    if (GET_CODE(XEXP(x, 0)) == REG) {
	return (strict ? REG_OK_FOR_BASE_P_STRICT (XEXP (x, 0))
		: REG_OK_FOR_BASE_P (XEXP (x, 0)));
      }
  }
  return 0;
}

legitimate_address(mode, x, strict)
    enum machine_mode mode;
    rtx             x;
    int strict;
{
  if (REG_P(x)
      && (strict ? REG_OK_FOR_BASE_P_STRICT(x) : REG_OK_FOR_BASE_P(x)))
   return 1;


  
  if (CONSTANT_ADDRESS_P(x))
   return 1;
  if (incok(x,strict,mode))
   return 1;
  if (GET_CODE(x) == PLUS) {
      rtx             p1 = XEXP(x, 0);
      rtx             p2 = XEXP(x, 1);
      if (CONSTANT_ADDRESS_P(p1) && REG_P(p2)
	  && (strict ? REG_OK_FOR_BASE_P_STRICT(p2) : REG_OK_FOR_BASE_P(p2)))
       return 1;
      if (CONSTANT_ADDRESS_P(p2) && REG_P(p1)
	  && (strict ? REG_OK_FOR_BASE_P_STRICT(p1) : REG_OK_FOR_BASE_P(p1)))
       return 1;
      return 0;
    }
  return 0;
}

asm_output_external(file, decl, name)
    FILE           *file;
    char           *name;
{
    fprintf(file, "\t.global\t");
    assemble_name(file, name);
    fprintf(file, "\n");

}





asm_file_end(file)
    FILE           *file;
{
  int             i;

  if (had_any_float_operand) 
   fprintf(file,"\t.word	__use_fp\n");
   fprintf(file, "\t.end\n");
}

char           *string[] =
{"__fpreg0", "__fpreg1", "__fpreg2", "__fpreg3", "__fpreg4", "__fpreg5", "__fpreg6"};
char           *malloc();



int
power_of_two_operand(op, mode)
    rtx             op;
    enum machine_mode mode;
{
    if (GET_CODE(op) == CONST_INT) {
	unsigned char   value = INTVAL(op);
	return (value != 0 && (value & (value - 1)) == 0);
    }
    return (FALSE);
}				/* power_of_two_operand */




int 
potl8(value)
{
switch (value) {
  case 1:
  case 2:
  case 4:
  case 8:
  case 16:
  case 32:
  case 64:
  case 128:
    return 1;
    
  }

return 0;

}

int potg8(value)
{
  switch (value) 
  {
  case 256:
  case 512:
  case 1024:
  case 2048:
  case 4096:
  case 8192:
  case 16384:
  case 32768:
    return 1;
    
  }
  
  return 0;
  
}

    
    
  
int neg_power_of_two_operand(op, mode)
    rtx             op;
    enum machine_mode mode;
{
    if (GET_CODE(op) == CONST_INT) {
	unsigned char   value = ~INTVAL(op);
	return (value != 0 && (value & (value - 1)) == 0);
    }
    return (FALSE);



}

fix_operand_dst(op, mode)
rtx op;
enum machine_mode mode;
{
  if (GET_CODE(op) == MEM
      && (GET_CODE(XEXP(op,0)) == POST_INC
	  || GET_CODE(XEXP(op,0)) == PRE_DEC))
    return 0;
  return general_operand(op,mode);
}

int 
float_operand_src(op, mode)
    rtx             op;
    enum machine_mode mode;
{
  if (GET_CODE(op) == MEM) 
  {
    
    rtx x = XEXP(op,0);
    if (GET_CODE(x) == PRE_DEC 
	|| GET_CODE(x) == POST_DEC 
	|| GET_CODE(x) == PRE_DEC 
	|| GET_CODE(x) == PRE_INC) { 
	return 0;
		    
      }
  }
  
  return float_operand(op, mode);
}

int 
float_move_operand_src(op, mode)
    rtx             op;
    enum machine_mode mode;
{
  if (GET_CODE(op) == MEM) 
  {
    
    rtx x = XEXP(op,0);
    if (GET_CODE(x) == PRE_DEC 
	|| GET_CODE(x) == POST_DEC 
	|| GET_CODE(x) == PRE_INC) { 
	return 0;
		    
      }
  }
  
  return float_operand(op, mode);
}

int 
float_operand(op, mode)
    rtx             op;
    enum machine_mode mode;
{
  had_float_operand = 1;
  had_any_float_operand = 1;
  
  if (GET_MODE(op) != mode)
   return 0;
  if (GET_CODE(op) == REG)
   return 1;
  if (GET_CODE(op) == SUBREG)
   return 1;
    
  if (GET_CODE(op) == MEM) {
      rtx             inside = XEXP(op, 0);
      /*      if (!incok(op,1)) return 0;*/
      
      return legitimate_address(mode, inside,0);

    }
  return 0;
}

int 
float_operand_dst(op, mode)
    rtx             op;
    enum machine_mode mode;
{
  if (GET_CODE(op) == MEM) 
  {
    rtx x = XEXP(op,0);
    
    if (GET_CODE(x) == POST_DEC 
	|| GET_CODE(x) == POST_INC 
	|| GET_CODE(x) == PRE_DEC 
	|| GET_CODE(x)	== PRE_INC) 
    { 
      return 0;
		    
    }
  }
  
  
  return float_operand(op, mode);

}

int 
float_move_operand_dst(op, mode)
    rtx             op;
    enum machine_mode mode;
{
  if (GET_CODE(op) == MEM) 
  {
    rtx x = XEXP(op,0);
    
    if (GET_CODE(x) == POST_DEC 
	|| GET_CODE(x) == POST_INC 
	|| GET_CODE(x)	== PRE_INC) 
    { 
      return 0;
		    
    }
  }
  
  
  return float_operand(op, mode);

}

int reg_ok_for_index(reg)
rtx reg;
{
  extern int reload_in_progress;
  if (REG_P(reg) && REGNO(reg) < 7) return 1;
  /*
  if (REG_P(reg)  && (reload_in_progress? REG_OK_FOR_BASE_P_STRICT(reg) : REG_OK_FOR_BASE_P(reg)))
   return 1;
  */
  return 0;
  
}

/* Operand valid for a bit instruction,
   either (reg:HI n), (reg:QI n) or (mem:QI (reg))
 */
int 
bit_operand(op)
    rtx             op;
{
  extern int strict;
  
  if (REG_P(op) && REGNO(op) < 7) return 1;

  if (GET_CODE(op) == MEM && GET_MODE(op) == QImode) 
  {
    rtx inside = XEXP(op,0);

    if (REG_P(inside)) {
	if (reload_completed) 
	 if (REGNO(inside) > 7) return 0;
	return 1;
      }
    
    
    
  }

  return 0;
}

/*
 * Letters in the range `Q' through `U' in a register constraint string may
 * be defined in a machine-dependent fashion to stand for arbitrary operand
 * types.
 */
extra_constraint(op, c)
    rtx             op;
    char            c;
{
  switch (c) {
    case 'T':
      /* Somethig real easy to get at */
      if (GET_CODE(op) == MEM ) 
      {
	op = XEXP(op, 0);
	if (GET_CODE(op) == CONST) return 1;
	if (GET_CODE(op) == SYMBOL_REF) return 1;
      }
      break;
	
	
      /* A P is valid destination for a bclr instruction */
    case 'S':
      /* True only for a reg which isn't  stack pointer */
      if (GET_CODE(op) == REG && REGNO(op) <STACK_POINTER_REGNUM+1)
       return 1;
      return 0;
	
	
    case 'U':
      /*
       * Rn @Rn @aa
       */
      if (GET_CODE(op) == REG)
       return 1;
      if (GET_CODE(op) == MEM) {
	  rtx             inside = XEXP(op, 0);
	  if (GET_CODE(inside) == REG)
	   return 1;
	}
      /* ALso check for the @aa:8 bit */
    }
  return 0;
}



optimization_options(opt)
    int             opt;
{
    extern int      flag_omit_frame_pointer;
    extern int      flag_cse_follow_jumps;
    extern int flag_float_store;
    
    extern int      optimize;
    extern int      obey_regdecls;
    flag_omit_frame_pointer = (optimize >0);
    flag_cse_follow_jumps = (optimize >= 1);
/*    flag_float_store = 1;*/
    
    if (optimize == 0) {
/*	obey_regdecls = 0;*/
/*	optimize = 1;*/
    }
}

asm_output_local(file, name, size, rounded)
    FILE           *file;
    char           *name;
    int             size;
    int             rounded;
{
    assemble_name(file, name);
    fprintf(file, ":\t.res.b\t%d\n", rounded);
}




int 
return_in_memory(exp)
    tree            exp;
{
    return GET_MODE_SIZE(TYPE_MODE(exp)) > 4;
}


handle_pragma(file)
    FILE           *file;
{
  int             c;
  char pbuf[20];
  int psize = 0;
    
  c = getc(file);
  while (c == ' ' || c == '\t')
   c = getc(file);

  if (c == '\n' || c == EOF) {
      return c;
    }
  /* The only pragmas we understand are interrupt and saveall */
  while (psize < sizeof(pbuf) -1
	 && c != '\n'
	 && c != EOF) 
  {
    pbuf[psize++] = c;
    c = getc(file);
  }
  pbuf[psize] = 0;

  if (strcmp(pbuf,"interrupt") == 0) 
  {
    
   pragma_interrupt = 1;
}
    if (strcmp(pbuf,"saveall") == 0) {
 	pragma_saveall = 1;
    }
    return c;

}

int             better = 0;


void 
storeem(o, mode)
    rtx             o;
    enum machine_mode mode;
{
    if (mode == HImode) {
	if (better == 0)
	    abort();
	better = 0;
	emit(gen_movhi(o, gen_rtx(REG, HImode, 0)));
    }
}

char           *
fakemove(operands)
    rtx            *operands;
{
    rtx             op[2];
    op[1] = operands[0];
    if (GET_CODE(operands[1]) == MEM) {
	rtx             sub = XEXP(operands[1], 0);
	if (GET_CODE(sub) == PLUS) {
	    rtx             sub1 = XEXP(sub, 0);
	    rtx             sub2 = XEXP(sub, 1);
	    if (GET_CODE(sub1) == REG &&
		GET_CODE(sub2) == CONST_INT) {

		op[0] = sub2;

		output_asm_insn("mov.w	%0,%1; fake", op);
		op[0] = sub1;
		output_asm_insn("add.w	%0,%1; fake", op);
		return "";
	    }
	}
	if (GET_CODE(sub) == REG) {
	    op[0] = sub;
	    output_asm_insn("mov.w %0,%1; fake ", op);
	    return "";
	}
    }
    if (GET_CODE(operands[1]) == REG)
	return "fail";
}

rtx 
function_value(valtype, func)
    tree            valtype;
    tree            func;
{
if (TYPE_MODE(valtype) == DFmode) 
      return gen_rtx(REG, DFmode,8);
else      return gen_rtx(REG, TYPE_MODE(valtype),0);


}


char            initial_fixed_regs[] = FIXED_REGISTERS;
char           *
initial_fixed_regs_func()
{
    return initial_fixed_regs;
}

char            initial_call_used_regs[] = CALL_USED_REGISTERS;
char           *
initial_call_used_regs_func()
{
    return initial_call_used_regs;
}
extern FILE    *asm_out_file;
char           *
output_float_op3(operands, name)
    rtx            *operands;
    char           *name;
{

    fprintf(asm_out_file, " call %s ", name);
    output_asm_insn("; floatop %0 %1 %2 ", operands);
}

#define FP_CODE	1
#define IR_CODE 3
#define IRI_CODE 5
#define IRD_CODE 7
#define NSHIFT 3


const_costs(r, c)
rtx r;
int c;
{
  switch (c) {

  case CONST_INT:						
    switch (INTVAL(r)) {
    case 0:
    case 1:
    case 2:
    case -1:
    case -2:
      return 1;
    default:
     return 1;

    }
  case CONST:							
  case LABEL_REF:						
  case SYMBOL_REF:						

    return 3;							
  case CONST_DOUBLE:						
    return 20;
  default:
    return 0;
  }
}

general_operand_src(op, mode)
rtx op;
enum machine_mode mode;
  {
if (reload_in_progress)
    if (GET_CODE(op) == MEM &&
	GET_CODE(XEXP(op,0)) == PRE_DEC) return 0;
    return general_operand(op,mode);
  }

general_operand_dst(op, mode)
rtx op;
enum machine_mode mode;
  {
if (reload_in_progress)
    if (GET_CODE(op) == MEM &&
	GET_CODE(XEXP(op,0)) == POST_INC) return 0;
    return general_operand(op,mode);
  }

static int old = 0;
final_prescan_insn(insn, operand, nop)
rtx insn;
 {
   extern int *insn_addresses;
   int uid = INSN_UID(insn);
   
if (TARGET_ADDRESSES) {   fprintf(asm_out_file,"; %d %d\n", insn_addresses[uid],

	   insn_addresses[uid] - old);
   old = insn_addresses[uid];
			}
   

 }

/************************************************************************

 'R' remove any pre or post inc thing, and treat just like the
 register. This is useful when pretending to have pre and post
 everything.

 'M' turn the constat into it's neg
 'A' if it's a reg, print as rnl, else it must be a constant, print
 the bottom 8 bits of it.

'B' if it'sa reg print as rnh, else print the top 8 bits of it
'j' print operand as condition code
'k' print operand as reverse condition code
'L' fake label, changed after used twice

 'W' treat arg as a 16 bit word 
 'N' treat are as the next 16 bit word in a 32 bit thing 

 'O' find the clear bt, and print it's number

 'E' like A but -ve
 'F' like B but -ve
 'C' print the operand -2.
 'P' if operand is incing/decing sp, print .w, otherwise .b
 'U' if operand is incing/decing sp, print l, otherwise nothing

'Z' print int & 7 
'Y' print either l or h depending on above int <8 >=8

 */




static log2(x)
{
  int i;
  int n = 1;
  
  for (i = 0; i < 32;i ++)
  {
    if (n == x) return i;
    n <<=1;
    
  }
  abort();  
}

void print_operand(file, x, code)
    FILE           *file;
    rtx             x;
    int             code;
{
  static int lab = 1000;
  static char *last_p;
  static int bitint;
  
  switch (code) 
  {
  case 'Z':
    bitint = INTVAL(x);
    fprintf(file,"#%d",bitint & 7);
    break;
    
  case 'Y':
    fprintf(file,"%c", bitint >7 ? 'h':'l');
    break;
    
case 'O':  
    bitint = log2((~INTVAL(x)) & 0xff);
    fprintf(file,"#%d", bitint & 7);

    break;

  case 'V':
    bitint = log2(INTVAL(x));
    
    fprintf(file,"#%d", bitint & 7);

    break;
    
  case 'P':
    if (REGNO(XEXP(XEXP(x,0),0)) == STACK_POINTER_REGNUM) 
    {
      last_p = "";
      fprintf(file,".w");
    }
    
    else 
    {
      last_p = "l";
      fprintf(file,".b");
    }
    
    break;
    
  case 'U':
    fprintf(file,"%s%s", regs_stuff[REGNO(x)].name_big, last_p);
    break;
    
    
  case 'Q':
    /* Special item for a fake fp instruction */
    while (GET_CODE(x) == CONST) 
    {
      x = XEXP(x,0);
      
    }
    
    switch (GET_CODE(x)) 
    {
    case REG:
      /* Output the address of the reg involved */
      if (REGNO(x) < 8) 
      {
	fprintf(file,".word\t0x%0x ; r%d",(REGNO(x)<<4)|7, REGNO(x));
      }
      else 
      {
	fprintf(file,".word\t__fpreg%d",REGNO(x));
      }
      
      break;
      
    case MEM:

      x = XEXP(x,0);
      while (GET_CODE(x) == CONST) 
      {
	x = XEXP(x,0);
      }
      
      switch (GET_CODE(x)) 
      {
	rtx reg;
	int rn;

      case PRE_DEC:
	abort();
	reg = XEXP(x,0);
	rn = REGNO(reg);
	fprintf(file,".word\t0x%0x ; -r%d", (rn << 4)+ 5, rn);
	break;
	
      case PLUS:


	
	if (GET_CODE(XEXP(x,0)) == SYMBOL_REF) 	
	{
	  fprintf(asm_out_file, ".word	");
	  print_operand_address(asm_out_file, x);
	}
	
	else   
	{
	  reg = XEXP(x,0);

	  if (GET_CODE(XEXP(x,0)) != REG) abort();

	  if ((GET_CODE(XEXP(x,1)) == CONST_INT)
	      && INTVAL(XEXP(x,1)) >= -128 
	      && INTVAL(XEXP(x,1)) <= 127) 
	  {
	    fprintf(file,".word\t0x%0x ; @(%d,r%d)",( (REGNO(reg)<<4) |
						     (INTVAL(XEXP(x,1))<< 8  ) |   1) & 0xffff,
		    INTVAL(XEXP(x,1)), REGNO(reg));
	  }
	
	  else  
	  {
	    fprintf(file,".word\t0x%0x\n", (REGNO(reg) <<4) +3);
	    fprintf(file,"\t.word\t");
	    print_operand_address(file, XEXP(x,1));

	  }
	}
	

	break;
	
      case REG:
	/* MEM(REG) */
	fprintf(file,".word\t0x%0x ; (r%d)",
		(REGNO(x) <<4) +1,
		REGNO(x));
	break;
	/* If simple, then just output the address */


      default:
	abort();
      case SYMBOL_REF:

	
	fprintf(file,".word\t");
	print_operand_address(file, x);
	break;
      }
      break;
      
    default:
      abort();
      
    }
    
	
	
    break;
    
  case 'R':
    /* Turn (mem(post_inc(x))) into x */
    x = XEXP(x,0);
    if (GET_CODE(x) == PRE_DEC ||
	GET_CODE(x) == POST_INC) {
	print_operand(file, XEXP(x, 0), 0);
      }
    else 
     abort();
    break;
  case 'M':
    switch (INTVAL(x)) 
    {
    case -2:
    case -4:
      
      fprintf(file,"#2");
      break;
    case -1:
    case -3:
      
      fprintf(file,"#1");
      break;
    default:
      abort();

    }


    break;
  case 'W':

    switch (GET_CODE(x)) 
    {
    case REG:
      fprintf(file,"%s", regs_stuff[REGNO(x)].name_big);      
      break;
    case MEM:
      x = adj_offsettable_operand(x, 0);
      print_operand(file,x,0);
      break;
    case CONST_INT:
    {
      int y = (INTVAL(x) & 0xffff);
      if (y & 0x8000)
      {
	fprintf(file,"#-%d", ((~y)+1)&0xffff);
      }
      
      else {     
	  fprintf(file,"#%d",y);
	}
    }      
      break;
    default:
      abort();
      
    }
    
    break;


  case 'N':
    
    switch (GET_CODE(x)) 
    {
    case REG:
      fprintf(file,"%s", regs_stuff[REGNO(x)+1].name_big);      
      break;
    case MEM:

          
      x = adj_offsettable_operand(x, 2);
      print_operand(file,x,0);
      break;
    case CONST_INT:
      fprintf(file,"#%d", INTVAL(x) >> 16);
      break;
      
    default:
      abort();
      
    }
    
    break;
  case 'C':
    fprintf(file,"#%d", INTVAL(x) - 2);
    break;
    
    
  case 'E':
    switch (GET_CODE(x)) 
    {
    case REG:
      fprintf(file,"%sl", regs_stuff[REGNO(x)].name_big);
      break;
    case CONST_INT:
      fprintf(file, "#%d", (-INTVAL(x)) & 0xff);
      break;
    default:
      abort();
    }
    break;    
  
  case 'A':
    switch (GET_CODE(x)) 
    {
    case REG:
      fprintf(file,"%sl", regs_stuff[REGNO(x)].name_big);
      break;
    case CONST_INT:
      fprintf(file, "#%d", INTVAL(x) & 0xff);
      break;
    case MEM:
      if (GET_CODE(XEXP(x,0))==REG) 
      {
	fprintf(file,"@r%d", REGNO(XEXP(x,0)));
      }
      else abort();
      
      break;
      
    default:
      abort();
    }
    break;    

  case      'B':
    switch (GET_CODE(x)) 
    {
    case REG:
      fprintf(file,"%sh", regs_stuff[REGNO(x)].name_big);
      break;
    case CONST_INT:
      fprintf(file, "#%d", (INTVAL(x) & 0xff00) >> 8);
      break;
    default:
      abort();
    }
    break;

  case      'F':
    switch (GET_CODE(x)) 
    {
    case REG:
      fprintf(file,"%sh", regs_stuff[REGNO(x)].name_big);
      break;
    case CONST_INT:
      fprintf(file, "#%d", ((-INTVAL(x)) & 0xff00) >> 8);
      break;
    default:
      abort();
    }
    break;
    
  case 'j':
    asm_fprintf(file, print_cond(file, GET_CODE(x), 0));
    break;
    
  case 'k':
    asm_fprintf(file, print_cond(file, reverse_condition(GET_CODE(x))));
    break;
    
  case 'L':

    asm_fprintf(file,"tl%d", (lab++)/2);
    break;
  default:
    switch (GET_CODE(x)) 
    {

      
    case REG:
      fprintf(file,"%s", regs_stuff[REGNO(x)].name_big);
      break;
    case MEM:
      fprintf(file,"@");
      output_address(XEXP(x,0));
      break;
    case CONST_INT:
    case SYMBOL_REF:      
    case CONST:
    case LABEL_REF:
      fprintf(file,"#");
      print_operand_address(file,x);
      break;
    }    
  }

}


print_operand_address(file, addr)
FILE *file;
rtx addr;
{
  register rtx reg1, reg2, breg, ireg;					
  rtx offset;								
  switch (GET_CODE (addr))						
  {									
  case REG:								
    fprintf (file, "%s", regs_stuff[REGNO (addr)].name_big);			
    break;								
  case PRE_DEC:							
    fprintf (file, "-%s", regs_stuff[REGNO (XEXP (addr, 0))].name_big);	
    break;								
  case POST_INC:							
    fprintf (file, "%s+", regs_stuff[REGNO (XEXP (addr, 0))].name_big);	
    break;								
  case PLUS:
    fprintf(file, "(");
    if (GET_CODE(XEXP(addr,0)) == REG) {
	print_operand_address(file, XEXP(addr, 1));
	fprintf(file,",");
	print_operand_address(file, XEXP(addr, 0));
      }
    else {
	print_operand_address(file, XEXP(addr, 0));
	fprintf(file, "+");
	print_operand_address(file, XEXP(addr, 1));
      }
    fprintf(file, ")");
    break;
  case CONST_INT:
    if (INTVAL(addr) < 0) 
    {
      int v = - INTVAL(addr);
      
          fprintf(file,"-%d", v);
	}
    else 
    {
      
    fprintf(file,"%d", INTVAL(addr));
  }
    
    break;

  default:								
    output_addr_const (file, addr);
    break;
      
  }
}




is_power_of_two(x)
{
    int             i;
    int             b = 1;
    for (i = 0; i < 32; i++) {
	if (x == b)
	    return 1;
	if (b > x)
	    return 0;
	b <<= 1;
    }
    return 0;
}
legitimate_constant_p(x)
rtx x;
{

  if (GET_CODE(x) == CONST_DOUBLE) return 0;
  return 1;
}



char *ashift(operands, string, mode)
rtx *operands;
char *string;
enum machine_mode mode;
{
  static int lshift_label;
  unsigned int n;
  char * treg = regs_stuff[REGNO(operands[3])].name_small;

  if (GET_CODE(operands[2]) !=CONST_INT) 
  {
    char *oreg;
	
    /* A variable length loop */
    oreg = regs_stuff[REGNO(operands[2])].name_small;    
    fprintf(asm_out_file,"\tmov.b	%s,%s\n",oreg, treg);
	
  }
  else {
    unsigned     int n = INTVAL(operands[2]);
    if (n < 3) {
      int i;
      for (i = 0; i < n; i++) {
	output_asm_insn( string, operands);
      }
      return "";
    }
    fprintf(asm_out_file,"\tmov.b	#%d,%s\n", n, treg);
  }
  /* Generate a little loop, with a test at the top  */

  fprintf(asm_out_file,"lshift%ds:\n", lshift_label);
  fprintf(asm_out_file,"\tdec\t%s\n", treg);
  fprintf(asm_out_file,"\tblt	lshift%de\n", lshift_label);
  output_asm_insn( string, operands);
  fprintf(asm_out_file,"\tbra	lshift%ds\n", lshift_label);
  fprintf(asm_out_file,"lshift%de:\n", lshift_label);
  lshift_label++;
  return "";

}

hard_regno_nregs(regno, mode)
int regno;
enum machine_mode mode;
{
  if (regno < 8) {
      return  ((GET_MODE_SIZE (mode) + UNITS_PER_WORD - 1) /   UNITS_PER_WORD) ;
    
    }
  return 1;

}


hard_regno_mode_ok(regno, mode)
{
  switch (regno) 
  {
   case 0:
   case 1:
   case 2:
   case 3:
   case 4:
   case 5:
   case 6:
   case 7:
    if (mode==DFmode) return 0;
    return (1);
   case 8:
   case 9:
   case 10:
    return mode==DFmode || mode == SFmode;
  }
  return 0;
}

reg_class_from_letter(C)
{
  return  ((C)=='f' ? FP_REGS:  GENERAL_REGS);
}

class_max_nregs(class, mode) 
{
  if (class == FP_REGS) 
  {
    return 1;
  }
  return ((GET_MODE_SIZE (mode) + UNITS_PER_WORD - 1) /    UNITS_PER_WORD);
}




int fp_offset;
int stack_size;
int reg_size;


use_return_insn()
{
  int             regno;
  return 0;
    
  if (!reload_completed || frame_pointer_needed || get_frame_size() != 0)
   return 0;

  /*
   * Copied from output_function_epilogue ().  We should probably create a
   * separate layout routine to perform the common work.
   */

  for (regno = 0; regno < 8; regno++)
   if (regs_ever_live[regno] && !call_used_regs[regno])
    return 0;

  if (pragma_interrupt || pragma_saveall )
   return 0;
  return 1;
}

rtx function_arg(cum, mode, type, named)
CUMULATIVE_ARGS  *cum;
enum machine_mode mode;
tree type;
int named;
{
  rtx result = 0;
  
  if (mode != VOIDmode && cum->libcall && mode != DFmode && mode != SFmode)
  {
    switch (cum->nbytes) 
    {
     case 0:
      result = gen_rtx (REG, mode, 0);
      break;
     case 2:
      result = gen_rtx (REG, mode, 1);      
      break;
     case 4:
      result = gen_rtx (REG, mode, 4);      
      break;
     case 6:
      result = gen_rtx (REG, mode, 5);            
      break;
     default:
      return 0;
    }
  }
  return result;
}

rtx double_acc;
rtx libcall_value(mode)
enum machine_mode mode;
{
  if (mode == DFmode)
  {
    return gen_rtx(REG, DFmode, 8);
  
  }
  return gen_rtx(REG, mode, 0);
}


char *gen_move_4bytes(which_alternative, operands)
int which_alternative;
rtx operands[];
{
  /* Most things can be copied in any order, special care has to taken
     for things like
     mov.w  r3,r4
     mov.w  r2,r3
     done the other way, trashes the dst reg. Also pushing requires that
     the regs be pushed in the other order */
  switch (which_alternative) 
  {
   case 0:
    if (REGNO(operands[1]) > REGNO(operands[0]))
    {
      return "mov.w\t%W1,%W0\n\tmov.w\t%N1,%N0";
    }
    else  
    {
      return "mov.w\t%N1,%N0\n\tmov.w\t%W1,%W0";

    }
   case 1:
    return "mov.w\t%N1,%W0\n\tmov.w\t%W1,%N0";
   case 2:
    return "mov.w\t%N1,%0\n\tmov.w\t%W1,%0";
   case 3:
    return "mov.w\t%1,%W0\n\tmov.w\t%1,%N0";
   case 4:
    return "mov.w\t%W1,@%R0\n\tadds\t#2,%R0\n\tmov.w\t%N1,@%R0\n\tadds\t#2,%R0";
   case 5:
    return "subs\t#2,%R1\n\tmov.w\t@%R1,%N0\n\tsubs\t#2,%R1\n\tmov.w\t@%R1,%W0";
   default:
    return "mov.w\t%N1,%N0\n\tmov.w\t%W1,%W0";
  }

}


/* This code is necessary because you have to be carefull when you add
   to the stack pointer

code like

add  #x,r7l
addx #y,r7h

can fail if there's an interrupt between x and y. The SP will not be
in a valid place if there was carry.

We fix it by forcing the number into a register 
mov.w #yx, rn
add.w rn,r7

*/

int addhi3(code,operands)
int code;
rtx operands[];
{
  if (REGNO(operands[0]) == STACK_POINTER_REGNUM 
      && GET_CODE(operands[2]) != REG)
  {
    extern int reload_in_progress;
    if (reload_in_progress) 
    {
      /* Ouch ! Didn't find out until too late. */
      int n;
      /* Can not use a temp reg now, we'll have to do it in bits */
      if (GET_CODE(operands[2]) != CONST_INT)
       abort();
      n = INTVAL(operands[2]);
      if (n < 0)
      {
	code = code == PLUS ? MINUS : PLUS;
	n = -n;
      }

      while (n >= 2) 
      {
	emit_insn(gen_rtx(SET, HImode, operands[0],
			  gen_rtx(code, HImode, operands[1],
				  gen_rtx(CONST_INT, VOIDmode, 2))));
	n-=2;
      }
      if (n==1) 
      {
	emit_insn(gen_rtx(SET, HImode, operands[0],
			  gen_rtx(code, HImode, operands[1],
				  gen_rtx(CONST_INT, VOIDmode, 1))));
      }

      /* Trust that the emitter will clean up the mess */
      return 1;
    }
    else 
    {
      /* Before reload, but let's look at the possibilities */
      if (GET_CODE(operands[2]) == CONST_INT && INTVAL(operands[2]) <=4) 
      {
	/* This won't need a temp reg anyway */
	return 0;
      }	   
      else if (GET_CODE(operands[2]) == CONST_INT)
      {
	/* Not too late to allocate a reg to keep the number in, we
	   leave the instruction alone though, so that optimizations
	   which look at stack pointer adjustments can note that a
	   constant is being added to the SP. We keep the reg around,
	   and always use r5  so we can dig it out later */

	emit (
	      gen_rtx
	      (PARALLEL, VOIDmode,
	       gen_rtvec(2,
			 gen_rtx
			 (SET, HImode, operands[0],
			  gen_rtx(code, HImode, operands[1], operands[2])),
			 gen_rtx(CLOBBER, VOIDmode, gen_rtx(REG,HImode,5)))));
	return 1;
      }
      
    }
  }
  return 0;
}



h8_saved_reg_size()
{
  if (frame_pointer_needed) return 4;
  return  REG_USED(2) + REG_USED(3);
}
pc_size()
{
  return 2;
}
/*
 before we try and remove the registers, the frame would look like
   	argn
 ap->	arn0
	pc
	r2
	r3
 fp->	ofp
	auto0
	...
 sp->   autoN

 after it goes to
 	argn
 ap->   argn0
        savedn
 fp->   saved0
        auto0
 sp->

*/


initial_elimination_offset(FROM, TO)
{
  int offset;
  
  int auto_size = get_frame_size();
  int save_size ;

  save_size = 2;		/* PC is always saved */
   
  if (frame_pointer_needed) 
  {
    save_size += 2 + 2 + 2;	/* 2 for r2, 2 for r3, 2 for fp */
  }
  else 
  {
    save_size += REG_USED(2) + REG_USED(3); /* No FP */
  }
  
  if (FROM == ARG_POINTER_REGNUM && TO == STACK_POINTER_REGNUM) 
  {
    offset = auto_size + save_size;
    
  }
  else if (FROM== ARG_POINTER_REGNUM &&  TO == FRAME_POINTER_REGNUM)
  {
    offset =  save_size;
  }
  else if (FROM == FRAME_POINTER_REGNUM &&  TO == STACK_POINTER_REGNUM) 
  {
    offset = auto_size;
  }
  else abort();
  return offset;
}


frame_pointer_required()
{
  return 0;  
}




/* build the most efficient op we can for XOR, AND and IOR H */
char *gen_subop(op, operands)
int op;
rtx *operands;
{
  char *name;
  int hit;  
  switch (op) 
  {
  case AND:
    name = "and";
    hit = 0xff;
    break;
  case IOR:
    name = "or";
    hit = 0;
    break;
  case XOR:
    name = "xor";
    hit = 0;
    break;
  }
  
    
  if (GET_CODE(operands[2]) == CONST_INT) {
      int val  = INTVAL(operands[2]);
      if (((val >> 8) & 0xff) != hit)
      {
	fprintf(asm_out_file, "\t%s\t#%d,r%dh\n",
		name, (val >> 8) & 0xff, REGNO(operands[0]));
      }
      if ((val & 0x00ff) != hit)
      {
	fprintf(asm_out_file, "\t%s\t#%d,r%dl\n",
		name, (val) & 0xff, REGNO(operands[0]));
      }
    }
  
  else 
  {
    fprintf(asm_out_file,"\t%s\tr%dl,r%dl\n",
	    name, REGNO(operands[2]), REGNO(operands[0]));

    fprintf(asm_out_file,"\t%s\tr%dh,r%dh\n",
	    name, REGNO(operands[2]), REGNO(operands[0]));
    
  }
  
  return "";

}


void select_section(DECL, RELOC, ptr, user)
tree DECL;
int RELOC;
int *ptr;
int user;
{
  if (TREE_CODE (DECL) == VAR_DECL || TREE_CODE (DECL) == FUNCTION_DECL)
  {									
    if (GET_CODE (DECL_RTL (DECL)) == MEM				
	&& GET_CODE (XEXP (DECL_RTL (DECL), 0)) == SYMBOL_REF		
	&& SYMBOL_REF_FLAG (XEXP (DECL_RTL (DECL), 0)))		
    {								
      rtx sym = XEXP (DECL_RTL (DECL), 0);				
      int len = strchr (XEXP (sym, 0), ' ') - XSTR (sym, 0);	
      char *section_name = (char *)alloca (len+1);			
      strncpy (section_name, XEXP (sym, 0), len);			
      section_name[len] = '\0';					
      *ptr  = user;						
      fprintf (asm_out_file, "\t.section\t%s\n", section_name);	
      return;
    }								
  }									
  if (TREE_CODE (DECL) == VAR_DECL)					
  {									
    if (TREE_READONLY (DECL) && ! TREE_SIDE_EFFECTS (DECL)		
	&& DECL_ALIGN (DECL) <= MAX_TEXT_ALIGN			
	&& ! (flag_pic && RELOC))					
     text_section ();						
    else								
     data_section ();						
  }									
  else if (TREE_CODE (DECL) == CONSTRUCTOR)				
  {									
    if (flag_pic != 0 && RELOC != 0)					
     data_section ();						
    else								
     text_section ();						
  }									
  else if (*tree_code_type[(int) TREE_CODE (DECL)] == 'c')		
  {									
    if ((TREE_CODE (DECL) == STRING_CST && flag_writable_strings)	
	|| TYPE_ALIGN (TREE_TYPE (DECL)) > MAX_TEXT_ALIGN)		
     data_section ();						
    else								
     text_section ();						
  } 
  else text_section() ;
}



asm_output_labelref(file, name)
FILE *file;
char *name;
{
  char *p;
  fputc('_', file);
  
  for (p = name; *p; p++)
  {
    if (*p == ' ') 
    {
      /* Found a space in the name, then we've skipped over the
	 section encoding */
      fputs(p+1, file);
      return;
    }
  }    
  /* No space, so no section */
  fputs(name, file);
}

  
