;;!emacs
;;
;; FILE:         hpath.el
;; SUMMARY:      Hyperbole support routines for handling UNIX paths.  
;; USAGE:        GNU Emacs Lisp Library
;;
;; AUTHOR:       Bob Weiner
;; ORG:          Brown U.
;;
;; ORIG-DATE:     1-Nov-91 at 00:44:23
;; LAST-MOD:     13-Dec-91 at 14:52:06 by Bob Weiner
;;
;; This file is part of Hyperbole.
;; 
;; Copyright (C) 1991, Brown University, Providence, RI
;; Developed with support from Motorola Inc.
;; 
;; Permission to use, modify and redistribute this software and its
;; documentation for any purpose other than its incorporation into a
;; commercial product is hereby granted without fee.  A distribution fee
;; may be charged with any redistribution.  Any distribution requires
;; that the above copyright notice appear in all copies, that both that
;; copyright notice and this permission notice appear in supporting
;; documentation, and that neither the name of Brown University nor the
;; author's name be used in advertising or publicity pertaining to
;; distribution of the software without specific, written prior permission.
;; 
;; Brown University makes no representations about the suitability of this
;; software for any purpose.  It is provided "as is" without express or
;; implied warranty.
;;
;;
;; DESCRIPTION:  
;; DESCRIP-END.

;;; ************************************************************************
;;; Public functions
;;; ************************************************************************

(defun hpath:absolute-to (path &optional default-dirs)
  "Returns PATH as an absolute path relative to one directory from optional DEFAULT-DIRS or 'default-directory'.
Returns PATH unchanged when it is not a valid path or when DEFAULT-DIRS
is invalid.  DEFAULT-DIRS when non-nil may be a single directory or a list of
directories.  The first one in which PATH is found is used."
  (if (not (stringp path))
      path
    (if (not (cond ((null default-dirs)
		    (setq default-dirs (cons default-directory nil)))
		   ((stringp default-dirs)
		    (setq default-dirs (cons default-dirs nil)))
		   ((listp default-dirs))
		   (t nil)))
	path
      (let ((rtn) dir)
	(while (and default-dirs (null rtn))
	  (setq dir (expand-file-name
		     (file-name-as-directory (car default-dirs)))
		rtn (expand-file-name path dir)
		default-dirs (cdr default-dirs))
	  (or (file-exists-p rtn) (setq rtn nil)))
	(or rtn path)))))

(defun hpath:at-p (&optional type non-exist)
  "Returns delimited path at point, if any.
Delimiters may be:  double quotes, open and close single quote, or
Texinfo file references.
If optional TYPE is the symbol 'file or 'directory, then only that path type is
accepted as a match.  Only locally reachable paths are checked for existence.
With optional NON-EXIST, nonexistent local paths are allowed.
Absolute pathnames must begin with a '/' or '~'.  Relative pathnames
must begin with a './' or '../' to be recognized."
  (hpath:is-p (or (hargs:delimited "\"" "\"") 
		  ;; Filenames in Info docs
		  (hargs:delimited "\`" "\'")
		  ;; Filenames in TexInfo docs
		  (hargs:delimited "@file{" "}"))
	      type non-exist))

(defun hpath:ange-ftp-p ()
  "Returns an ange-ftp pathname that point is within or nil.
See the 'ange-ftp' Elisp package for pathname format details."
  (save-excursion
    (skip-chars-backward "^ \t\n\"")
    (if (looking-at "/[a-zA-Z][^@]*@[^: \t\n\^M\"]+:[^ \t\n\^M\"]*")
	(buffer-substring (match-beginning 0) (match-end 0)))))

(defun hpath:is-p (path &optional type non-exist)
  "Returns PATH if PATH is a Unix path, else nil.
If optional TYPE is the symbol 'file or 'directory, then only that path
type is accepted as a match.
The existence of the path is checked only for locally reachable paths.
With optional NON-EXIST, nonexistent local paths are allowed."
  (and (stringp path) (not (string= path ""))
       (not (string-match "[ \t\n\^M\"`'{}()|\\]" path))
       (let ((remote-path (string-match "@.+:\\|.+:/" path)))
	 (cond (remote-path
		(cond ((eq type 'file)
		       (if (not (string= "/" (substring path -1))) path))
		      ((eq type 'directory)
		       (if (string= "/" (substring path -1)) path))
		      (path)))
	       ((or non-exist (file-exists-p path))
		(cond ((eq type 'file)
		       (and (not (file-directory-p path)) path))
		      ((eq type 'directory)
		       (and (file-directory-p path) path))
		      (path)))))))

(defun hpath:relative-to (path &optional default-dir)
  "Returns PATH relative to optional DEFAULT-DIR or 'default-directory'.
Returns PATH unchanged when it is not a valid path."
  (if (not (and (stringp path) (file-exists-p path)))
      path
    (setq default-dir
	  (expand-file-name
	   (file-name-as-directory (or default-dir default-directory)))
	  path (expand-file-name path))
    (and path default-dir
	 (let ((end-dir (min (length path) (length default-dir))))
	   (if (string= (substring path 0 end-dir) default-dir)
	       (concat "./" (substring path end-dir)) path)))))

;; Handles UNIX and Apollo Domain-isms in pathnames.
(defun hpath:validate (path)
  "Returns t if PATH is a valid, readable path, else signals error."
  (if (file-readable-p path)
      t
    (error "(hpath:validate): \"%s\" is not readable." path)))


(provide 'hpath)
