;;!emacs
;;
;; FILE:         hmouse-key.el
;; SUMMARY:      Key bindings for Hyperbole mouse control.
;; USAGE:        GNU Emacs Lisp Library
;;
;; AUTHOR:       Bob Weiner
;; ORG:          Brown U.
;;
;; ORIG-DATE:     3-Sep-91 at 21:40:58
;; LAST-MOD:     13-Dec-91 at 14:51:34 by Bob Weiner
;;
;; This file is part of Hyperbole.
;; 
;; Copyright (C) 1991, Brown University, Providence, RI
;; Developed with support from Motorola Inc.
;; 
;; Permission to use, modify and redistribute this software and its
;; documentation for any purpose other than its incorporation into a
;; commercial product is hereby granted without fee.  A distribution fee
;; may be charged with any redistribution.  Any distribution requires
;; that the above copyright notice appear in all copies, that both that
;; copyright notice and this permission notice appear in supporting
;; documentation, and that neither the name of Brown University nor the
;; author's name be used in advertising or publicity pertaining to
;; distribution of the software without specific, written prior permission.
;; 
;; Brown University makes no representations about the suitability of this
;; software for any purpose.  It is provided "as is" without express or
;; implied warranty.
;;
;;
;; DESCRIPTION:  
;;
;;   'sm-mouse-setup' globally binds middle and right mouse buttons as
;;   primary and secondary Smart Keys respectively.
;;
;;   'sm-mouse-toggle-bindings' may be bound to a key.  It switches between
;;   Smart Key mouse bindings and previous mouse key bindings any time
;;   after 'sm-mouse-setup' has been called.
;;
;; DESCRIP-END.

;;; ************************************************************************
;;; Required Elisp Libraries
;;; ************************************************************************

(defun sm-term-machine-type-p (machine-type)
  "Returns t if single string argument MACHINE-TYPE matches the first
part of the term-type; where a part in the term-type is delimited by a
'-' or  an '_'."
  (interactive "sMachine type: ")
  (let ((term (if (eq window-system 'x)
		  "xterm"
		(getenv "TERM"))))
    (and term
	 (equal (substring term 0 (string-match "[-_]" term)) machine-type))))

(cond
 ;; Epoch
 ((boundp 'epoch::version)         (require 'mouse))
 ;; X
 ((sm-term-machine-type-p "xterm") (require 'x-mouse))
 ;; SunView
 ((sm-term-machine-type-p "sun")   (require 'sun-mouse))
 )

;;; ************************************************************************
;;; Public functions
;;; ************************************************************************

(defun sm-mouse-get-bindings ()
  "Returns list of bindings for mouse keys prior to their use as Smart Keys."
  (cond
   ;; Epoch
   ((boundp 'epoch::version)
    (mapcar '(lambda (key) (cons key (aref mouse::global-map key)))
	    (list (mouse::index mouse-middle mouse-down)
		  (mouse::index mouse-middle mouse-up)
		  (mouse::index mouse-right mouse-down)
		  (mouse::index mouse-right mouse-up))))
   ;; X
   ((sm-term-machine-type-p "xterm")
    (mapcar '(lambda (key) (cons key (lookup-key mouse-map key)))
	    (list x-button-middle x-button-middle-up
		  x-button-right  x-button-right-up)))
   ;; SunView
   ((sm-term-machine-type-p "sun")
    (mapcar '(lambda (key)
	       (setq key (mouse-list-to-mouse-code key))
	       (cons key (mousemap-get key current-global-mousemap)))
	    '((text        middle) (text     up middle)
	      (text         right) (text     up  right))))
   ;; Apollo Display Manager
   ((and (sm-term-machine-type-p "apollo") (fboundp 'bind-apollo-mouse-button))
    (mapcar '(lambda (key-str) (apollo-mouse-key-and-binding key-str))
	    '("M2D" "M2U" "M3D" "M3U")))
   ))

(defun sm-mouse-set-bindings (key-binding-list)
  "Sets mouse keys used as Smart Keys to bindings in KEY-BINDING-LIST.
KEY-BINDING-LIST is the value returned by 'sm-mouse-get-bindings' prior to
Smart Key setup."
  (cond
   ;; Epoch
   ((boundp 'epoch::version)
    (mapcar
     '(lambda (key-and-binding)
	(aset mouse::global-map (car key-and-binding) (cdr key-and-binding)))
     key-binding-list))
   ;; X
   ((sm-term-machine-type-p "xterm")
    (mapcar
     '(lambda (key-and-binding)
	(define-key mouse-map (car key-and-binding) (cdr key-and-binding)))
     key-binding-list))
   ;; SunView
   ((sm-term-machine-type-p "sun")
    (mapcar
     '(lambda (key-and-binding)
	(global-set-mouse (car key-and-binding) (cdr key-and-binding)))
     key-binding-list))
   ;; Apollo Display Manager
   ((and (sm-term-machine-type-p "apollo") (fboundp 'bind-apollo-mouse-button))
    (mapcar
     '(lambda (key-and-binding)
	(global-set-key (car key-and-binding) (cdr key-and-binding)))
     key-binding-list))
   ))

(defun sm-mouse-setup ()
  "Binds mouse keys for use as Smart Keys."
  (interactive)
  (or sm-mouse-bindings-p
      (setq sm-mouse-previous-bindings (sm-mouse-get-bindings)))
  (cond ((boundp 'epoch::version)
	 (setq mouse-set-point-command 'mouse::set-point)
	 (global-set-mouse mouse-middle mouse-down  'sm-depress)
	 (global-set-mouse mouse-middle mouse-up    'smart-key-mouse)
	 (global-set-mouse mouse-right  mouse-down  'sm-depress-meta)
	 (global-set-mouse mouse-right  mouse-up    'smart-key-mouse-meta))
	((sm-term-machine-type-p "xterm")
	 (setq mouse-set-point-command 'x-mouse-set-point)
	 (define-key mouse-map x-button-middle 'sm-depress)
	 (define-key mouse-map x-button-middle-up 'smart-key-mouse)
	 (define-key mouse-map x-button-right 'sm-depress-meta)
	 (define-key mouse-map x-button-right-up 'smart-key-mouse-meta)
	 ;; Use these instead of the above for a true META-BUTTON binding.
	 ;; (define-key mouse-map x-button-m-middle 'sm-depress-meta)
	 ;; (define-key mouse-map x-button-m-middle-up 'smart-key-mouse-meta)
	 )
	((sm-term-machine-type-p "sun")
	 (setq mouse-set-point-command 'mouse-move-point)
	 (global-set-mouse '(text        middle) 'sm-depress)
	 (global-set-mouse '(text     up middle) 'smart-key-mouse)
	 (global-set-mouse '(text         right) 'sm-depress-meta)
	 (global-set-mouse '(text     up  right) 'smart-key-mouse-meta)
	 ;; Use these instead of the above for a true META-BUTTON binding.
	 ;; (global-set-mouse '(text meta   middle) 'sm-depress-meta)
	 ;; (global-set-mouse '(text meta up middle) 'smart-key-mouse-meta)
	 )
	((and (sm-term-machine-type-p "apollo")
	      (fboundp 'apollo-mouse-move-point))
	 (setq mouse-set-point-command 'apollo-mouse-move-point)
	 (bind-apollo-mouse-button "M2D" 'sm-depress)
	 (bind-apollo-mouse-button "M2U" 'smart-key-mouse)
	 (bind-apollo-mouse-button "M3D" 'sm-depress-meta)
	 (bind-apollo-mouse-button "M3U" 'smart-key-mouse-meta)
	 ;; Use these instead of the above for a true META-BUTTON binding.
	 ;; (bind-apollo-mouse-button "M2U" 'smart-key-mouse 'smart-key-mouse-meta)
	 ;; (bind-apollo-mouse-button "M2D" 'sm-depress 'sm-depress-meta)
	 ))
  (setq sm-mouse-bindings (sm-mouse-get-bindings)
	sm-mouse-bindings-p t))

(defun sm-mouse-toggle-bindings ()
  "Toggles between Smart Key mouse settings and their prior bindings."
  (interactive)
  (let ((key-binding-list (if sm-mouse-bindings-p
			      sm-mouse-previous-bindings
			    sm-mouse-bindings))
	(other-list-var (if sm-mouse-bindings-p
			    'sm-mouse-bindings
			  'sm-mouse-previous-bindings)))
      (if key-binding-list
	  (progn
	    (set other-list-var (sm-mouse-get-bindings))
	    (sm-mouse-set-bindings key-binding-list)
	    (message "%s mouse bindings in use."
		     (if (setq sm-mouse-bindings-p (not sm-mouse-bindings-p))
			 "Smart Key" "Personal")))
	(error "(sm-mouse-toggle-bindings): Null %s." other-list-var))))

;;; ************************************************************************
;;; Private functions
;;; ************************************************************************

(if (fboundp 'bind-apollo-mouse-button)
(defun apollo-mouse-key-and-binding (mouse-button)
  "Returns binding for an Apollo MOUSE-BUTTON (a string) or nil if none."
  (interactive "sMouse Button: ")
  (let ((numeric-code (cdr (assoc mouse-button *apollo-mouse-buttons*))))
    (if (null numeric-code)
	(error "(hmouse-key): %s is not a valid Apollo mouse key name."
	       mouse-button))
    (if (stringp numeric-code)
	(setq numeric-code
	      (cdr (assoc numeric-code *apollo-mouse-buttons*))))
    (let ((key-sequence (concat "\M-*" (char-to-string numeric-code))))
      (cons key-sequence (global-key-binding key-sequence))))))

(defun sm-depress (&optional args)
  (interactive)
  (setq *smart-key-depressed* t)
  (if *smart-key-meta-depressed*
      (progn (smart-key-set-point args)
	     (smart-key-help 'meta))))

(defun sm-depress-meta (&optional args)
  (interactive)
  (setq *smart-key-meta-depressed* t)
  (if *smart-key-depressed*
      (progn (smart-key-set-point args)
	     (smart-key-help nil))))

;;; ************************************************************************
;;; Private variables
;;; ************************************************************************

(defvar sm-mouse-bindings nil
  "List of (key . binding) pairs for Smart Mouse Keys.")

(defvar sm-mouse-bindings-p nil
  "True if Smart Key mouse bindings are in use, else nil.")

(defvar sm-mouse-previous-bindings nil
  "List of previous (key . binding) pairs for mouse keys used as Smart Keys.")

(provide 'hmouse-key)
