;;; Core functions and entry points for VM
;;; Copyright (C) 1989, 1990 Kyle E. Jones
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 1, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;; Entry points to VM are the commands vm, vm-mode, vm-visit-folder,
;; vm-visit-virtual-folder, and vm-mail.
;;
;; If autoloading then the lines:
;;   (autoload 'vm "vm" nil t)
;;   (autoload 'vm-mode "vm" nil t)
;;   (autoload 'vm-visit-folder "vm" nil t)
;;   (autoload 'vm-visit-virtual-folder "vm-virtual" nil t)
;;   (autoload 'vm-mail "vm-reply" nil t)
;; should appear in a user's .emacs or in default.el in the lisp
;; directory of the Emacs distribution.
;;
;; VM uses Emacs' etc/movemail, therefore movemail must work in order
;; for VM to function properly on your system.

(provide 'vm)

(defun vm-parse (string regexp &optional matchn)
  (or matchn (setq matchn 1))
  (let (list)
    (store-match-data nil)
    (while (string-match regexp string (match-end 0))
      (setq list (cons (substring string (match-beginning matchn)
				  (match-end matchn)) list)))
    (nreverse list)))

(defvar vm-primary-inbox "~/INBOX"
  "*Mail is moved from the system mailbox to this file for reading.")

(defvar vm-crash-box "~/INBOX.CRASH"
  "*File in which to store mail temporarily while it is transferrred from
the system mailbox to the primary inbox.  If the something happens
during this mail transfer, any missing mail will be found in this file.
VM will do crash recovery from this file automatically at startup, as
necessary.")

(defvar vm-spool-files
  (or (and (setq vm-spool-files (getenv "MAILPATH"))
	   (setq vm-spool-files
		 (vm-parse vm-spool-files
			   "\\([^:%]+\\)\\([%?][^:]*\\)?\\(:\\|$\\)")))
      (and (setq vm-spool-files (getenv "MAIL"))
	   (setq vm-spool-files (list vm-spool-files))))
  "*If non-nil this variable's value should be a list of strings naming files
that VM will check for incoming mail instead of the where VM thinks your
system mailbox is.  This variable can be used to specify multiple spool files
or to point VM in the right direction if its notion of your system mailbox is
incorrect.

VM will default to the value of the shell environmental variables
MAILPATH or MAIL if either of these variables are defined.")

(defvar vm-visible-headers
  '("From:" "Sender:" "To:" "Apparently-To:" "Cc:" "Subject:" "Date:")
  "*List of headers that should be visible when VM first displays a message.
These should be listed in the order you wish them presented.
Regular expressions are allowed.")

(defvar vm-invisible-header-regexp nil
  "*Non-nil value should be a regular expression that tells what headers
VM should NOT normally display when presenting a message.  All other
headers will be displayed.  The variable vm-visible-headers specifies
the presentation order of headers; headers not matched by
vm-visible-headers are displayed last.

Nil value causes VM to display ONLY those headers specified in
vm-visible-headers.")

(defvar vm-highlighted-header-regexp nil
  "*Regular expression that matches the beginnings of headers that should
be highlighted when a message is first presented.  For exmaple setting
this variable to \"^From\\\\|^Subject\" causes the From: and Subject:
headers to be highlighted.")

(defvar vm-preview-lines 0
  "*Non-nil value N causes VM to display the visible headers + N lines of text
from a message when it is first presented.  The message is not actually
flagged as read until it is exposed in its entirety.  Nil causes VM not
to preview a message at all; it is displayed in its entirety when first
presented and is immediately flagged as read.")

(defvar vm-preview-read-messages t
  "*Non-nil value means to preview messages even if they've already been read.
A nil value causes VM to preview messages only if new or unread.")

(defvar vm-auto-next-message t
  "*Non-nil value causes VM to use vm-next-mesage to advance to the next
message in the folder if the user attempts to scroll past the end of the
current messages.  A nil value disables this behavior.")

(defvar vm-honor-page-delimiters nil
  "*Non-nil value causes VM to honor pages delimiters (as specified by the
page-delimiter variable) when scrolling through a message.")

(defvar vm-confirm-quit nil
  "*Value of t causes VM to always ask for confirmation before quitting
a VM visit of a folder.  A nil value means VM will ask only when messages
will be lost unwittingly by quitting, i.e. not removed by intentional
delete and expunge.  A value that is not nil and not t causes VM to ask
only when there are unsaved changes to message attributes.")

(defvar vm-folder-directory nil
  "*Directory where folders of mail are kept.")

(defvar vm-confirm-new-folders nil
  "*Non-nil value causes interactive calls to vm-save-message
to ask for confirmation before creating a new folder.")

(defvar vm-delete-empty-folders t
  "*Non-nil value causes VM to remove empty (zero length) folder files
after saving them.")

(defvar vm-visit-when-saving 0
  "*Value determines whether VM will visit folders when saving messages.
`Visiting' means that VM will read the folder into Emacs and append the
message to the buffer instead of appending to the folder file directly.
This behavior is ideal when folders are encrypted or compressed since
appending plaintext directly to such folders is a ghastly mistake.

A value of t means VM will always visit folders when saving.

A nil value means VM will never visit folders before saving to them, and
VM will generate an error if you attempt to save messages to a folder
that is being visited.  The latter restriction is necessary to insure
that the buffer and disk copies of the folder being visited remain
consistent.

A value that is not nil and not t means VM will save to a folder's
buffer if that folder is being visited, otherwise VM saves to the folder
file itself.")

(defvar vm-auto-folder-alist nil
  "*Non-nil value should be an alist that VM will use to choose a default
folder name when messages are saved.  The alist should be of the form
\((HEADER-NAME
   (REGEXP . FOLDER-NAME) ...
  ...))
where HEADER-NAME and REGEXP are strings, and FOLDER-NAME is a string or an
s-expression that evaluates to a string.

If any part of the contents of the message header named by HEADER-NAME
is matched by the regular expression REGEXP, VM will evaluate the
corresponding FOLDER-NAME and use the result as the default when
prompting for a folder to save the message in.  If the resulting folder
name is a relative pathname, then it will resolve to the directory named by
vm-folder-directory, or the default-directory of the currently visited
folder if vm-folder-directory is nil.

When FOLDER-NAME is evaluated, the current buffer will contain only the
contents of the header named by HEADER-NAME.  It is safe to modify this
buffer.  You can use the match data from any \\( ... \\) grouping
constructs in REGEXP along with the function buffer-substring to build a
folder name based on the header information.  If the result of
evaluating FOLDER-NAME is a list, then the list will be treated as
another auto-folder-alist and will be descended recursively.

Whether matching is case sensitive depends on the value of the variable
vm-auto-folder-case-fold-search.")

(defvar vm-auto-folder-case-fold-search nil
  "*Non-nil value means VM will ignore case when matching header
contents while doing automatic folder selection via the variable
vm-auto-folder-alist.")

(defvar vm-virtual-folder-alist nil
  "*Non-nil value should be a list of virtual folder definitions.
A virtual folder is a mapping of messages from one or more real folders
into what appears to be a single folder.  A virtual folder definition
specifes which real folders should be searched for prospective messages
and what the inclusion criteria are.

Each virtual folder definition should have the following form:

  (VIRTUAL-FOLDER-NAME
    ( (FOLDER-NAME ...)
      (SELECTOR [ARG]) ... )
    ... )

VIRTUAL-FOLDER-NAME is the name of the virtual folder being defined.
This is the name by which VM will refer to this folder.

FOLDER-NAME should be the name of a real folder.  There may be more than
one FOLDER-NAME listed, the SELECTORs within that sublist will apply to
them all.  If FOLDER-NAME is a directory, VM will assume this to mean that
all the folders in that directory should be searched.

The SELECTOR is a Lisp symbol that tells VM how to decide whether a message
from one of the folders specifed by the FOLDER-NAMEs should be included
in the virtual folder.  Some SELECTORs require an arguemnt ARG; unless
otherwise noted ARG may be omitted.

The recognized SELECTORs are:

   any       - matches any message.
   author    - matches message if ARG matches the author; ARG should be a
               regular expression.
   header    - matches message if ARG matches any part of the header
               portion of the message; ARG should be a regular expression.
   recipient - matches message if ARG matches any part of the recipient list;
               ARG should be a regular expression.
   subject   - matches message if ARG matches any part of the message's
               subject; ARG should be a regular expression.
   text      - matches message if ARG matches any part of the text
               portion of the message; ARG should be a regular expression.")

(defvar vm-virtual-mirror nil
  "*Non-nil value causes the attributes of messages in virtual folders
to mirror the changes in the attributes of the underlying real messages.
Similarly, changes in the attributes of virtual messages will change the
attributes of the underlying real messages.  A nil value causes virtual
messages to have their own dinstinct set of attributes, apart from the
nderlying real message.

The value of vm-virtual-mirror is considered only at the time a
virtaul folder is visited.  Changing the value of vm-virtual-mirror
does not affect the behavior of existing virtual folders.")

(defvar vm-folder-read-only nil
  "*Non-nil value causes a folder to be considered unmodifiable by VM.
Comamnds that modify message attributes or messages themselves are disallowed.
Commands that scan or allow the reading of messages are allowed but the
`new' and `unread' message flags are not changed by them.")

(defvar vm-included-text-prefix " > "
  "*String used to prefix included text in replies.")

(defvar vm-reply-subject-prefix nil
  "*Non-nil value should be a string that VM should add to the beginning
of the Subject header in replies, if the string is not already present.
Nil means don't prefix the Subject header.")

(defvar vm-reply-ignored-addresses nil
  "*Non-nil value should be a list of regular expressions that match
addresses that VM should automatically remove from the recipient headers
of replies.")

(defvar vm-in-reply-to-format "%i"
  "*String which specifies the format of the contents of the In-Reply-To
header that is generated for replies.  See the documentation for the
variable vm-summary-format for information on what this string may
contain.  The format should *not* end with a newline.
Nil means don't put an In-Reply-To header in replies.")

(defvar vm-included-text-attribution-format "%F writes:\n"
  "*String which specifies the format of the attribution that precedes the
included text from a message in a reply.  See the documentation for the
variable vm-summary-format for information on what this string may contain.
Nil means don't attribute included text in replies.")

(defvar vm-forwarding-subject-format "forwarded message from %F"
  "*String which specifies the format of the contents of the Subject
header that is generated for a forwarded message.  See the documentation
for the variable vm-summary-format for information on what this string
may contain.  The format should *not* end with nor contain a newline.
Nil means leave the Subject header empty when forwarding.")

(defvar vm-digest-preamble-format "\"%s\" (%F)"
  "*String which specifies the format of the preamble lines gneerated by 
vm-send-digest when it is invoked with a prefix argument.  See the
documentation for the variable vm-summary-format for information on what
this string may contain.  The format should *not* end with nor contain a
newline.")

(defvar vm-digest-center-preamble t
  "*Non-nil value means VM will center the preamble lines that precede
the start of a digest.  How the lines will be centered depends on the
ambient value of fill-column.   A nil value suppresses centering.")

(defvar vm-summary-format "%2n %*%a %-17.17F %-3.3m %2d %4l/%-5c \"%s\"\n"
  "*String which specifies the message summary line format.
The string may contain the printf-like `%' conversion specifiers which
substitute information about the message into the final summary line.

Recognized specifiers are:
   a - attribute indicators (always four characters wide)
       The first char is  `D', `N', `U' or ` ' for deleted, new, unread
       and read messages respectively.
       The second char is `F', `W' or ` ' for filed (saved) or written
       messages.
       The third char is `R', `Z' or ` ' for messages replied to,
       and forwarded messages.
       The fourth char is `E' if the message has been edited, ` ' otherwise.
   A - longer version of attributes indicators (six characters wide)
       The first char is  `D', `N', `U' or ` ' for deleted, new, unread
       and read messages respectively.
       The second is `r' or ` ', for message replied to.
       The third is `z' or ` ', for messages forwarded.
       The fourth is `f' or ` ', for messages filed.
       The fifth is `w' or ` ', for messages written.
       The sixth is `e' or ` ', for messages that have been edited.
   c - number of characters in message (ignoring headers)
   d - numeric day of month message sent
   f - author's address
   F - author's full name (same as f if full name not found)
   h - hour:min:sec message sent
   i - message ID
   l - number of lines in message (ignoring headers)
   m - month message sent
   M - numeric month message sent (January = 1)
   n - message number
   s - message subject
   t - addresses of the recipients of the message, in a comma-separated list
   T - full names of the recipients of the message, in a comma-separated list
       If a full name cannot be found, the corresponding address is used
       instead.
   w - day of the week message sent
   y - year message sent
   z - timezone of date when the message was sent
   * - `*' if the message is marked, ` ' otherwise

Use %% to get a single %.

A numeric field width may be specified between the `%' and the specifier;
this causes right justification of the substituted string.  A negative field
width causes left justification.

The field width may be followed by a `.' and a number specifying the maximum
allowed length of the substituted string.  If the string is longer than this
value it is truncated.

The summary format need not be one line per message but it must end with
a newline, otherwise the message pointer will not be displayed correctly
in the summary window.")

(defvar vm-auto-center-summary nil
  "*Value controls whether VM will keep the summary arrow vertically
centered within the summary window. A value of t causes VM to always
keep arrow cenered.  A value of nil means VM will never bother centering
the arrow.  A value that is not nil and not t causes VM to center the
arrow only if the summary window is not the only existing window.")

(defvar vm-mail-window-percentage 75
  "*Percentage of the screen that should be used to show mail messages.
The rest of the screen will be used by the summary buffer, if displayed.")

(defvar vm-mutable-windows t
  "*This variable's value controls VM's window usage.

A value of t gives VM free run of the Emacs display; it will commandeer
the entire screen for its purposes.

A value of nil restricts VM's window usage to the window from which
it was invoked.  VM will not create, delete, or use any other windows,
nor will it resize its own window.

A value that is neither t nor nil allows VM to use other windows, but it
will not create new ones, or resize or delete the current ones.")

(defvar vm-startup-with-summary nil
  "*Value tells VM what to display when a folder is visited.
Nil means display folder only, t means display the summary only.  A
value that is neither t not nil means to display both folder and summary.
The latter only works if the variable pop-up-windows's value is non-nil.
See the documentation for vm-mail-window-percentage to see how to change how
the screen is apportioned between the folder and summary windows.")

(defvar vm-follow-summary-cursor nil
  "*Non-nil value causes VM to select the message under the cursor in the
summary window before executing commands that operate on the current message.
This occurs only when the summary buffer window is the selected window.")

(defvar vm-group-by nil
  "*Non-nil value tells VM how to group message presentation.
Currently, the valid non-nil values for this variable are
  \"subject\", which causes messages with the same subject (ignoring
    Re:'s) to be presented together,
  \"author\", which causes messages with the same author to be presented
    together,
  \"recipient\", which causes messages with the same set of recipients to
    be presented together,
  \"date-sent\", which causes message sent on the same day to be
    presented together,
  \"physical-order\" which appears only for completeness, this is the
    default behavior and is the same as nil.

The ordering of the messages in the folder itself is not altered, messages
are simply numbered and ordered differently internally.")

(defvar vm-skip-deleted-messages t
  "*Non-nil value causes VM's `n' and 'p' commands to skip over
deleted messages.  A value of t causes deleted message to always be skipped.
A value that is not nil and not t cuases deleted messages to be skipped only
if there are other messages that are not flagged for deletion in the desired
direction of motion.")

(defvar vm-skip-read-messages nil
  "*Non-nil value causes VM's `n' and `p' commands to skip over
messages that have already been read, in favor of new or unread messages.
A value of t causes read messages to always be skipped.  A value that is
not nil and not t causes read messages to be skipped only if there are
unread messages in the desired direction of motion.")

(defvar vm-move-after-deleting nil
  "*Non-nil value causes VM's `d' command to automatically invoke
vm-next-message or vm-previous-message after deleting, to move
past the deleted messages.")

(defvar vm-move-after-undeleting nil
  "*Non-nil value causes VM's `u' command to automatically invoke
vm-next-message or vm-previous-message after undeleting, to move
past the undeleted messages.")

(defvar vm-delete-after-saving nil
  "*Non-nil value causes VM automatically to mark messages for deletion
after successfully saving them to a folder.")

(defvar vm-delete-after-archiving nil
  "*Non-nil value causes VM automatically to mark messages for deletion
after successfully auto-archiving them with the vm-auto-archive-messages
command.")

(defvar vm-delete-after-bursting nil
  "*Non-nil value causes VM automatically to mark messages for deletion
after successfully bursting them with the vm-burst-digest command.")

(defvar vm-circular-folders 0
  "*Value determines whether VM folders will be considered circular by
various commands.  `Circular' means VM will wrap from the end of the folder
to the start and vice versa when moving the message pointer or deleting,
undeleting or saving messages before or after the current message.

A value of t causes all VM commands to consider folders circular.

A value of nil causes all of VM commands to signal an error if the start
or end of the folder would have to be passed to complete the command.
For movement commands, this occurs after the message pointer has been
moved as far as possible in the specified direction.  For other commands,
the error occurs before any part of the command has been executed, i.e.
no deletions, saves, etc. will be done unless they can be done in their
entirety.

A value that is not nil and not t causes only VM's movement commands to
consider folders circular.  Saves, deletes and undelete commands will
behave the same as if the value is nil.")

(defvar vm-search-using-regexps nil
  "*Non-nil value causes VM's search command to interpret user input as a
regular expression instead of as a literal string.")

(defvar vm-edit-message-mode 'text-mode
  "*Major mode to use when editing messages in VM.")

(defvar vm-mode-hooks nil
  "*List of hook functions to run when a buffer enters vm-mode.
These hook functions should generally be used to set key bindings
and local variables.  Mucking about in the folder buffer is certainly
possible but it is not encouraged.")

(defvar vm-berkeley-mail-compatibility
  (memq system-type '(berkeley-unix))
  "*Non-nil means to read and write BSD Mail(1) style Status: headers.
This makes sense if you plan to use VM to read mail archives created by
Mail.")

(defvar vm-gargle-uucp nil
  "*Non-nil value means to use a crufty regular expression that does
surprisingly well at beautifying UUCP addresses that are substitued for
%f and %t as part of summary and attribution formats.")

(defvar vm-strip-reply-headers nil
  "*Non-nil value causes VM to strip away all comments and extraneous text
from the headers generated in reply messages.  If you use the \"fakemail\"
program as distributed with Emacs, you probably want to set this variable to
to t, because as of Emacs v18.52 \"fakemail\" could not handle unstripped
headers.")

(defvar vm-rfc934-forwarding t
  "*Non-nil value causes VM to use char stuffing as described in RFC 934
when packaging a message to be forwarded.  This will allow the recipient
to use a standard bursting agent on the message and act upon it as if it
were sent directly.")

(defvar vm-inhibit-startup-message nil
  "*Non-nil causes VM not to display its copyright notice, disclaimers
etc. when started in the usual way.")

(defvar mail-yank-hooks nil
  "*List of hook functions called after yanking a message into a *mail*
buffer.  See the documentation for the function vm-yank-message for details.")

(defvar vm-mode-map nil
  "Keymap for VM mode and VM Summary mode.")

(defvar vm-movemail-program "movemail"
  "Name of program to use to move mail from the system spool
to another location.  Normally this shoul be the movemail program
distributed with Emacs.")

(defvar vm-mode-line-format
  '("" (vm-buffer-modified-p vm-buffer-modified-p "-----")
    ("VM " vm-version ": %b"
     (vm-local-message-list
      ("   " vm-ml-message-number
       " (of " vm-ml-highest-message-number ")")
      "  (no messages)"))
    "   "
    global-mode-string
    (vm-local-message-list
     ("   %[{" vm-ml-attributes-string "}%]----")
     ("   %[%]----"))
    (-3 . "%p") "-%-"))

(defconst vm-version "5.15 (beta)"
  "Version number of VM.")

;; internal vars
(defvar vm-folder-type nil)
(make-variable-buffer-local 'vm-folder-type)
(defvar vm-message-list nil)
(defvar vm-local-message-list nil)
(make-variable-buffer-local 'vm-local-message-list)
(defvar vm-virtual-buffers nil)
(make-variable-buffer-local 'vm-virtual-buffers)
(defvar vm-real-buffers nil)
(make-variable-buffer-local 'vm-real-buffers)
(defvar vm-message-pointer nil)
(defvar vm-local-message-pointer nil)
(make-variable-buffer-local 'vm-local-message-pointer)
(defvar vm-folder-read-only nil)
(make-variable-buffer-local 'vm-folder-read-only)
(defvar vm-buffer-modified-p nil)
(make-variable-buffer-local 'vm-buffer-modified-p)
(defvar vm-last-message-pointer nil)
(make-variable-buffer-local 'vm-last-message-pointer)
(defvar vm-primary-inbox-p nil)
(make-variable-buffer-local 'vm-primary-inbox-p)
(defvar vm-visible-header-alist nil)
(make-variable-buffer-local 'vm-visible-header-alist)
(defvar vm-mail-buffer nil)
(make-variable-buffer-local 'vm-mail-buffer)
(defvar vm-summary-buffer nil)
(make-variable-buffer-local 'vm-summary-buffer)
(defvar vm-summary-pointer nil)
(make-variable-buffer-local 'vm-summary-pointer)
(defvar vm-system-state nil)
(make-variable-buffer-local 'vm-system-state)
(defvar vm-undo-record-list nil)
(make-variable-buffer-local 'vm-undo-record-list)
(defvar vm-undo-record-pointer nil)
(make-variable-buffer-local 'vm-undo-record-pointer)
(defvar vm-current-grouping nil)
(make-variable-buffer-local 'vm-current-grouping)
(defvar vm-last-save-folder nil)
(make-variable-buffer-local 'vm-last-save-folder)
(defvar vm-last-pipe-command nil)
(make-variable-buffer-local 'vm-last-pipe-command)
(defvar vm-messages-not-on-disk 0)
(make-variable-buffer-local 'vm-messages-not-on-disk)
(defvar vm-totals nil)
(make-variable-buffer-local 'vm-totals)
(defvar vm-messages-needing-display-update nil)
(make-variable-buffer-local 'vm-messages-needing-display-update)
(defvar vm-inhibit-write-file-hook nil)
(defvar vm-session-beginning t)
(defvar vm-rc-loaded nil)
(defconst vm-spool-directory
  (or (and (boundp 'rmail-spool-directory) rmail-spool-directory)
      "/usr/spool/mail/"))
(defconst vm-attributes-header-regexp
  "^X-VM-\\(Attributes\\|v5-Data\\):\\(.*\n\\([ \t]+.*\n\\)*\\)")
(defconst vm-attributes-header "X-VM-v5-Data:")
(defconst vm-bookmark-header-regexp
  "X-VM-Bookmark:\\(.*\n\\([ \t]+.*\n\\)*\\)")
(defconst vm-bookmark-header "X-VM-Bookmark:")
(defconst vm-vheader-header-regexp
  "X-VM-VHeader:\\(.*\n\\([ \t]+.*\n\\)*\\)")
(defconst vm-vheader-header "X-VM-VHeader:")
(defconst vm-berkeley-mail-status-header "Status: ")
(defconst vm-berkeley-mail-status-header-regexp "^Status: \\(..?\\)\n")
(defconst vm-generic-header-regexp
  "^\\([^: \t\n]+\\)[\t ]*:\\(.*\n\\([ \t]+.*\n\\)*\\)")
(defconst vm-header-regexp-format "^%s:[ \t]*\\(.*\\(\n[ \t]+.*\\)*\\)")
(defconst vm-supported-groupings-alist
  '(("physical-order") ("subject") ("author") ("date-sent") ("recipient")))
(defconst vm-attributes-vector-length 9)
(defconst vm-cache-vector-length 16)
(defconst vm-softdata-vector-length 4)
(defconst vm-virtual-data-vector-length 8)
(defconst vm-startup-message-lines
  (if (string-match "alpha\\|beta" vm-version)
      '("This is prerelease software.  Giving out copies of it means DEATH!"
	"See Revelations 6:8 for full details"
	"VM comes with ABSOLUTELY NO WARRANTY; type \\[vm-show-no-warranty] for full details"
	"In Stereo (where available)")
    '("You may give out copies of VM.  Type \\[vm-show-copying-restrictions] to see the conditions"
      "VM comes with ABSOLUTELY NO WARRANTY; type \\[vm-show-no-warranty] for full details"
      "In Stereo (where available)")))
(defconst vm-startup-message-displayed nil)
;; for the mode line
(defvar vm-ml-message-number nil)
(make-variable-buffer-local 'vm-ml-message-number)
(defvar vm-ml-highest-message-number nil)
(make-variable-buffer-local 'vm-ml-highest-message-number)
(defvar vm-ml-attributes-string nil)
(make-variable-buffer-local 'vm-ml-attributes-string)

;; general purpose macros and functions
(defmacro vm-marker (pos &optional buffer)
  (list 'set-marker '(make-marker) pos buffer))

(defmacro vm-increment (variable)
  (list 'setq variable (list '1+ variable)))

(defmacro vm-decrement (variable)
  (list 'setq variable (list '1- variable)))

(defmacro vm-set-folder-variables ()
  '(setq vm-message-list vm-local-message-list
	 vm-message-pointer vm-local-message-pointer))

(defmacro vm-select-folder-buffer ()
  '(progn
     (and vm-mail-buffer (buffer-name vm-mail-buffer)
	  (set-buffer vm-mail-buffer))
     (and (interactive-p) (vm-set-folder-variables))))

(defmacro vm-check-for-killed-summary ()
  '(and (bufferp vm-summary-buffer) (null (buffer-name vm-summary-buffer))
	(setq vm-summary-buffer nil)))

(defmacro vm-error-if-folder-read-only ()
  '(while vm-folder-read-only
     (signal 'folder-read-only (list (current-buffer)))))

(put 'folder-read-only 'error-conditions '(folder-read-only error))
(put 'folder-read-only 'error-message "Folder is read-only")

(defmacro vm-error-if-referenced-virtually ()
  '(and (setq vm-virtual-buffers (vm-trim-dead-buffers vm-virtual-buffers))
	(error "Can't execute command: folder is referenced virtually.")))

(defmacro vm-error-if-virtual-folder ()
  '(and (eq major-mode 'vm-virtual-mode)
	(error "%s cannot be applied to virtual folders." this-command)))

(defmacro vm-nuke-dead-virtual-buffers ()
  '(setq vm-virtual-buffers (vm-trim-dead-buffers vm-virtual-buffers)))

(defmacro vm-check-message-clipping ()
  '(and vm-virtual-buffers
	(or (< (point-min) (vm-start-of (car vm-message-pointer)))
	    (> (point-max) (vm-text-end-of (car vm-message-pointer))))
	(vm-preview-current-message)))

(defun vm-trim-dead-buffers (list)
  (let ((curr list) (prev nil))
    (while curr
      (if (buffer-name (car curr))
	  (setq prev curr
		curr (cdr curr))
	(if (null prev)
	    (setq list (cdr list)
		  curr list)
	  (setcdr prev (cdr curr))
	  (setq curr (cdr curr)))))
    list ))

(defun vm-abs (n) (if (< n 0) (- n) n))

;; save-restriction flubs restoring the clipping region if you
;; (widen) and modify text outside the old region.
;; This should do it right.
(defmacro vm-save-restriction (&rest forms)
  (let ((vm-sr-clip (make-symbol "vm-sr-clip"))
	(vm-sr-min (make-symbol "vm-sr-min"))
	(vm-sr-max (make-symbol "vm-sr-max")))
    (list 'let (list (list vm-sr-clip '(> (buffer-size)
					  (- (point-max) (point-min))))
		     ;; this shouldn't be necessary but the
		     ;; byte-compiler turns these into interned symbols
		     ;; which utterly defeats the purpose of the
		     ;; make-symbol calls above.  Soooo, until the compiler
		     ;; is fixed, these must be made into (let ...)
		     ;; temporaries so that nested calls to this macros
		     ;; won't misbehave.
		     vm-sr-min vm-sr-max)
	  (list 'and vm-sr-clip
		(list 'setq vm-sr-min '(set-marker (make-marker) (point-min)))
		(list 'setq vm-sr-max '(set-marker (make-marker) (point-max))))
	  (list 'unwind-protect (cons 'progn forms)
		'(widen)
		(list 'and vm-sr-clip
		      (list 'progn
			    (list 'narrow-to-region vm-sr-min vm-sr-max)
			    (list 'set-marker vm-sr-min nil)
			    (list 'set-marker vm-sr-max nil)))))))

(defmacro vm-save-buffer-excursion (&rest forms)
  (list 'let '((vm-sbe-buffer (current-buffer)))
	(list 'unwind-protect
	      (cons 'progn forms)
	      '(and (not (eq vm-sbe-buffer (current-buffer)))
		    (buffer-name vm-sbe-buffer)
		    (set-buffer vm-sbe-buffer)))))

(defmacro vm-within-current-message-buffer (&rest forms)
  (list 'let '((vm-sbe-buffer (current-buffer)))
	'(and (eq major-mode 'vm-virtual-mode) vm-message-list
	      (set-buffer (marker-buffer (vm-start-of
					  (car vm-message-pointer)))))
	(list 'unwind-protect
	      (cons 'progn forms)
	      '(and (not (eq vm-sbe-buffer (current-buffer)))
		    (buffer-name vm-sbe-buffer)
		    (set-buffer vm-sbe-buffer)))))

;; macros and functions dealing with accessing message struct fields

;; data that is always shared with virtual folders
(defmacro vm-virtual-data-of (message) (list 'aref message 0))
;; where message begins (From_ line)
(defmacro vm-start-of (message) (list 'aref (list 'aref message 0) 0))
;; where visible headers start
(defun vm-vheaders-of (message)
  (or (aref (aref message 0) 1)
      (progn (vm-reorder-message-headers message)
	     (aref (aref message 0) 1))))
;; where text section starts
(defun vm-text-of (message)
  (or (aref (aref message 0) 2) (progn (vm-find-and-set-text-of message)
				       (aref (aref message 0) 2))))
;; where message ends
(defmacro vm-end-of (message) (list 'aref (list 'aref message 0) 3))
;; if message is being edited, this is the buffer being used.
(defmacro vm-edit-buffer-of (message) (list 'aref (list 'aref message 0) 4))
;; message type; if nil, then default to folder type
(defmacro vm-message-type-of (message) (list 'aref (list 'aref message 0) 5))
;; link to real message
(defmacro vm-real-message-of (message) (list 'aref (list 'aref message 0) 6))
;; list of virtual messages referencing the above real message
(defmacro vm-virtual-messages-of (message) (list 'aref (list 'aref message 0) 7))
;; soft data vector
(defmacro vm-softdata-of (message) (list 'aref message 1))
(defmacro vm-number-of (message) (list 'aref (list 'aref message 1) 0))
(defmacro vm-mark-of (message) (list 'aref (list 'aref message 1) 1))
;; start of summary line
(defmacro vm-su-start-of (message) (list 'aref (list 'aref message 1) 2))
;; end of summary line
(defmacro vm-su-end-of (message) (list 'aref (list 'aref message 1) 3))
;; message attribute vector
(defmacro vm-attributes-of (message) (list 'aref message 2))
(defmacro vm-new-flag (message) (list 'aref (list 'aref message 2) 0))
(defmacro vm-unread-flag (message) (list 'aref (list 'aref message 2) 1))
(defmacro vm-deleted-flag (message) (list 'aref (list 'aref message 2) 2))
(defmacro vm-filed-flag (message) (list 'aref (list 'aref message 2) 3))
(defmacro vm-replied-flag (message) (list 'aref (list 'aref message 2) 4))
(defmacro vm-written-flag (message) (list 'aref (list 'aref message 2) 5))
(defmacro vm-forwarded-flag (message) (list 'aref (list 'aref message 2) 6))
(defmacro vm-edited-flag (message) (list 'aref (list 'aref message 2) 7))
;; message cached data
(defmacro vm-cache-of (message) (list 'aref message 3))
;; message size in bytes (as a string)
(defmacro vm-byte-count-of (message) (list 'aref (list 'aref message 3) 0))
;; weekday sent
(defmacro vm-weekday-of (message) (list 'aref (list 'aref message 3) 1))
;; month day
(defmacro vm-monthday-of (message) (list 'aref (list 'aref message 3) 2))
;; month sent
(defmacro vm-month-of (message) (list 'aref (list 'aref message 3) 3))
;; year sent
(defmacro vm-year-of (message) (list 'aref (list 'aref message 3) 4))
;; hour sent
(defmacro vm-hour-of (message) (list 'aref (list 'aref message 3) 5))
;; timezone
(defmacro vm-zone-of (message) (list 'aref (list 'aref message 3) 6))
;; message author's full name (Full-Name: or gouged from From:)
(defmacro vm-full-name-of (message) (list 'aref (list 'aref message 3) 7))
;; message author address (gouged from From:)
(defmacro vm-from-of (message) (list 'aref (list 'aref message 3) 8))
;; message ID (Message-Id:)
(defmacro vm-message-id-of (message) (list 'aref (list 'aref message 3) 9))
;; number of lines in message (as a string)
(defmacro vm-line-count-of (message) (list 'aref (list 'aref message 3) 10))
;; message subject (Subject:)
(defmacro vm-subject-of (message) (list 'aref (list 'aref message 3) 11))
;; Regexp that can be used to find the start of the already ordered headers.
(defmacro vm-vheaders-regexp-of (message)
  (list 'aref (list 'aref message 3) 12))
;; Addresses of recipients in a comma separated list
(defmacro vm-to-of (message) (list 'aref (list 'aref message 3) 13))
;; Full names of recipients in a comma separated list.  Addresses if
;; full names not available.
(defmacro vm-to-names-of (message) (list 'aref (list 'aref message 3) 14))
;; numeric month sent
(defmacro vm-month-number-of (message) (list 'aref (list 'aref message 3) 15))
;; modificaiton flag for this message
(defmacro vm-modflag-of (message) (list 'aref message 4))

(defmacro vm-set-virtual-data-of (message vdata) (list 'aset message 0 vdata))
(defmacro vm-set-start-of (message start) (list 'aset (list 'aref message 0) 0 start))
(defmacro vm-set-vheaders-of (message vh) (list 'aset (list 'aref message 0) 1 vh))
(defmacro vm-set-text-of (message text) (list 'aset (list 'aref message 0) 2 text))
(defmacro vm-set-end-of (message end) (list 'aset (list 'aref message 0) 3 end))
(defmacro vm-set-edit-buffer-of (message buf) (list 'aset (list 'aref message 0) 4 buf))
(defmacro vm-set-message-type-of (message type) (list 'aset (list 'aref message 0) 5 type))
(defmacro vm-set-real-message-of (message type) (list 'aset (list 'aref message 0) 6 type))
(defmacro vm-set-virtual-messages-of (message type) (list 'aset (list 'aref message 0) 7 type))
(defmacro vm-set-softdata-of (message data) (list 'aset message 1 data))
(defmacro vm-set-number-of (message n) (list 'aset (list 'aref message 1) 0 n))
(defmacro vm-set-mark-of (message val) (list 'aset (list 'aref message 1) 1 val))
(defmacro vm-set-su-start-of (message pos) (list 'aset (list 'aref message 1) 2 pos))
(defmacro vm-set-su-end-of (message pos) (list 'aset (list 'aref message 1) 3 pos))
(defmacro vm-set-attributes-of (message attrs) (list 'aset message 2 attrs))
;; The other routines in attributes group are part of the undo system.
(defmacro vm-set-edited-flag (message flag)
  (list 'aset (list 'aref message 2) 7 flag))
(defmacro vm-set-cache-of (message cache) (list 'aset message 3 cache))
(defmacro vm-set-byte-count-of (message count)
  (list 'aset (list 'aref message 3) 0 count))
(defmacro vm-set-weekday-of (message val)
  (list 'aset (list 'aref message 3) 1 val))
(defmacro vm-set-monthday-of (message val)
  (list 'aset (list 'aref message 3) 2 val))
(defmacro vm-set-month-of (message val)
  (list 'aset (list 'aref message 3) 3 val))
(defmacro vm-set-year-of (message val)
  (list 'aset (list 'aref message 3) 4 val))
(defmacro vm-set-hour-of (message val)
  (list 'aset (list 'aref message 3) 5 val))
(defmacro vm-set-zone-of (message val)
  (list 'aset (list 'aref message 3) 6 val))
(defmacro vm-set-full-name-of (message author)
  (list 'aset (list 'aref message 3) 7 author))
(defmacro vm-set-from-of (message author)
  (list 'aset (list 'aref message 3) 8 author))
(defmacro vm-set-message-id-of (message id)
  (list 'aset (list 'aref message 3) 9 id))
(defmacro vm-set-line-count-of (message count)
  (list 'aset (list 'aref message 3) 10 count))
(defmacro vm-set-subject-of (message subject)
  (list 'aset (list 'aref message 3) 11 subject))
(defmacro vm-set-vheaders-regexp-of (message regexp)
  (list 'aset (list 'aref message 3) 12 regexp))
(defmacro vm-set-to-of (message recips)
  (list 'aset (list 'aref message 3) 13 recips))
(defmacro vm-set-to-names-of (message recips)
  (list 'aset (list 'aref message 3) 14 recips))
(defmacro vm-set-month-number-of (message val)
  (list 'aset (list 'aref message 3) 15 val))
(defmacro vm-set-modflag-of (message val) (list 'aset message 4 val))

(defun vm-text-end-of (message)
  (- (vm-end-of message)
     (cond ((eq vm-folder-type 'mmdf) 5)
	   (t 1))))

(defun vm-make-message ()
  (let ((v (make-vector 5 nil)))
    (vm-set-softdata-of v (make-vector vm-softdata-vector-length nil))
    (vm-set-virtual-data-of v (make-vector vm-virtual-data-vector-length nil))
    (vm-set-real-message-of v v)
    v ))

;; init
(if vm-mode-map
    ()
  (setq vm-mode-map (make-keymap))
  (suppress-keymap vm-mode-map)
  (define-key vm-mode-map "h" 'vm-summarize)
  (define-key vm-mode-map "\M-n" 'vm-next-unread-message)
  (define-key vm-mode-map "\M-p" 'vm-previous-unread-message)
  (define-key vm-mode-map "n" 'vm-next-message)
  (define-key vm-mode-map "p" 'vm-previous-message)
  (define-key vm-mode-map "N" 'vm-Next-message)
  (define-key vm-mode-map "P" 'vm-Previous-message)
  (define-key vm-mode-map "\t" 'vm-goto-message-last-seen)
  (define-key vm-mode-map "\r" 'vm-goto-message)
  (define-key vm-mode-map "t" 'vm-expose-hidden-headers)
  (define-key vm-mode-map " " 'vm-scroll-forward)
  (define-key vm-mode-map "b" 'vm-scroll-backward)
  (define-key vm-mode-map "\C-?" 'vm-scroll-backward)
  (define-key vm-mode-map "d" 'vm-delete-message)
  (define-key vm-mode-map "\C-d" 'vm-delete-message-backward)
  (define-key vm-mode-map "u" 'vm-undelete-message)
  (define-key vm-mode-map "U" 'vm-unread-message)
  (define-key vm-mode-map "e" 'vm-edit-message)
  (define-key vm-mode-map "j" 'vm-discard-cached-data)
  (define-key vm-mode-map "k" 'vm-kill-subject)
  (define-key vm-mode-map "f" 'vm-followup)
  (define-key vm-mode-map "F" 'vm-followup-include-text)
  (define-key vm-mode-map "r" 'vm-reply)
  (define-key vm-mode-map "R" 'vm-reply-include-text)
  (define-key vm-mode-map "\M-r" 'vm-resend-bounced-message)
  (define-key vm-mode-map "z" 'vm-forward-message)
  (define-key vm-mode-map "@" 'vm-send-digest)
  (define-key vm-mode-map "*" 'vm-burst-digest)
  (define-key vm-mode-map "m" 'vm-mail)
  (define-key vm-mode-map "g" 'vm-get-new-mail)
  (define-key vm-mode-map "G" 'vm-group-messages)
  (define-key vm-mode-map "v" 'vm-visit-folder)
  (define-key vm-mode-map "V" 'vm-visit-virtual-folder)
  (define-key vm-mode-map "s" 'vm-save-message)
  (define-key vm-mode-map "w" 'vm-save-message-sans-headers)
  (define-key vm-mode-map "A" 'vm-auto-archive-messages)
  (define-key vm-mode-map "S" 'vm-save-folder)
  (define-key vm-mode-map "|" 'vm-pipe-message-to-command)
  (define-key vm-mode-map "#" 'vm-expunge-folder)
  (define-key vm-mode-map "q" 'vm-quit)
  (define-key vm-mode-map "x" 'vm-quit-no-change)
  (define-key vm-mode-map "?" 'vm-help)
  (define-key vm-mode-map "\C-_" 'vm-undo)
  (define-key vm-mode-map "\C-xu" 'vm-undo)
  (define-key vm-mode-map "!" 'shell-command)
  (define-key vm-mode-map "<" 'vm-beginning-of-message)
  (define-key vm-mode-map ">" 'vm-end-of-message)
  (define-key vm-mode-map "\M-s" 'vm-isearch-forward)
  (define-key vm-mode-map "=" 'vm-summarize)
  (define-key vm-mode-map "L" 'vm-load-rc)
  (define-key vm-mode-map "\C-c\r" 'vm-next-command-uses-marks)
  (define-key vm-mode-map "\C-c\C-@" 'vm-mark-message) 
  (define-key vm-mode-map "\C-c " 'vm-unmark-message)
  (define-key vm-mode-map "\C-c\C-a" 'vm-mark-all-messages)
  (define-key vm-mode-map "\C-ca" 'vm-clear-all-marks)
  (define-key vm-mode-map "\M-C" 'vm-show-copying-restrictions)
  (define-key vm-mode-map "\M-W" 'vm-show-no-warranty))

(defun vm-mark-for-display-update (message)
  (setq vm-messages-needing-display-update
	(cons message vm-messages-needing-display-update)))

(defun vm-last (list) (while (cdr-safe list) (setq list (cdr list))) list)

(defun vm-vector-to-list (vector)
  (let ((i (1- (length vector)))
	list)
    (while (>= i 0)
      (setq list (cons (aref vector i) list))
      (vm-decrement i))
    list ))

(defun vm-extend-vector (vector length &optional fill)
  (let ((vlength (length vector)))
    (if (< vlength length)
	(apply 'vector (nconc (vm-vector-to-list vector)
			      (make-list (- length vlength) fill)))
      vector )))

(put 'folder-empty 'error-conditions '(folder-empty error))
(put 'folder-empty 'error-message "Folder is empty")

(defun vm-error-if-folder-empty ()
  (while (null vm-message-list)
    (signal 'folder-empty nil)))

(defun vm-proportion-windows ()
  (vm-select-folder-buffer)
  (vm-within-current-message-buffer
   ;; don't attempt proportioning if there aren't exactly two windows.
   (if (and (not (one-window-p t))
	    (eq (selected-window)
		(next-window (next-window (selected-window) 0) 0)))
       (if (= (window-width) (screen-width))
	   (let ((mail-w (get-buffer-window (current-buffer)))
		 (n (- (window-height (get-buffer-window (current-buffer)))
		       (/ (* vm-mail-window-percentage
			     (- (screen-height)
				(window-height (minibuffer-window))))
			  100)))
		 (old-w (selected-window)))
	     (if mail-w
		 (save-excursion
		   (select-window mail-w)
		   (shrink-window n)
		   (select-window old-w)
		   (and (memq major-mode '(vm-summary-mode vm-virtual-mode))
			(vm-auto-center-summary)))))
	 (let ((mail-w (get-buffer-window (current-buffer)))
	       (n (- (window-width (get-buffer-window (current-buffer)))
		     (/ (* vm-mail-window-percentage (screen-width))
			100)))
	       (old-w (selected-window)))
	   (if mail-w
	       (save-excursion
		 (select-window mail-w)
		 (shrink-window-horizontally n)
		 (select-window old-w)
		 (and (memq major-mode '(vm-summary-mode vm-virtual-mode))
		      (vm-auto-center-summary)))))))))

(defun vm-number-messages (&optional start-point)
  (setq vm-totals nil)
  (let ((n 1) (message-list vm-local-message-list))
    (if start-point
	(while (not (eq message-list start-point))
	  (setq n (1+ n) message-list (cdr message-list))))
    (while message-list
      (vm-set-number-of (car message-list) (int-to-string n))
      (setq n (1+ n) message-list (cdr message-list)))
    (setq vm-ml-highest-message-number (int-to-string (1- n)))
    (if vm-summary-buffer
	(let ((ml-h vm-ml-highest-message-number))
	  (save-excursion
	    (set-buffer vm-summary-buffer)
	    (setq vm-ml-highest-message-number ml-h))))))

(defun vm-match-visible-header (alist)
  (catch 'match
    (while alist
      (if (looking-at (car (car alist)))
	  (throw 'match (car alist)))
      (setq alist (cdr alist)))
    nil))

(defun vm-get-folder-type ()
  (save-excursion
    (save-restriction
      (widen)
      (goto-char (point-min))
      (skip-chars-forward "\n")
      (cond ((looking-at "From ") 'From_)
	    ((looking-at "\001\001\001\001\n") 'mmdf)
	    ((looking-at "^$") 'From_)))))

;; Build a chain of message structures.
;; Find the start and end of each message and fill in the relevant
;; fields in the message structures.

(defun vm-build-message-list ()
  (save-excursion
    (vm-build-visible-header-alist)
    (let (tail-cons message prev-message case-fold-search marker
	  start-regexp sep-pattern trailer-length)
      (if (eq vm-folder-type 'mmdf)
	  (setq start-regexp "^\001\001\001\001\n"
		separator-string "\n\001\001\001\001\n\001\001\001\001"
		trailer-length 6)
	(setq start-regexp "^From "
	      separator-string "\n\nFrom "
	      trailer-length 2))
      (if vm-local-message-list
	  (let ((mp vm-local-message-list)
		(end (point-min)))
	    (while mp
	      (if (< end (vm-end-of (car mp)))
		  (setq end (vm-end-of (car mp))))
	      (setq mp (cdr mp)))
	    ;; move back past trailer so separator-string will match below
	    (goto-char (- end trailer-length))
	    (setq tail-cons (vm-last vm-local-message-list)))
	(goto-char (point-min))
	(if (looking-at start-regexp)
	    (progn
	      (setq message (vm-make-message) prev-message message)
	      (vm-set-start-of message (vm-marker (match-beginning 0)))
	      (setq vm-local-message-list (list message)
		    tail-cons vm-local-message-list))))
      (while (search-forward separator-string nil t)
	(setq marker (vm-marker (+ trailer-length (match-beginning 0)))
	      message (vm-make-message))
	(vm-set-start-of message marker)
	(if prev-message
	    (vm-set-end-of prev-message marker))
	(if tail-cons
	    (progn
	      (setcdr tail-cons (list message))
	      (setq tail-cons (cdr tail-cons)
		    prev-message message))
	  (setq vm-local-message-list (list message)
		tail-cons vm-local-message-list
		prev-message message)))
      (if prev-message
	  (vm-set-end-of prev-message (vm-marker (point-max)))))))

(defun vm-build-visible-header-alist ()
  (let ((header-alist (cons nil nil))
	(vheaders vm-visible-headers)
	list)
    (setq list header-alist)
    (while vheaders
      (setcdr list (cons (cons (car vheaders) nil) nil))
      (setq list (cdr list) vheaders (cdr vheaders)))
    (setq vm-visible-header-alist (cdr header-alist))))

;; Group the headers that the user wants to see at the end of the headers
;; section so we can narrow to them.  The vheaders field of the
;; message struct is set.  This function is called on demand whenever
;; a vheaders field is discovered to be nil for a particular message.

(defun vm-reorder-message-headers (message)
  (save-excursion
    ;; if there is a cached regexp that points to the ordered headers
    ;; then use it and avoid a lot of work.
    (if (and (vm-vheaders-regexp-of message)
	     (progn (goto-char (vm-start-of message))
		    (re-search-forward (vm-vheaders-regexp-of message)
				       (vm-text-of message) t)))
	(vm-set-vheaders-of message (vm-marker (match-beginning 0)))
      ;; oh well, we gotta do it the hard way.
      ;;
      ;; vm-visible-header-alist is an assoc list version of
      ;; vm-visible-headers.  When a matching header is found,
      ;; the header is stuffed into its corresponding assoc cell
      ;; and the header text is deleted from the buffer.  After all
      ;; the visible headers have been collected, they are inserted
      ;; into the buffer in a clump at the end of the header section.
      (vm-save-restriction
       (let ((header-alist vm-visible-header-alist)
	     list buffer-read-only match-end-0 extras
	     (inhibit-quit t)
	     ;; This prevents file locking from occuring.  Disabling
	     ;; locking can speed things noticably if the lock directory
	     ;; is on a slow device.  We don't need locking here because
	     ;; in a mail context reordering headers is harmless.
	     (buffer-file-name nil)
	     (old-buffer-modified-p (buffer-modified-p)))
	 (goto-char (vm-start-of message))
	 (forward-line)
	 (while (and (not (= (following-char) ?\n))
		     (looking-at vm-generic-header-regexp))
	   (setq match-end-0 (match-end 0)
		 list (vm-match-visible-header header-alist))
	   (if (and (null list)
		    (or (null vm-invisible-header-regexp)
			(looking-at vm-invisible-header-regexp)))
	       (goto-char match-end-0)
	     (if list
		 (if (cdr list)
		     (setcdr list 
			     (concat
			      (cdr list)
			      (buffer-substring (point) match-end-0)))
		   (setcdr list (buffer-substring (point) match-end-0)))
	       (setq extras
		     (cons (buffer-substring (point) match-end-0) extras)))
	     (delete-region (point) match-end-0)))
	 (vm-set-vheaders-of message (point-marker))
	 (save-excursion
	   ;; now dump out the visible headers
	   ;; the vm-visible-headers go first
	   (setq list header-alist)
	   (while list
	     (if (cdr (car list))
		 (progn
		   (insert (cdr (car list)))
		   (setcdr (car list) nil)))
	     (setq list (cdr list)))
	   ;; now the headers that were not explicitly ignored, if any.
	   (if extras
	       (progn
		 (setq extras (nreverse extras))
		 (while extras
		   (insert (car extras))
		   (setq extras (cdr extras)))))
	   (set-buffer-modified-p old-buffer-modified-p))
	 ;; cache a regular expression that can be used to find the start of
	 ;; the reordered header the next time this folder is visited.
	 (if (looking-at vm-generic-header-regexp)
	     (vm-set-vheaders-regexp-of
	      message
	      (concat "^" (buffer-substring (match-beginning 1) (match-end 1))
		      ":"))))))))

(defun vm-find-and-set-text-of (m)
  (save-excursion
    (save-restriction
      (widen)
      (goto-char (vm-start-of m))
      (forward-line 1)
      (search-forward "\n\n" (vm-text-end-of m) t)
      (vm-set-text-of m (point-marker)))))

;; Reads the message attributes and cached header information from the
;; header portion of the each message, if our X-VM- attributes header is
;; present.  If the header is not present, assume the message is new,
;; unless we are being compatible with Berkeley Mail in which case we
;; also check for a Status header.
;;
;; If a message already has attributes don't bother checking the
;; headers.
;;
;; This function also discovers and stores the position where the
;; message text begins.
;;
;; Totals are gathered for use by vm-emit-totals-blurb.
;;
;; Supports version 4 format of attribute storage, for backward compatibility.

(defun vm-read-attributes ()
  (save-excursion
    (let ((mp vm-local-message-list)
	  (vm-new-count 0)
	  (vm-unread-count 0)
	  (vm-total-count 0)
	  data)
      (while mp
	(vm-increment vm-total-count)
	(if (vm-attributes-of (car mp))
	    ()
	  (goto-char (vm-start-of (car mp)))
	  ;; find start of text section and save it
	  (search-forward "\n\n" (vm-text-end-of (car mp)) 0)
	  (vm-set-text-of (car mp) (point-marker))
	  ;; now look for our header
	  (goto-char (vm-start-of (car mp)))
	  (cond
	   ((re-search-forward vm-attributes-header-regexp
			       (vm-text-of (car mp)) t)
	    (goto-char (match-beginning 2))
	    (condition-case ()
		(setq data (read (current-buffer)))
	      (error (setq data
			   (list
			    (make-vector vm-attributes-vector-length nil)
			    (make-vector vm-cache-vector-length nil)))
		     ;; In lieu of a valid attributes header
		     ;; assume the message is new.
		     (aset (car data) 0 t)))
	    ;; support version 4 format
	    (cond ((vectorp data)
		   (setq data (vm-convert-v4-attributes data)))
		  (t
		   ;; extend vectors if necessary to accomodate
		   ;; more caching and attributes without alienating
		   ;; other version 5 folders.
		   (cond ((< (length (car data))
			     vm-attributes-vector-length)
			  (setcar data (vm-extend-vector
					(car data)
					vm-attributes-vector-length))))
		   (cond ((< (length (car (cdr data)))
			     vm-cache-vector-length)
			  (setcar (cdr data)
				  (vm-extend-vector
				   (car (cdr data))
				   vm-cache-vector-length))))))
	    (vm-set-attributes-of (car mp) (car data))
	    (vm-set-cache-of (car mp) (car (cdr data))))
	   ((and vm-berkeley-mail-compatibility
		 (re-search-forward vm-berkeley-mail-status-header-regexp
				    (vm-text-of (car mp)) t))
	    (goto-char (match-beginning 1))
	    (vm-set-attributes-of
	     (car mp)
	     (make-vector vm-attributes-vector-length nil))
	    (vm-set-unread-flag (car mp) (not (looking-at ".*R.*")) t)
	    (vm-set-cache-of (car mp) (make-vector vm-cache-vector-length
						   nil)))
	   (t
	    (vm-set-attributes-of
	     (car mp)
	     (make-vector vm-attributes-vector-length nil))
	    (vm-set-new-flag (car mp) t t)
	    (vm-set-cache-of (car mp) (make-vector vm-cache-vector-length
						   nil)))))
	(cond ((vm-deleted-flag (car mp))) ; don't count deleted messages
	      ((vm-new-flag (car mp))
	       (vm-increment vm-new-count))
	      ((vm-unread-flag (car mp))
	       (vm-increment vm-unread-count)))
	(setq mp (cdr mp)))
      (setq vm-totals (list vm-total-count vm-new-count vm-unread-count)))))

(defun vm-convert-v4-attributes (data)
  (list (apply 'vector
	       (nconc (vm-vector-to-list data)
		      (make-list (- vm-attributes-vector-length
				    (length data))
				 nil)))
	(make-vector vm-cache-vector-length nil)))

;; Go to the message specified in a bookmark and eat the bookmark.
;; Returns non-nil if sucessful, nil otherwise.
(defun vm-gobble-bookmark ()
  (let ((old-buffer-modified-p (buffer-modified-p))
	buffer-read-only n
	;; This prevents file locking from occuring.  Disabling
	;; locking can speed things noticably if the lock
	;; directory is on a slow device.  We don't need locking
	;; here because the user shouldn't care about VM removing
	;; its own status headers.
	(buffer-file-name nil)
	(inhibit-quit t))
    (save-excursion
      (vm-save-restriction
       (let (lim)
	 (widen)
	 (goto-char (point-min))
	 (forward-line)
	 (search-forward "\n\n" nil t)
	 (setq lim (point))
	 (goto-char (point-min))
	 (forward-line)
	 (if (re-search-forward vm-bookmark-header-regexp lim t)
	     (progn
	       (setq n (string-to-int
			(buffer-substring (match-beginning 1) (match-end 1))))
	       (delete-region (match-beginning 0) (match-end 0))))))
    (set-buffer-modified-p old-buffer-modified-p)
    (if (null n)
	nil
      (condition-case ()
	  (vm-goto-message n)
	(error nil))
      t ))))

(defun vm-check-header-variables ()
  (save-excursion
    (vm-save-restriction
     (let (lim)
       (widen)
       (goto-char (point-min))
       (forward-line)
       (search-forward "\n\n" nil t)
       (setq lim (point))
       (goto-char (point-min))
       (forward-line)
       (if (re-search-forward vm-vheader-header-regexp lim t)
	   (let ((old-buffer-modified-p (buffer-modified-p))
		 ;; This prevents file locking from occuring.  Disabling
		 ;; locking can speed things noticably if the lock
		 ;; directory is on a slow device.  We don't need locking
		 ;; here because the user shouldn't care about VM removing
		 ;; its own status headers.
		 (buffer-file-name nil)
		 buffer-read-only vis invis got
		 (inhibit-quit t))
	     (goto-char (match-beginning 1))
	     (condition-case ()
		 (setq vis (read (current-buffer))
		       invis (read (current-buffer))
		       got t)
	       (error nil))
	     (delete-region (match-beginning 0) (match-end 0))
	     ;; if the variables don't match the values stored when this
	     ;; folder was saved, then we have to discard any cached
	     ;; vheader info so the user will see the right headers.
	     (and got (or (not (equal vis vm-visible-headers))
			  (not (equal invis vm-invisible-header-regexp)))
		  (let ((mp vm-local-message-list))
		    (message "Discarding visible header info...")
		    (while mp
		      (vm-set-vheaders-regexp-of (car mp) nil)
		      (vm-set-vheaders-of (car mp) nil)
		      (setq mp (cdr mp)))))
	     (set-buffer-modified-p old-buffer-modified-p)))))))

;; Stuff the message attributes back into the message as headers.
(defun vm-stuff-attributes (m &optional suppress-delete)
  (save-excursion
    (vm-save-restriction
     (widen)
     (let ((old-buffer-modified-p (buffer-modified-p))
	   attributes cache buffer-read-only
	   ;; This prevents file locking from occuring.  Disabling
	   ;; locking can speed things noticably if the lock
	   ;; directory is on a slow device.  We don't need locking
	   ;; here because the user shouldn't care about VM stuffing
	   ;; its own status headers.
	   (buffer-file-name nil)
	   (delflag (vm-deleted-flag m))
	   (inhibit-quit t))
       (setq attributes (vm-attributes-of m)
	     cache (vm-cache-of m))
       (and delflag suppress-delete
	    (vm-set-deleted-flag-in-vector attributes nil))
       (goto-char (vm-start-of m))
       (forward-line)
       (if (re-search-forward vm-attributes-header-regexp
			      (vm-text-of m) t)
	   (delete-region (match-beginning 0) (match-end 0)))
       (insert-before-markers vm-attributes-header " ("
			      (let ((print-escape-newlines t))
				(prin1-to-string attributes))
			      "\n\t"
			      (let ((print-escape-newlines t))
				(prin1-to-string cache))
			      ")\n")
       (cond (vm-berkeley-mail-compatibility
	      (goto-char (vm-start-of m))
	      (forward-line)
	      (if (re-search-forward vm-berkeley-mail-status-header-regexp
				     (vm-text-of m) t)
		  (delete-region (match-beginning 0) (match-end 0)))
	      (cond ((not (vm-new-flag m))
		     (insert-before-markers
		      vm-berkeley-mail-status-header
		      (if (vm-unread-flag m) "" "R")
		      "O\n")))))
       (and delflag suppress-delete
	    (vm-set-deleted-flag-in-vector attributes t))
       (set-buffer-modified-p old-buffer-modified-p)))))

;; Insert a bookmark into the argument message.
(defun vm-stuff-bookmark ()
  (if vm-message-pointer
      (save-excursion
	(vm-save-restriction
	 (widen)
	 (let ((old-buffer-modified-p (buffer-modified-p))
	       ;; This prevents file locking from occuring.  Disabling
	       ;; locking can speed things noticably if the lock
	       ;; directory is on a slow device.  We don't need locking
	       ;; here because the user shouldn't care about VM stuffing
	       ;; its own status headers.
	       (buffer-file-name nil)
	       buffer-read-only lim
	       (inhibit-quit t))
	   (goto-char (point-min))
	   (forward-line)
	   (search-forward "\n\n" nil t)
	   (setq lim (point))
	   (goto-char (point-min))
	   (forward-line)
	   (if (re-search-forward vm-bookmark-header-regexp lim t)
	       (delete-region (match-beginning 0) (match-end 0)))
	   (insert-before-markers vm-bookmark-header " "
				  (vm-number-of (car vm-message-pointer))
				  "\n")
	   (set-buffer-modified-p old-buffer-modified-p))))))

;; stuff the current values of the header variables for future messages.
(defun vm-stuff-header-variables ()
  (if vm-message-pointer
      (save-excursion
	(vm-save-restriction
	 (widen)
	 (let ((old-buffer-modified-p (buffer-modified-p))
	       (print-escape-newlines t)
	       buffer-read-only lim
	       ;; This prevents file locking from occuring.  Disabling
	       ;; locking can speed things noticably if the lock
	       ;; directory is on a slow device.  We don't need locking
	       ;; here because the user shouldn't care about VM stuffing
	       ;; its own status headers.
	       (buffer-file-name nil)
	       (inhibit-quit t))
	   (goto-char (point-min))
	   (forward-line)
	   (search-forward "\n\n" nil t)
	   (setq lim (point))
	   (goto-char (point-min))
	   (forward-line)
	   (if (re-search-forward vm-vheader-header-regexp lim t)
	       (delete-region (match-beginning 0) (match-end 0)))
	   (insert-before-markers vm-vheader-header " "
				  (prin1-to-string vm-visible-headers) " "
				  (prin1-to-string vm-invisible-header-regexp)
				  "\n")
	   (set-buffer-modified-p old-buffer-modified-p))))))

(defun vm-change-all-new-to-unread ()
  (let ((mp vm-message-list))
    (while mp
      (if (vm-new-flag (car mp))
	  (progn
	    (vm-set-new-flag (car mp) nil)
	    (vm-set-unread-flag (car mp) t)))
      (setq mp (cdr mp)))))

(defun vm-force-mode-line-update ()
  (save-excursion
    (set-buffer (other-buffer))
    (set-buffer-modified-p (buffer-modified-p))))

(defun vm-update-summary-and-mode-line ()
  (setq vm-ml-message-number (vm-number-of (car vm-message-pointer)))
  (cond ((vm-new-flag (car vm-message-pointer))
	 (setq vm-ml-attributes-string "new"))
	((vm-unread-flag (car vm-message-pointer))
	 (setq vm-ml-attributes-string "unread"))
	(t (setq vm-ml-attributes-string "read")))
  (cond ((vm-edited-flag (car vm-message-pointer))
	 (setq vm-ml-attributes-string
	       (concat vm-ml-attributes-string " edited"))))
  (cond ((vm-filed-flag (car vm-message-pointer))
	 (setq vm-ml-attributes-string
	       (concat vm-ml-attributes-string " filed"))))
  (cond ((vm-written-flag (car vm-message-pointer))
	 (setq vm-ml-attributes-string
	       (concat vm-ml-attributes-string " written"))))
  (cond ((vm-replied-flag (car vm-message-pointer))
	 (setq vm-ml-attributes-string
	       (concat vm-ml-attributes-string " replied"))))
  (cond ((vm-forwarded-flag (car vm-message-pointer))
	 (setq vm-ml-attributes-string
	       (concat vm-ml-attributes-string " forwarded"))))
  (cond ((vm-deleted-flag (car vm-message-pointer))
	 (setq vm-ml-attributes-string
	       (concat vm-ml-attributes-string " deleted"))))
  (cond ((vm-mark-of (car vm-message-pointer))
	 (setq vm-ml-attributes-string
	       (concat vm-ml-attributes-string " MARKED"))))
  (if (and vm-summary-buffer (not vm-real-buffers))
      (let ((ml-a vm-ml-attributes-string)
	    (ml-n vm-ml-message-number)
	    (ml-h vm-ml-highest-message-number)
	    (ml-b vm-buffer-modified-p)
	    (ml-l vm-local-message-list))
	(save-excursion
	  (set-buffer vm-summary-buffer)
	  (setq vm-ml-attributes-string ml-a
		vm-ml-message-number ml-n
		vm-ml-highest-message-number ml-h
		vm-buffer-modified-p ml-b
		vm-local-message-list ml-l))))
  (while vm-messages-needing-display-update
    (vm-update-message-summary (car vm-messages-needing-display-update))
    (setq vm-messages-needing-display-update
	  (cdr vm-messages-needing-display-update)))
  (vm-force-mode-line-update))

(defun vm-highlight-headers (message window)
  (and vm-highlighted-header-regexp window
       (<= (window-start window) (vm-text-of message))
       (save-excursion
	 ;; As of v18.52, this call to save-window-excursion is needed!
	 ;; Somehow window point can get fouled in here, and drag the
	 ;; buffer point along with it.  This problem only manifests
	 ;; itself when operating VM from the summary buffer, subsequent
	 ;; to using vm-beginning-of-message or vm-end-of-message.
	 ;; After running a next or previous message command, point
	 ;; somehow ends up at the end of the message.
	 (save-window-excursion
	   (goto-char (window-start window))
	   (while (re-search-forward vm-highlighted-header-regexp
				     (vm-text-of message) t)
	     (save-restriction
	       (goto-char (match-beginning 0))
	       (if (looking-at vm-generic-header-regexp)
		   (progn
		     (goto-char (match-beginning 2))
		     (narrow-to-region (point-min) (point))
		     (sit-for 0)
		     (setq inverse-video t)
		     (goto-char (point-min))
		     (widen)
		     (narrow-to-region (point) (match-end 2))
		     (sit-for 0)
		     (setq inverse-video nil)
		     (goto-char (match-end 0)))
		 (goto-char (match-end 0)))))))))

(defun vm-preview-current-message ()
  (setq vm-system-state 'previewing)
  (vm-within-current-message-buffer
   (widen)
   (narrow-to-region
    (vm-vheaders-of (car vm-message-pointer))
    (if vm-preview-lines
	(min
	 (vm-text-end-of (car vm-message-pointer))
	 (save-excursion
	   (goto-char (vm-text-of (car vm-message-pointer)))
	   (forward-line (if (natnump vm-preview-lines) vm-preview-lines 0))
	   (point)))
      (vm-text-of (car vm-message-pointer))))
   (if vm-honor-page-delimiters
       (vm-narrow-to-page))
   (goto-char (vm-text-of (car vm-message-pointer)))
   ;; If we have a window, set window start appropriately.
   ;; Highlight appropriate headers if current buffer is visible.
   (let ((w (get-buffer-window (current-buffer))))
     (if w (set-window-start w (point-min)))
     (vm-highlight-headers (car vm-message-pointer) w)))
  ;; De Morgan's Theorems could clear away most of the following negations,
  ;; but the resulting code would be horribly obfuscated.
  (if (or (null vm-preview-lines)
	  (and (not vm-preview-read-messages)
	       (not (vm-new-flag (car vm-message-pointer)))
	       (not (vm-unread-flag (car vm-message-pointer)))))
      (vm-show-current-message)
    (vm-update-summary-and-mode-line)))

(defun vm-show-current-message ()
  (setq vm-system-state 'showing)
  (vm-within-current-message-buffer
   (save-excursion
     (goto-char (point-min))
     (widen)
     (narrow-to-region (point) (vm-text-end-of (car vm-message-pointer))))
   (if vm-honor-page-delimiters
       (progn
	 (if (looking-at page-delimiter)
	     (forward-page 1))
	 (vm-narrow-to-page))))
  (cond ((vm-new-flag (car vm-message-pointer))
	 (vm-set-new-flag (car vm-message-pointer) nil))
	((vm-unread-flag (car vm-message-pointer))
	 (vm-set-unread-flag (car vm-message-pointer) nil)))
  (vm-update-summary-and-mode-line))

(defun vm-unread-message (&optional count)
  "Set the `unread' attribute for the current message.  If the message is
already new or unread, then it left unchanged.

Numeric prefix argument N mans to unread the current message plus the
next N-1 messages.  A negative N means unread the current message and
the previous N-1 messages.

When invoked on marked messages (via vm-next-command-uses-marks),
all marked messages are affected, other messages are ignored."
  (interactive "p")
  (or count (setq count 1))
  (vm-follow-summary-cursor)
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (vm-error-if-folder-empty)
  (let ((mlist (vm-select-marked-or-prefixed-messages count)))
    (while mlist
      (if (and (not (vm-unread-flag (car mlist)))
	       (not (vm-new-flag (car mlist))))
	  (vm-set-unread-flag (car mlist) t))
      (setq mlist (cdr mlist))))
  (vm-update-summary-and-mode-line))

(defun vm-expose-hidden-headers ()
  "Toggle exposing and hiding message headers that are normally not visible."
  (interactive)
  (vm-follow-summary-cursor)
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (vm-error-if-folder-empty)
  (vm-within-current-message-buffer
   (goto-char (point-min))
   (vm-widen-page)
   (let ((exposed (= (point-min) (vm-start-of (car vm-message-pointer)))))
     (goto-char (point-max))
     (widen)
     (if exposed
	 (narrow-to-region (point) (vm-vheaders-of (car vm-message-pointer)))
       (narrow-to-region (point) (vm-start-of (car vm-message-pointer))))
     (goto-char (point-min))
     (let (w)
       (setq w (get-buffer-window (current-buffer)))
       (and w (set-window-point w (point-min)))
       (and w
	    (= (window-start w) (vm-vheaders-of (car vm-message-pointer)))
	    (not exposed)
	    (set-window-start w (vm-start-of (car vm-message-pointer)))))
     (if vm-honor-page-delimiters
	 (vm-narrow-to-page)))))

(defun vm-display-current-message-buffer (&optional no-highlighting)
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (vm-within-current-message-buffer
   (if (null (get-buffer-window (current-buffer)))
       (if vm-mutable-windows
	   (let ((pop-up-windows (and pop-up-windows
				      (eq vm-mutable-windows t))))
	     (display-buffer (current-buffer)))
	 (switch-to-buffer (current-buffer))))
   (let ((w (get-buffer-window (current-buffer))))
     (if (>= (window-start w) (point-max))
	 (set-window-start w (point-min) t))))
  (if (and vm-summary-buffer (get-buffer-window vm-summary-buffer)
	   (eq vm-mutable-windows t))
      (vm-proportion-windows))
  (vm-within-current-message-buffer
   (if (not no-highlighting)
       (vm-highlight-headers (car vm-message-pointer)
			     (get-buffer-window (current-buffer))))))

(defun vm-widen-page ()
  (if (or (> (point-min) (vm-text-of (car vm-message-pointer)))
	  (/= (point-max) (vm-text-end-of (car vm-message-pointer))))
      (narrow-to-region (vm-vheaders-of (car vm-message-pointer))
			(vm-text-end-of (car vm-message-pointer)))))

(defun vm-narrow-to-page ()
  (save-excursion
    (let (min max (omin (point-min)) (omax (point-max)))
      (if (or (bolp) (not (save-excursion
			    (beginning-of-line)
			    (looking-at page-delimiter))))
	  (forward-page -1))
      (setq min (point))
      (forward-page 1)
      (beginning-of-line)
      (setq max (point))
      (narrow-to-region min max))))

(defun vm-scroll-forward (&optional arg)
  "Scroll forward a screenful of text.
If the current message is being previewed, the message body is revealed.
If at the end of the current message, move to the next message."
  (interactive "P")
  (let ((mp-changed (vm-follow-summary-cursor)) was-invisible do-next-message)
    (vm-select-folder-buffer)
    (vm-sanity-check-modification-flag)
    (vm-check-for-killed-summary)
    (vm-error-if-folder-empty)
    (if (vm-within-current-message-buffer
	 (null (get-buffer-window (current-buffer))))
	(progn
	  (vm-display-current-message-buffer)
	  (setq was-invisible t)))
    (if (eq vm-system-state 'previewing)
	(vm-show-current-message)
      (if (or mp-changed was-invisible)
	  ()
	(setq vm-system-state 'reading)
	(vm-within-current-message-buffer
	 (let ((w (get-buffer-window (current-buffer)))
	       (old-w (selected-window))
	       (direction (prefix-numeric-value arg))
	       error-data)
	   (unwind-protect
	       (progn
		 (select-window w)
		 (while
		     (catch 'tryagain
		       (if (not
			    (eq
			     (condition-case error-data
				 (scroll-up arg)
			       (error
				(if (or (and (< direction 0)
					     (> (point-min)
						(vm-text-of
						 (car vm-message-pointer))))
					(and (>= direction 0)
					     (/= (point-max)
						 (vm-text-end-of
						  (car vm-message-pointer)))))
				    (progn
				      (vm-widen-page)
				      (if (>= direction 0)
					  (progn
					    (forward-page 1)
					    (set-window-start w (point))
					    nil )
					(if (or
					     (bolp)
					     (not
					      (save-excursion
						(beginning-of-line)
						(looking-at page-delimiter))))
					    (forward-page -1))
					(beginning-of-line)
					(set-window-start w (point))
					(throw 'tryagain t)))
				  (if (eq (car error-data) 'end-of-buffer)
				      (if vm-auto-next-message
					  (progn (setq do-next-message t)
						 'next-message)
					(message "End of message %s from %s"
						 (vm-number-of
						  (car vm-message-pointer))
						 (vm-su-full-name
						  (car vm-message-pointer)))
					nil )))))
			     'next-message))
			   (progn
			     (if vm-honor-page-delimiters
				 (progn
				   (vm-narrow-to-page)
				   ;; This voodoo is required!  For some
				   ;; reason the 18.52 emacs display
				   ;; doesn't immediately reflect the
				   ;; clip region change that occurs
				   ;; above without this mantra. 
				   (scroll-up 0)))))
		       nil )))
	     (select-window old-w))))))
    (if do-next-message
	(vm-next-message)))
  (if (not (or vm-startup-message-displayed vm-inhibit-startup-message))
      (vm-display-startup-message)))

(defun vm-scroll-backward (arg)
  "Scroll backward a screenful of text."
  (interactive "P")
  (vm-scroll-forward (cond ((null arg) '-)
			   ((symbolp arg) nil)
			   ((consp arg) (list (- (car arg))))
			   (t arg))))

(defun vm-beginning-of-message ()
  "Moves to the beginning of the current message."
  (interactive)
  (vm-follow-summary-cursor)
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (vm-error-if-folder-empty)
  (setq vm-system-state 'reading)
  (vm-within-current-message-buffer
   (vm-widen-page)
   (goto-char (point-min))
   (if vm-honor-page-delimiters
       (vm-narrow-to-page)))
  (vm-display-current-message-buffer))

(defun vm-end-of-message ()
  "Moves to the end of the current message, exposing and marking it read
as necessary."
  (interactive)
  (vm-follow-summary-cursor)
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (vm-error-if-folder-empty)
  (if (eq vm-system-state 'previewing)
      (vm-show-current-message))
  (setq vm-system-state 'reading)
  (vm-within-current-message-buffer
   (vm-widen-page)
   (goto-char (point-max))
   (if vm-honor-page-delimiters
       (vm-narrow-to-page)))
  (vm-display-current-message-buffer t))

(defun vm-quit-no-change ()
  "Exit VM without saving changes made to the folder."
  (interactive)
  (vm-quit t))

(defun vm-quit (&optional no-change)
  "Quit VM, saving changes and expunging deleted messages."
  (interactive)
  (vm-select-folder-buffer)
  (if (not (memq major-mode '(vm-mode vm-virtual-mode)))
      (error "%s must be invoked from a VM buffer." this-command))
  (vm-check-for-killed-summary)
  (vm-error-if-referenced-virtually)
  (if (eq major-mode 'vm-virtual-mode)
      (vm-virtual-quit)
    (cond
     ((and no-change (buffer-modified-p)
	   (not (zerop vm-messages-not-on-disk))
	   ;; Folder may have been saved with C-x C-s and attributes may have
	   ;; been changed after that; in that case vm-messages-not-on-disk
	   ;; would not have been zeroed.  However, all modification flag
	   ;; undos are cleared if VM actually modifies the folder buffer
	   ;; (as opposed to the folder's attributes), so this can be used
	   ;; to verify that there are indeed unsaved messages.
	   (null (assq 'vm-set-buffer-modified-p vm-undo-record-list))
	   (not
	    (y-or-n-p
	     (format
	      "%d message%s have not been saved to disk, quit anyway? "
	      vm-messages-not-on-disk
	      (if (= 1 vm-messages-not-on-disk) "" "s")))))
      (error "Aborted"))
     ((and no-change (buffer-modified-p) vm-confirm-quit
	   (not (y-or-n-p "There are unsaved changes, quit anyway? ")))
      (error "Aborted"))
     ((and (eq vm-confirm-quit t)
	   (not (y-or-n-p "Do you really want to quit? ")))
      (error "Aborted")))
    (let ((inhibit-quit t))
      (if (not no-change)
	  (progn
	    ;; this could take a while, so give the user some feedback
	    (message "Quitting...")
	    (or vm-folder-read-only (vm-change-all-new-to-unread))
	    (if (not (buffer-modified-p))
		(message ""))))
      (if (and (buffer-modified-p) (not no-change))
	  (vm-save-folder t))
      (let ((summary-buffer vm-summary-buffer)
	    (mail-buffer (current-buffer)))
	(if summary-buffer
	    (progn
	      (if (eq vm-mutable-windows t)
		  (delete-windows-on vm-summary-buffer))
	      (kill-buffer summary-buffer)))
	(set-buffer mail-buffer)
	(set-buffer-modified-p nil)
	(kill-buffer (current-buffer))))))

;; This allows C-x C-s to do the right thing for VM mail buffers.
;; Note that deleted messages are not expunged.
(defun vm-write-file-hook ()
  (if (and (eq major-mode 'vm-mode) (not vm-inhibit-write-file-hook))
    ;; The vm-save-restriction isn't really necessary here, since
    ;; the stuff routines clean up after themselves, but should remain
    ;; as a safeguard against the time when other stuff is added here.
    (vm-save-restriction
     (let ((inhibit-quit t)
	   (mp vm-message-list)
	   (buffer-read-only))
	(while mp
	  (if (vm-modflag-of (car mp))
	      (vm-stuff-attributes (car mp)))
	  (setq mp (cdr mp)))
	(if vm-message-list
	    (progn
	      (vm-stuff-bookmark)
	      (vm-stuff-header-variables)))
	;; We can't be sure the write is going to succeed, so we can't set
	;; this variable to nil.  We can't leave it set to t either since
	;; the user will be confused, since she thought the folder was saved.
	;; The solution, set vm-buffer-modified-p to a value that indicates
	;; uncertainty.
	(setq vm-buffer-modified-p "--??-")
	(if vm-summary-buffer
	    (save-excursion
	      (set-buffer vm-summary-buffer)
	      (setq vm-buffer-modified-p "--??-")))
	nil ))))

(defun vm-save-folder (&optional quitting)
  "Save current folder to disk."
  (interactive)
  (vm-select-folder-buffer)
  (vm-sanity-check-modification-flag)
  (vm-check-for-killed-summary)
  (vm-error-if-virtual-folder)
  (if (buffer-modified-p)
      (let ((inhibit-quit t) mp)
	;; may get error if folder is emptied by the expunge.
	(condition-case ()
	    (vm-expunge-folder quitting t)
	  (error nil))
	;; stuff the attributes of messages that need it.
	(setq mp vm-message-list)
	(while mp
	  (if (vm-modflag-of (car mp))
	      (vm-stuff-attributes (car mp)))
	  (setq mp (cdr mp)))
	;; stuff bookmark and header variable values
	(if vm-message-list
	    (progn
	      (vm-stuff-bookmark)
	      (vm-stuff-header-variables)))
	(let ((vm-inhibit-write-file-hook t))
	  (save-buffer))
	(vm-set-buffer-modified-p nil t)
	(setq vm-messages-not-on-disk 0)
	(and (zerop (buffer-size)) vm-delete-empty-folders
	     (condition-case ()
		 (progn
		   (delete-file buffer-file-name)
		   (message "%s removed" buffer-file-name))
	       (error nil)))
	(if (not quitting)
	    (if vm-message-pointer
		(vm-update-summary-and-mode-line)
	      (vm-next-message))))))

(defun vm-visit-folder (folder)
  "Visit a mail file.
VM will parse and present its messages to you in the usual way."
  (interactive
   (save-excursion
     (vm-load-rc)
     (vm-select-folder-buffer)
     (let ((dir (if vm-folder-directory
		    (expand-file-name vm-folder-directory)
		  default-directory)))
       (list (read-file-name
	      (if vm-last-save-folder
		  (format "Visit folder: (default %s) " vm-last-save-folder)
		"Visit folder: ")
	      dir vm-last-save-folder t)))))
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (vm folder))

(defun vm-help ()
  "Display VM command and variable information."
  (interactive)
  (if (and vm-mail-buffer (get-buffer-window vm-mail-buffer))
      (set-buffer vm-mail-buffer))
  (let ((pop-up-windows (and pop-up-windows (eq vm-mutable-windows t))))
    (cond
     ((eq last-command 'vm-help)
      (describe-mode))
     ((eq vm-system-state 'previewing)
      (message "Type SPC to read message, n previews next message   (? gives more help)"))
     ((memq vm-system-state '(showing reading))
      (message "SPC and b scroll, (d)elete, (s)ave, (n)ext, (r)eply   (? gives more help)"))
     (t (describe-mode)))))

(defun vm-move-mail (source destination)
  (call-process vm-movemail-program nil nil nil
		(expand-file-name source) (expand-file-name destination)))

(defun vm-gobble-crash-box ()
  (save-excursion
    (vm-save-restriction
     (widen)
     (let ((opoint-max (point-max)) crash-buf buffer-read-only
	   (old-buffer-modified-p (buffer-modified-p))
           ;; crash box could contain a letter bomb...
	   ;; force user notification of file variables.
	   (inhibit-local-variables t))
       (setq crash-buf (find-file-noselect vm-crash-box))
       (goto-char (point-max))
       (insert-buffer-substring crash-buf
				1 (1+ (save-excursion
					(set-buffer crash-buf)
					(widen)
					(buffer-size))))
       (write-region opoint-max (point-max) buffer-file-name t t)
       ;; make sure primary inbox is private.  384 = octal 600
       (condition-case () (set-file-modes buffer-file-name 384) (error nil))
       (set-buffer-modified-p old-buffer-modified-p)
       (kill-buffer crash-buf)
       (condition-case () (delete-file vm-crash-box)
	 (error nil))))))

(defun vm-compatible-folder-p (file)
  (while (not (string= file (setq file (expand-file-name file)))))
  (let (buffer (type vm-folder-type))
    (if (zerop (buffer-size))
	t
      (if (null (setq buffer (get-file-buffer file)))
	  (if (not (file-exists-p file))
	      t
	    (unwind-protect
		(progn
		  (setq buffer (generate-new-buffer " *vm work*"))
		  (call-process "sed" file buffer nil "-n" "1p")
		  (save-excursion
		    (set-buffer buffer)
		    (or (zerop (buffer-size))
			(eq type (vm-get-folder-type)))))
	      (and buffer (kill-buffer buffer))))
	(save-excursion
	  (set-buffer buffer)
	  (or (zerop (buffer-size))
	      (eq type (vm-get-folder-type))))))))

(defun vm-get-spooled-mail ()
  (let ((spool-files (or vm-spool-files
			 (list (concat vm-spool-directory (user-login-name)))))
	(inhibit-quit t)
	(got-mail))
    (if (file-exists-p vm-crash-box)
	(progn
	  (message "Recovering messages from crash box...")
	  (vm-gobble-crash-box)
	  (message "Recovering messages from crash box... done")
	  (setq got-mail t)))
    (while spool-files
      (if (and (file-readable-p (car spool-files))
	       (vm-compatible-folder-p (car spool-files)))
	  (progn
	    (message "Getting new mail from %s..." (car spool-files))
	    (vm-move-mail (car spool-files) vm-crash-box)
	    (vm-gobble-crash-box)
	    (message "Getting new mail from %s... done" (car spool-files))
	    (setq got-mail t)))
      (setq spool-files (cdr spool-files)))
    got-mail ))

(defun vm-get-new-mail (&optional arg)
  "Move any new mail that has arrived in the system mailbox into the
primary inbox.  New mail is appended to the disk and buffer copies of
the primary inbox.

Prefix arg means to gather mail from a user specified folder, instead of
the usual spool file(s).  The file name will be read from the minibuffer.
Unlike when getting mail from a spool file, in this case the folder is left
undisturbed after its messages have been copied."
  (interactive "P")
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (vm-error-if-virtual-folder)
  (vm-error-if-folder-read-only)
  (and (null arg) (not vm-primary-inbox-p)
       (error "This is not your primary inbox."))
  (if (null arg)
      (if (not (and (vm-get-spooled-mail) (vm-assimilate-new-messages)))
	  (message "No new mail.")
	(vm-set-folder-variables)
	(vm-emit-totals-blurb)
	(vm-thoughtfully-select-message))
    (let (folder mcount buffer-read-only)
      (setq folder (read-file-name "Gather mail from folder: "
				   vm-folder-directory t))
      (if (not (vm-compatible-folder-p folder))
	  (error "Folder %s is not the same format as this folder." folder))
      (save-excursion
	(vm-save-restriction
	 (widen)
	 (goto-char (point-max))
	 (insert-file-contents folder)))
      (if (null vm-totals)
	  (vm-read-attributes))
      (setq mcount (car vm-totals))
      (if (vm-assimilate-new-messages)
	  (progn
	    (vm-set-folder-variables)
	    ;; Message is actually still on disk unless the user
	    ;; deletes the folder himself.  However, users may not
	    ;; understand what happened if the messages go away
	    ;; after a "quit, no save".
	    (setq vm-messages-not-on-disk
		  (+ vm-messages-not-on-disk (- (car vm-totals) mcount)))
	    (vm-emit-totals-blurb))
	(message "No new messages.")))))

(defun vm-emit-totals-blurb ()
  (save-excursion
    (vm-select-folder-buffer)
    (if (null vm-totals)
	(vm-read-attributes))
    (message "%d message%s, %d new, %d unread."
	     (car vm-totals) (if (= (car vm-totals) 1) "" "s") 
	     (car (cdr vm-totals))
	     (car (cdr (cdr vm-totals))))))

;; returns non-nil if there were any new messages
(defun vm-assimilate-new-messages ()
  (let ((tail-cons (vm-last vm-local-message-list))
	(new-messages-p (null vm-local-message-list)))
    (save-excursion
      (vm-save-restriction
       (widen)
       (vm-build-message-list)
       (vm-read-attributes)
       (setq new-messages-p (or new-messages-p (cdr tail-cons)))
       (cond ((and vm-current-grouping new-messages-p)
	      (condition-case data
		  (vm-group-messages vm-current-grouping)
		;; presumably an unsupported grouping
		(error (message (car (cdr data)))
		       (sleep-for 2)
		       (vm-number-messages)
		       (if vm-summary-buffer
			   (progn
			     (vm-do-summary (cdr tail-cons))
			     (vm-set-summary-pointer
			      (car vm-local-message-pointer)))))))
	     ((and vm-summary-buffer new-messages-p)
	      (vm-number-messages (cdr tail-cons))
	      (if vm-summary-buffer
		  (progn
		    (vm-do-summary (cdr tail-cons))
		    (vm-set-summary-pointer
		     (car vm-local-message-pointer)))))
	     (new-messages-p (vm-number-messages (cdr tail-cons)))))
      new-messages-p )))

;; return a list of all marked messages or the messages indicated by a
;; prefix argument.
(defun vm-select-marked-or-prefixed-messages (prefix)
  (let (mlist)
    (if (eq last-command 'vm-next-command-uses-marks)
	(setq mlist (vm-marked-messages))
      (let* ((direction (if (< prefix 0) 'backward 'forward))
	     (count (vm-abs prefix))
	     (vm-message-pointer vm-message-pointer))
	(if (not (eq vm-circular-folders t))
	    (vm-check-count prefix))
	(while (not (zerop count))
	  (setq mlist (cons (car vm-message-pointer) mlist))
	  (vm-decrement count)
	  (if (not (zerop count))
	      (vm-move-message-pointer direction))))
      (nreverse mlist))))

(defun vm-display-startup-message ()
  (if (sit-for 5)
      (let ((lines vm-startup-message-lines))
	(message "VM %s, Copyright (C) 1990 Kyle E. Jones; type ? for help"
		 vm-version)
	(setq vm-startup-message-displayed t)
	(while (and (sit-for 4) lines)
	  (message (substitute-command-keys (car lines)))
	  (setq lines (cdr lines)))))
  (message ""))

(defun vm-load-rc (&optional interactive)
  (interactive "p")
  (if (or (not vm-rc-loaded) interactive)
      (load "~/.vm" (not interactive) (not interactive) t))
  (setq vm-rc-loaded t))

(defun vm (&optional folder)
  "Read mail under Emacs.
Optional first arg FOLDER specifies the folder to visit.  It defaults
to the value of vm-primary-inbox.  The folder buffer is put into VM
mode, a major mode for reading mail.

Visiting the primary inbox causes any contents of the system mailbox to
be moved and appended to the resulting buffer.

All the messages can be read by repeatedly pressing SPC.  Use `n'ext and
`p'revious to move about in the folder.  Messages are marked for
deletion with `d', and saved to another folder with `s'.  Quitting VM
with `q' expunges deleted messages and saves the buffered folder to
disk.

See the documentation for vm-mode for more information."
  (interactive)
  ;; If this is the first time VM has been run in this Emacs session,
  ;; do some necessary preparations.
  (if vm-session-beginning
      (progn
	(random t)
	(vm-load-rc)
	;; Load these files conditionally since a clever user may have already
	;; loaded them and customized a few functions.
	(and (not (commandp 'vm-next-message)) (load "vm-motion"))
	(and (not (commandp 'vm-undo)) (load "vm-undo"))
	(and (not (commandp 'vm-summarize)) (load "vm-summary"))))
  ;; set inhibit-local-variables non-nil to protect
  ;; against letter bombs.
  (let ((inhibit-local-variables t)
	(full-startup (not (bufferp folder)))
	mail-buffer)
    (setq mail-buffer
	  (if (bufferp folder)
	      folder
	    (let ((file (or folder (expand-file-name vm-primary-inbox))))
	      (if (file-directory-p file)
		  ;; MH code perhaps... ?
		  (error "%s is a directory" file)
		(or (get-file-buffer file) (find-file-noselect file))))))
    (set-buffer mail-buffer)
    (vm-sanity-check-modification-flag)
    (vm-check-for-killed-summary)
    ;; If the buffer's not modified then we know that there can be no
    ;; messages in the folder that are not on disk.
    (or (buffer-modified-p) (setq vm-messages-not-on-disk 0))
    (let ((first-time (not (eq major-mode 'vm-mode)))
	  (inhibit-quit t))
      ;; If this is not a VM mode buffer then some initialization
      ;; needs to be done 
      (if first-time
	  (progn
	    (buffer-flush-undo (current-buffer))
	    (abbrev-mode 0)
	    (auto-fill-mode 0)
	    (vm-mode-internal))
	;; make sure vm-message-{pointer,list} are set correctly
	(vm-set-folder-variables))
      (if (or (and vm-primary-inbox-p (vm-get-spooled-mail)) first-time
	      ;; If the message list is empty, take a second look: the
	      ;; buffer may have been encrypted the first time around.
	      ;; This is a concession to users who use crypt.el and put
	      ;; vm-mode into auto-mode-alist.
	      (null vm-message-list))
	  (progn
	    (vm-assimilate-new-messages)
	    (vm-set-folder-variables)
	    ;; Can't allow a folder-empty error here because execution
	    ;; would abort before the session startup code below.
	    (if (null vm-message-list)
		(and full-startup (message "Folder is empty."))
	      (if first-time
		  (vm-check-header-variables))
	      (and full-startup (vm-emit-totals-blurb))
	      (set-buffer mail-buffer)
	      (if first-time
		  (vm-gobble-bookmark))
	      (vm-thoughtfully-select-message))))
      (and full-startup (switch-to-buffer mail-buffer))
      (if (and full-startup vm-message-list vm-startup-with-summary)
	  (progn
	    (vm-summarize t)
	    (vm-emit-totals-blurb)
	    (and (eq vm-startup-with-summary t)
		 (eq vm-mutable-windows t)
		 ;; If we're not in the summary window go there
		 ;; and nuke all other windows.
		 ;; If we are in the summary window then vm-proportion-windows
		 ;; must have forced the mail window off the screen, so don't
		 ;; bother deleting other windows. 
		 (if (not (eq major-mode 'vm-summary-mode))
		     (select-window (get-buffer-window vm-summary-buffer)))
		 (progn
		   (delete-other-windows)
		   ;; force vertical centering at startup as a courtesy
		   (recenter '(4)))))
	(if (and full-startup (eq vm-mutable-windows t))
	    (delete-other-windows)))
      (if vm-session-beginning
	  (progn
	    (setq vm-session-beginning nil)
	    ;; Display copyright and copying info unless
	    ;; user says no.
	    (if (and (not vm-inhibit-startup-message) full-startup)
		(progn
		  (vm-display-startup-message)
		  ;; If there are any messages, redisplay the totals blurb.
		  ;; w.r.t the major mode check below: we won't be in the
		  ;; summary buffer if there are no messages...
		  ;; Forget it if we're not doing a full-startup.
		  (if (and full-startup
			   (or vm-message-list
			       (eq major-mode 'vm-summary-mode))
			   (not (input-pending-p)))
		      (vm-emit-totals-blurb)))))))))

(defun vm-mode ()
  "Major mode for reading mail.

Commands:
   h - summarize folder contents
   j - discard cached information about the current message

   n - go to next message
   p - go to previous message
   N - like `n' but ignores skip-variable settings
   P - like `p' but ignores skip-variable settings
 M-n - go to next unread message
 M-p - go to previous unread message
 RET - go to numbered message (uses prefix arg or prompts in minibuffer)
 TAB - go to last message seen
 M-s - incremental search through the folder

   t - display hidden headers
 SPC - scroll forward a page (if at end of message, then display next message)
   b - scroll backward a page
   > - go to end of current message

   d - delete message, prefix arg deletes messages forward (flag as deleted)
 C-d - delete message, prefix arg deletes messages backward (flag as deleted)
   u - undelete
   k - flag for deletion all messages with same subject as the current message

   r - reply (only to the sender of the message)
   R - reply with included text for current message
 C-r - extract and resend bounced message
   f - followup (reply to all recipients of message)
   F - followup with included text from the current message
   z - forward the current message
   m - send a message

   @ - digestify and mail entire folder contents (the folder is not modified)
   * - burst a digest into individual messages, and append and assimilate these
       message into the current folder.

   G - group messages according to some criteria

   g - get any new mail that has arrived in the system mailbox
       (new mail is appended to the disk and buffer copies of the
       primary inbox.)
   v - visit another mail folder
   V - visit a virtual folder

   e - edit the current message

   s - save current message in a folder (appends if folder already exists)
   w - write current message to a file without its headers (appends if exists)
   S - save entire folder to disk, expunging deleted messages
   A - save unfiled messages to their vm-auto-folder-alist specified folders
   # - expunge deleted messages (without saving folder)
   q - quit VM, deleted messages are expunged, folder saved to disk
   x - exit VM with no change to the folder

   M - use marks; the next vm command will affect only marked messages
       if it makes sense for the command to do so

       C-c C-@ - mark the current message
       C-c SPC - unmark the current message
       C-c C-a - mark all messsages
       C-c a   - unmark all messsages

 C-_ - undo, special undo that retracts the most recent
             changes in message attributes.  Expunges and saves
             cannot be undone.

   L - reload your VM init file, ~/.vm

   ? - help

   ! - run a shell command
   | - run a shell command with the current message as input

 M-C - view conditions under which you may redistribute VM
 M-W - view the details of VM's lack of a warranty

Variables:
   vm-auto-center-summary
   vm-auto-folder-alist
   vm-auto-folder-case-fold-search
   vm-auto-next-message
   vm-berkeley-mail-compatibility
   vm-circular-folders
   vm-confirm-new-folders
   vm-confirm-quit
   vm-crash-box
   vm-delete-after-archiving
   vm-delete-after-bursting
   vm-delete-after-saving
   vm-delete-empty-folders
   vm-digest-center-preamble
   vm-digest-preamble-format
   vm-folder-directory
   vm-folder-read-only
   vm-follow-summary-cursor
   vm-forwarding-subject-format
   vm-gargle-uucp
   vm-group-by
   vm-highlighted-header-regexp
   vm-honor-page-delimiters
   vm-in-reply-to-format
   vm-included-text-attribution-format
   vm-included-text-prefix
   vm-inhibit-startup-message
   vm-invisible-header-regexp
   vm-mail-window-percentage
   vm-mode-hooks
   vm-move-after-deleting
   vm-move-after-undeleting
   vm-mutable-windows
   vm-preview-lines
   vm-preview-read-messages
   vm-primary-inbox
   vm-reply-ignored-addresses
   vm-reply-subject-prefix
   vm-rfc934-forwarding
   vm-search-using-regexps
   vm-skip-deleted-messages
   vm-skip-read-messages
   vm-spool-files
   vm-startup-with-summary
   vm-strip-reply-headers
   vm-summary-format
   vm-virtual-folder-alist
   vm-virtual-mirror
   vm-visible-headers
   vm-visit-when-saving"
  (interactive)
  (vm (current-buffer)))

;; this does the real major mode scutwork.
(defun vm-mode-internal ()
  (widen)
  (make-local-variable 'require-final-newline)
  (make-local-variable 'file-precious-flag)
  (setq
   case-fold-search t
   file-precious-flag t
   major-mode 'vm-mode
   mode-line-format vm-mode-line-format
   mode-name "VM"
   buffer-read-only t
   require-final-newline nil
   vm-local-message-list nil
   vm-local-message-pointer nil
   vm-message-list nil
   vm-message-pointer nil
   vm-current-grouping vm-group-by
   vm-folder-type (vm-get-folder-type)
   vm-primary-inbox-p (equal buffer-file-name
			     (expand-file-name vm-primary-inbox)))
  (use-local-map vm-mode-map)
  (run-hooks 'vm-mode-hooks))

(put 'vm-mode 'mode-class 'special)

(autoload 'vm-group-messages "vm-group" nil t)

(autoload 'vm-reply "vm-reply" nil t)
(autoload 'vm-reply-include-text "vm-reply" nil t)
(autoload 'vm-followup "vm-reply" nil t)
(autoload 'vm-followup-include-text "vm-reply" nil t)
(autoload 'vm-mail "vm-reply" nil t)
(autoload 'vm-resend-bounced-message "vm-reply" nil t)
(autoload 'vm-forward-message "vm-reply" nil t)
(autoload 'vm-send-digest "vm-reply" nil t)

(autoload 'vm-isearch-forward "vm-search" nil t)

(autoload 'vm-burst-digest "vm-digest" nil t)
(autoload 'vm-rfc934-char-stuff-region "vm-digest")
(autoload 'vm-digestify-region "vm-digest")

(autoload 'vm-show-no-warranty "vm-license" nil t)
(autoload 'vm-show-copying-restrictions "vm-license" nil t)

(autoload 'vm-auto-archive-messages "vm-save" nil t)
(autoload 'vm-save-message "vm-save" nil t)
(autoload 'vm-save-message-sans-headers "vm-save" nil t)
(autoload 'vm-pipe-message-to-command "vm-save" nil t)

(autoload 'vm-delete-message "vm-delete" nil t)
(autoload 'vm-delete-message-backward "vm-delete" nil t)
(autoload 'vm-undelete-message "vm-delete" nil t)
(autoload 'vm-kill-subject "vm-delete" nil t)
(autoload 'vm-expunge-folder "vm-delete" nil t)

(autoload 'vm-mark-message "vm-mark" nil t)
(autoload 'vm-unmark-message "vm-mark" nil t)
(autoload 'vm-clear-all-marks "vm-mark" nil t)
(autoload 'vm-mark-all-messages "vm-mark" nil t)
(autoload 'vm-next-command-uses-marks "vm-mark" nil t)

(autoload 'vm-edit-message "vm-edit" nil t)
(autoload 'vm-discard-cached-data "vm-edit" nil t)

(autoload 'mail-mode "sendmail" nil t)

(if (not (memq 'vm-write-file-hook write-file-hooks))
    (setq write-file-hooks
	  (cons 'vm-write-file-hook write-file-hooks)))
