;;; -*- Package: Apollo; Mode: Common-Lisp -*-
(in-package 'apollo)
(provide 'apollo-lisp-support)
;; Copyright (c) 1988,1989 Apollo Computer, Inc. 
;; 
;; The software and information herein are the property
;; of Apollo Computer, Inc.
;;
;; This file may be distributed without further permission 
;; from Apollo Computer, Inc. as long as:
;; 
;;    * all copyright notices and this notice are preserved
;;    * it is not part of a product for resale
;;    * a detailed description of all modifications to this
;;      file is included in the modified file.
;;  
;; Version 1.1
;; Fri Sep  8 10:11:26 1989
;;
;; This file provides a gnuemacs interface to Domain/CommonLISP
;; designed for use on APOLLO workstations.  This code should
;; work with LUCID Common Lisp with X windows and other Common
;; Lisp's with some porting effort.
;; This software is normally unsupported and without warranty.
;; Support will be provided only for customers of 
;; Domain/CommonLISP from APOLLO Computers Inc.
;;
;; Questions, comments and enhancements are welcome from
;; all sources.
;;
;; Send questions, comments and enhancements to:
;;  lisp-tools:@apollo.com
;;  mit-eddie!ulowell!apollo!lisp-tools:
;; or 
;;  Domain/CommonLISP Group
;;  Apollo Computer Inc.
;;  330 Billerica Road
;;  Chelmsford MA 01824
;;
;;
;; 
;;; Known Deficiencies:
;;  Symbol-Count is ignored from attribute list
;;  DOMIAN IX may lose its ability to fork processes
;;  Emacs may lose when killing too large a block of text
;;  Emacs sometimes terminates with IOT for no known reason
;;  Some of the description commands only work for common lisp not emacs lisp
;;  No Source compare merge facility
;;  No Patch file facility exists
;;  No Edit Warnings facility
;;  If user changes keys, may give incorrect messages
;;  May lose by renaming inferior lisp buffer.
;;  Inferior lisp communication loses if lisp ignores errors or in GC
;;  Changed definitions only knows files from load-file or load-compile-file.
;;  Changed definitions won't know about recursively lisp-loaded files

(defvar *TOP-LEVEL-READTABLE* LISP:*READTABLE*)
(defvar *RESET* nil)
(defvar *called-from-system-packages* nil)
(defvar *called-from-user-packages* t)
(defvar *abort* (symbol-function 'abort))
(defvar *TRAPPING-ABORTS* nil)
(defvar *UNBOUND* (intern "<UNBOUND>"))
(defvar *NAMED-OBJECTS* (make-hash-table :test #'eq :size 1))
(defvar *HASHING-NAMED-OBJECTS-P* nil)

#+LUCID #+LCL3.0 #+HASH-NAMED-OBJECTS
(progn
  ;; When displaying certain output in lisp, display names are used
  ;; for many objects.  These names are not useful for accessing
  ;; the object.  This code within this PROGN will hash these
  ;; names and values when the names are output.
  ;; These may then be referenced with the function
  ;; OBJECT-NAMED which takes a string such as "#<Array F431>" and
  ;; returns the object.  The problem with this is that since a 
  ;; pointer to this object is maintained, it will never be GC'd.
  ;; When debugging code within the APOLLO GNUEMACS Common Lisp support
  ;; package, this may be turned on so that <C-c v>, describe variable
  ;; at point may be used to describe these objects.
  ;; To turn this feature on place the following in your
  ;; ~/user_data/common_lisp/startup.lisp file before loading or compiling
  ;; /sys/common_lisp/gnuemacs_support/apollo-lisp-support.lisp:
  ;;    (pushnew :HASH-NAMED-OBJECTS *FEATURES*)
  ;; and in your ~/.emacs file:
  ;;    (setq apollo:grabbing-dispatched-names-p t)
  ;; It may also be a good idea to periodically run:
  ;;    (APOLLO:RESET-NAMED-OBJECTS)
  ;; which will clear out the table of named objects so they may be 
  ;; GC'd.  Objects displayed prior to this point will no longer be
  ;; accessable.

  (export 'object-named)

  (setq *HASHING-NAMED-OBJECTS-P* t)
  (defun object-named (name)
    (or (gethash (intern name (find-package "APOLLO")) *NAMED-OBJECTS*)
	(error "No Object named ~A." name)))

  (defun reset-named-objects ()
    (clrhash *NAMED-OBJECTS*))

  (defvar *STRING-STREAM* (make-string-output-stream))

  (defvar *BROADCAST-STREAM* nil)

  (defvar *LAST-PRINT-OUTPUT* nil)

  (defmacro def-advice-hash-named-object (function)
    (let ((args (gensym)))
      `(defadvice (,function hash-named-object) (&rest ,args)
	 (if *HASHING-NAMED-OBJECTS-P*
	     (progn
	       (unless (eq *LAST-PRINT-OUTPUT* LUCID::*PRINT-OUTPUT*)
		 (setf *LAST-PRINT-OUTPUT* LUCID::*PRINT-OUTPUT*
		       *BROADCAST-STREAM* (make-broadcast-stream LUCID::*PRINT-OUTPUT* *STRING-STREAM*)))
	       (let ((LUCID::*PRINT-OUTPUT* *BROADCAST-STREAM*))
		 (multiple-value-prog1
		     (apply-advice-continue ,args)
		   (let ((string (get-output-stream-string *STRING-STREAM*))
			 (package (find-package "APOLLO")))
		     (setf (gethash (intern string package) *NAMED-OBJECTS*) (car ,args))))))
	     (apply-advice-continue ,args)))))

  (def-advice-hash-named-object LUCID::OUTPUT-TERSE-OBJECT)
  (def-advice-hash-named-object LUCID::OUTPUT-BIT-VECTOR)
  (def-advice-hash-named-object LUCID::OUTPUT-TERSE-ARRAY)
  (def-advice-hash-named-object LUCID::OUTPUT-VECTOR)
  (def-advice-hash-named-object LUCID::OUTPUT-ARRAY)
  (def-advice-hash-named-object LUCID::OUTPUT-A-STRUCTURE)
  (def-advice-hash-named-object LUCID::OUTPUT-FOREIGN-POINTER)
  )

(defun kill-process-named (name)
  (dolist (p system::*all-processes*)
    (when (equal (system::process-name p) name)
      (format t "~&;;; Killing Process: ~A ~A~%" name (sys::process-initial-arguments p))
      (system::kill-process p :SUPPRESS-UNWIND-PROTECTS nil))))

(defvar *THROW-TO* nil)

(defun interrupt-process-named (&optional (name "LISP-SERVER"))
  (dolist (p system::*all-processes*)
    (when (equal (system::process-name p) name)
      (when (and (sys::symbol-process-value '*throw-to* p) (not (eq p sys::*current-process*)))
	(sys::interrupt-process
	  p 
	  #'(lambda ()
	      (let ((tag (sys::symbol-process-value '*throw-to* sys::*CURRENT-PROCESS*)))
		(when tag (throw tag sys::*current-process*)))))))))

(defun create-lisp-servers (&rest names)
  (kill-process-named "LISP-SERVER")
  (dolist (name names names)
    (format t "~&;;; Starting Process: ~A~%" name)
    (make-process :function 'lcl::lisp-server :args (list name))))

(defmacro with-lisp-servers-deactivated (&body body)
  `(unwind-protect
	(progn
	  (dolist (p sys::*all-processes*)
	    (if (equal (sys::process-name p) "LISP-SERVER") (sys::deactivate-process p)))
	  ,@body)
     (dolist (p sys::*all-processes*)
       (if (equal (sys::process-name p) "LISP-SERVER") (sys::activate-process p)))))

(let ((*ERROR-OUTPUT* nil))
  (defun abort (&rest args) (if *TRAPPING-ABORTS* (throw :ABORT-TRAP t) (apply *ABORT* args))))

(defmacro with-abort-trapping (&rest body)
  `(let* ((*TRAPPING-ABORTS* t)
	  (result-list nil)
	  (abort-p (catch :ABORT-TRAP (setq result-list (multiple-value-list ,@body)) nil)))
     (values (if abort-p nil result-list) abort-p)))
	 
(defun macro-expand-expression (foo)
  (with-output-to-string (*standard-output*)
    (pprint (macroexpand-1 foo))))

(defun macro-expand-expression-all (foo)
  (with-output-to-string (*standard-output*)
    (pprint (code-walk foo))))

(defun lisp-package () 
  #+LUCID #+LCL3.0 (package-name (symbol-process-value '*package* *initial-process*))
  #+LUCID #-LCL3.0 (package-name *PACKAGE*)
  #-LUCID (error "(LISP-PACKAGE ~S) not defined for this implementation" x))

(shadow "SOURCE-CODE")
(defun source-code (x) 
  #+LUCID #+LCL3.0 (system::source-code x)
  #+LUCID #-LCL3.0 (lucid::%svref x 0)
  #-LUCID (error "(SOURCE-CODE ~S) not defined for this implementation" x))

(defun handle-error (form)
  (let ((result nil) (error nil))
    #+LUCID #-LCL3.0 
    (progn
      (multiple-value-setq (result error) (lucid::with-error-trapping (funcall form)))
      (when error
	(setq result 
	      (concatenate 'string
			   (symbol-name (car result)) ": "
			   (apply #'format nil (caddr result) (cadddr result))))))
    #+LUCID #+LCL3.0 
    (catch :ERROR
      (handler-bind 
	((t (function (lambda (c)
	      (let ((format-string nil) (format-arguments nil))
		(dotimes (n (lucid::defstruct-info (type-of c)))
		  (multiple-value-bind (name accessor) (lucid::defstruct-slot-info (type-of c) n)
		    (case name
		      (lucid::format-arguments (setq format-arguments (funcall accessor c)))
		      (lucid::format-string    (setq format-string    (funcall accessor c))))))
		(let ((message (if format-string
				   (concatenate 'string
						(symbol-name (type-of c))
						": "
						(apply (function format) 
						       nil 
						       format-string 
						       format-arguments))
				   (prin1-to-string (type-of c)))))
		  (cond ((typep c 'warning)
			 (format *ERROR-OUTPUT* "~&;;; ~A~%" message)
			 (multiple-value-setq (result error) (handle-error (muffle-warning))))
			(t
			 (format *ERROR-OUTPUT* "~&;;; ~A~%" message)
			 (setq result message error t)
			 (throw :ERROR message)))))))))
	(multiple-value-bind (r abort-p) (with-abort-trapping (funcall form))
	  (if abort-p
	      (progn
		(format *ERROR-OUTPUT* "&;;; Aborted Operation~%")
		(setq result "Aborted Operation" error t))
	      (setq result r)))))
    #-LUCID `(error "Define apollo:handle-error for this lisp")
    (values result error)))

(shadow "TRAPPING-ERRORS")
(defmacro trapping-errors (&rest body)
  `(handle-error #'(lambda () ,@body)))

(shadow "IGNORE-ERRORS")
(defmacro ignore-errors (&rest body)
  #+LUCID   (let ((result (gensym)) (error (gensym)))
	      `(let ((*ERROR-OUTPUT* nil))
		 (multiple-value-bind (,result ,error) (trapping-errors ,@body)
		   (if ,error nil (apply #'values ,result)))))
  #-LUCID `(error "Define apollo:ignore-errors for this lisp"))

(defvar *GET-SOURCE-FILE-HOOK* nil)
(shadow "GET-SOURCE-FILE")
(defun get-source-file (object &optional type want-list)
  (or (and *GET-SOURCE-FILE-HOOK* (funcall *GET-SOURCE-FILE-HOOK* object type want-list))
      #+LUCID (ignore-errors (system::get-source-file object type want-list))
      #-LUCID (error "Define apollo:get-source-file for this lisp implementation")))

(defmacro safe-progn (&rest body)
  (let ((result (gensym)) (error (gensym)))
    `(multiple-value-bind (,result ,error) (trapping-errors ,@body)
       (if ,error 
	   (progn (print ,error) nil)
	   (apply #'values ,result)))))

(defvar *INITIAL-PACKAGES* (list-all-packages))
(do ((l *initial-packages* (cdr l)))
    ((null l) *INITIAL-PACKAGES*)
  (when (eq (car l) *package*) 
    (setf (car l) (cadr l)) (setf (cdr l) (cddr l))))

(defmacro with-package ((package) . body) 
  `(let ((*package* (or (find-package ,package) *package*))) ,@body))

(defun eval-string (string &key package ibase base readtable used-packages (quiet nil) compile)
  (declare (ignore used-packages compile))
  (let ((*READTABLE*  (or readtable *TOP-LEVEL-READTABLE*))
	(*READ-BASE*  (or ibase 10))
	(*PRINT-BASE* (or base  10))
	(*PRINT-LEVEL* 1000)
	(*PRINT-LENGTH* 1000)
	(eof (gensym))
	output result error forms)
    (catch :EVAL-STRING-ABORT
      (let ((*THROW-TO* :EVAL-STRING-ABORT))
	(sys::with-interruptions-allowed
	    (with-input-from-string (stream string)
	      (let ((*ERROR-OUTPUT* nil))
		(loop 
		  (multiple-value-setq (result error) 
		    (trapping-errors (with-package (package) (read stream nil eof))))
		  (cond (error
			 (push `(error ,result) forms)
			 (return 
			   (setq forms (if (> (length forms) 1) `(progn ,@(nreverse forms)) (car forms)))))
			((and (consp result) (eq (car result) eof))
			 (return 
			   (setq forms 
				 (if forms
				     (if (> (length forms) 1) `(progn ,@(nreverse forms)) (car forms))
				     `(error "Unexpected End of file encountered")))))
			(t (push (car result) forms))))))
	  (if (and compile LUCID::*IN-THE-COMPILER* nil)
	      (let ((initial-process 
		     (dolist (p *all-processes*) (when (equal (process-name p) "Initial") (return p)))))
		(let ((done nil))
		  (interrupt-process initial-process
				     #'(lambda () 
					 (multiple-value-setq (result error) 
					   (trapping-errors (with-package (package) (eval forms))))
					 (setq done t)))
		  (loop (when done (return)))))
	      (progn
		(when (and compile LUCID::*IN-THE-COMPILER*)
		  (setq forms '(error "Lisp is already compiling!")))
		(if quiet 
		    (setq output
			  (with-output-to-string (*TERMINAL-IO*) 
			    (multiple-value-setq (result error) 
			      (trapping-errors (with-package (package) (eval forms))))))
		    (multiple-value-setq (result error) 
		      (trapping-errors (with-package (package) (eval forms)))))))
	  (if error
	      (unless quiet
		(format *ERROR-OUTPUT* "~&;;; ~A~%" result))
	      (setq result (mapcar #'(lambda (s) (prin1-to-string s)) result))))))
    (values result error output)))

(defmacro recoverable-eval (&rest args)
  `(let ((apollo::*reset* t) (*debug-print-length* 1000) (*debug-print-level* 1000))
    (catch :gnuemacs-return
      (unwind-protect
	   (progn ,@args)
	(when apollo::*reset* (throw :gnuemacs-return ':done))))))

(defun recover-from-eval () 
  (when apollo::*reset* 
    (unwind-protect 
	 (setq apollo::*reset* nil) 
      (throw :gnuemacs-return :gnuemacs-error-recovered))))

(defmacro interpretted-function-p (v) 
  #+LUCID `(lucid::procedurep ,v )
  #-LUCID `(error "Define apollo:interpretted-function-p for this lisp")
  ) 

(defun emacs-lisp-complete-symbol (f c p arg)
  (declare (optimize (speed 3) (safety 1) (compilation-speed 0)))
  (let ((*print-length* most-positive-fixnum))
    (mapcan
      #'(lambda (a) 
	  (let* ((sf (and (fboundp a) (symbol-function a)))
		 (n (symbol-name a))
		 (is-f (and sf (or (compiled-function-p sf) (interpretted-function-p sf)))))
	    (if (and (or (not f) is-f) (or arg (eq 0 (search c n))))
		(list (list n (if is-f " <f>" "    "))))))
      (sort 
	(apropos-list c p)
	#'(lambda (a b) (string< (symbol-name a) (symbol-name b)))))))

(defun where-is-symbol (s)
  (declare (optimize (speed 3) (safety 1) (compilation-speed 0)))
  (cond ((stringp s) (setq s (symbol-name (read-from-string s))))
	((symbolp s) (setq s (symbol-name s)))
	(t (error "~S must be a string or symbol!")))
  (let ((result nil))
    (dolist (p (list-all-packages))
      (multiple-value-bind (symbol loc)
	  (find-symbol s p)
	(when (and symbol (member loc '(:INTERNAL :EXTERNAL))) 
	  (push (format nil "~A::~A" (package-name p) s) result))))
    result))

(defvar *RECURSIVE-OBJECTS* nil)
(defun objects-referenced-by-function (v &optional object)
  (cond ((memq v *RECURSIVE-OBJECTS*) nil)
	((typep v 'lucid::compiled-function) 
	 (let ((*RECURSIVE-OBJECTS* (cons v *RECURSIVE-OBJECTS*)))
	   (objects-referenced-by-compiled-function v object)))
	((lucid::procedurep v)
	 (objects-referenced-by-interpretted-function v object))
	((not (symbolp v)) (error "Unknown function Object: ~S" v))
	((fboundp v)
	 (let ((*RECURSIVE-OBJECTS* (cons v *RECURSIVE-OBJECTS*)))
	   (objects-referenced-by-function (symbol-function v) object)))
	(t nil)))
	 
(defun objects-referenced-by-compiled-function (v &optional object)
  (declare (optimize (speed 3) (safety 1) (compilation-speed 0)))
  #+LUCID (do ((n lucid::procedure-literals (1+ n))
	       (l nil)
	       (end (lucid::procedure-length v)))
	      ((>= n end) l)
	    (let ((literal (lucid::procedure-ref v n)))
	      (if (lucid::procedurep literal)
		  (let ((r (objects-referenced-by-function literal object)))
		    (if object 
			(if (eq r object) (return t))
			(setq l (nconc l r))))
		  (if object
		      (when (eq object literal) (return t))
		      (push literal l)))))
  #-LUCID `(error "Define apollo:objects-referenced-by-compiled-function for this lisp"))

(defun objects-referenced-by-interpretted-function (l &optional object)
  (declare (optimize (speed 3) (safety 1) (compilation-speed 0)))
  (multiple-value-bind (form info) 
      (code-walk (source-code l))
    (declare (ignore form))
    (let ((r (mapcan #'(lambda (i) (when (consp i) i)) info)))
      (if object (member object r) r))))

(defun called-functions (s)
  (declare (optimize (speed 3) (safety 1) (compilation-speed 0)))
  (delete-if-not
    #'(lambda (o)
	(and (symbolp o) (fboundp o)
	     (or (compiled-function-p     (symbol-function o))
		 (interpretted-function-p (symbol-function o)))))
    (referenced-objects s)))

(defun referenced-objects (s &optional object)
  (declare (optimize (speed 3) (safety 1) (compilation-speed 0)))
  (flet ((sort-and-select-from-appropriate-packages (l)
	   (sort 
	     (delete-if-not 
	       #'(lambda (o) 
		   (if (symbolp o)
		       (if (member (symbol-package o) *INITIAL-PACKAGES*)
			   *called-from-system-packages*
			   *called-from-user-packages*)))
	       l)
	     #'(lambda (a b) 
		 (setq a (cond ((stringp a) a) 
			       ((symbolp a) (symbol-name a))
			       (t "")))
		 (setq b (cond ((stringp b) b) 
			       ((symbolp b) (symbol-name b))
			       (t "")))
		 (string< a b)))))
    (let ((r (objects-referenced-by-function s object)))
      (if object 
	  r
	  (sort-and-select-from-appropriate-packages r)))))

(system::defsubst referenced-object-p (s f)
  (referenced-objects s f))

(defun who-calls (f &optional (package nil) inheritors-p inherited-p)
  (declare (optimize (speed 3) (safety 1) (compilation-speed 0)))
  (declare (ignore  inheritors-p inherited-p)) ; I don't know what these are for
  (let ((l nil)
	(*called-from-system-packages* *called-from-system-packages*))
    (unless (packagep package) (setq package (find-package package)))
    (if (member package *INITIAL-PACKAGES*) (setq *called-from-system-packages* t))
    (labels ((safe-source (s type)
	       (let ((f (ignore-errors (get-source-file s type))))
		 (if f (namestring f) "Unknown Source")))
	     (add-entry (s)
	       (push 
		 (cond ((macro-function s) 
			`(,s ,(safe-source s 'macro) "Macro"))
		       ((and (fboundp s) (compiled-function-p     (symbol-function s))) 
			`(,s ,(safe-source s 'function) "Compiled-Function"))
		       ((and (fboundp s) (interpretted-function-p (symbol-function s))) 
			`(,s ,(safe-source s 'function) "Interpretted-Function"))
		       (t 
			`(,s ,(safe-source s 'variable) "Variable")))
		 l)))
      (declare (inline add-entry calls-function))
      (if package  ; this is called with a c-u emacs arg for all packages
	  (do-symbols (s package) (when (referenced-object-p s f) (add-entry s)))
	  (do-all-symbols (s)     (when (referenced-object-p s f) (add-entry s)))))
    l))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;                   Beginning of source for the code walker                  ;;;
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *SPECIAL-FORMS* '(;; ;;;;; Special forms
			  lisp::block lisp::catch 
			  lisp::compiler-let
			  lisp::declare lisp::eval-when lisp::flet lisp::function
			  lisp::go
			  lisp::if
			  lisp::labels 
			  lisp::let lisp::let*
			  lisp::macrolet 
			  lisp::multiple-value-call lisp::multiple-value-prog1
			  lisp::progn lisp::progv
			  lisp::quote lisp::return-from lisp::setq
			  lisp::tagbody lisp:the lisp::throw lisp::unwind-protect
			  ))

(defvar *PUSH-VARIABLE-BINDINGS-ACTION* nil)
(defvar *POP-VARIABLE-BINDINGS-ACTION*  nil)
(defvar *PUSH-FUNCTION-BINDINGS-ACTION* nil)
(defvar *POP-FUNCTION-BINDINGS-ACTION*  nil)
(defvar *BOUND-VARIABLES* nil)     ; These are the bound local variables
;;; The globals bound for this scope (may override *BOUND-VARIABLES*)
(defvar *SCOPED-GLOBALS* nil)
;;; Boolean: T when walking a LET's variables list or a LAMBDA's argument list
(defvar *PROCESSING-LOCALLY-DECLARED-SYMBOLS* nil)
(defvar *LET-VARS-SET* nil)        ; The variables set within a let var list
(defvar *LET-VARS-ACCESSED* nil)   ; The variables accessed within a let var list
(defvar *SPECIAL-ACTIONS* nil)
(defvar *EXPAND-QUOTED-LAMBDAS-P* 't)
(defvar *UNBOUND-VARIABLE-ACTION* nil)
(defvar *FUNCTION-ACTION* nil)
(defvar *SPECIAL-FORMS-USED* nil)
(defvar *FUNCTIONS-USED* nil)
(defvar *FUNCTIONS-USED-AT-COMPILE-TIME* nil)
(defvar *MACROS-USED* nil)
(defvar *GLOBALS-ACCESSED* nil)
(defvar *KEYWORDS-USED* nil)
(defvar *GLOBALS-SET*  nil)
(defvar *MACROEXPAND-1-BEFORE-ACTION* nil)
(defvar *CODE-WALK-PARENT* nil)
(defvar *CODE-WALK-CHILDREN* nil)
(defun code-walk (form &key 
		       (special-actions               *SPECIAL-ACTIONS*)
		       (bound-variables               *BOUND-VARIABLES*)
		       (function-action               *FUNCTION-ACTION*)
		       (expand-quoted-lambdas-p       *EXPAND-QUOTED-LAMBDAS-P*)
		       (MACROEXPAND-1-BEFORE-ACTION   *MACROEXPAND-1-BEFORE-ACTION*)
		       (push-variable-bindings-action *push-variable-bindings-action*)
		       (pop-variable-bindings-action  *pop-variable-bindings-action*)
		       (push-function-bindings-action *PUSH-FUNCTION-BINDINGS-ACTION*)
		       (pop-function-bindings-action  *POP-FUNCTION-BINDINGS-ACTION*)
		       (unbound-variable-action       *UNBOUND-VARIABLE-ACTION*))
  (let ((*SPECIAL-ACTIONS*                 special-actions)
	(*UNBOUND-VARIABLE-ACTION*         unbound-variable-action)
	(*PUSH-VARIABLE-BINDINGS-ACTION*   push-variable-bindings-action)
	(*POP-VARIABLE-BINDINGS-ACTION*    pop-variable-bindings-action)
	(*PUSH-FUNCTION-BINDINGS-ACTION*   push-function-bindings-action)
	(*POP-FUNCTION-BINDINGS-ACTION*    pop-function-bindings-action)
	(*BOUND-VARIABLES*                 bound-variables)
	(*FUNCTION-ACTION*                 function-action)
	(*EXPAND-QUOTED-LAMBDAS-P*         expand-quoted-lambdas-p)
	(*MACROEXPAND-1-BEFORE-ACTION*     macroexpand-1-before-action)
	(*PROCESSING-LOCALLY-DECLARED-SYMBOLS*       nil)
	(*SCOPED-GLOBALS*                  nil)
	(*LET-VARS-SET*                    nil)
	(*LET-VARS-ACCESSED*               nil)
	(*CODE-WALK-PARENT*                nil)
	(*CODE-WALK-CHILDREN*              nil)
	(*FUNCTIONS-USED*                  nil)
	(*FUNCTIONS-USED-AT-COMPILE-TIME*  nil)
	(*SPECIAL-FORMS-USED*              nil)
	(*KEYWORDS-USED*                   nil)
	(*MACROS-USED*                     nil)
	(*GLOBALS-SET*                     nil)
	(*GLOBALS-ACCESSED*                nil))
    (values (expand form)
	    (list 
	      :FUNCTIONS               *FUNCTIONS-USED*
	      :COMPILE-TIME-FUNCTIONS  *FUNCTIONS-USED-AT-COMPILE-TIME*
	      :MACROS                  *MACROS-USED*
	      :SPECIAL-FORMS           *SPECIAL-FORMS-USED*
	      :KEYWORDS-USED           *KEYWORDS-USED*
	      :GLOBALS-ACCESSED        *GLOBALS-ACCESSED*
	      :GLOBALS-SET             *GLOBALS-SET*))))

(defmacro with-processing-locally-declared-symbols (&rest body)
  ;; argument-variables-from-arglist will treat as though they were let-bindings
  `(let ((*processing-locally-declared-symbols* t))
     ,@body))

(defmacro with-variable-bindings-preserved (&rest body)
  (let ((old-bound-variables (make-symbol "OLD-BOUND-VARIABLES")))
    `(let ((*BOUND-VARIABLES* *BOUND-VARIABLES*)
	   (*SCOPED-GLOBALS* *SCOPED-GLOBALS*)
	   (*LET-VARS-ACCESSED* nil)
	   (*LET-VARS-SET* nil)
	   (,old-bound-variables *BOUND-VARIABLES*))
       (multiple-value-prog1
	   (progn ,@body)
	 (when *POP-VARIABLE-BINDINGS-ACTION* 
	   (funcall *POP-VARIABLE-BINDINGS-ACTION* ,old-bound-variables))))))

(defsubst code-walk-global-variable-p (symbol)
  (or (lucid::proclaimed-special-p symbol)
      (memq symbol *scoped-globals*)
      (not (memq symbol *bound-variables*))))

(defsubst code-walk-collect-scoped-globals (locally-declared-symbols)
  ;; The local let or argument bindings (a.k.a. the locally-declared-symbols of the
  ;; current scope) override globals. Therefore the scoped globals [i.e. a global
  ;; declared in a "(declare (special ... ))"] for this scope is the set of scoped 
  ;; globals in the outer scope not local declared in the current scope. Additionally,
  ;; they include any further symbols declared global in the current scope.
  (mapcan #'(lambda (global) (unless (memq global locally-declared-symbols) `(,global)))
	  *scoped-globals*))

(defun code-walk-lambda (form)
  (declare (optimize (speed 3) (safety 0) (compilation-speed 0)))
  (with-variable-bindings-preserved
      (let* ((lambda-type (car form))
	     (lambda-name (when (eq lambda-type 'lucid::named-lambda)
			    (list (cadr form))))
	     (lambda-args (if (eq lambda-type 'lucid::named-lambda)
			      (caddr form) (cadr form)))
	     (lambda-body (if (eq lambda-type 'lucid::named-lambda)
			      (cdddr form) (cddr form)))
	     (args (argument-variables-from-arglist lambda-args))
	     (body (mapcar #'expand lambda-body)))
	(let ((*scoped-globals* (code-walk-collect-scoped-globals args)))
	  `(,lambda-type ,@lambda-name ,args ,@body)))))

(defun argument-variables-from-arglist (arglist)
  "Strip Off optional and key initial values and &key, &optional ... "
  (declare (optimize (speed 3) (safety 0) (compilation-speed 0)))
  (let ((seen-ampersand-p nil))
    (with-processing-locally-declared-symbols
	(mapcan #'(lambda (a)
		    (cond ((consp a)
			   (if seen-ampersand-p
			       (let ((var (car a))
				     (rest (mapcar #'expand (cdr a))))
				 (pushnew var *LET-VARS-SET*)
				 (if *PUSH-VARIABLE-BINDINGS-ACTION*
				     (funcall *PUSH-VARIABLE-BINDINGS-ACTION* var)
				     (pushnew var *BOUND-VARIABLES*))
				 (list (cons var rest)))
			       (argument-variables-from-arglist a)))
			  ((and (symbolp a) (eq (aref (symbol-name a) 0) #\&)) 
			   (setq seen-ampersand-p t)
			   (list a))
			  ((symbolp a) 
			   (pushnew a *LET-VARS-SET*)
			   (if *PUSH-VARIABLE-BINDINGS-ACTION*
			       (funcall *PUSH-VARIABLE-BINDINGS-ACTION* a)
			       (pushnew a *BOUND-VARIABLES*)) 
			   (list a))
			  (t (error "Invalid arg in lambda list: ~A" a))))
		arglist))))

(defun expand (form)
  (declare (optimize (speed 3) (safety 0) (compilation-speed 0)))
  (cond ((keywordp form)
	 (pushnew form *KEYWORDS-USED*)
	 form)
	((and (symbolp form) (code-walk-global-variable-p form))
	 (pushnew form *GLOBALS-ACCESSED*)
	 (if *unbound-variable-action* 
	     (funcall *UNBOUND-VARIABLE-ACTION* form)
	     form))
	((and (symbolp form) *processing-locally-declared-symbols*)
	 (pushnew form *LET-VARS-ACCESSED*)
	 form)
	((not (consp form))  ;; this is to eat up strings, numbers, etc
	 form)
	((consp (car form)) 
	 (mapcar #'expand form))
	((let* ((a (assoc (car form) *SPECIAL-ACTIONS*))
		(test (and a (sys:procedurep (third a)) (third a))))
	   (and a (or (null test) (funcall test form))))
	 (funcall (cadr (assoc (car form) *SPECIAL-ACTIONS*)) form))
	((member (car form) *SPECIAL-FORMS*) 
	 (pushnew (car form) *SPECIAL-FORMS-USED*)
	 (expand-special-form form))
	((macro-function (car form))
	 (pushnew (car form) *MACROS-USED*)
	 (when *MACROEXPAND-1-BEFORE-ACTION*
	   (setq form (funcall *MACROEXPAND-1-BEFORE-ACTION* form)))
	 (expand (macroexpand-1 form)))
	((member (car form) '(lisp::lambda lucid::named-lambda))
	 (code-walk-lambda form))
	((member (car form) '(LISP::EVAL LISP::APPLY LISP::FUNCALL))
	 (let ((form (expand (cadr form))))
	   (if (and (consp form) (member (car form) '(lisp::quote)))
	       (if (consp (cadr form)) 
		   (expand (cadr form)) ;; Just to figure out functions used
		   (pushnew (cadr form) *FUNCTIONS-USED*))))
	 (pushnew (car form) *FUNCTIONS-USED*)
	 `(,(car form) ,@(mapcar #'expand (cdr form))))
	(*FUNCTION-ACTION* (funcall *FUNCTION-ACTION* form))
	(t ;;; Plain old function call
	 (pushnew (car form) *FUNCTIONS-USED*)
	 `(,(car form) ,@(mapcar #'expand (cdr form))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;         Functions in the *code-walk-special-form-hashtable*                   ;;;
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun code-walk-simple-walk (form)
  `(,(car form) ,@(mapcar #'expand (cdr form))))

(defun code-walk-let (form)
  (declare (optimize (speed 3) (safety 0) (compilation-speed 0)))
  (with-variable-bindings-preserved
      (let ((name (car form))
	    (vars (cadr form))
	    (body (cddr form))
	    (varlist nil))
	(unless (eq name 'lisp::let*)
	  (with-processing-locally-declared-symbols
	      (setq vars (mapcar 
			   #'(lambda (var) 
			       (if (consp var)
				   (let ((varname (car var)))
				     (pushnew varname *LET-VARS-SET*)
				     (push varname varlist)
				     (cons varname (mapcar #'expand (cdr var))))
				   (progn (pushnew var *LET-VARS-SET*)
					  (push var varlist)
					  var)))
			   vars))))
	(setq vars
	      (let ((let-star-p (eq name 'lisp::let*)))
		(mapcar 
		  #'(lambda (var)
		      (when let-star-p
			(with-processing-locally-declared-symbols
			    (if (consp var)
				(let ((varname (car var)))
				  (push varname varlist)
				  (setq var (cons (car var)
						  (mapcar #'expand (cdr var)))))
				(push var varlist))))
		      (let ((v (if (consp var) (car var) var)))
			(when let-star-p (pushnew v *LET-VARS-SET*))
			(if *PUSH-VARIABLE-BINDINGS-ACTION* 
			    (setq v (funcall *PUSH-VARIABLE-BINDINGS-ACTION* v))
			    (pushnew v *BOUND-VARIABLES*))
			(if (consp var) (setf (car var) v) (setq var v)))
		      var)
		  vars)))
	(let ((*scoped-globals* (code-walk-collect-scoped-globals varlist)))
	  `(,name ,vars ,@(mapcar #'expand body))))))

(defun code-walk-macrolet (form)
  (declare (optimize (speed 3) (safety 0) (compilation-speed 0)))
  (let ((*FUNCTIONS-USED* nil))
    (prog1
	(code-walk-local-functions-and-macros form)
      (setf *FUNCTIONS-USED-AT-COMPILE-TIME*
	    (nconc *FUNCTIONS-USED* *FUNCTIONS-USED-AT-COMPILE-TIME*)))))

(defun code-walk-local-functions-and-macros (form)
  (declare (optimize (speed 3) (safety 0) (compilation-speed 0)))
  (flet ((guys-used () 
	   (ecase (car form) 
	     (lisp::macrolet *MACROS-USED*)
	     ((lisp::flet lisp::labels) *FUNCTIONS-USED*)))
	 (set-guys-used (value) 
	   (ecase (car form) 
	     (lisp::macrolet (setf *MACROS-USED* value))
	     ((lisp::flet lisp::labels) (setf *FUNCTIONS-USED* value)))))
    (declare (inline guys-used set-guys-used))
    (let ((flet-functions nil)
	  (*special-actions* *special-actions*)
	  (ofunctions (guys-used)))
      (flet ((delete-flet-functions ()
	       (do ((f (guys-used) (cdr f)))
		   ((eq f ofunctions) f)
		 (when (memq (car f) flet-functions)
		   (set-guys-used (delete (car f) (guys-used) :count 1))))))
	(let ((flets
	       (mapcar 
		 #'(lambda (flet) 
		     (let ((flet-name (car flet))
			   (flet-args (cadr flet))
			   (flet-body (cddr flet)))
		       (ecase (car form)
			 ((lisp::flet lisp::labels)
			  (push flet-name flet-functions))
			 ((lisp::macrolet)
			  (push (make-symbol (string flet-name)) flet-functions)))
		       (let ((l (expand `(lambda ,flet-args ,@flet-body))))
			 (when (eq (car form) 'lisp::macrolet)
			   (let ((name (car flet-functions)))
			     (when *PUSH-FUNCTION-BINDINGS-ACTION*
			       (funcall *PUSH-FUNCTION-BINDINGS-ACTION* name))
			     ;; Whenever the code walker sees the old name, have it 
			     ;; glue the new name in instead.
			     (push `(,flet-name
				       ,#'(lambda (form)
					    (expand (cons name (cdr form))))
				       :FLET-LABEL-OR-MACRO)
				   *special-actions*)
			     (eval `(defmacro ,name ,@(cdr l)))))
			 (when (memq (car form) 
				     '(;; Lucid treats macrolet ala labels?
				       #+LUCID lisp::macrolet
				       lisp::labels))
			   (when *POP-FUNCTION-BINDINGS-ACTION*
			     (funcall *POP-FUNCTION-BINDINGS-ACTION* flet-functions))
			   (delete-flet-functions))
			 (setq ofunctions (guys-used))
			 `(,flet-name ,@(cdr l)))))
		 (cadr form))))
	  (unwind-protect
	       `(,(car form) ,flets ,@(mapcar #'expand (cddr form)))
	    (when *POP-FUNCTION-BINDINGS-ACTION*
	      (funcall *POP-FUNCTION-BINDINGS-ACTION* flet-functions))
	    (delete-flet-functions)))))))

(defun code-walk-setq (form)
  (declare (optimize (speed 3) (safety 0) (compilation-speed 0)))
  (do ((f (cdr form) (cddr f))
       (result (list 'setq)))
      ((null f) (nreverse result))
    (let ((set-variable (car f))
	  (set-to (cadr f)))
      (push set-variable result)
      (push (expand set-to) result)
      (cond
	((lucid::proclaimed-special-p set-variable)
	 (pushnew set-variable *GLOBALS-SET*))
	((member set-variable *scoped-globals*)
	 (pushnew set-variable *GLOBALS-SET*))
	((member set-variable *BOUND-VARIABLES* :test #'eq)
	 (when *processing-locally-declared-symbols*
	   (pushnew set-variable *LET-VARS-SET*)))
	(t (pushnew set-variable *GLOBALS-SET*))))))

(defun code-walk-declare (form)
  (declare (optimize (speed 3) (safety 0) (compilation-speed 0)))
  (dolist (declaration (cdr form))
    (when (consp declaration)
      (case (car declaration)
	(lisp::special
	 (dolist (global (cdr declaration))
	   (push global *SCOPED-GLOBALS*)
	   (when (memq global *let-vars-set*)
	     (pushnew global *globals-set*))
	   (when (memq global *let-vars-accessed*)
	     (pushnew global *globals-accessed*)))))))
  form)

(defun code-walk-function (form)
;  (if (consp (cadr form))
;      (expand (cadr form)) ;; Just to figure out functions used
;      (pushnew (cadr form) *FUNCTIONS-USED*))
  (when (symbolp (cadr form))
    (pushnew (cadr form) *FUNCTIONS-USED*))
  (code-walk-simple-walk form))

(defun code-walk-if (form)
  `(,(car form)
     ,(expand (cadr form))
     ,(with-variable-bindings-preserved (expand (caddr form)))
     ,(with-variable-bindings-preserved (expand (cadddr form)))))

(defun code-walk-quote (form)
  (cond ((and *EXPAND-QUOTED-LAMBDAS-P*
	      (consp (cadr form))
	      (eq (caadr form) 'lisp::lambda))
	 ;; Do I really want to expand lambdas?
	 (let ((*special-actions* 
		(remove-if #'(lambda (a) (eq (third a) :FLET-LABEL-OR-MACRO))
			   *special-actions*)))
	   `(,(car form) ,@(mapcar #'expand (cdr form)))))
	(t form)))

(defun code-walk-tagbody (form)
  `(,(car form)
     ,@(mapcar #'(lambda (form) (if (consp form) (expand form) form)) (cdr form))))

(defun code-walk-return-from (form)
  `(,(car form) ,(cadr form) ,@(mapcar #'expand (cddr form))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;        End of functions in the *code-walk-special-form-hashtable*             ;;;
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *code-walk-special-form-hashtable* (make-hash-table :test #'eq :size 25))

(defmacro code-walk-register-special-form (form-name function-name)
  `(setf (gethash ',form-name *code-walk-special-form-hashtable*)
	 (symbol-function ',function-name)))

(progn
  (code-walk-register-special-form lisp::block code-walk-return-from)
  (code-walk-register-special-form lisp::catch code-walk-simple-walk)
  (code-walk-register-special-form lisp::compiler-let code-walk-let)
  (code-walk-register-special-form lisp::declare code-walk-declare)
  (code-walk-register-special-form lisp::eval-when code-walk-return-from)
  (code-walk-register-special-form lisp::flet code-walk-local-functions-and-macros)
  (code-walk-register-special-form lisp::function code-walk-function)
  (code-walk-register-special-form lisp::go code-walk-return-from)
  (code-walk-register-special-form lisp::if code-walk-if)
  (code-walk-register-special-form lisp::labels code-walk-local-functions-and-macros)
  (code-walk-register-special-form lisp::let code-walk-let)
  (code-walk-register-special-form lisp::let* code-walk-let)
  (code-walk-register-special-form lisp::macrolet code-walk-macrolet)
  (code-walk-register-special-form lisp::multiple-value-call code-walk-simple-walk)
  (code-walk-register-special-form lisp::multiple-value-prog1 code-walk-simple-walk)
  (code-walk-register-special-form lisp::progn code-walk-simple-walk)
  (code-walk-register-special-form lisp::progv code-walk-simple-walk)
  (code-walk-register-special-form lisp::quote code-walk-quote)
  (code-walk-register-special-form lisp::return-from code-walk-return-from)
  (code-walk-register-special-form lisp::setq code-walk-setq)
  (code-walk-register-special-form lisp::tagbody code-walk-tagbody)
  (code-walk-register-special-form lisp::the code-walk-return-from)
  (code-walk-register-special-form lisp::throw code-walk-simple-walk)
  (code-walk-register-special-form lisp::unwind-protect code-walk-simple-walk))

(defun expand-special-form (form)
  (declare (optimize (speed 3) (safety 0) (compilation-speed 0)))
  (let ((code-walk-function (gethash (car form) *code-walk-special-form-hashtable*)))
    (if code-walk-function
	(funcall code-walk-function form)
	(error "The special form ~A has no entry in the hashtable."
	       code-walk-function))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;                   End of source for the code walker                        ;;;
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defmacro export-package-symbols ()
  (let ((g (gensym)))
    `(eval-when (load eval)
       (do-symbols (,g *package*)
	 (export ,g)))))

(export-package-symbols)

