"======================================================================
|
|   Generic visitor pattern for STParseNodes
|
|
 ======================================================================"


"======================================================================
|
| Copyright 1999, 2000, 2001, 2002 Free Software Foundation, Inc.
| Written by Paolo Bonzini.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
| 
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
| 
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"

STParser subclass: #STVisitingParser
       instanceVariableNames: 'compiler evaluator'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'System-Compiler'
!

STVisitingParser comment:
'I provide means to pass a parse tree to a STParseNodeVisitor.  I am used
internally by that class.'!

!STVisitingParser methodsFor: 'error checking'!

sanityCheck
    (compiler isNil eqv: evaluator isNil)
	ifTrue: [ self error: 'badly initialized STVisitingParser' ]! !

!STVisitingParser methodsFor: 'accessing'!

compileThrough: aVisitor
    compiler := aVisitor!

evaluateThrough: aVisitor
    evaluator := aVisitor! !

!STVisitingParser methodsFor: 'pluggable STParser behavior'!

searchMethodListHeader
    self sanityCheck.

    "No evaluator: assumes all bang-separated method definitions."
    evaluator isNil ifTrue: [ ^self atEnd not ].

    "No compiler, evaluator: assumes all doits."
    ^super searchMethodListHeader!

comment: commentTok
    self lastNode addComment: commentTok value!

compile: aNode
    "#compile: should be called only if compiler is not nil!"
    compiler visit: aNode!

evaluate: aNode
    "#evaluate: should be called only if evaluator is not nil!"
    evaluator visit: aNode.
    ^false! !

Object subclass: #STParseNodeVisitor
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: 'VMOtherConstants VMByteCodeNames'
       category: 'System-Compiler'
!

STParseNodeVisitor comment:
'I provide generic callbacks to visit a Smalltalk parse tree.'!

!STParseNodeVisitor methodsFor: 'visiting sets of nodes'!

visitMethodDefinition: aStream
    (STVisitingParser onStream: aStream)
	compileThrough: self;
	parseSmalltalk!

visitDoitCode: aStream
    (STVisitingParser onStream: aStream)
	evaluateThrough: self;
	parseSmalltalk! !

!STParseNodeVisitor methodsFor: 'visiting sets of nodes'!

arguments: argumentsCollection
    argumentsCollection do: [ :each | self argument: each ]
!

assignments: assignmentsCollection
    assignmentsCollection do: [ :each | self assignment: each ]
!

beginMessage: aMessageNode
!

endMessage: aMessageNode
!

sequence: nodeCollection
    nodeCollection
	do: [ :each | self visit: each ]
	separatedBy: [ self endOfStatement ]
!

temporaries: temporariesCollection
    temporariesCollection do: [ :each | self temporary: each ]
!

!STParseNodeVisitor methodsFor: 'misc. '!

endOfStatement!

scopeEnter
!

scopeLeave
! !

!STParseNodeVisitor methodsFor: 'visiting particular nodes'!

argument: aSymbol
!

assignment: assignedSymbol
!

beginMessage: messageNode
!

braceArray: nodeCollection
    self sequence: nodeCollection
!

cascadeReceiver: anExpressionNode
    "Treat as a #messageReceiver: by default"
    self messageReceiver: anExpressionNode
!

comment: aString
!

endMessage: messageNode
!

literal: aLiteral
!

message: aMessageSendNode
!

messageReceiver: anExpressionNode
!

primitive: anIntegerOrNil
!

return: aNode
!

selector: aSymbol
!

temporary: aSymbol
!

variable: aSymbol
!

variableBinding: anOrderedCollection
!

visit: aNode
    aNode visitThrough: self
! !


!STParseNode methodsFor: 'visiting'!

passCommentsTo: aVisitor
    comments isNil ifTrue: [ ^self ].
    self comments do: [ :each | aVisitor comment: each ]
!

visitThrough: aVisitor
    self subclassResponsibility
! !

!STBindingNode methodsFor: 'visiting'!

visitThrough: aVisitor
    aVisitor variableBinding: self binding.
    self passCommentsTo: aVisitor
! !

!STBlockNode methodsFor: 'visiting'!

visitThrough: aVisitor
    self passCommentsTo: aVisitor.
    aVisitor scopeEnter.
    1 to: self numArgs do: [ :each |
    	aVisitor argument: (self parameters at: each)
    ].
    self numArgs + 1 to: self numArgs + self numTemps do: [ :each |
    	aVisitor temporary: (self parameters at: each)
    ].
    aVisitor sequence: self statements.
    aVisitor scopeLeave.
! !

!STCascadeNode methodsFor: 'visiting'!

visitThrough: aVisitor
    aVisitor
	beginMessage: self;
	cascadeReceiver: self receiver;
	message: self message.

    self otherMessages do: [ :each | aVisitor message: each ].

    aVisitor endMessage: self.
    self passCommentsTo: aVisitor.
! !

!STArrayNode methodsFor: 'visiting'!

visitThrough: aVisitor
    self passCommentsTo: aVisitor.
    aVisitor braceArray: self statements.
! !

!STConstNode methodsFor: 'visiting'!

visitThrough: aVisitor
    aVisitor literal: self value.
    self passCommentsTo: aVisitor.
! !

!STExpressionNode methodsFor: 'visiting'!

visitThrough: aVisitor
    self assigns isEmpty ifFalse: [ aVisitor assignments: self assigns ].
    aVisitor visit: self expression.
    self passCommentsTo: aVisitor.
! !

!STIdentifierNode methodsFor: 'visiting'!

visitThrough: aVisitor
    aVisitor variable: self id.
    self passCommentsTo: aVisitor.
! !

!STSpecialIdentifierNode methodsFor: 'visiting'!

visitThrough: aVisitor
    | literal |
    self id = -1 ifTrue: [ ^aVisitor variable: #super ].
    self id = 0 ifTrue: [ ^aVisitor variable: #self ].
    self id = 25 ifTrue: [ ^aVisitor variable: #thisContext ].
    aVisitor literal: (VMSpecialLiterals at: self id).
    self passCommentsTo: aVisitor.
! !

!STMessageNode methodsFor: 'visiting'!

visitThrough: aVisitor
    aVisitor
	beginMessage: self;
	messageReceiver: self receiver;
	message: self message;
	endMessage: self.

    self passCommentsTo: aVisitor.
! !

!STMessageSendNode methodsFor: 'visiting'!

visitThrough: aVisitor
    aVisitor message: self.
    self passCommentsTo: aVisitor.
! !

!STMethodDefinitionNode methodsFor: 'visiting'!

visitThrough: aVisitor
    self passCommentsTo: aVisitor.
    aVisitor
	scopeEnter;
	selector: self selector selector;
	arguments: self selector args;
	temporaries: self body temporaries;
	primitive: self body primitive;
	sequence: self body statements;
	scopeLeave.
! !

!STReturnNode methodsFor: 'visiting'!

visitThrough: aVisitor
    aVisitor return: self expression.
    self passCommentsTo: aVisitor.
! !
