/*
 * control-proto.h - control protocol header definitions
 *
 * Copyright (C) 2000 Stefan Jahn <stefan@lkcc.org>
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this package; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 *
 * $Id: control-proto.h,v 1.7 2000/09/15 08:22:50 ela Exp $
 *
 */

#ifndef __CONTROL_PROTO_H__
#define __CONTROL_PROTO_H__

#if HAVE_CONFIG_H
# include <config.h>
#endif

#define _GNU_SOURCE
#include "socket.h"
#include "server.h"

#define STAT_BUFFER_SIZE 256

/*
 * System statistics structure.
 */
typedef struct
{
#if HAVE_PROC_STAT
  char *cpufile;                /* CPU state file under Linux */
  char *cpuline;                /* CPU line format under Linux */
#endif /* HAVE_PROC_STAT */
  char *cpuinfoline;            /* the info format string */
  char info[STAT_BUFFER_SIZE];  /* the info line itself */
  char pinfo[STAT_BUFFER_SIZE]; /* process info itself */
  unsigned long cpu[2][4];      /* cpu values */
  unsigned long total[2];
  unsigned long proc[2][4];     /* process values */
  unsigned long ptotal[2];
  int index;                    /* index for cpu differences */
}
cpu_state_t;

/*
 * Control protocol server configuration.
 */
typedef struct
{
  portcfg_t *netport; /* port configuration only (yet) */
}
ctrl_config_t;

/* Export the control server definition to `server.c'. */
extern server_definition_t ctrl_server_definition;

/* server functions */
int ctrl_init (server_t *server);
int ctrl_finalize (server_t *server);
char *ctrl_info_server (server_t *server);
char *ctrl_info_client (void *ctrl_cfg, socket_t sock);

/* basic protocol functions */
int ctrl_detect_proto (void *cfg, socket_t sock);
int ctrl_connect_socket (void *cfg, socket_t sock);

int ctrl_idle (socket_t sock);
int ctrl_handle_request (socket_t sock, char *request, int len);

#define CTRL_FLAG_PASSED          0x0001
#define CTRL_PACKET_DELIMITER     "\n"
#define CTRL_PACKET_DELIMITER_LEN 1

/*
 * Format string for system business output on different systems.
 */
#ifdef HAVE_LIBKSTAT
# define CPU_FORMAT \
  "user %ld.%01ld%%, sys %ld.%01ld%%, wait %ld.%01ld%%, idle %ld.%01ld%%"
#elif HAVE_PROC_STAT
# define CPU_FORMAT \
  "user %ld.%01ld%%, nice %ld.%01ld%%, sys %ld.%01ld%%, idle %ld.%01ld%%"
#elif HAVE_PSTAT
# define CPU_FORMAT \
  "user %ld.%01ld%%, nice %ld.%01ld%%, sys %ld.%01ld%%, idle %ld.%01ld%%"
#elif HAVE_SYSGET
# define CPU_FORMAT \
  "user %ld.%01ld%%, sys %ld.%01ld%%, wait %ld.%01ld%%, idle %ld.%01ld%%"
#else
# define CPU_FORMAT "no cpu info available"
#endif

#if HAVE_TIMES
# define PROC_FORMAT \
  "user %ld.%01ld%%, sys %ld.%01ld%%, " \
  "child user %ld.%01ld%%, child sys %ld.%01ld%%"
#else
# define PROC_FORMAT \
  "process %ld.%01ld%%"
#endif

#if HAVE_PROC_STAT
# define CPU_FILE_NAME   "/proc/stat"
# define CPU_LINE_FORMAT "cpu  %lu %lu %lu %lu"
#endif /* HAVE_PROC_STAT */

/* welcome message */
#define CTRL_PASSWD "Welcome to serveez control center. " \
                    "Please login.\r\n\r\n" \
                    "Password: "

/* the control protocol client prompt */
#define CTRL_PROMPT "ctrl-sh $ "

/* the receive and send buffer size of an control protocol client */
#define CTRL_RECV_BUFSIZE 512
#define CTRL_SEND_BUFSIZE 1024 * 100

/* how often we update the CPU information (in seconds) */
#define CTRL_LOAD_UPDATE 1

/* these are all available instructions */
#define CTRL_CMD_HELP          "help"
#define CTRL_CMD_QUIT          "quit"
#define CTRL_CMD_EXIT          "exit"
#define CTRL_CMD_BYE           "bye"
#define CTRL_CMD_STAT          "stat"
#define CTRL_CMD_STAT_CON      "stat con"
#define CTRL_CMD_STAT_ALL      "stat all"
#define CTRL_CMD_STAT_ID       "stat id"
#define CTRL_CMD_STAT_COSERVER "stat coserver"
#define CTRL_CMD_STAT_CACHE    "stat cache"
#define CTRL_CMD_KILL_CACHE    "kill cache"
#define CTRL_CMD_KILLALL       "killall"
#define CTRL_CMD_KILL_ID       "kill id"
#define CTRL_CMD_RESTART_RDNS  "restart reverse dns"
#define CTRL_CMD_RESTART_DNS   "restart dns"
#define CTRL_CMD_RESTART_IDENT "restart ident"

#endif /* __CONTROL_PROTO_H__ */
