%{
/*
     spar - Show Process Accounting Records
     Copyright (C) 1993 Douglas Lee Schales, David K. Hess, David R. Safford

     Please see the file `COPYING' for the complete copyright notice.

grammar.y - 10/28/93

*/
#include <stdio.h>
#include <netdb.h>
#include <sys/types.h>
#include <netinet/in.h>

#include "config.h"
#include "lex.h"
#include "chario.h"
#include "parser.h"
#include "builder.h"
#include "timesub.h"
#include "spar.h"

extern int parse_error;
extern int quietflag;

extern int cgetuidname(char *, unsigned long *);
extern int cgetgidname(char *, unsigned long *);
extern int cgetttyname(char *, unsigned long *);
%}

%union {
     unsigned int intval;
     unsigned long *intpval;
     char *strval;
}

%token USER GROUP COMMAND TTY TIME DATE PRIV CPUTIME USERTIME SYSTIME
%token TODAY YEST INCLUDE
%token OR AND MASKOPER NOT SINCE BEFORE
%token NEQ LEQ GEQ STREQ STRNE
%token INTEGER STRING BYTEVAL
%token PRINT NEXT
%type <intval> INTEGER BYTEVAL
%type <intval> OR AND
%type <intval> oper NEQ LEQ GEQ '=' '<' '>' '!' NOT STREQ STRNE
%type <intval> DATE TIME TODAY YEST SINCE BEFORE
%type <intval> USER COMMAND TTY GROUP PRIV CPUTIME USERTIME SYSTIME
%type <intval> INCLUDE
%type <intval> PRINT NEXT
%type <strval> STRING
%type <intval> datespec integer cputime timespec
%left OR
%left AND
%right '!'
%%
file
     : stmt file
     | stmt 
;

stmt
     : cond '{' actionset '}' {
	  queuestmt();
     }
     | INCLUDE STRING ';' {
	  if(includefile($2) != 0){
	       (void)fprintf(stderr, "\"%s\", line %d: include file error.\n",
		       getfilename(), getlinenum());
	       parse_error = 1;
	  }
     }
;

cond
     : /* */
     | cond OR cond {
          pushoper($2);
     }
     | cond AND cond {
	  pushoper($2);
     }
     | term
;

actionset
     : action ';' actionset
     | action ';'
     | action
;

action
     : PRINT {
	  queueact($1);
     }
     | NEXT {
          queueact($1);
     }
     | error
;

term
     : USER oper STRING {
	  unsigned long uid;
	  if(cgetuidname($3, &uid) != -1){
	       pushkey(USER);
	       pushval(uid);
	       pushoper($2);
	  }
	  else {
	       if(!quietflag)
		    (void)fprintf(stderr,
				  "\"%s\", line %d: User %s unknown.\n",
				  getfilename(), getlinenum(), $3);
	       switch($2){
	       case '=':
		    pushval(0);
		    break;
	       default:
		    pushval(1);
		    break;
	       }
	  }
	  free($3);
     }
     | USER oper integer {
	  pushkey(USER);
	  pushval($3);
	  pushoper($2);
     }
     | GROUP oper STRING {
	  unsigned long gid;
	  if(cgetgidname($3, &gid) != -1){
	       pushkey(GROUP);
	       pushval(gid);
	       pushoper($2);
	  }
	  else switch($2){
	  case '=':
	       pushval(0);
	       break;
	  default:
	       pushval(1);
	       break;
	  }
	  free($3);
     }
     | GROUP oper integer {
	  pushkey(GROUP);
	  pushval($3);
	  pushoper($2);
     }
     | COMMAND oper STRING {
	  int soper = 0;

	  switch($2){
	  case '=':
	       soper = STREQ;
	       break;
	  case NEQ:
	       soper = STRNE;
	       break;
	  }
	  if(soper){
	       pushkey(COMMAND);
	       pushval((unsigned long)$3); /* XXX - need pushstr() */
	       pushoper(soper);
	  }
     }
     | TTY oper STRING {
	  unsigned long ttynum;
	  
	  if(cgetttyname($3, &ttynum) != -1){
	       pushkey(TTY);
	       pushval(ttynum);
	       pushoper($2);
	  }
	  else switch($2){
	  case '=':
	       pushval(0);
	       break;
	  default:
	       pushval(1);
	       break;
	  }
	  free($3);
     }
     | DATE oper datespec {
	  unsigned long sday, eday;
	  sday = $3;
	  eday = timeofday(sday, 23, 59, 59);
          switch($2){
	  case '>':
	       pushkey(STARTDATE);
	       pushval(eday);
	       pushoper($2);
	       pushkey(ENDDATE);
	       pushval(eday);
	       pushoper($2);
	       pushoper(OR);
	       break;
	  case GEQ:
	       pushkey(STARTDATE);
	       pushval(sday);
	       pushoper($2);
	       pushkey(ENDDATE);
	       pushval(eday);
	       pushoper($2);
	       pushoper(OR);
	       break;
	  case '<':
	       pushkey(STARTDATE);
	       pushval(sday);
	       pushoper($2);
	       pushkey(ENDDATE);
	       pushval(sday);
	       pushoper($2);
	       pushoper(OR);
	       break;
	  case LEQ:
	       pushkey(STARTDATE);
	       pushval(eday);
	       pushoper($2);
	       pushkey(ENDDATE);
	       pushval(eday);
	       pushoper($2);
	       pushoper(OR);
	       break;
	  case '=':
	       pushkey(STARTDATE);
	       pushval(eday);
	       pushoper(LEQ);

	       pushkey(ENDDATE);
	       pushval(sday);
	       pushoper(GEQ);

	       pushoper(AND);
	       break;
	  case NEQ:
	       pushkey(ENDDATE);
	       pushval(sday);
	       pushoper('<');
	       pushkey(STARTDATE);
	       pushval(eday);
	       pushoper('>');
	       pushoper(OR);
	       break;
	  }
     }
     | PRIV {
          pushkey(PRIV);
     }
     | TIME oper timespec {
          switch($2){
	  case '>':
	  case GEQ:
	       pushkey(STARTTIME);
	       pushval($3);
	       pushoper($2);
	       pushkey(ENDTIME);
	       pushval($3);
	       pushoper($2);
	       pushoper(OR);
	       break;
	  case '<':
	  case LEQ:
	       pushkey(STARTTIME);
	       pushval($3);
	       pushoper($2);
	       pushkey(ENDTIME);
	       pushval($3);
	       pushoper($2);
	       pushoper(OR);
	       break;
	  case '=':
	       pushkey(STARTTIME);
	       pushval($3);
	       pushoper(GEQ);
	       pushkey(ENDTIME);
	       pushval($3);
	       pushoper(LEQ);
	       pushoper(AND);
	       break;
	  case NEQ:
	       pushkey(ENDTIME);
	       pushval($3);
	       pushoper('<');
	       pushkey(STARTTIME);
	       pushval($3);
	       pushoper('>');
	       pushoper(OR);
	       break;
	  }
     }
     | cputime oper timespec {
          pushkey($1);
          pushval($3);
          pushoper($2);
     }
     | SINCE datespec BYTEVAL ':' BYTEVAL {
	  unsigned long when;

	  when = timeofday($2, $3, $5, 0);
          pushkey(STARTDATE);
          pushval(when);
          pushoper(GEQ);
	  pushkey(ENDDATE);
	  pushval(when);
	  pushoper(GEQ);
	  pushoper(OR);
     }
     | SINCE datespec BYTEVAL ':' BYTEVAL ':' BYTEVAL {
	  unsigned long when;

	  when = timeofday($2, $3, $5, $7);
          pushkey(STARTDATE);
          pushval(when);
          pushoper(GEQ);
	  pushkey(ENDDATE);
	  pushval(when);
	  pushoper(GEQ);
	  pushoper(OR);
     }
     | SINCE datespec {
          pushkey(STARTDATE);
          pushval($2);
          pushoper(GEQ);
	  pushkey(ENDDATE);
	  pushval($2);
	  pushoper(GEQ);
	  pushoper(OR);
     }
     | BEFORE datespec BYTEVAL ':' BYTEVAL {
	  unsigned long when;
	  when = timeofday($2, $3, $5, 0);

	  pushkey(STARTDATE);
	  pushval(when);
	  pushoper('<');
          pushkey(ENDDATE);
          pushval(when);
          pushoper('<');
	  pushoper(OR);
     }
     | BEFORE datespec BYTEVAL ':' BYTEVAL ':' BYTEVAL {
	  unsigned long when;
	  when = timeofday($2, $3, $5, $7);

	  pushkey(STARTDATE);
	  pushval(when);
	  pushoper('<');
          pushkey(ENDDATE);
          pushval(when);
          pushoper('<');
	  pushoper(OR);
     }
     | BEFORE datespec {
          pushkey(STARTDATE);
          pushval($2);
          pushoper('<');
	  pushkey(ENDDATE);
	  pushval($2);
	  pushoper('<');
	  pushoper(OR);
     }
     | '(' cond ')'
     | '!' term { pushuoper($1); }
     | NOT term { pushuoper('!'); }
     | error { pushval(0); }
;

oper
     : '=' {$$ = $1;}
     | '>' {$$ = $1;}
     | '<' {$$ = $1;}
     | NEQ {$$ = $1;}
     | LEQ {$$ = $1;}
     | GEQ {$$ = $1;}
     | error { pushoper(0); }
;

cputime
     : CPUTIME
     | USERTIME
     | SYSTIME
;

datespec
     : BYTEVAL '/' BYTEVAL {
	  int year = getyear();
	  if(year < 100)
	       year += 1900;
	  if($1 < 1 || $1 > 12 ||
	     !validmday($1, $3, year)){
	       (void)fprintf(stderr, "\"%s\", line %d: malformed date.\n",
		       getfilename(), getlinenum());
	       parse_error = 1;
	  }
	  else
	       $$ = makedate($1, $3, year);
     }
     | BYTEVAL '/' BYTEVAL '/' integer {
	  int year = $5;
	  if(year < 100)
	       year += 1900;
	  if($1 < 1 || $1 > 12 ||
	     !validmday($1, $3, year)){
	       (void)fprintf(stderr, "\"%s\", line %d: malformed date.\n",
		       getfilename(), getlinenum());
	       parse_error = 1;
	  }
	  else
	       $$ = makedate($1, $3, year);
     }
     | TODAY {
          $$ = today();
     }
     | YEST {
          $$ = yesterday();
     }
     | error { $$ = 0; }
;	       

integer
     : INTEGER {
          $$ = $1;
     }
     | BYTEVAL {
          $$ = $1;
     }
;

timespec
     : BYTEVAL ':' BYTEVAL {
	  $$ = $1*3600 + $3*60;
     }
     | BYTEVAL ':' BYTEVAL ':' BYTEVAL {
	  $$ = $1*3600 + $3*60 + $5;
     }
     | error { $$ = 0; }
;

%%
     struct keywords keywords[] = {
     { "user", USER },
     { "cmd", COMMAND },
     { "tty", TTY },
     { "time", TIME },
     { "date", DATE },
     { "since", SINCE },
     { "before", BEFORE },
     { "today", TODAY },
     { "yesterday", YEST },
     { "include", INCLUDE },
     { "group", GROUP },
     { "priv", PRIV },
     { "cputime", CPUTIME },
     { "usertime", USERTIME },
     { "systime", SYSTIME },
     { "su", PRIV },
     { "or", OR },
     { "and", AND },
     { "not", NOT }, 
     { "print", PRINT },
     { "next", NEXT },
     { (char *)0, 0 }
     };

void
yyerror()
{
     (void)fprintf(stderr, "\"%s\", line %d: syntax error.\n",
	     getfilename(), getlinenum());
     parse_error = 1;
}
