package WebService::GData::YouTube;
use WebService::GData;
use base 'WebService::GData';

use WebService::GData::Base;
use WebService::GData::YouTube::Constants;
use WebService::GData::YouTube::Query;
use WebService::GData::YouTube::Feed;
use WebService::GData::YouTube::Feed::PlaylistLink;
use WebService::GData::YouTube::Feed::Video;
our $PROJECTION = WebService::GData::YouTube::Constants::PROJECTION;
our $BASE_URI   = WebService::GData::YouTube::Constants::BASE_URI;
our $VERSION    = 0.01_07;

sub __init {
    my ( $this, $auth ) = @_;

    $this->{_baseuri} = $BASE_URI . $PROJECTION . '/';
    $this->{_request}    = new WebService::GData::Base();
    $this->{_request}->auth($auth) if ($auth);

    #overwrite default query engine to support youtube extra feature
    my $query = new WebService::GData::YouTube::Query();
    $query->key( $auth->key ) if ($auth);
    $this->query($query);
}

sub query {
    my ( $this, $query ) = @_;
    return $this->{_request}->query($query);
}

sub base_uri {
    my $this = shift;
    return $this->{_baseuri};
}

sub base_query {
    my $this = shift;
    return $this->query->to_query_string;
}

#playlist related

sub get_user_playlist_by_id {
    my ( $this, $playlistid, $full ) = @_;

    my $res =
      $this->{_request}->get( $this->{_baseuri} . 'playlists/' . $playlistid );

    my $playlists =
      new WebService::GData::YouTube::Feed( $res, $this->{_request} );

    return $playlists if ($full);

    return $playlists->entry->[0];
}

sub get_user_playlists {
    my ( $this, $channel, $full ) = @_;

    #by default, the one connected is returned
    my $uri = $this->{_baseuri} . 'users/default/playlists';
    $uri = $this->{_baseuri} . 'users/' . $channel . '/playlists' if ($channel);

    my $res = $this->{_request}->get($uri);

    my $playlists =
      new WebService::GData::YouTube::Feed( $res, $this->{_request} );

    return $playlists if ($full);

    return $playlists->entry;
}

#video related

sub video {
    my $this = shift;
    return new WebService::GData::YouTube::Feed::Video($this->{_request});
}

sub search_video {
    my ( $this, $query ) = @_;
    $this->query($query) if ($query);
    my $res = $this->{_request}->get( $this->{_baseuri} . 'videos/' );
    my $playlists =
      new WebService::GData::YouTube::Feed( $res, $this->{_request} );
    return $playlists->entry;
}

sub get_video_by_id {
    my ( $this, $id ) = @_;

    my $uri = $this->{_baseuri} . 'videos/' . $id;

    my $res = $this->{_request}->get($uri);

    my $playlists =
      new WebService::GData::YouTube::Feed( $res, $this->{_request} );

    return $playlists->entry->[0];
}

sub get_user_video_by_id {
    my ( $this, $id, $channel ) = @_;

    my $uri = $this->{_baseuri} . 'users/default/uploads/' . $id;
    $uri = $this->{_baseuri} . 'users/' . $channel . '/uploads/' . $id
      if ($channel);

    my $res = $this->{_request}->get($uri);

    my $playlists =
      new WebService::GData::YouTube::Feed( $res, $this->{_request} );

    return $playlists->entry->[0];
}

sub get_user_videos {
    my ( $this, $channel ) = @_;

    my $uri = $this->{_baseuri} . 'users/default/uploads';
    $uri = $this->{_baseuri} . 'users/' . $channel . '/uploads' if ($channel);

    my $res = $this->{_request}->get($uri);

    my $playlists =
      new WebService::GData::YouTube::Feed( $res, $this->{_request} );

    return $playlists->entry;
}

sub get_user_favorite_videos {
    my ( $this, $channel ) = @_;

    my $uri = $this->{_baseuri} . 'users/default/favorites/';
    $uri = $this->{_baseuri} . 'users/' . $channel . '/favorites/'
      if ($channel);

    my $res = $this->{_request}->get($uri);

    my $playlists =
      new WebService::GData::YouTube::Feed( $res, $this->{_request} );

    return $playlists->entry;
}

sub move_video {
    my ( $this, %params ) = @_;

    my $playlistLink =
      new WebService::GData::YouTube::Feed::PlaylistLink( {}, $this->{_request} );

    #delete old one
    $playlistLink->playlistId( $params{'from'} );
    $playlistLink->deleteVideo( videoId => $params{'videoid'} );

    #put in new one
    $playlistLink->playlistId( $params{'to'} );
    $playlistLink->addVideo( videoId => $params{'videoid'} );
}

#standard feeds
no strict 'refs';
foreach my $stdfeed (
    qw(top_rated top_favorites most_viewed most_popular most_recent most_discussed most_responded recently_featured watch_on_mobile)
  )
{

    *{ __PACKAGE__ . '::get_' . $stdfeed . '_videos' } = sub {
        my ( $this, $region, $category, $time ) = @_;

        my $uri = $this->{_baseuri} . 'standardfeeds/';
        $uri .= $region . '/'   if ($region);
        $uri .= $stdfeed;
        $uri .= '_' . $category if ($category);
        $this->query->time($time) if ($time);
        my $res = $this->{_request}->get($uri);

        my $playlists =
          new WebService::GData::YouTube::Feed( $res, $this->{_request} );
        return $playlists->entry;
      }
}

#to do: comments returns comments feeds! responses,related sends back video feed so are ok!
foreach my $feed (qw(comments responses related)) {

    *{ __PACKAGE__ . '::get_' . $feed . '_for_video_id' } = sub {
        my ( $this, $id ) = @_;

        my $uri = $this->{_baseuri} . 'video/' . $id . '/' . $feed;
        my $res = $this->{_request}->get($uri);

        my $playlists =
          new WebService::GData::YouTube::Feed( $res, $this->{_request} )
          ;
        return $playlists->entry;
      }
}

"The earth is blue like an orange.";

__END__


=pod

=head1 NAME

WebService::GData::YouTube - Access YouTube contents(read/write) with API v2.

=head1 SYNOPSIS

    use WebService::GData::YouTube;

    #create an object that only has read access
    my $yt = new WebService::GData::YouTube();

    #get a feed response from YouTube;
    my $videos  = $yt->get_top_rated_videos;
    #more specific:
    my $videos  = $yt->get_top_rated_videos('JP','Comedy');

    foreach my $video (@$videos) {
        $video->video_id;
        $video->title;
        $video->content;
    }

    #connect to a YouTube account
    my $auth = new WebService::GData::ClientLogin(
        email   =>'...'
        password=>'...',
        key     =>'...'
    );

    #give write access
    my $yt = new WebService::GData::YouTube($auth);

    #returns all the videos from the logged in user 
    #including private ones.
    my $videos  = $yt->get_user_videos();

    #update the videos by adding the common keywords if they are public
    #delete a certain video by checking its id.
    foreach my $video (@$videos) {

        if($video->video_id eq $myid) {

            $video->delete();

        } else {

            if($video->is_listing_allowed){

                $video->keywords('music,live,guitar,'.$video->keywords);
                $video->save();
            }
        }
    }
	 

=head1 DESCRIPTION

!WARNINGS! Developer release. Starting refactoring.
Even if the interface should not change too much, things may change or break but feel free to give me some feedbacks!

I<inherits from L<WebService::GData>>

This package is a point of entry giving access to general YouTube feeds.
Passing an optional authorization object (WebService::GData::ClientLogin) will allow you to access private contents.
It also offers some helper methods to shorten up the code.
Most of the methods will return one of the following object:

=over 

=item L<WebService::GData::YouTube::Feed::Video>

This object handles the manipulation of the video data such as inserting/editing the metadata, uploading a video,etc.

=item L<WebService::GData::YouTube::Feed::Playlist>

This object inherits from L<WebService::GData::YouTube::Feed::Video>.
It contains a list of all the videos within this particular playlist.
The only difference with L<WebService::GData::YouTube::Feed::Video> is that it offers the position tag that specifies
the position of the video within the playlist.

=item L<WebService::GData::YouTube::Feed::PlaylistLink>

This object represents all the playlists metadata of a user.
It is not possible to get the metadata of one playlist. You need to query them all and search for the one you're interested in.

See also:

=over 

=item * L<WebService::GData::YouTube::Doc::BrowserbasedUpload> - overview of the browser based upload mechanism

=back



=back


=head2 CONSTRUCTOR


=head3 new

=over

Create a L<WebService::GData::YouTube> instance.

B<Parameters>

=over 4

=item C<auth:Object> (optional)  - Accept an optional authorization object.

Only L<WebService::GData::ClientLogin> is available for now but OAuth should come anytime soon.
Passing an auth object allows you to access private contents and insert/edit/delete data.

=back

B<Returns> 

=over 4

=item C<WebService::GData::YouTube> instance 

=back

Example:

    use WebService::GData::ClientLogin;
    use WebService::GData::YouTube;

    #create an object that only has read access
    my $yt = new WebService::GData::YouTube();

    #connect to a YouTube account
    my $auth = new WebService::GData::ClientLogin(
        email=>'...'
        password=>'...',
        key        =>'...'
    );

    #give write access with a $auth object that you created
    my $yt = new WebService::GData::YouTube($auth);

=over


=head2 GENERAL METHODS

=head3 query

=over

Set/get a query object that handles the creation of the query string sent to the service.
The query object will build the query string required to access the data.
All queries contain some default parameters like the alt,v,strict parameters.
You can add other parameters in order to do a search.

B<Parameters>

=over 4

=item C<none> - getter context

=item C<query:Object> - setter context accept a query string builder instance. Default to L<WebService::GData::YouTube::Query>

=back

B<Returns>

=over 4

=item C<query:Object> in both setter/getter context the query object. Default to L<WebService::GData::YouTube::Query>

=back

Example:

	use WebService::GData::YouTube;

   	my $yt = new WebService::GData::YouTube();

	$yt->query()->q("ski")->limit(10,0);

	#or set your own query object
	$yt->query($myquery);

	my $videos = $yt->search_video();

=over


=head3 base_uri

=over

Get the base uri used to query the data.

B<Parameters>

=over 4

=item C<none>

=back

B<Returns>

=over 4

=item  C<url:Scalar> the root uri

=back


=head3 base_query

=over

Get the base query string used to get the data.

B<Parameters>

=over 4

=item C<none>

=back

B<Returns>

=over

=item C<url:Scalar> - default to ?alt=json&prettyprint=false&strict=true

=back

=back


=head2 STANDARD FEED METHODS

YouTube offers some feeds regarding videos like the most discussed videos or the most viewed videos.
All the standard feed methods are implemented:

B<methods>

=head3 get_top_rated_videos

=head3 get_top_favorites_videos

=head3 get_most_viewed_videos

=head3 get_most_popular_videos

=head3 get_most_recent_videos

=head3 get_most_discussed_videos

=head3 get_most_responded_videos 

=head3 get_recently_featured_videos

=head3 get_watch_on_mobile_videos 

=over

All the above standard feed methods accept the following optional parameters:

B<Parameters>

=over 4

=item C<region_zone:Scalar> - a country code - ie:JP,US.

=item C<category:Scalar> - a video category - ie:Comedy,Sports.

=item C<time:Scalar> - a time  - ie:today,this_week,this_month,all_time

=back

B<Returns>

=over 4

=item L<WebService::GData::Youtube::Feed::Video> objects

=back

B<Throws>

=over 5

=item L<WebService::GData::Error>

=back

Example:

    use WebService::GData::YouTube;
	
    my $yt   = new WebService::GData::YouTube();
    my $videos = $yt->get_top_rated_videos();
    my $videos = $yt->get_top_rated_videos('JP');#top rated videos in Japan
    my $videos = $yt->get_top_rated_videos('JP','Comedy');#top rated videos in Japanese Comedy 
    my $videos = $yt->get_top_rated_videos('JP','Comedy','today');#top rated videos of the day in Japanese Comedy 


B<See also>:

Explanation of the different standard feeds:

L<http://code.google.com/intl/en/apis/youtube/2.0/reference.html#Standard_feeds>

=back


=head2 VIDEO FEED METHODS

These methods allow you to access videos.
You do not need to be logged in to use these methods.

=head3 get_video_by_id

=over

Get a video by its id.

B<Parameters>

=over 4

=item C<video_id:Scalar> - the unique id of the video- ie:Displayed in the url when watching a video. Looks like:Xzek3skD

=back

B<Returns>

=over 4

=item L<WebService::GData::YouTube::Feed::Video>

=back

B<Throws>

=over 4

=item L<WebService::GData::Error>

=back


Example:

    use WebService::GData::YouTube;
	
    my $yt   = new WebService::GData::YouTube();

    my $video = $yt->get_video_by_id('Xzek3skD');

=back


=head3 search_video

=over

Send a request to search for videos.
You create the query by calling $yt->query and by setting the available parameters.

B<Parameters>

=over

=item C<query:Object> (optional) - a query builder instance

=back

B<Returns>

=over 4

=item L<WebService::GData::YouTube::Feed::Video>

=back

B<Throws>

=over 4

=item L<WebService::GData::Error>

=back

Example:

    use WebService::GData::YouTube;
    
    my $yt   = new WebService::GData::YouTube();

       $yt->query->q("ski")->limit(10,0);

    my $videos = $yt->search_video();

    #or

    my $yt     = new WebService::GData::YouTube();
    my $query  = $yt->query;
       $query -> q("ski")->limit(10,0);
    my $videos = $yt->search_video();

    #or set a new query object
    #it could be a sub class that has predefined value

    my $query  = new WebService::GData::YouTube::Query();

       $query -> q("ski")->limit(10,0);

    my $videos = $yt->search_video($query);#this is a helper the same as doing: $yt->query($query); $yt->search_video();

B<See also>:

A list of all the query parameters and related methods you can use with the default query object:

L<WebService::GData::YouTube::Query>

=back

=head3 get_related_for_video_id

=over 

Get the related videos for a video.
These videos are returned by following YouTube's own algorithm.

B<Parameters>

=over 4

=item C<video_id:Scalar> - the unique identifier of the video.

=back

B<Returns>

=over 4

=item L<WebService::GData::YouTube::Feed::Video> objects 

=back

B<Throws>

=over 4

=item L<WebService::GData::Error> 

=back

Example:

    use WebService::GData::Base;
    
    my $yt   = new WebService::GData::YouTube();
    
    my $videos = $yt->get_related_for_video_id('Xz2eFFexA');

=back


=head2 USER VIDEO FEED METHODS

All these methods allow you to access the videos of the programmaticly logged in user.
Being logged in allow you to access private contents or contents that have been uploaded but is not public yet.
The responses will also have a read/write access so you will be able to edit the videos.

It does not mean that you need to be logged in to use these methods.
By setting the name of the user (channel name),you will only get a read access to the public data.

=head3 get_user_video_by_id

=over

Get a video for the logged in user or for the user name you specified.
It queries the uploads feed which can be more up to date than the feed used with C<get_video_by_id()>.

B<Parameters>

=over

=item C<video_id:Scalar> - the id of the video

=item C<user_name:Scalar> (optional) - the name of the user (channel name)

=back

B<Returns>

=over

=item L<WebService::GData::YouTube::Feed::Video> objects 

=back

B<Throws>

=over

=item L<WebService::GData::Error> 

=back

Example:

    use WebService::GData::Base;

    my $auth = new WebService::GData::ClientLogin(email=>...);
    
    my $yt   = new WebService::GData::YouTube($auth);
    
    my $videos = $yt->get_user_video_by_id('Xz2eFFexA');

    #if not logged in.
    my $videos = $yt->get_user_video_by_id('Xz2eFFexA','live');#you must specify the user if not logged in!

=back

=head3 get_user_videos

=over

Get the videos for the logged in user or for the user name you specified.

B<Parameters>

=over 4

=item C<user_name:Scalar> (optional) - the user name/channel name

=back

B<Returns>

=over 4 

=item L<WebService::GData::YouTube::Feed::Video> objects 

=back

B<Throws>

=over 4

=item L<WebService::GData::Error> 

=back

Example:

    use WebService::GData::Base;

    my $auth = new WebService::GData::ClientLogin(email=>...);
    
    my $yt   = new WebService::GData::YouTube($auth);
    
    my $videos = $yt->get_user_videos();

    #if not logged in, pass the user name as the first parameter
    my $videos = $yt->get_user_videos('live');

=back


=head3 get_user_favorite_videos

Get the videos that user specificly set a favorites (meaning that you may not have write access to the content even if you are logged in!).

B<Parameters>

=over 4

=item C<user_name:Scalar> (optional) - the user name/channel name

=back

B<Returns>

=over

=item L<WebService::GData::YouTube::Feed::Video> objects 

=back

B<Throws>

=over 4

=item L<WebService::GData::Error> 

=back

Example:

    use WebService::GData::Base;

    my $auth = new WebService::GData::ClientLogin(email=>...);
    
    my $yt   = new WebService::GData::YouTube($auth);
    
    my $videos = $yt->get_user_favorite_videos();

    #if not logged in, pass the user name as the first parameter
    my $videos = $yt->get_user_favorite_videos('live');

=back


=head2 USER PLAYLIST METHODS

=over

These methods allow you to access the videos in a playlist or a list of playlists created by a user.
If you are logged in, you will be able to modify the data.
If you are not logged in,you will only have a read access and you must set the user name.

=back

=head3 get_user_playlist_by_id

=over

Retrieve the videos in a playlist by passing the playlist id.

B<Parameters>

=over 4

=item C<playlist_id:Scalar> - the id of the playlist, looks like 'CFESE01KESEQE'

=back

B<Returns>

=over 4

=item L<WebService::GData::YouTube::Feed::Playlist>

A L<WebService::GData::YouTube::Feed::Playlist> contains the same information as a L<WebService::GData::YouTube::Feed::Video> instance

but adds the position information of the video within the playlist.

=back

B<Throws>

=over 4

=item L<WebService::GData::Error> 

=back

Example:

    use WebService::GData::Base;

    my $auth = new WebService::GData::ClientLogin(email=>...);
    
    my $yt   = new WebService::GData::YouTube($auth);
    
    my $videos_in_playlist = $yt->get_user_playlist_by_id('CFESE01KESEQE');
	
=back


=head3 get_user_playlists

=over

Get the playlists metadata for the logged in user or the user set as a parameter.

B<Parameters>

=over 4

=item C<user_name:Scalar> (optional) - the user name/channel name

=back

B<Returns>

=over 4

=item L<WebService::GData::YouTube::Feed::PlaylistLink> objects

If you are logged in, you can access private playlists.

L<WebService::GData::YouTube::Feed::PlaylistLink> is a list of playlists. 
If you want to modify one playlist metadata, you must get them all, loop until you find the one you want and then edit.

=back

B<Throws>

=over 4 

=item L<WebService::GData::Error> 

=back

Example:

    use WebService::GData::Base;

    my $auth = new WebService::GData::ClientLogin(email=>...);
    
    my $yt   = new WebService::GData::YouTube($auth);
    
    my $playlists = $yt->get_user_playlists;
	
    #or if you did not pass a $auth object:
    my $playlists = $yt->get_user_playlists('live');	

=back


=head2  HANDLING ERRORS

Google data APIs relies on querying remote urls on particular services.
Some of these services limits the number of request with quotas and may return an error code in such a case.
All queries that fail will throw (die) a L<WebService::GData::Error> object. 
You should enclose all code that requires connecting to a service within eval blocks in order to handle it.


Example:

    use WebService::GData::Base;

    my ($auth,$videos);

    eval {
        $auth = new WebService::GData::ClientLogin(email=>...);
    };
    
    my $yt   = new WebService::GData::YouTube($auth);
    
    #the server is dead or the url is not available anymore or you've reach your quota of the day.
    #boom the application dies and your program fails...

    my $videos = $yt->get_user_videos;

    #with error handling...

    #enclose your code in a eval block...
    eval {
        $videos = $yt->get_user_videos;;
    }; 

    #if something went wrong, you will get a WebService::GData::Error object back:

    if(my $error = $@){

        #do whatever you think is necessary to recover (or not)
        #print/log: $error->content,$error->code
    }	


=head2  TODO

Many things are left to be implemented!

The YouTube API is very thorough and it will take some time but by priority:

=over 4

=item * OAuth authorization system

=item * Channel search

=item * Playlist search

=item * know the status of a video

=item * Partial Upload

=item * Partial feed read/write

=back

Certainly missing many other stuffs...

=head1  DEPENDENCIES

=over

=item L<JSON>

=item L<LWP>

=back

=head1 BUGS AND LIMITATIONS

If you do me the favor to _use_ this module and find a bug, please email me
i will try to do my best to fix it (patches welcome)!

=head1 AUTHOR

shiriru E<lt>shirirulestheworld[arobas]gmail.comE<gt>

=head1 LICENSE AND COPYRIGHT

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself. 

=cut
