package MooseX::InstanceTracking;
use Moose::Role;
use Set::Object::Weak;

our $VERSION = '0.01';

has _instances => (
    isa     => 'Set::Object::Weak',
    default => sub { Set::Object::Weak->new },
    lazy    => 1,
    handles => {
        instances         => 'members',
        _track_instance   => 'insert',
        _untrack_instance => 'remove',
    },
);

sub get_all_instances {
    my $self = shift;
    map { $_->meta->instances } $self->name, $self->subclasses;
}

around 'construct_instance', 'clone_instance' => sub {
    my $orig = shift;
    my $self = shift;

    my $instance = $orig->($self, @_);
    $self->_track_instance($instance);

    return $instance;
};

after rebless_instance => sub {
    my $self     = shift;
    my $instance = shift;

    $self->_track_instance($instance);
};

before rebless_instance_away => sub {
    my $self     = shift;
    my $instance = shift;

    $self->_untrack_instance($instance);
};

before make_immutable => sub {
    confess "Instance tracking does not yet work with make_immutable.";
};

1;

__END__

=head1 NAME

MooseX::InstanceTracking - Trait for tracking all instances of a class

=head1 SYNOPSIS

    package Employee;
    use Moose -traits => 'MooseX::InstanceTracking';
    
    my $jerry = Employee->new;
    my $george = Employee->new;
    
    Employee->meta->instances; # $jerry, $george (or $george, $jerry)
    Employee->meta->get_all_instances; # $jerry, $george (or $george, $jerry)
    
    
    package Employee::Chef;
    extends 'Employee';
    
    my $emerill = Employee::Chef->new;
    
    Employee->meta->instances; # $jerry, $george (or $george, $jerry)
    Employee->meta->get_all_instances; # $jerry, $george, $emerill

=head1 DESCRIPTION

This trait extends your metaclass by providing instance tracking. Every object
that is instantiated will be tracked on the metaclass. This can be useful if
you need to interact with all the live objects for some reason.

This trait has a limitation in that it does not work with
L<Moose::Meta::Class/make_immutable>. This is because a metaclass trait has no
easy way to influence the code of inlined constructors, so we cannot begin
tracking instances generated by the inlined constructor. To avoid introducing
subtle bugs, this trait throws an error if you attempt to make_immutable. This
does mean that instance construction will be slower.

=head1 PUBLIC METHODS

=over 4

=item instances

Returns the B<unordered> set of instances of this direct class. Instances of
subclasses are not included in this set.

=item get_all_instances

Returns the B<unordered> set of instances of this direct class and all of its
subclasses.

=head1 PRIVATE METHODS

You should probably not call these methods. If you extend Moose by adding some
way to construct objects outside of L<Moose::Meta::Class/construct_instance>,
you I<are> crazy but you'll need to call these methods.

=over 4

=item _track_instance

Begins tracking the instance(s) passed.

=item _untrack_instance

Explicitly stops tracking the instance(s) passed. You do not need to call this
if your instance is garbage collected, since we use L<Set::Object::Weak>.
However if your instance leaves the class some other way, you may need to
explicitly call this. For example, this can happen in core Moose when an
instance is reblessed.

=back

=head1 AUTHOR

Shawn M Moore, C<sartak@bestpractical.com>

=cut

